function parse_compound,input,symbols,fractions,fractions_weight,molwt=molwt
; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       PARSE_COMPOUND
;
; PURPOSE:
;	Returns the number of elements (and optionally their symbols 
;	and fractions) from a chemical formula. Returns 0 (zero)
;	if the input expression is not understood.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = parse_compound(input [,symbols,fractions])
; INPUTS:
;	input: a string with the formula to be processed.
; OPTIONAL OUTPUTS:
;	symbols: a string array with the chamical symbols of the elements
;		in the compoud formula.
;	fractions: a integer array with the number of atoms for
;		each respective element.
;	fractions_weight = a double array with the ratio in weight of 
;		each element.
;	
; KEYWORDS:
;	molwt = Set this keyword to a named variable which will contain
;		the molecular weight. 
; OUTPUT:
;	out: the number of elements that constitute the compound.
;
; PROCEDURE:
;	Straightforward
;
; EXAMPLES:
;	Example 1:
;	   print,parse_compound('H2O')
;	   (returns 2)
;	Example 2:
;	print,parse_compound('H2O',sym,frac,fracw,molwt=molwt)
;	print,sym,frac,fracw,molwt
;	H  O 
;	       2.0000000       1.0000000
;	      0.11189834      0.88810166
;	       18.015280
;
;
; MODIFICATION HISTORY:
;       96-10-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;		Corrects a bug when single character symbols were used.
;       98-03-15 srio@esrf.fr include the "forward_function atomic_constants"
;		statement to allow parse_compound to be used as standalone
;		function (without the weight kw) in the SHADOW VUI application.
;	01-02-28 returns the fraction in weight independently of the fraction
;-
on_error,2
forward_function atomic_constants
idebug = 0

; calculate the number of elements (or uppercase letters) and their positions
nchar = n_elements(byte(input))
nelem=0
for  i=0,nchar-1 do  begin
  char_dec = (byte(strmid(input,i,1)))(0)
  if char_dec ge (byte('A'))(0) and char_dec le (byte('Z'))(0) then begin
     nelem = nelem+1 
     if nelem eq 1 then pos = i else pos=[pos,i]
  endif
endfor
if nelem LE 0 then goto,error
if idebug then print,'Number of element in '+input+' compound: ',nelem
if n_params() eq 1 then goto,fin  ; everything we wanted
;
; extract symbols and fractions
;
parts = strarr(nelem)
if nelem eq 1 then parts = input else begin
  for i=0,nelem-2 do parts(i) = strmid(input,pos(i),pos(i+1)-pos(i))
  if nelem gt 1 then parts(nelem-1) = strmid(input,pos(nelem-1),strlen(input)-1)
endelse

for i=0,nelem-1 do begin
  sb = byte(parts(i))
  iletters = intarr(n_elements(sb))
  iletters(where( sb lt (byte('.'))(0) or sb gt (byte('9'))(0) )) = 1
  letters = where(iletters eq 1)
  numbers = where(iletters ne 1)
  if letters(0) eq -1 then goto,error
  symbol = string(sb(letters))
  if strlen(symbol) eq 1 then symbol = symbol+' '
  if numbers(0) eq -1 then frac = 1.0 else frac = double(string(sb(numbers)))
  if idebug then print,'>> symbol: ',symbol
  if idebug then print,'>> frac: ',frac
  if i eq 0 then symbols=symbol else symbols = [symbols,symbol]
  if i eq 0 then fractions=frac else fractions = [fractions,frac]
endfor

fractions = double(fractions)
fractions_weight = fractions
for i=0,nelem-1 do begin
  atwt = atomic_constants(symbols(i),hh,return='AtomicMass')
  fractions_weight(i) = atwt*fractions(i)
endfor
molwt = total(fractions_weight)
fractions_weight = fractions_weight/molwt

IF idebug THEN BEGIN
  for i=0,nelem-1 do print,symbols(i),' ',fractions(i)
  print,'Molecular weight is: ',molwt
ENDIF

goto,fin
error:
message,/info,'Expression not understood: '+input
return,0

fin:
return,nelem

end


