Function f0_calc_mix,input,descriptor,ratio,NAME=NAME,group=group

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       F0_CALC_MIX
;
; PURPOSE:
;	calculate the elastic Photon-Atom F0 coefficient
;	as a function of ratio = sin(theta)/lambda[Angs]
;	for compounds and mixtures.
;	It users data from DABAX data-base.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = f0_calc_mix(input,descriptor[,ratio])
; INPUTS:
;	input: a dabax input file for f0 data as accepted by dabax_access().
;	descriptor: a string  with either the descriptor or the name of the 
;	  mixture. 
;	  The descriptor sintax is the following:
;	  string with the chemical formula of the compounds with their weight 
;	  fraction between parenthesis.
;               Descriptor examples:
;               SiO2(0.807)B2O3(0.129)Na2O(0.038)Al2O3(0.022)K2O(0.004)
;               H2O(0.99)Cl2(0.01)
;               H(0.11189834)O(0.88810166)
;	  The name is one of the names defines in the DABAX file Compounds.dat
;	  When using the name input option, the keyword NAME must be set.
;		
; OPTIONAL INPUTS:
;	ratio: if undefined, it uses the standard ratio grid,
;		which is the grid in the data file for the tabulated
;		cases, and  dindgen(100)/99.*3.0 for the parametrized
;		data. When ratio is defined, this value is used as a grid
;		for the results (it interpolates in the case of 
;		tabulated data)
;	
; KEYWORDS:
;	NAME: Set this keyword to a valid mixture name in the file 
;		Compounds.dat. If not set, it will interpret
;		"descriptor" as a mixture formula.
;       GROUP = The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;
; OUTPUT:
;	out: an array with the F0 values for the selected mixture.
;
; PROCEDURE:
;	Calls f0_calc for each element of the compound and make the
;	weighted average.
;
; EXAMPLES:
;	delvar,ratio ; cleans energy variable
;	f0 = f0_calc_mix('f0_CromerMann.dat','water',ratio,/NAME)
;	plot,ratio,f0
;	; the same resul can be get with:
;	f0 = f0_calc_mix('f0_CromerMann.dat','H2O',ratio)
;	;or
;	f0 = f0_calc_mix('f0_CromerMann.dat','H(0.11189834)O(0.88810166)',ratio)
;
; MODIFICATION HISTORY:
;       96-12-03 Written by M. Sanchez del Rio (srio@esrf.fr)
;	97-02-20 MSR ports to Windows95.
;       97-10-16 srio@esrf.fr adds GROUP keyword.
;
;-
on_error,2

if keyword_set(name) then begin
  if (sdep() EQ 'WINDOWS' and sdep(/vs) EQ '4') then $
    sfile='compnds.dat' else sfile='Compounds.dat'
  h = dabax_access(sfile)
  index = dabax_select(h,descriptor,/casematch,group=group)
  tmp = spec_data(h,index,/index)
  z = fix(tmp(0,*))
  f = tmp(1,*)
  s = atomic_symbols(z)
endif else begin
  n = parse_mixture(descriptor,s,f,z)
endelse


if n_elements(s) LT 1 then begin
  message,/info,'Error in mixture descriptor/name (Returning 0): '+mixture
  return,0
endif

for i=0,n_elements(s)-1 do begin
  out = f0_calc(input,s(i),ratio,group=group)
  if i eq 0 then begin
     out_tot = out*f(i) 
  endif else begin
     out_tot = out_tot + out*f(i)
  endelse
endfor
return,out_tot

end
