FUNCTION DABAX_DEFAULTS,program,group=group
;+
; NAME:
;	DABAX_DEFAULTS
;
; PURPOSE:
;	Define the data structure for each program of the Xop application.
;	The purpose of this function is to contain all the default input 
;	parameter definition in order to allow different Xop users to
;	customize their particular input defaults. 
;
; CATEGORY:
;	Xop application.
;
; CALLING SEQUENCE:
;
;	inp = DABAX_DEFAULTS('xprogram')
;
; INPUTS:
;	xprogram:	Name of the program interface in Xop.
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;       group = The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;
; OUTPUTS:
;	The input structure with the data necessary to run the program.
;
; OPTIONAL OUTPUTS:
;
; COMMON BLOCKS:
;
; SIDE EFFECTS:
;
; RESTRICTIONS: *** WARNING  ***
;	If the user changes this defaults, be careful with the type
;	of variable. Saying for instance a:4 initializes a as short
;	integer and WILL NEVER be a float. Place a:4.0 to set it as
;	float.
;
; PROCEDURE:
;	Straightforward.
;
; EXAMPLE:
;	inp = DABAX_DEFAULTS('xfh')
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio (srio@esrf.fr) 11 July, 1996	
;	96/11/25 MSR Updates for xf1f2
;	97/01/15 MSR adapts to Windows.
;	97/03/17 MSR adds the option to read file ~/.dabax_defaults in
;		the $DABAX_HELP directory and in $HOME directory (the
;		last one only under Unix).
;	97/10/15 uses file .xop_defaults instead of .dabax_defaults.
;		Reads it from $XOP_HOME instead from $DABAX_HELP.  Uses sdep.
;	97/10/16 keyword group added.
;	98/09/04 better error management
;	98/12/14 Adapted for XOP2.0. Use of anonymous structures. New
;		input files.
;	02/06/06 srio@esrf.fr checks for output in dabax_ls(). Remove
;		idl 4.x dependencies. 
;	02/08/29 srio@esrf.fr updates xfh
;-


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'DABAX_DEFAULTS: error caught (returned 0): '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF

program = strupcase(program)

case program of
	'BRAGG': begin
		;
		list = dabax_ls(NF=nf)
		IF nf EQ 0 THEN Message,'No dabax files found'

		xtalfile = 'CrystalStructures.dat'
		if0 = where(strpos(list,'f0') NE -1)
		IF N_Elements(if0) EQ 1 AND if0[0] EQ -1 THEN Message,'F0 dabax file not found' 
		filef0 = list(where(strpos(list,'f0') NE -1))
		; remove Kissel data (not parametrized)
		filef0=filef0(where(strpos(filef0,'Kissel') EQ -1) )
		f1f2prefix = 'f1f2'

		; get list of crystals
		h = dabax_access(xtalfile,group=group)
		n = spec_access(h)
		xtals = strarr(n)
		for i=0,n-1 do xtals(i)=spec_name(h,i+1,/index)
		;
		; list = dabax_ls()
		;
		; get f1f2 file list
		filef1f2 = list(where(strpos(list,f1f2prefix) NE -1))
		;
		; titles and flags
		bragg_titles = [ 'DABAX f0 file:','DABAX f1f2 file:',$
		'Crystal:','h miller index','k miller index','l miller index',$
		'E min [eV]','E max [eV]','E step [eV]',$
		'Include absorption:','Temperat. fact. [see help]:', $
		'Output file:']
		flags=replicate('1',n_elements(bragg_titles))
		;
		; define structure
		;
		bragg = { titles:bragg_titles, $
		flags:flags, $
		parameters:{ filef0:['0',filef0],$
		  filef1f2:['0',filef1f2], $
		  ilattice:['0',xtals], $
		  hmiller:1, $
		  kmiller:1, $
		  lmiller:1,$
		  EMIN:1.0D3, $
		  EMAX:1.0D4, $
		  ESTEP:1.0D2, $
		  I_ABSORP:['1','No','Yes'], $
		  TEMPER:'1.0', $
		  OUTFIL:'bragg.out'} $ ;end parameters
		}
		end
	'XF0': begin
		;get the list of compounds
		list = dabax_ls(NF=nf)
		IF nf EQ 0 THEN Message,'No dabax files found'

		if0 = where(strpos(list,'f0') NE -1)
		IF N_Elements(if0) EQ 1 AND if0[0] EQ -1 THEN Message,'F0 dabax file not found' 

		datasets = list(where(strpos(list,'f0') NE -1))
		file = dabax_pathf('Compounds.dat',group=group)
		h = dabax_access(file,group=group)
		n = spec_access(h)
		mat_list = strarr(n)
		for i=0,n-1 do mat_list(i)=spec_name(h,i+1,/index)
		;set titles and flags
		titles = ['datasets:','material','table','formula', $
		 'sin_theta/lambda grid:','From: ','To: ','Number of points']
		flags = ['1','1','w(1) Eq 2','w(1) Ne 2','1','w(4) NE 0',$
			'w(4) NE 0','w(4) NE 0']
		;
		; define structure
		;
		xf0 = { titles:titles, $
			flags:flags, $
			parameters:{ datasets:['0','all',datasets], $
			  mat_flag:['0','Element(formula)','Mixture(formula)',$
                                'Mixture(table)'], $
                          mat_list:['0',mat_list], $
			  descriptor:'Si', $
			  GRID:['0','Standard','User defined'], $
                	  GRIDSTART:0.0, $
			  GRIDEND:4.0, $
			  GRIDN:100L }}
		end

	'XCROSSSEC': begin
		list = dabax_ls(NF=ng)
		IF ng EQ 0 THEN Message,'No dabax files found'
		  iden = 'CrossSec_'
		  ipos = 9
		  filecomp = 'Compounds.dat'
		datasets = list(where(strpos(list,iden) NE -1))
		; take only names with a single "_"
		datasets = datasets(where( strpos(datasets,'_',ipos) EQ -1 ))
		file = dabax_pathf(filecomp,group=group)
		h = dabax_access(file,group=group)
		n = spec_access(h)
		mat_list = strarr(n)
		for i=0,n-1 do mat_list(i)=spec_name(h,i+1,/index)
		titles = ['datasets:','material','table','formula','density',$
			'Calculate (i.e. "all","total","compton")', $
			'Energy [eV] grid:', 'Starting Energy [eV]: ','To: ', $
			'Number of points','Units']
		flags = ['1','1','w(1) EQ 2','w(1) LE 1 ',$
			'w(1) EQ 1', '1','1',$
			 'w(6) NE 0','w(6) Eq 1','w(6) Eq 1','1']
		;
		; define structure
		;
		xcrosssec= { titles:titles, $
		flags:flags, $
		parameters:{ datasets:['1','all',datasets], $
                  mat_flag:['0','Element(formula)','Mixture(formula)',$
			'Mixture(table)'], $
                  mat_list:['0',mat_list], $
                  descriptor:'Si', $
                  density:1.0, $
                  calculate:'all',$
                  GRID:['0','Standard','User defined','Single Value'], $
                  GRIDSTART:100.0, GRIDEND:10000.0, GRIDN:200L,$
		  unit:['0','barn/atom [Cross Section]',$
			    'cm^2 [Cross Section]','cm^2/g [Mass abs coef]',$
			    'cm^-1 [Linear abs coef]'] } }
		end
	'XF1F2': begin
		list = dabax_ls(NF=nf)
		IF nf EQ 0 THEN Message,'No dabax files found'
		datasets = list(where(strpos(list,'f1f2') NE -1))
		file = dabax_pathf('Compounds.dat',group=group)
		h = dabax_access(file,group=group)
		n = spec_access(h)
		mat_list = strarr(n)
		for i=0,n-1 do mat_list(i)=spec_name(h,i+1,/index)
		titles = ['datasets:','material','table','formula','density',$
			'Calculate', $
			'Energy [eV] grid:', 'Starting Energy [eV]: ','To: ', $
			'Number of points',$
			'Grazing angle','Roughness rms [A]',$
			'Starting Graz angle [mrad]','To [mrad]',$
			'Number of angular points']
		flags = ['1','1','w(1) EQ 2','w(1) LE 1','w(1) EQ 1', '1','1',$
			'w(6) NE 0','w(6) Eq 1','w(6) Eq 1',$ ;energy
			'w(5) eq 0 or w(5) gt 7','w(5) eq 0 or w(5) gt 7',$
			'w(5) eq 0 or w(5) gt 7',$
			'(w(5) eq 0 or w(5) gt 7) and w(10) eq 1',$
			'(w(5) eq 0 or w(5) gt 7) and w(10) eq 1']
		;
		; define structure
		;
		xf1f2= { titles:titles, $
		flags:flags, $
		parameters:{ datasets:['1','all',datasets], $
                  mat_flag:['0','Element(formula)','Mixture(formula)',$
			'Mixture(table)'], $
                  mat_list:['0',mat_list], $
                  descriptor:'Si', $
                  density:1.0, $
                  calculate:['1','all','f1','f2','delta','beta',$
			'mu [cm^-1]','mu [cm^2/g]','Cross Section[barn]', $
			'reflectivity-s','reflectivity-p',$
			'reflectivity-unpol'], $
                  GRID:['0','Standard','User defined','Single Value'], $
                  GRIDSTART:5000.0, GRIDEND:25000.0, GRIDN:100L, $
		  THETAGRID:['0','Single value','User Defined'], $
		  ROUGH:0.0, THETA1:2.0,THETA2:5.0,THETAN:50 } }
		end
	'XFH': begin
		titles = [ 'DABAX f0 file:','DABAX f1f2 file:',$
		'Crystal:','h miller index','k miller index','l miller index',$
		'Temperature factor [see help]:','From Energy [eV]', $
		'To energy [eV]','Number of points']
		flags = replicate('1',n_elements(titles))
		tmp = dabax_defaults('bragg')
		str = tmp.parameters
		energy =  8000.0D0
		energy_end =  18000.0D0
		;
		; define structure
		;
		xfh= { titles:titles, $
		flags:flags, $
		parameters:{filef0:str.filef0, $
		  filef1f2:str.filef1f2, $
		  ilattice:str.ilattice, $
		  hmiller:str.hmiller, $
		  kmiller:str.kmiller, $
		  lmiller:str.lmiller, $
		  temper:str.temper, $
		  energy:energy, $
		  energy_end:energy_end, $
		  npoints: 20L} } 
		end

	'CRL': begin
		list = dabax_ls(NF=nf)
		IF nf EQ 0 THEN Message,'No dabax files found.'
		;
		; database parameters
		;
		f1f2 = list(where(strpos(list,'f1f2') NE -1))
		CS = list(where(strpos(list,'CrossSec') NE -1))
		CS = CS(where( strpos(CS,'_',9) EQ -1 ))
		n_database = 2
		;
		; material parameters
		;
		filecomp = 'Compounds.dat'
		file = dabax_pathf(filecomp,group=group)
		h = dabax_access(file,group=group)
		n = spec_access(h)
		mat_list = strarr(n)
		for i=0,n-1 do mat_list(i)=spec_name(h,i+1,/index)
		titles_material = ['material','table','formula','density']
		flags_material = ['1','w(0) EQ 2','w(0) LE 1','w(0) EQ 1']
		n_material = 4
		;
		; pars
		;
		n_pars = 6
		;
		; define structure
		;
		crl= { n_database:n_database, $
		  n_material:n_material, $
		  n_pars:n_pars, $
;
		  titles_material:titles_material, $
		  flags_material:flags_material, $
		  parameters:{ $
		    f1f2:['0',f1f2], $		; database
		    CS:['0',CS], $		; database
		    mat_flag:['0','Element(formula)','Mixture(formula)',$
		    'Mixture(table)'], $	; material
		    mat_list:['0',mat_list], $	; material
		    descriptor:'Be', $		; material
		    density:1.0, $		; material
;
		    calc_type:1, $		; pars (0=main, 1=ray-tracing)
		    R:0.4, $			; pars
		    N:51L, $			; pars
		    E:24000.0, $		; pars
		    p:24.0, $			; pars
		    q:9.15, $			; pars
		    dholes:100.0, $		; pars
		    SourceSize:10.0, $		; pars
		    SourceDiv:7.5, $		; pars
		    parab_aperture:0.5, $	; pars
		    UND_flag:['0','No','Yes...'], $	; pars
		    UND_emittance:0.039, $	; pars
		    UND_beta:2.5, $		; pars
		    UND_length:1.6, $		; pars
		    cyl_flag:['0','Circular','Parabolic'], $	; pars
		    calc_method:['1','by weighting','by inverse CDF'],$	; pars
		    rt_show:['0','image profile','image phase-space',$
			'source phase-space','profile+phase-space'],$	; pars
		    nrays:5000L,$		; pars
		    seed:10822541L }} 		;pars
		end


	'XMCKALE': begin
		titles = ['Edge:','Absorber atom Z:','Scatter atom Z:',$
		'Distance [A]:']
		flags = replicate('1',n_elements(titles))
		;
		; define structure
		;
		xmckale= { titles:titles, $
		flags:flags, $
		parameters:{ edge:['0','K','L'], $
		  at1:14, $
		  at2:14, $
		  r:3.0} }
		end
	else:   begin
		message,/info,'ERROR: Program not found'
		out = 0
		end
	endcase

;
;----------------------- WARNING !! -----------------------------
; If this section is modified, please check also xop_defaults.pro
;----------------------------------------------------------------
;
; rename the structure to out
;
out = 0
command = 'out = '+program
;print,command 
tmp = Execute(command)

;
; Overwrite data with values from defaults files
; (same section in dabax_defaults, xop_defaults, xplot_defaults 
;  and xnlfit_defaults)
;
sep = sdep(/ds)
filedir = [Xop_GetEnv('XOP_HOME')+sep+'defaults'] 
filedir = [filedir]
IF Xop_GetEnv('XOP_DEFAULTS_DIR') NE '' THEN $
   filedir = [filedir,Xop_GetEnv('XOP_DEFAULTS_DIR')]

nfiles = n_elements(filedir)
if nfiles EQ 1 then filedir = [filedir]
tmp=0
for i=0,nfiles-1 do begin
  file = filedir[i]+sep+StrLowCase(program)+'.xop'
  tmp = checkfile(file)
  if tmp EQ 1 then begin
    str_par = Xop_Input_Load(inputFile=file)
    ;IF N_Tags(out.parameters) NE 0 THEN BEGIN
    IF (Where(StrUpCase(Tag_Names(out)) EQ 'PARAMETERS'))(0) NE -1 THEN BEGIN
      tmp = out.parameters
      Copy_Structure,str_par, tmp
      out.parameters = tmp
    ENDIF ELSE BEGIN
      tmp = out
      Copy_Structure,str_par, tmp
      out = tmp
    ENDELSE
    message,/info,'Defaults from file: '+file+'
  endif
endfor
;
; return
;
return,out
end
