Function cross_calc,input,descriptor,energy,col_titles,partial=F,$
 density=idensity,casematch=casematch,unit=funit,group=group, $
 verbose=verbose

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       CROSS_CALC
;
; PURPOSE:
;	calculate the atomic cross sections and attenuation coefficients.
;	It uses data from DABAX data-base.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = cross_calc(input,descriptor[,energy,col_titles])
; INPUTS:
;	input: a dabax input file as accepted by dabax_access().
;	       The program also accepts the empty string '' and 
;		takes the first cross section file in the list.
;	descriptor: a string  with a description of the material.
;		The string is searched in the scan titles #S in order
;		to find the appropiated data.
; OPTIONAL INPUTS:
;	energy: if undefined, it uses the standard energy grid,
;		When energy is defined, this value is used as a grid
;		for the results (it interpolates in the case of 
;		tabulated data). Always in eV.
; OPTIONAL OUTPUTS:
;	col_titles: A array of strings with the labels of the data returned 
;		in "out"
;	
; KEYWORDS:
;	PARTIAL= Text string that has to be matched with the column titles
;		in the DABAX data file. It is used to extract partial cross
;		sections. For example PARTIAL='Total' (default value) will
;		output the columns having this word in their labels, thus
;		the total cross section. Use for example 'Compton', or
;		'Rayleigh' if you want to extract only one of the other partial
;		cross sections or absorption coefficients. This keyword is
;		case-insensitive unless you set the CASEMATCH keyword.
;	CASEMATCH= to be used together with the PARTIAL keyword.
;		When this keyword is set to 1, the matching for the text
;		in PARTIAL is case sensitive.
;	DENSITY= the density value to be used for the calculations. If
;		not defined, take the value given by Atomic_Constants().
;               (used only if UNIT=3)
;	UNIT= An integer indicating the unit of the output array
;		0 (default): barn/atom (Cross Section calculation)
;		1 : cm^2 (Cross Section calculation)
;		2 : cm^2/g (Mass Attenuation Coefficient)
;		3 : cm^-1 (Linear Attenuation Coefficient)
;       GROUP= The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;       VERBOSE= If set (default is verbose=1) prints some info.
; OUTPUT:
;	out: an array with the values of the selected return parameter(s).
;
; PROCEDURE:
;	Takes the CrossSection values from the DABAX files. It also takes the
;	Atomic constants from DABAX, and performs the classical 
;	operations to calculate the other parameters.
;
; EXAMPLES:
;	delvar,energy ; cleans energy variable
;	mu = cross_calc('CrossSec_XCOM.dat','Si',energy,col_title,unit=2)
;	help,mu  (You get: DOUBLE    = Array(2, 82), because in the 
;	  DABAX file CrossSec_XCOM.dat thay are two colums with the 
;	  word ('Total'). 
;	plot,energy,mu(0,*),/xlog,/ylog,ytitle=col_title(0)
;
; MODIFICATION HISTORY:
;       96-11-29 Written by M. Sanchez del Rio (srio@esrf.fr)
;       96-12-19 srio@esrf.fr fix a bug (when partial='all')
;       97-01-10 srio@esrf.fr fix a bug (in col_title output when unit=3)
;       97-10-14 srio@esrf.fr fix a bug with density.
;       97-10-16 srio@esrf.fr adds group kw.
;       2003-05-22 srio@esrf.fr adds verbose kw.
;       2003-07-04 srio@esrf.fr adds input=''
;
;-
;on_error,2
IF N_Elements(verbose) EQ 0 THEN verbose=1

IF input EQ '' THEN BEGIN
list = dabax_ls()
iden_mu = 'CrossSec_'
dataMu = list(where(strpos(list,iden_mu) NE -1))
; take only names with a single "_"
IF N_Elements(dataMu) GT 1 THEN BEGIN
  dataMu = dataMu(where( strpos(dataMu,'_',ipos) EQ -1))
ENDIF
input=dataMu[0]
IF verbose THEN Message,/Info,'Using DABAX file: '+input
ENDIF
h = dabax_access(input,group=group)

; on error, return 0
if not(keyword_set(h)) then return,0

;get the scan which title fits with descriptor
igood = dabax_select(h,descriptor,/casematch,group=group)
IF verbose THEN $
  message,/info,'Using scan: '+strcompress(spec_name(h,igood,/index))
z = double(strsubstitute(spec_headers(h,igood,'#S',/index),'#S',''))

data = spec_data(h,igood,/index)
cols = spec_labels(h,igood,/index)

if not(keyword_set(F)) then F='Total'

;finds column with energy
tmp = strpos(cols,'PhotonEnergy')
itot = where(tmp NE -1)

;finds column(s) with magnitude(s) to extract.
for ii=0,n_elements(F)-1 do begin
  if keyword_set(casematch) then tmp = strpos(cols,F(ii)) else $
    tmp = strpos(strupcase(cols),strupcase(F(ii)))
  if  F(0) ne 'all' then itot = [itot,where(tmp NE -1)] else $
                         itot = [itot,where(tmp EQ -1)]
endfor

; clean vector with column result.
itot = itot(where(itot NE -1))
itot = itot(UNIQ(itot, SORT(itot)))
if n_elements(itot) EQ 1 then itot=replicate(itot,2)

; get energy
fconv = 1.0
if (strpos(strupcase(cols(itot(0))),'KEV'))(0) NE -1  then fconv = 1000.0
if (strpos(strupcase(cols(itot(0))),'MEV'))(0) NE -1 then fconv = 1D6
energy1 = reform(data(itot(0),*)*fconv)


; interpolation (if necessary)
if n_elements(energy) EQ 0 then begin
  energy = energy1
  out = data(itot(1:n_elements(itot)-1),*)
endif else begin
  out = fltarr(n_elements(itot)-1,n_elements(energy))
  for kk = 1 ,n_elements(itot)-1 do begin
     out(kk-1,*) = interpol(reform(data(itot(kk),*)),energy1,energy)
  endfor
endelse
if n_elements(itot) EQ 1 then out = reform(out)

; prepare the output
col_titles=(cols(itot))(1:n_elements(itot)-1)
if not(keyword_set(funit)) then funit = 0
case funit of
  0: 
  1: begin		; cm^2
      out = out*1e-24
      for ii=0,n_elements(col_titles)-1 do col_titles(ii) =$
	strsubstitute(col_titles(ii),'[barn/atom]','[cm^2]')
     end
  2: begin		; cm^2/g
      cf = 1e-24*physical_constants('AVOGADRO')/$
        atomic_constants(descriptor,return='AtomicMass')
      IF verbose THEN message,/info,'Conversion factor (b/at->cm^2/g): '+$
	strcompress(cf)
      out = out*cf
      for ii=0,n_elements(col_titles)-1 do col_titles(ii) =$
	strsubstitute(col_titles(ii),'[barn/atom]','[cm^2/g]')
      end
  3: begin		; cm^-1
      if not(keyword_set(idensity)) then density=$
	atomic_constants(descriptor,return='Density') else $
	density = idensity
      IF verbose THEN message,/info,'Using density: '+strcompress(density,/rem)
      cf=1e-24*physical_constants('AVOGADRO')/$
	atomic_constants(descriptor,return='AtomicMass')*density
      IF verbose THEN message,/info,'Conversion factor (b/at->cm^-1): '+$
	strcompress(cf)
      out = out*cf
      for ii=0,n_elements(col_titles)-1 do col_titles(ii) =$
	strsubstitute(col_titles(ii),'[barn/atom]','[cm^-1]')
     end
endcase

return,out
end
