
;+
;==================================   CRL   ====================================
;
;
;   CRL is an widget based graphical interface to calculate the parameters
;   of Compound Refractive Lenses (CRL).
;
;   Two kind of calculations are allowed:
;	i) main parameters (focal distances, gain, acceptance, etc.)
;	   (calculated using Refs. 1 & 2)
;	ii) Pseudo-Ray-Tracing to compute the beam profile at the image
;	    plane and the phase-space plot. 
;	   (calculated using Ref. 2, formulas 9 and 32 for cylindrical 
;	    and parabolic lenses, respectively).
;
;    Reference 1: 
;    A. Snigirev, V. Kohn, I. Snigireva, A. Souvorov and B. Lengeler:
;    "Focusing High Energy X-rays by Compound Refractive Lenses",
;    Applied Optics 37 (1998) pp. 653-662) 
;
;    Reference 2: 
;    P. Elleaume:
;    Nucl. Instrum. and Meth in Phys. Research A 412 (1998) pp. 483-506
;
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    CRL input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;  Quit: to exit from the program
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main CRL window. 
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. After 
;		pressing the "GO CALCULATE" button, CRL starts running.
;		Note that, differently from most of other XOP applications, 
;		this "Set Parameters" window is a non-modal window.
;  Set Defaults: Sets the default parameters.
;
;  Help:   Shows the CRL help (this text).
;
;
; COPYRIGHT:
;	CRL  belongs to XOP package and it is distributed within XOP.
;	PLEASE REFER TO THE XOP COPYRIGHT NOTICE BEFORE USING IT.
;
; CREDITS:
;	Published calculations made with XOP should refer:
;
;	  M. Sanchez del Rio and R. J. Dejus "XOP: Recent Developments"
;	  SPIE proceedings vol. 3448, pp.340-345, 1998.
;
;	In addition, published calculations with the CRL aplication may 
;	also cite the reference mentioned before.
;
; LAST MODIFICATION: msr/msr/99-03-16
;
;
;-
;
; Written by M. Sanchez del Rio, srio@esrf.fr 97/12/17
;	98-12-22 srio@esrf.fr adapts for XOP2.0
;	99-03-16 srio@esrf.fr updates doc.
;
;
;=================================================================
;
FUNCTION CRL_Version
RETURN,'1.0'
END ; CRL_Version
;
;==========================================================================
;
PRO crl_event,event

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'CRL_EVENT: error caught: '+!err_string)
   IF Widget_Info(stateid,/Valid_Id) AND N_Elements(state) NE 0 THEN $
   Widget_control, stateid,  Set_uvalue=state,/No_Copy
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF
;
; register the events
;
Widget_control, event.id, Get_uvalue=eventuval

stateid = Widget_Info(event.handler,/Child)
Widget_control, stateid,  Get_uvalue=state,/No_Copy

CASE eventuval OF
	'QUIT': BEGIN
		IF Ptr_Valid(state.str_Ptr) THEN Ptr_Free,state.str_Ptr
		IF Ptr_Valid(state.str_defaults_Ptr) THEN $
		  Ptr_Free,state.str_defaults_Ptr
		WIDGET_CONTROL,event.top,/DESTROY
		Return
		END
	'FILEINPUT': BEGIN
		action=''
		Widget_Control,event.id, Get_Value=action
		CASE action OF
		'Load from file...': BEGIN
		  if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
		  str_par = Xop_Input_Load(Title=$
		    'Select CRL input file...',$
		    /NoConf,Filter=filter,Group=event.top)
		  IF Type(str_par) EQ 8 THEN BEGIN
		    IF Widget_Info(state.wids.crl_set,/Valid_Id) THEN $
		      Widget_Control,state.wids.crl_set,/Destroy
		    tmp = (*(state.str_Ptr)).parameters
		    Copy_Structure,str_par, tmp,Group=event.top, /OnlyFirstField
		    (*(state.str_Ptr)).parameters = tmp
		  ENDIF
		  END
		'Save to file...': BEGIN
		  IF Widget_Info(state.wids.crl_set,/Valid_Id) THEN $
		    CRL_Update,state.wids.crl_set,Group=event.top
		  str_par = (*(state.str_Ptr)).parameters
		  Xop_Input_Save,str_par,File='crl.xop',$
		    /Write, Group=event.top, Comment='; xop/crl(v'+$
		    crl_version()+') input file on '+SysTime()
		  END
		'Save as default': BEGIN
		  IF Widget_Info(state.wids.crl_set,/Valid_Id) THEN $
		    CRL_Update,state.wids.crl_set,Group=event.top
		  str_par = (*(state.str_Ptr)).parameters
		  Xop_Input_Save,str_par,Group=event.top, $
		    Default='crl.xop',Comment='; xop/crl(v'+$
		    crl_version()+') input file on '+SysTime()
		  END
		ENDCASE
		END
	'SETDEF': BEGIN
		itmp = Dialog_Message(Dialog_Parent=event.top, $
		  /Question,['This option initializes the',$
		  'CRL parameters to their default values.',$
		  'Then you must click Set_parameters to run the program.',$
		  'Please confirm'],Title='CRL: set defaults')
		IF itmp EQ 'No' THEN GoTo,out
		IF Widget_Info(state.wids.crl_set,/Valid_Id) THEN $
		  Widget_Control,state.wids.crl_set,/Destroy
		state.wids.crl_set=0L
		(*state.str_Ptr).parameters = (*state.str_defaults_Ptr)
		END
	'SETPAR': BEGIN
		IF Widget_Info(state.wids.crl_set,/Valid_Id) THEN BEGIN
		  itmp = Dialog_Message(/Info,Dialog_Parent=event.top,$
		    ['There is an open "Set CRL Parameters" window',$
		     'Only a single window is allowed.'])
		  GoTo,out
		ENDIF
		ptr = state.str_Ptr
		p=0L
		crl_set,ptr,Group=event.top,Parent=p,No_Block=state.no_Block
		state.wids.crl_set=p
		END
        'HELP': xhelp,'crl',Group=event.top
ENDCASE
out:
Widget_Control, stateid, Set_Uvalue=state, /No_Copy
END
;
;==============================================================================
;
PRO CRL, GROUP=group, INPUTFILE=inputfile, No_Block=no_Block
;
;
Forward_Function dabax_defaults
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'CRL: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

;
; widget definition
;

base=WIDGET_BASE(/COLUMN,TITLE='CRL '+crl_version(),$
	MBAR=wMenuBar)

wButtons = widget_base(base,/Column) ; also to store state

wFile = widget_button(wMenuBar,VALUE='File',/MENU)
  wtmp0 = widget_button(wFile,VALUE='CRL input parameters', /Menu)
  wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
  wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='QUIT',/SEPARATOR)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='SETPAR')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')

;wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
;  wSpec = widget_button(wResults,VALUE='Spectra',UVALUE='SHOW')
;  wTraj = widget_button(wResults,VALUE='Trajectories',UVALUE='TRAJ')
;  wtmp = widget_button(wResults,VALUE='Parameters',UVALUE='PARAMETERS')

wHelp = widget_button(wMenuBar,VALUE='Help', /Help)
  wtmp = widget_button(wHelp,VALUE='CRL', UVALUE='HELP')
;
wtmp = widget_button(BASE,VALUE='Set Parameters', UVALUE='SETPAR')

if sdep() EQ 'WINDOWS' then font = 'VERDANA*BOLD*ITALIC*24' else $
  font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'

junk = WIDGET_LABEL( BASE, FONT=font, VALUE='   CRL   ')
junk = WIDGET_LABEL( BASE, FONT=font, VALUE='  Compound Refractive Lenses  ')

;
;state
;
str = dabax_defaults('crl')
str_defaults = str.parameters
IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = str.parameters
    Copy_Structure,str_par, tmp, Group=group
    str.parameters = tmp
  ENDIF
ENDIF
wids = {crl_set:0L}
str_Ptr=Ptr_New(str)
str_defaults_Ptr=Ptr_New(str_defaults)
IF N_Elements(no_Block) EQ 0 THEN no_Block=0
state = { str_Ptr:str_Ptr, str_defaults_Ptr:str_defaults_Ptr, $
  wids:wids, no_block:no_block}
;
;actions
;
Widget_control,Widget_Info(base,/Child),Set_uvalue=state,/No_Copy
Widget_control,base,/Realize
Xmanager,'crl',base,Group_leader=group, No_Block=no_Block
END
