function bragg_calc,inp,group=group

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       BRAGG_CALC
;
; PURPOSE:
;	Preprocessor for Structure Factor (FH) calculations.
;	It calculates the basic ingredients of FH.
;	
; CATEGORY:
;       X-Ray optics.
;
; CALLING SEQUENCE:
;	out = bragg_calc(inp)
; INPUTS:
;	inp: a structure with the input data. This structure may
;		be created by either dabax_defaults('bragg') or
;		bragg_inp() and modified by bragg_inp(ask={1,2}.
;	
; KEYWORDS:
;
; OUTPUT:
;	out: a structure with the output data. This structure is the
;		input of other programs (as crystal_fh) and can be 
;		dumped to disk by using bragg_out.
;
;	group = The widget ID of the widget that calls the routine.
;		This is passed to widget_message in its dialog_parent
;		keyword to center the window message.
;
; PROCEDURE:
;	Reads the crystal structure, f0, f' and f" from the
;	DABAX library and returns some crystal parameters, the 
;	geometrical part of FH and the atomic part F' and F".
;	It has been adapted from bragg 2.2 (SHADOW ray-tracing package).
;	It uses the following DABAX files:
;	  CrystalStructures.dat, CrystalCell.dat, f1f2_*.dat, f0_*.dat
;	  (note that not all datafiles f0_* are used, only the ones
;	  having f0 parametrizated, not tabulated).
;
; EXAMPLE:
;	inp = bragg_defaults()
;	out = bragg_calc(inp)
;
; SEE ALSO:
;       bragg_inp, bragg_out, dabax_defaults.
;
; MODIFICATION HISTORY:
;       96-07-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;       96-12-03 srio@esrf.fr: adapt to new bragg structure. It allows
;		the selection of f1f2 file. It users now f1f2_calc.pro
;       97-01-14 srio@esrf.fr: adapts to Windows.
;       97-10-08 srio@esrf.fr: fixes no-absorption case.
;       97-10-16 srio@esrf.fr: Avoids to stop the program when a scan
;		(usually H) is not found. Uses sdep, adds group kw.
;	98-10-13 srio@esrf.fr uses catch. 
;	02-10-09 srio@esrf.fr adds calculation of DebyeWaller factor. 
;-
;on_error,2

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'BRAGG_CALC: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return,0
endif

if sdep(/w) then ask=2 else ask=1
if n_params() NE 1 then begin
  inp = bragg_inp(ask=ask)
endif


ifilef0 = inp.parameters.filef0(0) + 1
filef0 = inp.parameters.filef0(ifilef0)
ifilef1f2 = inp.parameters.filef1f2(0) + 1
filef1f2 = inp.parameters.filef1f2(ifilef1f2)
ilattice = inp.parameters.ilattice(0) + 1
hmiller=inp.parameters.hmiller 
 kmiller=inp.parameters.kmiller 
  lmiller=inp.parameters.lmiller
I_ABSORP = inp.parameters.i_absorp(0)
TEMPER = inp.parameters.temper
EMIN = inp.parameters.emin 
 EMAX = inp.parameters.emax 
  ESTEP = inp.parameters.estep
OUTFIL = inp.parameters.outfil

if (sdep() EQ 'WIN' and sdep(/vs) eq '4') then begin
  filestr='CRY_STR.dat'
  filecell='CRY_CELL.dat'
endif else begin
  filestr='CrystalStructures.dat'
  filecell='CrystalCell.dat'
endelse
hstr = dabax_access(filestr)
hcell = dabax_access(filecell)
;
; load crystal structure and cell parameters from files
;
index = ilattice
str = double(spec_data(hstr,index,/index))
cell = double(spec_data(hcell,index,/index))

if spec_name(hcell,index,/index) NE spec_name(hstr,index,/index) then $
	message,'Crystal structures are different in files '+$
	filestr+' ('+spec_name(hcell,index,/index)+') and '+$
	filecell+' ('+spec_name(hstr,index,/index)+').'
;
; Calculate the volume of the unit cell and interlattice distance d
; following, for example, "X-ray diffraction", B. E. Warren, Dover, 1990
;
	M_REF = double([0,hmiller,kmiller,lmiller])
	D_LATT_A = cell(0)*1.0D-8
	D_LATT_B = cell(1)*1.0D-8
	D_LATT_C = cell(2)*1.0D-8
	D_LATT_ALPHA  = cell(3)*!dpi/180.
	D_LATT_BETA  =  cell(4)*!dpi/180.
	D_LATT_GAMMA  = cell(5)*!dpi/180.



	d_latt_volume = D_LATT_A*D_LATT_B*D_LATT_C * SQRT(1+ $
	    2.0D0*cos(D_LATT_ALPHA)*cos(D_LATT_BETA)*cos(D_LATT_GAMMA) $
	    - cos(D_LATT_ALPHA)^2 - cos(D_LATT_BETA)^2 $
	    - cos(D_LATT_GAMMA)^2               )

	oneoverd2 = 1.0D0 + 2.0D0*cos(D_LATT_ALPHA)*cos(D_LATT_BETA)* $
		  cos(D_LATT_GAMMA) - cos(D_LATT_ALPHA)^2 - cos(D_LATT_BETA)^2 $
	    - cos(D_LATT_GAMMA)^2
	oneoverd2 = 1.0D0/oneoverd2       
	oneoverd2 = oneoverd2  * ( $
	 (M_REF(1)*sin(D_LATT_ALPHA)/D_LATT_A)^2 + $
	 (M_REF(2)*sin(D_LATT_BETA)/D_LATT_B)^2 + $
	 (M_REF(3)*sin(D_LATT_GAMMA)/D_LATT_C)^2 + $
	 2.0D0*M_REF(1)*M_REF(2)/D_LATT_A/D_LATT_B* $
	     (cos(D_LATT_ALPHA)*cos(D_LATT_BETA)-cos(D_LATT_GAMMA)) + $
	 2.0D0*M_REF(2)*M_REF(3)/D_LATT_B/D_LATT_C* $
	     (cos(D_LATT_BETA)*cos(D_LATT_GAMMA)-cos(D_LATT_ALPHA)) + $
	 2.0D0*M_REF(3)*M_REF(1)/D_LATT_A/D_LATT_C* $
	     (cos(D_LATT_GAMMA)*cos(D_LATT_ALPHA)-cos(D_LATT_BETA)) $
		  			  )      
	SP_HKL = 1.0D0/SQRT(oneoverd2)
	R_NATOM = 1.0D0/d_latt_volume

	message,/info,'Interplanar distances (Angs) :' +$
		strcompress(SP_HKL*1.0D8,/rem)
	message,/info,'d_latt_volume (cm^3) : ' +$
		strcompress(d_latt_volume,/rem)
	message,/info,'1/V=r_natom (cm^-3) :' +$
		strcompress(R_NATOM,/rem)

	E2_MC2  =  2.817939D-13    ; e^2/mc^2 classical e- radius in cm
	RN = E2_MC2*R_NATOM

;
; fo the non-dispersive part of the atomic scattering factor 
;
;
;        print, '************************************************'
;        print, 'Atomic scattering factor is defined by '
;        print, '            fo + f'' + if"'
;        print, ' where fo = fo(SIN(theta)/Lambda) '
;        print, ' is the non-dispersive part'
;        print, ' and f'', f" (Lambda) are the dispersive part.'
;        print, '***********************************************'
;

	zcol = str(0,*)
	fcol = str(1,*)
	batom =  zcol(uniq(zcol,sort(zcol)) )
	nbatom = n_elements( batom )
        ;f0coeff= dblarr(nbatom,9)

;        print,' fo is obtained by the parametrization values'
;        print,' read from file $SHADOW_DATA/TABLE0, which '
;        print,' stores data from elements and some ionized'
;        print,' states.'

	;tmp = spec_access(h,filef0)
	h = dabax_access(filef0)
	for ii=0,nbatom-1 do begin
	  tmp = double(spec_data(h,batom(ii)))
	  if ii EQ 0 then f0coeff= dblarr(nbatom,n_elements(tmp))
	  f0coeff(ii,*) = tmp
	endfor


;
; Retrieve the corrections to the atomic scattering factor
; from the SHADOW optical constant library.  Note that it is assumed
; that the corrections are independent of k, Also note that READLIB
; obtains the real part of f(k=0) = f0(k=0) + f1(k=0) =  Z  + f1(k=0)
; and so Z is subtracted from f1 to obtain f1(k=0).
; We call readlib for any element obtaining : an array with the 
; 420 energy points in eV and logarithmic scale, the atomic number 
; atnum, the atomic weight atwt, the absorption coefficient rmu, 
; and 420 values for the f1 and f2, the real and imaginary (respectively)
; dispersive parts of the atomic scattering factor. 
;
        NPOINT  = (EMAX - EMIN)/ESTEP + 1
	energy = dblarr(npoint)
	for i=0,npoint-1 do energy(i) = emin+estep*i

	fp = dblarr(nbatom,npoint)
	fpp = dblarr(nbatom,npoint)
	;h = dabax_access(filef1f2)
        ;for ii=0,nbatom-1 do begin
        ;  tmp = double(spec_data(h,batom(ii)))
        ;  earr = tmp(0,*)
        ;  ofp = tmp(1,*)
        ;  ofpp = tmp(2,*)
	;  ; make interpolation
	;  earr_log      = alog(earr)
	;  my_energy_log = alog(energy)
	;  fp(ii,*) = interpol(     ofp,   earr_log,my_energy_log)
	;  fpp(ii,*) = exp(interpol(alog(ofpp),  earr_log,my_energy_log))
        ;endfor
        ;end
        for ii=0,nbatom-1 do begin
	  sym = atomic_symbols(batom(ii))
	  f1f2 = f1f2_calc(filef1f2,sym,energy,group=group)
	  if ((n_elements(f1f2) EQ 1) and (f1f2(0) EQ 0)) then begin
	    if ask EQ 2 then begin
	      itmp = widget_message(dialog_parent=group,/Question,$
	      ['Error calculating f1f2 values for '+sym,'Continue?'])
	    endif else begin
	      itmp=0
	      message,/info,'Error calculating f1f2 values for '+sym+'.'
	      read,'>>> Continue? [1=Yes,0=No] ',itmp
	      if itmp EQ 1 then itmp='Yes' else itmp = 'No'
	    endelse
	    if itmp EQ 'No' then return,0
	    fp(ii,*) = 0.0+BATOM(ii)
	    fpp(ii,*) = 0.0
	  endif else begin
	    fp(ii,*) = f1f2(0,*)
	    fpp(ii,*) = f1f2(1,*)
	  endelse
        endfor
;
; Compute the geometrical part G's of the structure factor
;
	
	CI = dcomplex(0.0,1.0)
	G = replicate(dcomplex(0.0,0.0),nbatom)
	G_BAR = G
        for i = 0, nbatom-1 do begin
		ibatom = batom(i)
		tmp = where(zcol EQ ibatom)
		for j=0,n_elements(tmp)-1 do begin
			X = str(2,tmp(j))
			Y = str(3,tmp(j))
			Z = str(4,tmp(j))
			ttt = exp(ci*2*!dpi*  $
			  (X*M_REF(1) + Y*M_REF(2) + Z*M_REF(3)))
                	G(i) = G(i) + ttt
		endfor
        G_BAR(i) = CONJ(G(i))
	endfor
;
; Crystal absorption and Temperature factor
;
;

; If NO crystal absorption is selected, set to zero the imaginary
; part of all the atomic form factor
if i_absorp EQ 0 then fpp = fpp*0.0

;
; Now prepare the file for SHADOW.
;

ATNUM= fix(batom)

FRACT = dblarr(nbatom)
for i=0,nbatom-1 do FRACT(i)=(fcol( where(Zcol EQ batom(i))))(0)

G_0 = dblarr(nbatom)
for i=0,nbatom-1 do G_0(i)=n_elements( where(Zcol EQ batom(i)))

NPOINT = long(NPOINT)
F1 = dblarr(nbatom,npoint)
for j=0,nbatom-1 do F1(j,*) = FP(j,*)-BATOM(j)
F2 = FPP

;
; temperature parameter
;
temper1 = temper
IF StrCompress(temper1,/Rem) EQ '?' THEN BEGIN
  temper1 = -300.0
  editFlag=1
ENDIF ELSE BEGIN
  temper1 = Float(temper1)
  editFlag=0
ENDELSE

IF temper1 LT 0.0 THEN BEGIN
  ratio = 1D/ (2D*SP_HKL*1.0D8)
  temper=DebyeWaller(batom,Ratio=ratio, Temper=Abs(temper1), Edit=editFlag )
ENDIF ELSE BEGIN
  TEMPER = replicate(TEMPER1,nbatom)
ENDELSE



D_SPACING = sp_hkl

out = { outfil:outfil, RN:rn, d_spacing:d_spacing, nbatom:nbatom, $
 atnum:atnum, fract:fract, TEMPER:temper, g_0:g_0, $
 G:G, G_BAR:G_BAR, f0coeff:f0coeff, $
 NPOINT:NPOINT, energy:ENERGY, f1:f1, f2:f2 }

return,out

end

