;+
; =================================  EXODUS ===============================
;
;   EXODUS is a widget based application to manipulate SPEC files: 
;
;	1) It loads a SPEC file and user can  select, view, interpolate, 
;		average, and perform operations with data from multiple scans. 
;          It also loads data from ASCII files and from other XPlot windows.
;	2) It allows to export the selected data set from one or 
;		many SPEC files into files af different types
;		(Spec/Scan,Spac/Mesh, etc.)
;	3) It contains some tools to convert and operate with 
;		lists of Spec/MCA files. 
;
;		
; ============= Non interactive use of EXODUS  ============================
;
;	NAME:
;		EXODUS
;	PURPOSE:
;		Manipulate SPEC data.
; 	CATEGORY:
;		Widgets. SPEC tools.
;	CALLING SEQUENCE:
;		EXODUS 
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls Xplot.
;			When this ID is specified, a death of the caller 
;			results in a death of EXODUS.
;		NO_BLOCK = Set this keyword to start EXODUS and leaving
;			unblocked the IDL prompt (default: no_block=1)
;		XAFS = Set this keyword to start EXODUS with some
;			functionality for EXAFS data treatment. 
;
;	OUTPUTS:
;		Open the widget utility.
;	COMMON BLOCKS:
;		None
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;	RESTRICTIONS:
;		Unknown.
;	HISTORY:
;		EXODUS was  designed for being an application aiming
;		to average multiple scans for the ESRF beamline ID26
;		(at the time dedicated to EXafs Of DilUte Samples,
;		origin of the program title). However, as the
;		application revealed to be of general interest, it was 
;		promoted to be a general XOP tool. The "particularities" 
;		of EXAFS data analysis (i.e., interpolate and resample in 
;		k space, etc.), are hidden in the standard version, and are 
;		only available if EXODUS is started using the XAFS keyword on. 
;
;	MODIFICATION HISTORY:
;		Written by  Manuel Sanchez del Rio. ESRF. 13 April 2000
;		version 1.0 Beta2
;		00-05-10 srio@esrf.fr doc, added resample, 
;			and other changes. Release of version 1.0.
;		03-03-27 added MCA stuff. version 1.1
;			Bug fixed (did not use the first spectrum 
;			for average and sum...) M. Valvidares (id3).
;			use ompp inetead of mpp (obsoleted).
;			Export to mesh file. New routines:
;			exodus_interpolation and exodus_quit. 
;		03-03-31 Promoted from being an utility of xop/xaid
;			to be a general tool for XOP. 
;		03-11-17 srio@esrf.fr Added "Import from" options in
;			right panel.  Version 1.11
;		04-01-14 srio@esrf.fr Added Preferences (skip lines in ascii)
;		04-02-27 srio@esrf.fr Added import filter version 1.12
;
;-
;
;========================================================================
;
Function Exodus_Version
Return,'1.12'
END
;
;========================================================================
;
PRO exodus_quit,event
;
Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'Exodus_quit: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

;Widget_Control,event.top, Get_UValue=state
stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state ; , /No_Copy


IF Ptr_Valid(state.ptrSpecFile) EQ 1 THEN Ptr_Free,state.ptrSpecFile
IF Obj_Valid(state.ompp) EQ 1 THEN Obj_Destroy,state.ompp

Widget_Control,event.top,/Destroy
END ; Exodus_quit
;
;========================================================================
;
;
; calculate interpolated data (for sum, averaging, mesh writing, etc. )
;
PRO exodus_interpolate,event,state,isel,xInt=xInt,yInt=yInt, $
  leave_flag=leave_flag,ir_label=ir_label
	leave_flag=0
        ;isel = Widget_Info(event.id,/DropList_Select)
	IF isel EQ 0 THEN BEGIN
	  leave_flag = 1
	  GoTo,out
	ENDIF
        ;Widget_Control,event.id,Set_DropList_Select=0

	iList = Widget_Info(state.wids.PRList,/List_Select)
	IF isel EQ 5 THEN BEGIN
	  IF N_Elements(iList) LE 2 THEN BEGIN
 	    itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	      ['Please select at least three lines including:',$
		'- the data set with the target (to be fitted)',$
		'- two or more components ','',$
		'The fit will minimize the difference between the',$
		'target and a linear combination of the components'])
	    leave_flag=1
	    GoTo,out
	  ENDIF
	  ptrMPP = state.ompp
	  pText=''
	  tmp = ompp_lcfit(ptrMPP,iList,Dialog_Parent=event.top, $
		PText=pText)
	  ;
	  ; store result
	  ;
	  IF N_Elements(tmp) LE 1 THEN BEGIN
	    leave_flag=1
	    GoTo,out
	  ENDIF
	  ptrMPP->set,Value=tmp,Title='Fit result '+pText,/OverAdd
          state.ompp = ptrMPP
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,$
                  Set_List_Select=iList
	  leave_flag=1
	  GoTo,out
	ENDIF
 
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    'Please select some data lines')
	  leave_flag=1
	  GoTo,out
	ENDIF
	IF N_Elements(iList) LE 1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    ['Please select MULTIPLE data lines',$
	     '(use <Ctr>+Click or <Shift>+Click for multiple selection.)'])
	  leave_flag=1
	  GoTo,out
	ENDIF
	mpp = state.ompp
	;
	; Check if all abscissas are the same and calculate extrema
	; 
	ideff = 0
	nn = N_Elements(iList)
	xMin = DblArr(nn)
	xMax = DblArr(nn)
	xStep = DblArr(nn)
	xPts = LonArr(nn)
	FOR i=0,nn-1 DO BEGIN
	  data = mpp->value(iList[i])
	  IF i EQ 0 THEN BEGIN
	    xx = Reform(data[0,*]) 
	    xMax[i] = Max(xx)
	    xMin[i] = Min(xx)
	    xPts[i] = N_Elements(xx)
	    xStep[i] = (xMax[i] - xMin[i])/(xPts[i] - 1)
          ENDIF ELSE BEGIN
	    xi = Reform(data[0,*])
	    xMax[i] = Max(xi)
	    xMin[i] = Min(xi)
	    xPts[i] = N_Elements(xi)
	    xStep[i] = (xMax[i] - xMin[i])/(xPts[i] - 1)
	    IF N_Elements(xx) NE N_Elements(xi) THEN BEGIN
	      ideff =1 
	      txt = 'Abscissas arrays have different number of elements.'
	    ENDIF ELSE IF Total(Abs(xx-xi)) NE 0 THEN BEGIN
	      ideff =1 
	      txt = 'Abscissas arrays have different values.'
	    ENDIF
	  ENDELSE
	ENDFOR

	IF ideff NE 0 THEN BEGIN  ; interpolation
 	  itmp = Widget_Message(/Info,Dialog_Parent=event.top,/Cancel, $
	    [txt,'Data will be interpolated','','Information: ',$
	    'Minima extrema = '+Vect2String([Min(xMin),Max(xMin)]),$
	    'Maxima extrema = '+Vect2String([Min(xMax),Max(xMax)]),$
	    'Step = '+Vect2String([Min(xStep),Max(xStep)]), $
	    'Number of points in = '+Vect2String([Min(xPts),Max(xPts)])] )
	  IF itmp EQ 'Cancel' THEN BEGIN
	     leave_flag=1
	     GoTo,out
	  ENDIF
	  x1 = Max(xMin)
	  x2 = Min(xMax)
	  nPts = Min(xPts)
	  deltaX = Max(xStep)

	  IF N_Elements(intStr) EQ 0 THEN BEGIN
	    ;intStr = { type:['0','Like 1st data','Customized'], x1:x1, x2:x2,$ 
	    ;      nPts:nPts, step:['0','energy (E)','wavenumber (k)'],$ 
            ;      e0:x1, method:['0','Interpolation','Resample'] }
	    intStr = { type:['0','Like 1st data','Customized'], x1:x1, x2:x2,$ 
	          deltaX:deltaX, step:['0','energy (E)','wavenumber (k)'],$ 
		  e0:x1, method:['0','Interpolation','Resample'] }
	  ENDIF ELSE BEGIN
	    intStr.x1=x1
	    intStr.x2=x2
	    ;intStr.nPts=nPts
	    intStr.deltaX=deltaX
	    intStr.e0=x1
	  ENDELSE
	  IF state.xafs EQ 1 THEN BEGIN
	    flags = ['1','w(0) EQ 1','w(0) EQ 1','w(0) EQ 1', $
		'w(0) EQ 1','w(0) EQ 1 AND w(4) EQ 1','1' ]
	    titles = ['Interpolated abscissas','X Min [Energy]', $
		'X Max [Energy]', $
		'Step value [Energy or k]','Step constant in', $
		'E0 (energy edge)','Method']
	  ENDIF ELSE BEGIN
	    flags = ['1','w(0) EQ 1','w(0) EQ 1','w(0) EQ 1', $
		'0','0','1' ]
	    titles = ['Interpolated abscissas','X Min', $
		'X Max', $
		'Step value','****', '****','Method']
	  ENDELSE
	  action = 0
	  XScrMenu,intstr,/NoType,/Inter,Flags=flags,Titles=titles, $
	    Dialog_Parent=event.top,WTitle='Interpolation parameters',$
	    Action=action
	  IF action EQ 'DONT' THEN BEGIN
		leave_flag=1
		GoTo,out
	  ENDIF

	  ;
	  ; Interpolate
	  ;

	  IF StrCompress(intstr.type[0],/Rem) EQ '0' THEN xint = xx ELSE BEGIN
	    CASE StrCompress(intstr.step[0],/Rem) OF
	     '0': BEGIN
		  nPts = ((intstr.x2-intstr.x1)/intStr.deltaX)+1
		  ;xInt = Abscissas(intstr.x1,intstr.x2,intstr.nPts)
		  xInt = Abscissas(intstr.x1,intstr.x2,nPts)
		  END
	     '1': BEGIN
		  xlimE =[[intstr.x1,intstr.x1],[intstr.x2,intstr.x2]]
		  xlimK = e2k(xlimE,E0=intstr.e0)
		  nPts = ( (xlimK[0,1]-xlimK[0,0]) / intStr.deltaX)+1
		  ;xIntk = Abscissas(xlimK[0,0],xlimK[0,1],intstr.nPts) 
		  xIntk = Abscissas(xlimK[0,0],xlimK[0,1],nPts) 
		  xInt = Make_Set(xIntk,xIntk)
		  xInt = k2e(xInt)
		  xInt = Reform(xInt[0,*])+intstr.e0
	          END
	    ENDCASE
	  ENDELSE
	  Widget_Control,/HourGlass
	  nxInt = N_Elements(xInt)
	  yInt = FltArr(nn,nxInt)
	  FOR i=0,nn-1 DO BEGIN
	    data = mpp->value(iList[i])
	    xx = Reform(data[0,*])
	    yy = Reform(data[1,*])
	    CASE StrCompress(intstr.method[0],/Rem) OF
	    '0': BEGIN
		;Message,/Info,'Using Interpolate'
	        ; y = Interpol(yy,xx,xInt)
		y = Interpolate(yy,FIndex(xx,xInt))
	    	yInt[i,*] = y
		ir_label='i'
		END
	    '1': BEGIN
		;Message,/Info,'Using Xaid_Resample'
		y = XAID_Resample(xx,yy,xInt)
	    	yInt[i,*] = y
		ir_label='r'
		END
	     else:
	     ENDCASE
	  ENDFOR
	ENDIF ELSE BEGIN
	  data = mpp->value(iList[0])
          xInt = Reform(data[0,*])
	  nxInt = N_Elements(xInt)
	  yInt = FltArr(nn,nxInt)
	  IF nn GT 0 THEN BEGIN
	    FOR i=0,nn-1 DO BEGIN
	      data = mpp->value(iList[i])
	      yInt[i,*] = Reform(data[1,*])
	    ENDFOR
	  ENDIF
	ENDELSE
	; 
 out:
END

;
;========================================================================
;
PRO exodus_event,event
COMMON exodus, intStr

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'EXODUS_EVENT: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   goto,out
endif

IF Tag_Names(event,/Structure_Name) EQ 'WIDGET_KILL_REQUEST' THEN BEGIN
  Exodus_Quit,event
  RETURN
ENDIF

;
;
Widget_Control, event.id, get_UValue=eventUValue
;Widget_Control, event.id, get_Value=Value
 
if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''
 
stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state ; , /No_Copy

 
back:
Case eventUValue OF
    'QUIT': Exodus_Quit,event
    'PREFERENCES': BEGIN
	   tmp = state.pref
	   XScrMenu,tmp,/NoType,/Interp, Action=action,  $
		Titles=['Lines to skip in ASCII files:','Import filter:'], $
		WTitle='Exodus preferences',Dialog_Parent=event.top, $
		FieldLen=35
	   IF action EQ 'DONT' THEN GoTo,out
	   state.pref=tmp
	END
    'IMPORT': BEGIN
	isel = Widget_Info(event.id,/DropList_Select)
	Widget_Control,event.id,Set_DropList_Select=0
	CASE isel OF
	0: 
	1: BEGIN ; load spec file
	  IF Widget_Info(state.Xplot,/Valid_Id) EQ 1 THEN $
	    Xplot_Quit,state.Xplot
	  dir = state.importDir 
	  file = Dialog_PickFile(Title='Select a SPEC file',$
	    Dialog_Parent=event.top,PATH=dir,GET_PATH=dir)
	  IF file EQ '' THEN GoTo,out
	  Widget_Control,/Hourglass
          Widget_Control,event.top,TLB_Set_Title= $
	  'EXODUS '+Exodus_Version()+' - Imported: '+file
	  n = Spec_Access(hSpec,file)
	  IF n EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error, DIALOG_PARENT=event.top, $
              ['EXODUS_EVENT: Error reading SPEC data file: ',file])
	    GoTo,out
	  ENDIF
	  list0 = StrCompress(hSpec.scan.scan_n)+' - '
	  list1 = hSpec.scan.name
	  Widget_Control,state.wids.PLList,Set_Value=list0+list1
	  state.ptrSpecFile=Ptr_New(hSpec)
	  state.specFile=file
	  state.importDir=dir
	  END
	2: BEGIN  ; Reload spec file
	  Widget_Control,/Hourglass
	  file = state.specFile
	  n = Spec_Access(hSpec,file)
	  IF n EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error, DIALOG_PARENT=event.top, $
              ['EXODUS_EVENT: Error reading SPEC data file: ',file])
	    GoTo,out
	  ENDIF
	  list0 = StrCompress(hSpec.scan.scan_n)+' - '
	  list1 = hSpec.scan.name
	  Widget_Control,state.wids.PLList,Set_Value=list0+list1
	  state.ptrSpecFile=Ptr_New(hSpec)
	END

	3: BEGIN  ;  mca files
	  itmp = Dialog_Message(Dialog_parent=event.top, /Info,/Cancel,$
		['MCA files cannot be load directly.',$
		 'They must be converted into a SPEC standard scan file.',$
		 'Do you want to do it?'])
	  IF itmp EQ 'Cancel' THEN GoTo,out
	  outFile = 0 ; variable to receive the file name
	  mca2scan,Group=event.top,info=info
	  IF N_Elements(info) EQ 0 THEN GoTo,out
	  file = info.outFile
	  dir = info.pathout

	  ; from here, the same as CASE=0
	  IF Widget_Info(state.Xplot,/Valid_Id) EQ 1 THEN $
	    Xplot_Quit,state.Xplot
	  IF file EQ '' THEN GoTo,out
	  Widget_Control,/Hourglass
          Widget_Control,event.top,TLB_Set_Title= $
	  'EXODUS '+Exodus_Version()+' - Imported: '+file
	  n = Spec_Access(hSpec,file)
	  IF n EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error, DIALOG_PARENT=event.top, $
              ['EXODUS_EVENT: Error reading SPEC data file: ',file])
	    GoTo,out
	  ENDIF
	  list0 = StrCompress(hSpec.scan.scan_n)+' - '
	  list1 = hSpec.scan.name
	  Widget_Control,state.wids.PLList,Set_Value=list0+list1
	  state.ptrSpecFile=Ptr_New(hSpec)
	  state.specFile=file
	  state.importDir=dir
	  END
	ENDCASE
	END
    'PLLIST': BEGIN
	IF event.clicks NE 2 THEN GoTo,out
	eventUValue = 'XPLOT_LEFT'
	GoTo,back
	; plot
	END
    'XSPECVIEW': BEGIN
	hSpec = *state.ptrSpecFile
	iList = Widget_Info(state.wids.PLList,/List_Select)
	tmp = Spec_Labels(hSpec,1+iList[0],/Index) ; To set current scan
	XSpecView,hSpec,Group=event.top,No_Block=state.no_block
	END
    'EXODUS': BEGIN
	Widget_Control,/HourGlass
	Exodus,No_Block=state.no_Block
	END
    'XAID': BEGIN
	Widget_Control,/HourGlass
	Xaid,No_Block=state.no_Block
	END
    'XPLOT': BEGIN
	Widget_Control,/HourGlass
	;Xplot,Group=event.top,No_Block=state.no_Block
	  XPlot,Parent=p,Group=event.top,No_Block=state.no_block
          state.Xplot=p
	END
    'DISPLAYFILE': BEGIN
	file = Dialog_Pickfile(Dialog_Parent=event.top, /Read)
	IF file EQ '' THEN GoTo,out
	XDisplayFile1,file,Dialog_Parent=event.top
	END
    'XPLOT_LEFT': BEGIN
        Widget_Control,/HourGlass
	hSpec = *state.ptrSpecFile
	iList = Widget_Info(state.wids.PLList,/List_Select)

	tmp = Spec_Labels(hSpec,1+iList[0],/Index) ; To set current scan

        IF Widget_Info(state.Xplot,/Valid_Id) EQ 0 THEN BEGIN
          p=0
	  XPlot,Parent=p,Group=event.top,No_Block=state.no_block
          state.Xplot=p
        ENDIF
        Xplot_LoadFile,state.Xplot,SPEC=hSpec,ScanId='CURRENT'

	END
    'XSCALE': BEGIN
	CASE event.index OF
	0: state.xScale=[1.0D0,0.0D0]
	1: state.xScale=[1.0D3,0.0D0]
	2: BEGIN
	   itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
		['Enter the multiplicative factor A for abscissas (X)',$
		 'and a shift B (Xnew = A*X + B) ',$
		 'for applying when copying data to right panel'],/Cancel)
	   IF itmp EQ 'Cancel' THEN GoTo,out

	   tmp = {a:state.xScale[0], b:state.xScale[1]}
	   XScrMenu,tmp,Action=action, Titles=['A (x_new=Ax+B) : ', $
		'B (x_new=Ax+B) : '], Dialog_Parent=event.top,/NoType
	   IF action EQ 'DONT' THEN GoTo,out
	   state.xScale[0]=tmp.a
	   state.xScale[1]=tmp.b
	   END
	else: 
	ENDCASE
	END
    ; 
    ; Central panel
    ;
    'COPY1': BEGIN
	Widget_Control,event.id,Get_Value=val
	hSpec = *state.ptrSpecFile
	list = StrCompress(hSpec.scan.scan_n)+' - ' + hSpec.scan.name
	CASE val OF
	  '=>': iList = Widget_Info(state.wids.PLList,/List_Select)
	  '*=>': iList=LIndGen(N_Elements(list))
	  else: 
	ENDCASE
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    'Please select a data line')
	  GoTo,out
	ENDIF
	titles = list[ilist]
;	ptr = state.ptrMPP
	ptr = state.ompp
	Widget_Control,state.wids.xCol,Get_Value=xCol
	Widget_Control,state.wids.yCol,Get_Value=yCol
	cols = Fix([xcol[0],ycol[0]])
	Widget_Control,/HourGlass
	FOR i=0,N_Elements(iList)-1 DO BEGIN
	  index = iList[i]+1
	  IF Spec_Points(hSpec,index,/Index) LE 0 THEN BEGIN
	    txt = StrCompress(hSpec.scan[index-1].scan_n)+' - ' +$
	    Spec_Name(hSpec,index,/Index)
	    tmp = Dialog_Message(/Question,['Scan contains no data: ',$
	      txt,'','Abort? '],Dialog_Parent=event.top)
	    CASE tmp OF
	      'No': 
	      'Yes': GoTo,out
	    ENDCASE
	    Widget_Control,/HourGlass
	  ENDIF ELSE BEGIN
	    data = Spec_Data(hSpec,index,cols,/Index)
	    IF (state.xScale[0] NE 1.0) OR (state.xScale[1] NE 0.0) THEN BEGIN
		data[0,*]=data[0,*]*state.xScale[0]+state.xScale[1]
	    ENDIF
	    IF  Spec_Columns(hSpec,index,/Index) NE 2 THEN $
;	      MPP_Set,ptr,Value=data,/Add, $
	      ptr->set,Value=data,/OverAdd, $
	      Title=titles[i]+' Cols='+Vect2String(cols) ELSE $
;	      MPP_Set,ptr,Value=data,/OverAdd, Title=titles[i]
	      ptr->set,Value=data,/OverAdd, Title=titles[i]
	  ENDELSE
	ENDFOR
;	state.ptrMPP = ptr
	state.ompp = ptr
;	list = MPP_Value(state.ptrMPP,/Title)
	list = (state.ompp)->value(/Title)
	Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
	  N_Elements(list)-1
	END
    ; 
    ; Right panel
    ;
    'IMPORTRIGHT': BEGIN
	isel = Widget_Info(event.id,/DropList_Select)
	Widget_Control,event.id,Set_DropList_Select=0
	CASE isel OF
	 0:
	 1: BEGIN ; Import from current Xplot window
          IF Widget_Info(state.xplot,/Valid_Id) EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	     'No active xplot window created from Exodus')
	    Goto,out
	  ENDIF
          Widget_Control,/HourGlass
          Xplot_GetCurrentData,state.xplot,data
	  IF N_Elements(data) LE 1 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
             'No data imported')
            Goto,out
	  ENDIF
	  ptr = state.ompp
	  ptr->set,Value=data,/OverAdd, Title='Imported from xplot'
          state.ompp = ptr
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)
	  END
	 2: BEGIN ; Import Xplot window with ID...
	  id = state.xplot
	  XEdit,id,TEXT='XPlot window ID:',TITLE='Import from XPlot',$
	    Infotext=['Enter the Xplot ID number of the window from where'+$
		' you want to import data.', $
		'(See it in Xplot->View->XPlot ID number)'],Action=Action,$
		Dialog_Parent=event.top
	  IF action EQ 'CANCEL' THEN GoTo,out
          IF Widget_Info(id,/Valid_Id) EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	     'XPLOT Window does not exist: '+StrCompress(id) )
	    Goto,out
	  ENDIF
          Widget_Control,/HourGlass
          Xplot_GetCurrentData,id,data
	  IF N_Elements(data) LE 1 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
             'No data imported')
            Goto,out
	  ENDIF
	  ptr = state.ompp
	  ptr->set,Value=data,/OverAdd, Title='Imported from xplot '+ $
		StrCompress(id)
          state.ompp = ptr
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)
	  END

	 3: BEGIN ; Import from ASCII file
	  dir = state.importDir 
	  file = Dialog_PickFile(Title='Select an ASCII file',$
	    Dialog_Parent=event.top,PATH=dir,GET_PATH=dir,$
	    /Multiple_Files)
	  IF file[0] EQ '' THEN GoTo,out
	  Widget_Control,/Hourglass
	  FOR i=0,N_Elements(file)-1 DO BEGIN
	    data = rascii(file[i],skip=state.pref.rascii_skip)
	    ptr = state.ompp
	    ptr->set,Value=data,/OverAdd, Title=file[i]
            state.ompp = ptr
	  ENDFOR
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)
	  END

	 4: BEGIN ; Import ASCII from list
	  wildfile=state.wildfile ; '*.txt'
	  XEdit,wildfile,TEXT='Match string: ',TITLE= 'Import from file list',$
	    Infotext=['Enter a string describing the files you want to use'],$
		Action=Action,Dialog_Parent=event.top,XSize=25
	  IF action EQ 'CANCEL' THEN GoTo,out
          state.wildfile=wildfile
	  files = FindFile(wildfile)
	  IF files[0] EQ '' THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
             'No files found')
            Goto,out
	  ENDIF
	  XDisplayFile1,TEXT=files,/Modal,Title= $
	    'Selected files (File->Accept to proceed)',Action=action, $
	    Dialog_Parent=event.top
	  IF action EQ 'DONT' THEN GoTo,out
	  Widget_Control,/Hourglass
	  nfiles = N_Elements(files)
	  FOR i=0,nfiles-1 DO BEGIN
	    data = rascii(files[i],skip=state.pref.rascii_skip)
	    ptr = state.ompp
	    ptr->set,Value=data,/OverAdd, Title=files[i]
            state.ompp = ptr
	  ENDFOR
          list = (state.ompp)->value(/Title)
	  nlist = N_Elements(list)
          sele=IndGen(nfiles)+(nlist-nfiles)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
	  sele
	  END

	 5: BEGIN ; Import filter
	  Widget_Control,/Hourglass
	  x=0 & y = 0
	  xtmp=Temporary(x)
	  ytmp=Temporary(y)
	  command=state.pref.import_filter
	  itmp = Execute(command)
	  IF itmp NE 1 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
            'Error executing: '+command)
          Goto,out
	  ENDIF
	    
	  IF (size(y))[0] GT 1 THEN BEGIN
	     data = y 
          ENDIF ELSE BEGIN
	    IF N_Elements(x) EQ 0 THEN x=FindGen(N_Elements(y))
	    data = make_set(x,y)
	  ENDELSE
	  ptr = state.ompp
	  ptr->set,Value=data,/OverAdd, Title=command
          state.ompp = ptr

          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)
	  END


	 6: BEGIN ; Import filter from list
	  wildfile=state.wildfile ; '*.txt'
	  XEdit,wildfile,TEXT='Match string: ',TITLE= 'Import from file list',$
	    Infotext=['Enter a string describing the files you want to use'],$
		Action=Action,Dialog_Parent=event.top,XSize=25
	  IF action EQ 'CANCEL' THEN GoTo,out
	 state.wildfile=wildfile
	  files = FindFile(wildfile)
	  IF files[0] EQ '' THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
             'No files found')
            Goto,out
	  ENDIF
	  nfiles = N_Elements(files)
	  command  = Replicate(state.pref.import_filter,nfiles)
	  FOR i=0,nfiles-1 DO $
		command[i]=StrSubstitute(command[i],'file','"'+files[i]+'"')
	  XDisplayFile1,TEXT=command,/Modal,Title= $
	    'Selected files (File->Accept to proceed)',Action=action, $
	    Dialog_Parent=event.top
	  IF action EQ 'DONT' THEN GoTo,out
	  Widget_Control,/Hourglass
help,command
	  FOR i=0,N_Elements(command)-1 DO BEGIN
	    x=0 & y = 0
	    xtmp=Temporary(x)
	    ytmp=Temporary(y)
	    itmp = Execute(command[i])
	    IF itmp NE 1 THEN BEGIN
	      itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
              'Error executing: '+command[i])
              Goto,out
	    ENDIF
	    
	    IF (size(y))[0] GT 1 THEN BEGIN
	       data = y 
            ENDIF ELSE BEGIN
	      IF N_Elements(x) EQ 0 THEN x=FindGen(N_Elements(y))
	      data = make_set(x,y)
	    ENDELSE
	    ptr = state.ompp
	    ptr->set,Value=data,/OverAdd, Title=files[i]
            state.ompp = ptr
	  ENDFOR
          list = (state.ompp)->value(/Title)
	  nlist = N_Elements(list)
          sele=IndGen(nfiles)+(nlist-nfiles)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
	  sele
	  END
	 else: 
	ENDCASE
	END
    'VIEWEDIT': BEGIN
	isel = Widget_Info(event.id,/DropList_Select)
	Widget_Control,event.id,Set_DropList_Select=0
	CASE isel OF
	 0:
	 1: BEGIN ; View Info
;	    tmp = MPP_Info(state.ptrMPP)
	    tmp = (state.ompp)->info()
	    XDisplayFile1,Text=tmp, Group=event.top, Title='Information ', $
		Dialog_Parent=event.top
	    END
	 2: BEGIN  ; Edit Labels
	      iList = Widget_Info(state.wids.PRList,/List_Select)
	      IF iList[0] EQ -1 THEN BEGIN
 	        itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	          'Please select a data line')
	        GoTo,out
	      ENDIF
;	      txt_old = MPP_Value(state.ptrMPP,ilist,/Title)
	      txt_old = (state.ompp)->value(ilist,/Title)
	      txt_new = txt_old
	      action = ''
	      XDisplayFile1,Text=txt_new,Title='Edit data labels', $
		Group=event.top,/Modal,Action=action, $
		Dialog_Parent=event.top
	      IF action EQ 'DONT' THEN GoTo,out
	      IF N_Elements(txt_new) NE N_Elements(txt_old) THEN BEGIN
		itmp = Dialog_Message(/Error, Dialog_Parent=event.top, $
		 'Number of label lines has changed. Aborted')
		GoTo,out
	      ENDIF
;	      MPP_Set,state.ptrMPP,ilist,Title=txt_new
	      (state.ompp)->set,ilist,Title=txt_new
;	      list = MPP_Value(state.ptrMPP,/Title)
	      list = (state.ompp)->value(/Title)
	      Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
	        iList[0]
	    END
	 else:
	ENDCASE
	END
    'EXPORT': BEGIN
	isel = Widget_Info(event.id,/DropList_Select)
	Widget_Control,event.id,Set_DropList_Select=0
	iList = Widget_Info(state.wids.PRList,/List_Select)
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Please select a data line')
	  GoTo,out
	ENDIF

	CASE isel OF
	  1: flag_single=1
	  4: flag_single=1
	  5: flag_single=1
	  6: flag_single=1
	  else: flag_single=0
	ENDCASE
	IF flag_single AND N_Elements(iList) GT 1 THEN BEGIN
 	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Only the data from a single data line can be exported')
	  GoTo,out
	ENDIF
	
	CASE isel OF
	0: 
	1: BEGIN ; Save to a new ASCII file 
           file = Dialog_PickFile(File='exodus.ascii',Title= $
		'Select a file for writing (ASCII format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out

	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF
;	   set = MPP_Value(state.ptrMPP,iList[0])
	   set = (state.ompp)->value(iList[0])
;	   comm = ';' + MPP_Value(state.ptrMPP,iList[0],/Title)
	   comm = ';' + (state.ompp)->value(iList[0],/Title)
	   XEdit,comm,InfoText=['Type a 1-line comment for the new file',$
		'(Press CANCEL for skipping comment line)'],$
		Title='ASCII file comment',text='Comment: ',$
		Action=action,Format='(a100)',Dialog_Parent=event.top
	   IF action EQ 'CANCEL' THEN comm=''
	
	   OpenW,unit,file,/Get_Lun
	   format = '(2(G16.10," "))'
	   IF comm NE '' THEN printf,unit,comm
	   printf,unit,format=format,set 
	   Free_Lun,unit
	   END
	2: BEGIN ; Save to a file (SPEC format)
           file = Dialog_PickFile(File='exodus.spec',Title= $
		'Select a file for writing (SPEC format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out
	   IF checkfile(file) EQ 1 THEN BEGIN
             itmp=Dialog_Message(/Question,/Cancel,Dialog_Parent=$
                event.top,['File exists: ',file,'Append data?','',$ 
		'(Yes: data appended)','(No: file rewritten)', $
		'(Cancel: Abort operation)'])
	      CASE itmp OF
	        'Yes':
	        'No': Delete_Files,file
		'Cancel': GoTo,out
	      ENDCASE
	   ENDIF
	   nn = N_Elements(iList)
	   hspec = 0
;	   titles = MPP_Value(state.ptrMPP,iList,/Title)
	   titles = (state.ompp)->value(iList,/Title)
	   FOR i=0L,nn-1 DO BEGIN
;	     set = MPP_Value(state.ptrMPP,iList[i])
	     set = (state.ompp)->value(iList[i])
	     IF i EQ 0 THEN BEGIN
		Widget_Control,/Hourglass
		tmp=Spec_Save(hspec,set,file,Name=titles[i],Dialog_Parent=$
			event.top,/No_Confirm) 
		IF tmp EQ 0 THEN BEGIN
		  tmp=Dialog_Message(/Error,Dialog_Parent=$
		  event.top,'Writing operation aborted.')
		  GoTo,out
		ENDIF
	     ENDIF ELSE BEGIN
		tmp=Spec_Save(hspec,set,file,Name=titles[i],/No_Confirm)
	     ENDELSE
	   ENDFOR
	   END
	3: BEGIN ; Save to a file (SPEC/MESH format)
	   isel=1
	   exodus_interpolate,event,state,isel,yInt=yInt,xInt=xInt, $
		leave_flag=leave_flag
	   IF leave_flag EQ 1 THEN GoTo,out

           file = Dialog_PickFile(File='exodus.spec',Title= $
		'Select a file for writing (SPEC/MESH format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out

	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF
	   OpenW,unit,file,/Get_Lun
           printf,unit,'#F '+file
           printf,unit,'#D '+sysTime()
           printf,unit,'#UINFO File written by xop/exodus containing an interpolated mesh from:'
;	   list = MPP_Value(state.ptrMPP,/Title)
	   list = (state.ompp)->value(/Title)
           FOR i=0,N_Elements(list)-1 DO BEGIN
              printf,unit,'#UINFO      '+list[i]
           ENDFOR
           printf,unit,'#S  1 mesh from mca data'
           printf,unit,'#N  3'
           printf,unit,'#L X  index  Y'
           FOR i=0,N_Elements(yInt[*,0])-1 DO BEGIN ; columns
             FOR j=1,N_Elements(yInt[0,*])-1 DO BEGIN ; points=rows
	       printf,unit, String([xInt[j],i,yInt[i,j]],Format='(3G13.6)')
	     ENDFOR
	   ENDFOR
           Free_Lun,unit
           Print,'EXODUS: File written to disk: '+file

	   END
	4: BEGIN ; other applications
;	   set = MPP_Value(state.ptrMPP,iList[0])
	   set = (state.ompp)->value(iList[0])
	   x=set[0,*]
	   y=set[1,*]
	   txt = 'Xplot,x,y,/No_Block'
	   XEdit,txt,InfoText= $
		['Type the command that will accept the data','  Use: ',$
		 'appl,x,y [,Keywords]','  or','appl,set [,Keywords] '],$
		Title='Export to application',text='Command: ',$
		Action=action,Format='(a100)', Dialog_Parent=event.top
	   IF action EQ 'CANCEL' THEN GoTo,out
	   itmp = Execute(txt)	
	   END
	5: BEGIN ; mu2chi
;	   set = MPP_Value(state.ptrMPP,iList[0])
	   set = (state.ompp)->value(iList[0])
	   XAID_mu2chi,set,/No_Block
	   END
	6: BEGIN ; ff
;	   set = MPP_Value(state.ptrMPP,iList[0])
	   set = (state.ompp)->value(iList[0])
	   XAID_ff,set,/No_Block
	   END
	7: BEGIN ; Save to a file (DELIA binary format)

           file = Dialog_PickFile(File='exodus.delia',Title= $
		'Select a file for writing (DELIA binary format)',$
                Dialog_Parent=event.top)
           IF file EQ '' THEN GoTo,out

	   IF CheckFile(file) EQ 1 THEN BEGIN
	     itmp = Dialog_Message(/Question, ['File '+file+' exists.',$
		'Overwrite it?'],Dialog_Parent=event.top)
	     IF itmp EQ 'No' THEN BEGIN
	       itmp = Dialog_Message(/Error, 'Operation aborted',$
		Dialog_Parent=event.top)
	       GoTo,out
	     ENDIF
	   ENDIF

	   Widget_Control,/Hourglass
	   OpenW,unit,file,/Get_Lun,/Swap_If_Little_Endian

	   nn = N_Elements(iList)
	   FOR i=0L,nn-1 DO BEGIN
;	     set = MPP_Value(state.ptrMPP,iList[i])
	     set = (state.ompp)->value(iList[i])
	     set1=FltArr(3,N_Elements(set[0,*]))
	     set1[1:2,*]=set
	     ii = Float(i)
	     set1[0,*]=ii
	     WriteU,unit,set1
	   ENDFOR
	   Free_Lun,unit
	   END
	else:
	ENDCASE
	END
    'DELETEALL': BEGIN
	itmp = Dialog_Message(/Question, $
	  ['This option deletes all data in the right panel', $
	   'Please confirm'],Dialog_Parent=event.top)
	IF itmp EQ 'No' THEN GoTo,out
;	MPP_Free,state.ptrMPP
	(state.ompp)->remove
	Widget_Control,state.wids.PRList,Set_Value=''
	END
    'DELETE': BEGIN
	iList = Widget_Info(state.wids.PRList,/List_Select)
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    'Please select a data line')
	  GoTo,out
	ENDIF
	(state.ompp)->remove,iList
	IF (state.ompp)->info(/N_Good_Elements) EQ 0 THEN $
	  list = '' ELSE list = (state.ompp)->value(/Title)
	Widget_Control,state.wids.PRList,Set_Value=list
	END
    'GETXPLOT': BEGIN
          IF Widget_Info(state.xplot,/Valid_Id) EQ 0 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	     'No active xplot window created from Exodus')
	    Goto,out
	  ENDIF
          Widget_Control,/HourGlass
          Xplot_GetCurrentData,state.xplot,data
	  ptr = state.ompp
	  ptr->set,Value=data,/OverAdd, Title='Imported from xplot'
          state.ompp = ptr
          list = (state.ompp)->value(/Title)
          Widget_Control,state.wids.PRList,Set_Value=list,Set_List_Select=$
          Widget_Info(state.wids.PRList,/List_Select)

	END
    'OPERATIONS': BEGIN
        isel = Widget_Info(event.id,/DropList_Select)
	Widget_Control,event.id,Set_DropList_Select=0

	exodus_interpolate,event,state,isel,xInt=xInt,yInt=yInt, $
	  leave_flag=leave_flag,ir_label=ir_label

	IF leave_flag EQ 1 THEN GoTo,out
	nn = N_Elements(yInt[*,0])
	nxInt = N_Elements(xInt)
	iList = Widget_Info(state.wids.PRList,/List_Select)
	mpp = state.ompp
	; 
	; Perform requested operation
	;
	CASE isel OF
	0: 
	1: BEGIN  ; Interpolation
	   tmp = {output:['0','Add to List','Xplot 2D (multicolumn)',$
		'Xplot 3D view']}
	   action = ''
	   XScrMenu,tmp,Titles=['Output to:'],/NoType,/Interp,$
	     Dialog_Parent=event.top,Action=action,WTitle=$
		'Destination of interpolated data'
		
	   IF action EQ 'DONT' THEN GoTo,out
	   CASE StrCompress(tmp.output[0],/Rem) OF
	   '0': BEGIN ; Add to list
		FOR i=0,nn-1 DO BEGIN
		  title = mpp->value(iList[i],/Title)
	          data = Make_Set(xInt,Reform(yInt[i,*]))
		  mpp->set,Value=data,Title=ir_label+title,/OverAdd
		ENDFOR
	        state.ompp = mpp
                list = (state.ompp)->value(/Title)
                Widget_Control,state.wids.PRList,Set_Value=list,$
	          Set_List_Select=iList
		END
	   '1': BEGIN ; Xplot 2D
		yout = DblArr(nn+1,N_Elements(xint))
		labels = StrArr(nn+1)
		yout[0,*]=xint
		labels[0] = 'Interpolated abscissas'
		FOR i=1,nn DO BEGIN
		  labels[i] = mpp->value(iList[i-1],/Title)
	          yout[i,*] = Reform(yInt[i-1,*])
		ENDFOR
		XPlot,yout,No_Block=state.No_Block,ColTitles=labels,$
		 Wtitle='EXODUS: multicolumn 2D view'
		END
	   '2': BEGIN ; Xplot 3D
		npts = N_Elements(xint)
		yout = DblArr(3,npts*nn)
		FOR i=0,nn-1 DO BEGIN
	          yout[0,npts*i:(npts*i)+npts-1] = xint
	          yout[1,npts*i:(npts*i)+npts-1] = i
	          yout[2,npts*i:(npts*i)+npts-1] = Reform(yInt[i,*])
		ENDFOR
		p=0
		XPlot,yout,Parent=p,XCol=1,YCol=3,Wtitle='EXODUS: 3D view',$
		  /No_Block
		XPlot_Mesh,p,Flag=1,Kind=0,Col=2
		END
	   else: 
	   ENDCASE
	   END
	2: BEGIN  ; Average
	   xOut = xInt
	   yOut = FltArr(nxInt)
	   FOR i=0,nn-1 DO BEGIN
		print,'Index, values: ',i,yInt[i,*]
		yOut=yOut + Reform(yInt[i,*])
	   ENDFOR
	   yOut = yOut/Float(nn)
	   data = Make_Set(xOut,yOut)

	   title = 'Average sets: '
           FOR i=0,nn-1 DO BEGIN
             tmp = mpp->value(iList[i],/Title)
             tmp_pos = StrPos(tmp,'-')
             IF tmp_pos LE 0 THEN tmp_pos = 6
             title = title + StrMid(tmp,0,tmp_pos) +'; '
           ENDFOR

	   mpp->set,Value=data,Title=title, /OverAdd
	   state.ompp = mpp
           list = (state.ompp)->value(/Title)
           Widget_Control,state.wids.PRList,Set_Value=list,$
	          Set_List_Select=iList
	   END
	3: BEGIN  ; Sum
	   xOut = xInt
	   yOut = FltArr(nxInt)
	   FOR i=0,nn-1 DO yOut=yOut + Reform(yInt[i,*])
	   yOut = yOut
	   data = Make_Set(xOut,yOut)

	   title = 'Sum sets: '
           FOR i=0,nn-1 DO BEGIN
             tmp = mpp->value(iList[i],/Title)
             tmp_pos = StrPos(tmp,'-')
             IF tmp_pos LE 0 THEN tmp_pos = 6
             title = title + StrMid(tmp,0,tmp_pos) +'; '
           ENDFOR


	   mpp->set,Value=data,Title=title, /OverAdd
	   state.ompp = mpp
           list = (state.ompp)->value(/Title)
           Widget_Control,state.wids.PRList,Set_Value=list,$
	          Set_List_Select=iList
	   END
	4: BEGIN  ; Customized
	   xOut = xInt
	   yOut = DblArr(nxInt)
	   buffer = [';','; Define your exit ordinates as yOut',$
	      '; nn is the number of data lines you selected', $
	      '; xInt (yInt) are the interpolated abscissas (ordinates)',$
	      '; As an example, the following code sums all selected lines',$
	      '; ',$
	      'FOR i=0,nn-1 DO yOut=yOut + Reform(yInt[i,*]) ']
	   XDisplayFile1,Text=buffer,Title='Customized operation',$
	     /Modal,Action=action, Dialog_Parent=event.top
	   IF action EQ 'DONT' THEN GoTo,out
	   FOR iii=0,N_Elements(buffer)-1 DO BEGIN
	     itmp = Execute(buffer[iii])
	     IF itmp NE 1 THEN BEGIN
	       jtmp = Dialog_Message(/Question,Dialog_Parent=event.top,$
	       ['Error executing IDL command:'+command(i),'Continue?'])
	       if jtmp EQ 'No' then GoTo,out
	     ENDIF
	   ENDFOR
	   txt = 'Result of customized operation'
	   XEdit,txt,InfoText=['Type a label for the new data '],$
		Title='Store result of operation',text='Label: ',$
		Action=action,Format='(a30)',Dialog_Parent=event.top
	   IF action EQ 'CANCEL' THEN GoTo,out
	   tmp = Make_Set(xOut,yOut)
	   mpp->set,Value=tmp,Title=txt,/OverAdd
	   state.ompp = mpp
           list = (state.ompp)->value(/Title)
           Widget_Control,state.wids.PRList,Set_Value=list, $
	     Set_List_Select=iList
	   END

	else: BEGIN 
	   END
	ENDCASE

	  

	END
    'XPLOT_RIGHT': BEGIN
	iList = Widget_Info(state.wids.PRList,/List_Select)
	IF iList[0] EQ -1 THEN BEGIN
 	  itmp = Widget_Message(/Error,Dialog_Parent=event.top, $
	    'Please select a data line')
	  GoTo,out
	ENDIF
	IF N_Elements(iList) GT 1 THEN BEGIN ; multiple plot
	  Widget_Control,/HourGlass
	  ompp = state.ompp
	  XWindow,Buffer=$
	   [';','; It is NOT recommended  to change this line',$
	    '; Use the "Edit Parameters" button to customize',';',$
           "(data.mpp)->mplot,data.list,xRange=[parms.xMin,parms.xMax]"+$
	   ",yRange=[parms.yMin,parms.yMax],Legend=Fix(parms.legend[0])"+$
	   ",shiftValue=parms.shiftValue"],$
           Edit=3, Group=event.top,WTitle='EXODUS: Multiple plot', $
           Data={mpp:ompp,list:iList}, $
	   Parms={xMin:0.0,xMax:0.0,yMin:0.0,yMax:0.0,Legend:['1','No','Yes'],$
	   shiftValue:0.0},$
	   Titles=['X Min (Set Min=Max for defaults)','X Max', $
		'Y Min','Y Max','Legend',$
		'Shift spectrum factor (0<f<1)'],/Interp,/NoType, $
		/ZoomFlag,ZBoxColor=8

	ENDIF ELSE BEGIN   ; single plot
	  IF Widget_Info(state.xplot,/Valid_Id) EQ 0 THEN BEGIN
	    Widget_Control,/HourGlass
	    p=0
	    Xplot,Group=event.top,Parent=p,No_Block=state.no_block
	    state.xplot=p
	  ENDIF
	  ;data = MPP_value(state.ptrMPP,iList)
	  data = state.ompp->value(iList)
	  Xplot_LoadFile,state.xPlot,data
	ENDELSE
	END


    'MCA2SCAN': mca2scan,Group=event.top
    'MCA2MESH': mca2mesh,Group=event.top,/full_select,outFlag=1
    'MCA2MESH2': mca2mesh,Group=event.top,/full_select,outFlag=0
    'PRLIST': BEGIN
	IF event.clicks NE 2 THEN GoTo,out
	iList = event.Index
	
	IF Widget_Info(state.xplot,/Valid_Id) EQ 0 THEN BEGIN
	  Widget_Control,/HourGlass
	  p=0
	  Xplot,Group=event.top,Parent=p,No_Block=state.no_block
	  state.xplot=p
	ENDIF
	;data = MPP_value(state.ptrMPP,iList)
	data = state.ompp->value(iList)
	Xplot_LoadFile,state.xPlot,data
	END



    'ABOUT': XAID_Help,Group=event.top
    'HELP': BEGIN
	widget_control,event.id,get_value=tmp
	xhelp,tmp,GROUP=event.top
	END
    else: BEGIN
	;tmp=''
	;Widget_Control,event.id,Get_UValue=tmp,/Hourglass
	;command = tmp[0]+ $  ;',Group='+StrCompress(event.Id,/Rem)+'L'+$
	; ',No_Block='+StrCompress(state.no_block)
	;Message,/info,'Executing: '+command
	;itmp = Execute(command)
	END
EndCase


out:
if Widget_Info(stateid,/Valid_ID) then $
  Widget_Control, stateid, set_UValue=state ; , /No_Copy
end ; exodus_event
;
;====================================================================
;

PRO exodus, group=group, No_Block=no_Block, xafs=xafs

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'EXODUS: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return
endif
;
; color setting
;
Device,Get_Decomposed=usingDecomposed
IF usingDecomposed EQ 1 THEN BEGIN
  itmp = Dialog_Message(/Question, $
     ['EXODUS does not work properli with decomposed colors',$
     'Turn decomposed colors off?'])
  IF itmp EQ 'Yes' THEN Device,Decomposed=0  
ENDIF
Tek_Color

IF N_Elements(no_block) EQ 0 THEN no_block=1
IF N_Elements(xafs) EQ 0 THEN xafs=0

;
; define widgets
;
wbase=widget_base(/col,title='EXODUS '+Exodus_Version(),MBar=wMenuBar, $
	 /TLB_KILL_REQUEST_EVENTS, Event_Pro='Exodus_Quit')

wtmp = widget_base(wbase) ; to store state

wFileMenu = Widget_Button(wmenuBar,Value='File',/Menu)
  wtmp = Widget_Button(wFileMenu,Value='Preferences...',UValue='PREFERENCES',$
	/Separator)
  wtmp = Widget_Button(wFileMenu,Value='Quit',UValue='QUIT',/Separator)
  IF Keyword_Set(xafs) THEN BEGIN
    wExafsToolsMenu = Widget_Button(wmenuBar,Value='EXAFS_Tools',/Menu)
      wtmp = Widget_Button(wExafsToolsMenu,Value='EXODUS (New window)',$
	UValue='EXODUS')
      wtmp = Widget_Button(wExafsToolsMenu,Value='Main window (XAID)',$
	UValue='XAID')
  ENDIF
wToolsMenu = Widget_Button(wmenuBar,Value='Tools',/Menu)
  wtmp = Widget_Button(wToolsMenu,Value='EXODUS (New window)',$
	UValue='EXODUS')
  wtmp = Widget_Button(wToolsMenu,Value='Plotting tool...',$
	UValue='XPLOT')
  wtmp = Widget_Button(wToolsMenu,Value='Display file...',$
	UValue='DISPLAYFILE')
  wtmp = Widget_Button(wToolsMenu,Value='Convert MCA to SCAN',$
	UValue='MCA2SCAN')
  wtmpbase = Widget_Button(wToolsMenu,Value='Convert MCA to MESH',$
	/Menu)
    wtmp = Widget_Button(wtmpbase,Value='Output to file...',$
	UValue='MCA2MESH')
    wtmp = Widget_Button(wtmpbase,Value='Output to xplot window',$
	UValue='MCA2MESH2')

wHelpMenu = Widget_Button(wmenuBar,Value='Help',/Help)
  IF Keyword_Set(xafs) THEN BEGIN
    wtmp = Widget_Button(wHelpMenu,Value='About XAID',UValue='ABOUT')
  ENDIF
  wtmp = Widget_Button(wHelpMenu,Value='exodus',UValue='HELP')

wPannelMain = Widget_Base(wBase,Row=1)
wPannelLeft = Widget_Base(wPannelMain,Col=1,/Frame)
wPannelCenter = Widget_Base(wPannelMain,Col=1,/Frame)
wPannelRight = Widget_Base(wPannelMain,Col=1,/Frame)

yListSize = 25
xListSize=35
;
; Left Pannel
; 
wPLBaseTop = Widget_Base(wPannelLeft,/Row)
  wtmp = Widget_DropList(wPLBaseTop, UValue='IMPORT', $
    Value=['Import SPEC: ','SPEC file','ReLoad SPEC file','MCA/SPEC file list'])
wPLList = Widget_List(wPannelLeft,value=SIndGen(yListSize),UValue='PLLIST',$
  ysize=yListSize,xSize=xListSize,/Multiple)
wPLBaseBottom = Widget_Base(wPannelLeft,/Col)
  wtmp1= Widget_Base(wPLBaseBottom,/Row)
    wtmp = Widget_Label(wtmp1,Value='Cols X:')
    wXCol = Widget_Text(wtmp1,Value='1',UValue='XCOL',XSize=2,/Edit)
    wtmp = Widget_Label(wtmp1,Value=' Y:')
    wYCol = Widget_Text(wtmp1,Value='2',UValue='YCOL',XSize=2,/Edit)
    wXScale = Widget_DropList(wtmp1, UValue='XSCALE', $
      Value=['X * 1','X * 1000','A*X+B ?'])
  wtmp2= Widget_Base(wPLBaseBottom,/Row)
    wtmp= Widget_Button(wtmp2,Value='View Scan',UValue='XSPECVIEW')
    wtmp= Widget_Button(wtmp2,Value='Plot Scan',UValue='XPLOT_LEFT')

;
; Central Pannel
;
wPCCopy1= Widget_Button(wPannelCenter,Value='=>',UValue='COPY1')
wPCCopyAll= Widget_Button(wPannelCenter,Value='*=>',UValue='COPY1')

;
; Right Pannel
; 
wPRBaseTop = Widget_Base(wPannelRight,/Row)
  wtmp = Widget_DropList(wPRBaseTop, UValue='IMPORTRIGHT', $
    Value=['Import from: ',$
           'Current xplot window','Xplot window ID...',$
           'ASCII file(s) [browser]','ASCII files [wildcard/list]', $
	   'Import filter [one file]','Import filter [wildcard]'])
  wtmp = Widget_DropList(wPRBaseTop, UValue='VIEWEDIT', $
    Value=['View/Edit: ','View Info','Edit Title'])
IF Keyword_Set(xafs) THEN BEGIN
  wtmp = Widget_DropList(wPRBaseTop, UValue='EXPORT', $
    Value=['Export: ','ASCII file','SPEC file','SPEC/Mesh file',$
           'Application...','XAID/mu2chi','XAID/ff','DELIA binary file'])
ENDIF ELSE BEGIN
  wtmp = Widget_DropList(wPRBaseTop, UValue='EXPORT', $
    Value=['Export: ','ASCII file','SPEC file','SPEC/Mesh file',$
           'Application...'])
ENDELSE
wPRList = Widget_List(wPannelRight,value=SIndGen(yListSize),UValue='PRLIST',$
  ysize=yListSize,xsize=xListSize,/Multiple)
wPRBaseBottom = Widget_Base(wPannelRight,/Col)
  wtmp1= Widget_Base(wPRBaseBottom,/Row)
    wtmp= Widget_Button(wtmp1,Value='*Delete',UValue='DELETEALL')
    wtmp= Widget_Button(wtmp1,Value='Delete',UValue='DELETE')
    wtmp= Widget_Button(wtmp1,Value='Plot',UValue='XPLOT_RIGHT')
  wtmp2= Widget_Base(wPRBaseBottom,/Row)
    wtmp = Widget_DropList(wtmp2, UValue='OPERATIONS', $
      Value=['Operations: ','Interpolation','Average','Sum','Customized',$
	'Fit LinCom'])


;
; set widget values
;
Widget_Control,wPLList,Set_value=['']
Widget_Control,wPRList,Set_value=['']
IF xafs EQ 1 THEN BEGIN
  xScale=[1.0D3,0.0D0]
  Widget_Control,wXScale,Set_DropList_Select=1 
ENDIF ELSE BEGIN
  xScale=[1.0D0,0.0D0]
  Widget_Control,wXScale,Set_DropList_Select=0
ENDELSE
;
; parameters to store
;
dir=0 & cd,current=dir
ptrSpecFile = Ptr_New()
;ptrMPP = MPP_New()
wids = {Base: wBase, PLList:wPLList, PRList:wPRList, xCol:wXCol, yCol:wYCol}
pref = {rascii_skip:0, import_filter:'y=read_edf(file)'}
state = {wids:wids, pref:pref, no_block:no_block, ptrSpecFile:ptrSpecFile, $
  ompp:Obj_New('ompp'), Xplot:0L, xScale:xScale, $
  specFile:'', importDir:dir, xafs:xafs, wildfile:'*.txt' }
  ; ptrMPP:ptrMpp, Xplot:0L, XplotLeft:0L, xScale:xScale, $

;
;
widget_control,Widget_Info(wbase,/Child),set_uvalue=state ; ,/no_copy

widget_control,wbase,/realize
xmanager,'exodus',wbase,GROUP=group,No_Block=no_block
end
