
;+
; NAME:
;	KGAP_TOOL
;
; PURPOSE:
;	Calculates and displays either the K value or the Gap value
;	of an synchrotron radiation insertion device source.
;
; CATEGORY:
;	XOP Widgets.
;
; CALLING SEQUENCE:
;
;	KGAP_TOOL [, keywords] 
;
; INPUTS:
;
; KEYWORD PARAMETERS:
;	inpstr:	A structure containg at leas the KY (k-value) tag and 
;		the Period tag. 
;
;	Br:	The remanant field value.
;
;	Bp:	The Bp parameters
;
;	K:	The K value (overwrites the one in inpstr).
;
;	Period:	The Period value (overwrites the one in inpstr).
;
;	GROUP:	The parent widget id.
;
; OUTPUTS:
;	Updates the input and keywords with the calculated values.
;
; PROCEDURE:
;	Interactively apply the formulas:
;	Bo = Br exp(-Bp G / P) 
;	K = 93.4 P[m] Bo[T]
;
; EXAMPLE:
;	kgap_tool,inpstr={KY:1.9,Period:0.042},bp=2.0
;
; MODIFICATION HISTORY:
; 	Written by: Manuel Sanchez del Rio. (srio@esrf.fr) November 1995
;	97/11/10 srio@esrf.fr makes kgapt_tool a separate routine in xop.
;		It used to belong to xurgent. Adds doc.
;       01/01/17 srio@esrf.frslightly updates help text
;-

;===== 
; kgap_tool: tool to calculate k as a function of gap
;
; MSR November 1995
;=====

;
;=================================================================
;
FUNCTION kgap_tool_calc,get,p,br,bp,K=k,G=g, string=string
;
; FUNCTION kgap_tool_k_calc,get,p,br,bp,K=k,G=g, string=string
;  calculates the undulator Bo and K as a function of the
;	period[m], remanent field, Bp and gap [mm] or 
;  calculates the undulator Bo and G as a function of the
;	period, remanent field, Bp and K 
;
; usage result=kgap_tool_calc('K',0.04,1.57,!pi,G=20) or 
;       result=kgap_tool_calc('G',0.04,1.57,!pi,K=1.5)

if N_params() NE 4 then begin
  print,"kgap_tool_calc: Usage: result=kgap_tool_calc('K',0.04,1.57,!pi,G=20)"
  return,0
endif

; convert from text to float
P = float(P)
Br= float(Br)
Bp= float(Bp)
if keyword_set(K) then K = float(K)
if keyword_set(G) then G = float(G)

; gap mm-> m
if keyword_set(G) then G1 = G/1000.

;calculation of gap
if keyword_set(K) then G1 = (P/Bp) * alog(K/(93.4*P*Br))

; calculation of Bo
if keyword_set(G) then Bo = Br * exp(-Bp*G1/P)
if keyword_set(K) then Bo = K /93.4/P

; calculation of K
if not(keyword_set(k)) then K = 93.4 * P * Bo

;calculation of gap
if keyword_set(K) then G1 = (-1.*P/Bp) * alog(Bo/Br)



case strupcase(get) of
  'BO': out = Bo
  'G':  out = G1*1000.
  'K':  out = K
  'P':  out = P
  'BR':  out = Br
  'BP':  out = Bp
endcase

if keyword_set(string) then out = strcompress(out,/rem)
return,out
end

;
;=================================================================
;
pro kgap_tool_event,event
;
widget_control,event.id,get_uvalue=uvalue
widget_control,event.top,get_uvalue=state

if uvalue EQ 'CANCEL' then  begin
  widget_control,event.top,/DESTROY
  return
endif

if uvalue EQ 'DONE' then  begin
  widget_control,state.wK,get_value=K
  widget_control,state.wP,get_value=P
  widget_control,state.wBr,get_value=Br
  widget_control,state.wBp,get_value=Bp
  widget_control,state.wBaseData,get_uval = data
  data.inpstr.period = float(P(0))
  data.inpstr.ky = float(K(0))
  data.Br = float(Br(0))
  data.Bp = float(Bp(0))
  widget_control,state.wBaseData,set_uval = data
  widget_control,event.top,/DESTROY
  return
endif

widget_control,state.wK,get_value=K
widget_control,state.wG,get_value=G
widget_control,state.wP,get_value=P
widget_control,state.wBr,get_value=Br
widget_control,state.wBp,get_value=Bp


case uvalue of 
  'K': begin
       G = kgap_tool_calc('G',P,Br,Bp,K=K)
       Bo = kgap_tool_calc('Bo',P,Br,Bp,K=K)
       end
  'G': begin
       K = kgap_tool_calc('K',P,Br,Bp,G=G)
       Bo = kgap_tool_calc('Bo',P,Br,Bp,G=G)
       end
  'P': begin
       K = kgap_tool_calc('K',P,Br,Bp,G=G)
       Bo = kgap_tool_calc('Bo',P,Br,Bp,G=G)
       end
  'BR': begin
       K = kgap_tool_calc('K',P,Br,Bp,G=G)
       Bo = kgap_tool_calc('Bo',P,Br,Bp,G=G)
       end
  'BP': begin
       K = kgap_tool_calc('K',P,Br,Bp,G=G)
       Bo = kgap_tool_calc('Bo',P,Br,Bp,G=G)
       end
  'HELP': begin
	text = [' ',$
	'This applications calculates the Insertion Device Peak Magnetic ',$
	'field and K value as a function of the gap. For that it uses a ',$
        'variation of the semiempirical formula given in Refs 1 and 2. ',$
        '',$
        'Bo is parametrizated in the following way:',$
	'Bo= Br exp(- Bp G /P)  and K = 93.4 P[m] Bo[T], where ',$
	'  Bo is the Peak Magnetic Field (in Tesla), ',$
	'  Br is a parameter proportional to the Magnet Remanent Field, ',$
	'  G is the gap, P is the Insertion Device Period (same units as G)',$
	'  Bp is a parameter value (close to Pi) and K is the ID K-value.',$
	'  ',$
	'The application allows to change interactively a given parameters',$
	'and calculates automatically the others.', $
        'Note that this formula is not universal. For accurate values ',$
        'refer to the documentation of the insertion device in use.',$
        '', $
        'References: ',$
        '1) ESRF Red Book, pag CIV-314',$
        '2) K. Halbach, J. Phys. (Paris) Colloq. C1, 2 Vol 44 211 (1983)']
	
	 Xdisplayfile1,text=text,group=event.top,title='K(Gap) Tool Help'
	return
	end
endcase


widget_control,state.wK,set_value=strcompress(K,/rem)
widget_control,state.wG,set_value=strcompress(G,/rem)
widget_control,state.wP,set_value=strcompress(P,/rem)
widget_control,state.wBr,set_value=strcompress(Br,/rem)
widget_control,state.wBp,set_value=strcompress(Bp,/rem)

widget_control,state.wBo,set_value= $
  'Peak field [T] is:        Bo = Br exp(-Bp G / P) = '+$
  strcompress(Bo(0),/rem)

;
end
;
;
;
;=================================================================
;
pro kgap_tool,inpstr=inpstr, group = group, Br=Br, K=K, Period=Period, $
 Bp=Bp

if not(keyword_set(inpstr)) then $
  ;inpstr=xop_defaults('xurgent')
  inpstr={KY:1.7,Period:0.046}

if keyword_set(K) then inpstr.Ky=K
if keyword_set(Period) then inpstr.Period=Period
if not(keyword_set(Br)) then Br = 1.57 else Br = float(Br)
if not(keyword_set(Bp)) then Bp = !pi else Bp = float(Bp)
;
; widget def
;
wBaseData = widget_base()
wbase = widget_base(title='K(gap) tool',/col)
K = inpstr.ky
P = inpstr.period
Bo = kgap_tool_calc('Bo',P,Br,Bp,K=K,/string)
G = kgap_tool_calc('G',P,Br,Bp,K=K,/string)

wbuttons = widget_base(wbase,/ROW)
wcancel = widget_button(wbuttons,VALUE='Cancel',UVALUE='CANCEL')
wdone = widget_button(wbuttons,VALUE='Accept',UVALUE='DONE')
whelp = widget_button(wbuttons,VALUE='Help',UVALUE='HELP')

wtxt = widget_base(wbase,/COL)


wtmpbase = widget_base(wtxt,/COL,/FRAME)
wtmp = widget_label(wtmpbase,/align_left,VALUE= $
  'Deflection Parameter is:  K = 93.4 P[m] Bo[T]')
wBo = widget_label(wtmpbase,/align_left,VALUE= $
  'Peak field [T] is:        Bo = Br exp(-Bp G / P) = '+strcompress(Bo))


wtmp = widget_label(wtxt,VALUE='Modify value and type <return>:')
wtmpbase = widget_base(wtxt,/ROW,/FRAME)
wtmp = widget_label(wtmpbase,VALUE='K: Deflection Par:                    ')
wK = widget_text(wtmpbase,/EDIT,UVALUE='K',VALUE=strcompress(K,/rem) )

wtmpbase = widget_base(wtxt,/ROW,/FRAME)
wtmp = widget_label(wtmpbase,VALUE='G: Magnet Gap [mm]:                   ')
wG = widget_text(wtmpbase,/EDIT,UVALUE='G', VALUE=G)

wtmpbase = widget_base(wtxt,/ROW,/FRAME)
wtmp = widget_label(wtmpbase,VALUE='P: Insertion Device Period [m]:              ')
wP = widget_text(wtmpbase,/EDIT,UVALUE='P', $
	VALUE=strcompress(P,/rem))

wtmpbase = widget_base(wtxt,/ROW,/FRAME)
wtmp = widget_label(wtmpbase,VALUE='Br: Cte x Magnet Remanent Field [T]:  ')
wBr = widget_text(wtmpbase,/EDIT,UVALUE='BR', $
	VALUE=strcompress(Br,/REM) )

wtmpbase = widget_base(wtxt,/ROW,/FRAME)
wtmp = widget_label(wtmpbase,VALUE='Bp:                                   ')
wBp = widget_text(wtmpbase,/EDIT,UVALUE='BP', $
	VALUE=strcompress(Bp,/REM) )

;
; state
;
state = { wBaseData:wBaseData , wK:wK, wG:wG, wP:wP, wBr:wBr, wBo:wBo, wBp:wBp}

Widget_control,wbasedata,set_uval={inpstr:inpstr, Br:Br, Bp:Bp}
Widget_control,wbase,set_uval=state
widget_control,wbase,/REALIZE
xmanager, 'kgap_tool', wbase, group = group, /modal
;
; extract modified data
;
Widget_control,wbasedata,get_uval=data
inpstr = data.inpstr
Br = data.Br
Bp = data.Bp
if keyword_set(K) then K=inpstr.Ky
if keyword_set(Period) then Period=inpstr.Period

end


