pro HISTO1, shadow_in, col, shadow_out,XRANGE=xrange, YRANGE=yrange, $
NBINS=nbins,NOLOST=nolost, REF=ref,GAUSSFIT=gauss, $
TITLE=title,XTITLE=xtitle,YTITLE=ytitle,OVER=over,WRITE=write,  $
fwhm_fit=fwhm_fit,CALFWHM=CALFWHM,FWHM_VAL=fwhm_val,SAVE=save, $
X0=x0,PUBLISH=publish, GROUP=group
;+
; NAME:
;	HISTO1 
; PURPOSE:
;	an IDL implementation of the HISTO1 SHADOW's utility
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	histo1, shadow_in, col, shadow_out, keyword_parameters
; INPUTS:
;	shadow_in   IDL structure with SHADOW data
;	col   column to analyze (see doc in PLOTXY for a column description)
; OPTIONAL INPUTS:
;	shadow_out  name of the IDL-SHADOW structure where store the file
; KEYWORD PARAMETERS:
;	XRANGE = [min,max], limits of the histogram. 
;	YRANGE = [min,max], heigth limits of the histogram. 
;	NBINS = number of bins (def = 25)
;	NOLOST = consider losses
;		0 all rays (default option)
;		1 exclude losses 
;		2 consider only losses 
;	REF = include reflectivity
;		0 no (default)
;		1 yes (A**2 as weight=column 23)
;               n (21<n<34, n=column to weight (see PLOTXY for col description))
;	GAUSSFIT = makes a gaussian fit of the histo
;		0  No (default)
;		1  Yes: using gauss_fit routine
;		2  Yes: using gaussfit routine
;       TITLE = top title
;       XTITLE = title in X column
;       YTITLE = title in Y column
;       OVER   = overplots the previous histogram when set to 1
;       WRITE  = when set to 1 writes the histogram in the file HISTO1
;	CALFWHM = if set to 1 then calculates the fwhm of the histogram
;	FWHM_VAL = fwhm_val stores the FWHM value of the histogram in
;	          an IDL varianle
;	FWHM_FIT = fwhm_fit stores the FWHM value of the gauss fit 
;	           in an IDL varianle
;	IDL_VAR = idl_var srores the histogram in an IDL variable
;	SAVE = a named variable where to save the histogram (if
;		defined as a single element) or to accumulate the 
;		histogram in sucessive runs.
;	X0 = a numerical value to be deduced from the X array (useful 
;		when one wants to plot vs. E-Eo, for example).
;	PUBLISH = flag to supress graphic output not suitable for publication, 
;		i.e. username, date, time, filepath, etc.
;		When flag is present and non-zero, output is supressed.
;       GROUP The parent id for the caller widget (used to positioning
;               error and information windows).
;
;	OUTPUTS:
;	an histogram on the screen.
; OPTIONAL OUTPUT PARAMETERS:
;	shadow_out  name of the IDL-SHADOW structure where store the file
; COMMON BLOCKS:
;	Calls histogramw
; SIDE EFFECTS:
; 	None.
; RESTRICTIONS:
; 	None.
; KNOWN BUGS:
;	It has been found that the routine (or idl) gives bad results
;	when plotting vs col 11 using very small interval (very high
;	resolution). A temporal solution of this problem is to use the X0
;	keyword.
; PROCEDURE:
;	Similar to SHADOW's original.
; EXAMPLES:
;	1) Simplest run:
;		histo1,'begin.dat',1
;	2) Save variable:
;		histoall = 1
;		histo1,'begin.dat',1,xrange=[-1,1],save=histoall
;		plotfile,histoall,psym=10 ; to check if it contains data
;		histo1,'begin2.dat',1,xrange=[-1,1],save=histoall ;add
;							; a new histogram
;		(xrange must be defined to force histo1 to produce the
;		same gridding for the X values)
;
; MODOFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble May 1991
;       11-12-91 MSR output for Sigma in fit option
;       01-09-92 MSR  change input type.
;       03-09-92 MSR  introduce new columns. Few changes
;       02-10-92 MSR  added GETPHASE to correct phases for stokes parameters
;       06-10-92 MSR  added YRANGE and WRITE OPTIONS
;       19-04-93 MSR  changes GAUSSFIT by GAUSS_FIT
;       14-05-93 MSR  allows using either gaussfit or gauss_fit
;       29-07-93 MSR  adds CALFWHM,FWHM_VAL and FWHM_FIT
;       13-06-95 MSR  used getshcol, and allows external ref array.
;		      Use the histosh function.
;	26-06-96 MSR Adds save option. Speeds-up printing of time, date...
;	02-04-97 MSR Adds X0 keyword. Cosmetics.
;	03-03-98 MSR Adds PUBLISH keyword. Adapts to xsh_histo1 interface.
;	07-04-98 MSR uses column_name to display it. 
;	13-06-01 MSR sets the minimum for the plot different from zero
;		for the default case. Useful for negative values, like S3
;-
;on_error,2

catch, error_status
 if error_status ne 0 then begin
   catch, /cancel
   message,/info,'Error caught: '+!err_string
   if sdep(/w) then itmp = dialog_message(/Error,Dialog_Parent=group,$
	'HISTO1: Error caught: '+!err_string)
   goto,out
endif

p_position_old=!p.position
p_charsize_old=!p.charsize
;
; set graphic environment and font's size independently from device
;
if n_params() LT 2 then begin
  message,/info,'Usage: histo1,shadow_input,column'
  return
endif


!p.position = [.1,.1,.9,.9]
bfy = float(!d.y_ch_size)/!d.y_size
bfx = float(!d.x_ch_size)/!d.x_size
tfact = 2.0
xsize = tfact*0.0065/bfx
ysize = tfact*0.018/bfy
!p.charsize = min([xsize,ysize]) 
;
; defaults
;
if not(keyword_set(nbins)) then  nbins = 25
if not(keyword_set(title)) then  title = ' '
if (keyword_set(ref) and n_elements(ref) EQ 1) then begin
  if ref eq 1 then ref = 23
endif
if not(keyword_set(ytitle)) and not(keyword_set(publish)) then  begin
  if keyword_set(ref) then begin
    if n_elements(ref) EQ 1 then $
       ytitle = 'Histogram weight column = '+column_name(ref) else $
       ytitle = 'Histogram weight column: EXTERNAL' 
  endif else begin
    ytitle = 'Intensity [arb. units]'
  endelse
endif
if not(keyword_set(xtitle)) and not(keyword_set(publish)) then  begin
  xtitle = 'Column '+column_name(col)
  if keyword_set(x0) then xtitle=xtitle+' - '+strcompress(x0,/rem)
endif
;
; load shadow-idl structure and define arrays and constants
;
shadow_out = readsh(shadow_in,GROUP=group)
if type(shadow_out) NE 8 then message,'bad input'
nelem = shadow_out.npoint
flg = getshcol(shadow_out,10)
arr = getshcol(shadow_out,col,nolost=nolost)
if keyword_set(x0) then arr = arr-x0
;
if not(keyword_set(xrange)) then xrange=[min(arr),max(arr)]
if keyword_set(ref) then begin
  if n_elements(ref) EQ 1 then $
  cp =  getshcol(shadow_out,ref,nolost=nolost) else $
  cp = ref( where(flg > 0) )
endif else begin
  cp= arr*0.+1.
endelse

hh = histosh(arr,cp,nbins=nbins,xrange=xrange)
if not(keyword_set(yrange)) then yrange=[min(hh(1,*))<0,max(hh(1,*))]
if keyword_set(over) then begin
  oplot,hh(0,*),hh(1,*),psym=10
endif else begin
  plot,hh(0,*),hh(1,*),psym=10,xrange=xrange,yrange=yrange, $
     xtitle=xtitle,ytitle=ytitle
endelse

;
; writes file
;
if keyword_set(write) then begin
openw,Unit,'HISTO1',/get_lun
printf,Unit,hh
free_lun,Unit
message,/info,'File HISTO1 written to disk.'
endif
if keyword_set(calfwhm) then begin
 fwhm_val = getfwhm(hh,/plot)
 fwhm_text='FWHM of the histogram is : '+strcompress(fwhm_val,/rem)
 message,/info,fwhm_text
endif
;
; saves file
;
if N_Elements(save) GT 0 then begin
  if n_elements(save) eq 1 then save = hh else begin
    if n_elements(hh) NE n_elements(save) then $
	message,'Incompatible dimensions for saving data'
    save(1,*) = save(1,*) + hh(1,*)
  endelse
endif
;
; gauss fit
;
if keyword_set(gauss) then begin
 hx = reform(hh(0,*))  &  hy = reform(hh(1,*))
 if (gauss eq 1) then begin
  yfit = gauss_fit (hx,hy,afit)
  oplot, hx,yfit
 endif
 if (gauss eq 2) then begin
  yfit = gaussfit (hx,hy,afit)
  zgauss = (hx-afit(1))/afit(2)
  hfit2 = (exp(-zgauss*zgauss/2))*afit(0)
  oplot, hx,hfit2
 endif
 message,/info,'SIGMA of gaussian fit is '+strcompress(afit(2),/rem)
 text_fit = 'FWHM  of gaussian fit is: '+StrCompress( 2.35*afit(2),/Rem)
 message,/info, text_fit
 fwhm_fit= 2.35*afit(2)
 endif
;
; titles
;
if not(keyword_set(over)) and not(keyword_set(publish)) then begin
  ;spawn,'date',/noshell,date
  date = systime()

  cd,current=pwd
  ;spawn,'pwd',/noshell,pwd
  ;spawn,'echo $USER',user,/sh
  if sdep() EQ 'UNIX' then begin
    user_host = getenv('USER')+'@'+ getenv('HOST')
    host = getenv('HOST')
  endif else user_host=''
  subtitle=pwd+sdep(/ds)+shadow_out.name+'  '+date+'  '+user_host
  xyouts,.1,.94,/norm,subtitle
  tsize = !p.charsize * 1.43
  xyouts,.1,.97,/norm,siz=tsize,title
  ;if keyword_set(gauss) then xyouts,.1,.91,/norm,text_fit
  if keyword_set(gauss) or keyword_set(calfwhm) then begin
    ttext=''
    if keyword_set(gauss) then ttext=text_fit
    if keyword_set(calfwhm) then ttext=ttext+'    '+fwhm_text
    xyouts,.1,.91,/norm,ttext
  endif
endif

OUT:
!p.position = p_position_old
!p.charsize = p_charsize_old
end
