function getshcol,shadow_in,col, NOLOST=nolost
;+
; NAME:
;	GETSHCOL
; PURPOSE:
;	extract a column array(s) from a SHADOW file or structure.
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	output=getshcol(shadow_in,col,keyword_parameters)
; INPUTS:
;	shadow_in   may be:
;           a shadow file (between quotes) (e.g. 'begin.dat') or
;           an IDL-SHADOW structure
;	col    number of SHADOW column to extract or array with
;		the SHADOW columns to extract.
;           the colum meaning is the following:
;           1   X spatial coordinate [user's unit]
;           2   Y spatial coordinate [user's unit]
;           3   Z spatial coordinate [user's unit]
;           4   X' direction or divergence [rads]
;           5   Y' direction or divergence [rads]
;           6   Z' direction or divergence [rads]
;           7   X component of the electromagnetic vector (s-polariz)
;           8   Y component of the electromagnetic vector (s-polariz)
;           9   Z component of the electromagnetic vector (s-polariz)
;          10   Lost ray flag
;          11   Energy [eV]
;          12   Ray index
;          13   Optical path length
;          14   Phase (s-polarization)
;          15   Phase (p-polarization)
;          16   X component of the electromagnetic vector (p-polariz)
;          17   Y component of the electromagnetic vector (p-polariz)
;          18   Z component of the electromagnetic vector (p-polariz)
;          20   R= SQRT(X^2+Y^2+Z^2)
;          21   angle from Y axis
;          22   the magnituse of the Electromagnetic vector
;          23   |E|^2 (total intensity)
;          24   total intensity for s-polarization
;          25   total intensity for p-polarization
;          30   S0-stokes = |Es|^2 + |Ep|^2
;          31   S1-stokes = |Es|^2 - |Ep|^2
;          32   S2-stokes = 2 |Es| |Ep| cos(phase_s-phase_p)
;          33   S3-stokes = 2 |Es| |Ep| sin(phase_s-phase_p)
; KEYWORD PARAMETERS:
;        NOLOST = consider losses
;            0 all rays (default option)
;            1 exclude losses;
;            2 only losses;
; OUTPUTS:
;	an array with the extracted column(s)
; COMMON BLOCKS:
;	None.
; SIDE EFFECTS:
;	None.
; RESTRICTIONS:
;	Nono.
; PROCEDURE:
;	Straightforward.
; KNOWN BUGS:
;	None.
; MODIFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, June 1995
;	98-06-15 srio@esrf.fr fixes a bug when extracting an array of
;		columns being Energy one of those. Cosmetics.
;-
on_error,2
;
; load shadow-idl structure 
;
str = readsh(shadow_in,nolost=nolost)
if not(keyword_set(str)) then return,0
shadow_out = str

npoint = str.npoint
ncol = str.ncol
;
; extends rays with extra columns and defines output
;
extracols = where(col GT 18,nextracols)

if (nextracols EQ 0) then output = str.ray(col-1,*) else begin
   raysmore = fltarr(33,npoint)
   raysmore(0:ncol-1,*) = str.ray
   raysmore(0:ncol-1,*) = str.ray

   r_pol = sqrt(str.ray(0,*)^2+str.ray(1,*)^2+str.ray(2,*)^2)
   angle_pol = acos(str.ray(4,*))
   as_pol = sqrt(str.ray(6,*)^2+str.ray(7,*)^2+str.ray(8,*)^2)
   as2_pol = str.ray(6,*)^2+str.ray(7,*)^2+str.ray(8,*)^2
   if str.ncol eq 18 then begin
     ap_pol = sqrt(str.ray(15,*)^2+str.ray(16,*)^2+str.ray(17,*)^2)
     ap2_pol = str.ray(15,*)^2+str.ray(16,*)^2+str.ray(17,*)^2
     phase = getphase(str)
   endif else begin
     ap_pol = as_pol*0.0
     ap2_pol = as2_pol*0.0
     phase=fltarr(4,npoint)
   endelse
  cos_pol = cos(phase(0,*)-phase(1,*))
  sin_pol = sin(phase(0,*)-phase(1,*))
  
  raysmore(19,*) =  r_pol
  raysmore(20,*) =  angle_pol
  raysmore(21,*) =  sqrt(as2_pol+ap2_pol)  ;sqrt(int)
  raysmore(22,*) =  as2_pol+ap2_pol
  raysmore(23,*) =  as2_pol
  raysmore(24,*) =  ap2_pol

  raysmore(29,*) =  as2_pol+ap2_pol
  raysmore(30,*) =  as2_pol-ap2_pol
  raysmore(31,*) =  2*as_pol*ap_pol*cos_pol
  raysmore(32,*) =  2*as_pol*ap_pol*sin_pol

  output = raysmore(col-1,*) 
endelse
;
; if a column is energy, then change units to eV
;
ctmp = where(col eq 11,count)
if count GT 0 then begin
  for i=0L,n_elements(ctmp)-1 do begin
    message ,/info ,'GETSHCOL: Energy column selected: set units to eV'
    output(ctmp[i],*) = output(ctmp[i],*) $
	/6.283185307179586476925287D0*1.239852D-4
  endfor
endif
;
return,output
end
