FUNCTION MPP_LCFunct,x,a, init=mpp, index=index

;+
; 
; NAME: 
;	MPP_LCFunct
;
; PURPOSE:
;	Customized function for being used by MPP_LCFit
;	It returns the CL:
;	a(i)*y(i), with normalization Total(a)=1
;
;
; CATEGORY:
;	MPP utilities to handle multiple array data
;
; CALLING SEQUENCE:
;	out = MPP_LCFunct(x,a)
;
; INPUTS:
;	x: the abscissas array
;	a: the parameters array. 
;		or a pointer to a structure.
;
; KEYWORD PARAMETERS:
;	init: set this keyword to a valid mpp handle to initialize
;		the function. This keyword must be used before 
;		calling the function with arguments x,a
;
; EXAMPLES:
;	mpp = MPP_New()
;	x=findgen(100)
;	y=x*x+100
;	y1=x*x+150
;	mpp_set,mpp,Value=Make_Set(x,y),/Add
;	mpp_set,mpp,Value=Make_Set(x,y1),/Add
;	tmp = mpp_info(mpp,/Verbose)
;	tmp = mpp_lcfunct(init=mpp)
;	x1=x+0.1
;	plot,x,y,pSym=1
;	oplot,x,y1,pSym=1
;	yo  = mpp_lcfunct(x1,[5,5])
;	oplot,x1,yo,color=2
;	mpp_free,mpp
;
; MODIFICATION HISTORY:
;	Initial version by M. Sanchez del Rio, February 2000.
;	00-05-10 srio@esrf.fr uses Absolute value of coefficients.
;
;-

COMMON MPP_LC, mppstr


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'MPP_LCFUNCT: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF

IF N_Elements(mpp) GT 0 THEN BEGIN ; initialize
  mppType = Type(mpp)
  IF mppType EQ 10 THEN BEGIN
    IF Ptr_Valid(mpp) EQ 0 THEN Message,'MPP_Value: input not valid.'
    mppstr=*mpp 
  ENDIF ELSE mppstr=mpp
  RETURN,0
ENDIF

IF N_Elements(mppstr) EQ 0 THEN BEGIN
  Message,'First initialize with INIT keyword'
  RETURN,0
ENDIF


IF N_Elements(index) EQ 0 THEN ind=LIndGen(N_Elements(a)) ELSE ind=index

set = mpp_value(mppstr,ind[0])
y = Interpolate(Reform(set[1,*]),FIndex(Reform(set[0,*]),x))
aa = a/Double(Total(a))
; print,'a: ',a
; print,'aa: ',aa
; print,' '
out = aa[0]*y
IF N_ELEMENTS(a) GT 1 THEN BEGIN
  FOR i=1,N_Elements(aa)-1 DO BEGIN
    set = mpp_value(mppstr,ind[1])
    y = Interpolate(Reform(set[1,*]),FIndex(Reform(set[0,*]),x))

    out = out + Abs(aa[i])*y
  ENDFOR
ENDIF

RETURN,out
END
