
;+
; NAME:
;	ROWLAND
; PURPOSE:
;	Calculates the tangential and sagittal focusing radii, and 
;	Rowland condition for gratings and asymmetric crystals.
; CATEGORY:
;	Optics
; CALLING SEQUENCE:
;	rowland
; KEYWORD PARAMETERS:
;    GROUP
;	The widget id of the parent. Also used for centered the input/output 
;	windows.
; OUTPUTS:
;	The outputs are shown in a text window.
; COMMON BLOCKS:
;	COMMON rowland str_rowland   
;	(to store the inputs for following runs)
; SIDE EFFECTS:
;	None.
; RESTRICTIONS:
;	If the IDL session support widgets, the i/o is done via widgets, 
;	otherwise it is done by terminal.
;	None.
; PROCEDURE:
;	Use well known formulas.
; MODIFICATION HISTORY:
;	Created 2001-03-08 by M. Sanchez del Rio from a c code
;	01/03/14 srio@esrf.fr force to use cm and eV
;-
;/****************************************************************************
; *  
; *  File:                 rowland.c
; *
; *  Project:              X-ray optics utilities
; *
; *  Description:          Calculates the sagittal and tangential radii
; *                        for gratings or systems in which the incident
; *                        and reflected angles are different. It also
; *                        calculates the Rowland conditions and the step 
; *                        of the grating for a given energy.
; *
; *  Author:               Manolo Sanchez del Rio 
; *
; *  Date:                 December 1991
; *
; *****************************************************************************/

PRO rowland, Group=group

on_error,2

COMMON rowland,str_rowland

TOANGS = 12.39852D3
TORAD  = 0.017453292519943295769237D0

;
; Inputs
;
IF SDep(/w) EQ 1 THEN BEGIN
   IF N_Elements(str_rowland) EQ 0 THEN BEGIN
   str_rowland = {i_flagMenu:['1','Incident and reflected angle',$
			  'Energy for asymmetrical cut crystals'], $

	  t1:5.0, t2:5.0, $

	  e:10000.0, $
          nmenu:['0','Si 111','Si 220','Ge 111','Ge 220','Si 331','Si 777'],$
	  alfa:0.0, $

	  d1:3000.0, d2:3000.0 }

   ENDIF
   flags = ['1', $
	'w(0) EQ 0','w(0) EQ 0',$
	'w(0) EQ 1','w(0) EQ 1','w(0) EQ 1', $
	'1','1']

   titles = ['Select', $
	'incident grazing angle [deg]', 'reflected grazing angle [deg]', $
	'Energy (eV)','crystal','Asymmetric cut angle [deg]', $
	'Distance source-element [cm]','Distance element-image [cm]']
          

   action=''
   XScrMenu,str_rowland,/Inter,/NoType,Flags=flags,Titles=titles,Action=action,$
	Wtitle='Focusing radii', Dialog_Parent=group
   IF action EQ 'DONT' THEN RETURN

   CASE StrCompress(str_rowland.i_flagMenu[0],/Rem) OF
     '0': i_flag='I'
     '1': i_flag='E'
     else:
   ENDCASE
   t1 = str_rowland.t1
   t2 = str_rowland.t2

   e = str_rowland.e
   n = Fix(str_rowland.nmenu[0])
   alfa = str_rowland.alfa
   d1 = str_rowland.d1
   d2 = str_rowland.d2
   

ENDIF ELSE BEGIN

;
; input section 
;
    Print," Do you want to enter ?: "
    Print,"  Incident and reflected angle (i) "
    Print,"  Energy for asymmetrical cut crystals (e) "
    i_flag = ''
    Read,"  ?>", i_flag
    CASE StrCompress(StrUpCase(i_flag),/Rem) OF
      'I' : BEGIN
        Read,"Input incident angle (grazing, in degrees) : ",t1
        Read,"Input reflected angle (grazing, in degrees) : ",t2
	END
      'E': BEGIN
        Read,"Energy (eV) : ", e

        Print," Type of crystal:"
        Print,"       0 - Si 111."
        Print,"       1 - Si 220."
        Print,"       2 - Ge 111."
        Print,"       3 - Ge 220."
        Print,"       4 - Si 331."
        Print,"       5 - Si 777."
        Read,"       ?> ",n

        
        Read,"Asymmetric cut angle [deg] : ",alfa

	END
      default: Message," Urecognized input"
      ENDCASE

      read,"Distance source-element [cm] ? : ", d1
      read,"Distance element-image [cm] ? : ",d2
ENDELSE
;
; end inputs
;
    CASE StrCompress(StrUpCase(i_flag),/Rem) OF
     'I': BEGIN
	t1 = t1*TORAD 
	t2 = t2*TORAD
	END
     'E': BEGIN
        CASE Fix(n) OF
          0: d=3.135
          1: d=1.92
          2: d=3.262
          3: d=1.997
          4: d=1.246
          5: d=0.447946
          else: Message," Urecognized input "
        ENDCASE
	theta=asin(TOANGS/(e*2.0*d))

	alfa = alfa*TORAD
        t1 = theta+alfa
        t2 = theta-alfa
	END
      else: Message," Urecognized input "
    ENDCASE

;    /*
;     * Calculations
;     */
    s1 = sin(t1)
    s2 = sin(t2)
    s1_2 = s1*s1
    s2_2 = s2*s2
    r = s1_2/d1 + s2_2/d2
    r = (s1+s2)/r
    rs = (s1+s2)/(1.0/d1 + 1.0/d2)

;    /*
;     * Present output.
;     */
    out = [ " ******************************************************", $
    " energy is "+StrCompress(e,/Rem)+' eV',$
    " d-spacing is "+StrCompress(d)+ ' A', $
    " Bragg angle angle is "+StrCompress(theta/TORAD)+' degrees', $
    " Incident angle is "+StrCompress(t1/TORAD)+' degrees', $
    " Reflected angle is "+StrCompress(t2/TORAD)+' degrees', $
    " ", $
    " Tangential radius is "+StrCompress(r)+' cm', $
    " Sagittal radius is "+StrCompress(rs)+' cm', $
    " ", $
    " The Rowland Circle conditions are: ", $
    "     R1 = "+StrCompress(r*s1)+' cm', $
    "     R2 = "+StrCompress(r*s2)+' cm', $
    " ", $
    " For s1 = "+StrCompress(d1)+ "the Rowland Circle conditions are: ", $
    "     s2 = "+StrCompress(d1*s2/s1)+" cm ", $
    "      r = "+StrCompress(d1/s1)+" cm ", $
    " ******************************************************", $
    " " ]


    IF SDep(/w) THEN $
    XDisplayFile1,Text=out, Dialog_Parent=group, Title='Rowland' 

    FOR i=0,N_Elements(out)-1 DO Print,out[i]

;    Print, " Do you want to calculate the ruling of the grating? [y/n] ");
;    fflush(stdin);
;    scanf("%c",&i2);
;    if (i2=='y'||i2=='Y') {
;    printf(" Energy (eV) ? : ");
;    scanf("%f",&e);
;    dg = TOANGS/e/((float)cos((double)t1)-(float)cos((double)t2)) ;
;    printf(" Tne number of lines by cm is %f",1.0e8/dg);

END
