;+
;
; =================================  nMirror      ============================
;
; nMirror is a widget based graphical interface to calculate Neutron
; Scattering Factors, and their derived parameters
; (refraction index, absorption cross section, reflectivity).
;
; The data are taken from the DABAX data base.
;
; Possible calculations:
;
;	Real(b) [fm]   =  real part of the bound coherent scattering length
;	Imag(b) [fm]   =  imaginary part of the bound coh. sca. length
;	Scat. cross sec. [barns] = Scattering cross section (coherent+incoh)
;	Incoh. cross sec. [barns] = Incoherent scattering cross section
;	Coh. cross sec. [barns] = Coherent scattering cross section
;	Absorption. cross sec. [barns] = Absorption Cross Section
;	delta =  delta [n=1-delta-i beta]
;	beta  =  beta [n=1-delta-i beta]
;	mu [cm^-1]   =  Nuclear absorption coeff (mu [cm^-1)]
;	mu [cm^2/g]  =  Mass  absorption coeff (mu [cm^2/g)
;	Reflectivity =  Mirror reflectivity
;
;
; WARNING: These calculations can be made for unpolarized or polarized
; (Up(+1/2) and Down(-1/2)) neutrons (in the case that polarized
; data are available in the selected database, otherwise the 
; unpolarised data is selected WITHOUT ANY WARNING!!). 
;
;
; Nota: when chosing delta or beta (i.e., refraction index), the
;   program ask is the user wants to create a file fith the refraction index
;   for further use with IMD [multilayers].
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    nMirror input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where
;		"application " is the name of the current NOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;  Quit: to exit from the program
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main nMirror window.
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. After
;		pressing the ACCEPT button, nMirror start running and
;		presents a graphic display with results.
;  Set Defaults: Sets the default parameters.
;
;  Help:   Shows the nMirror help (this text).
;
;
; COPYRIGHT:
;	nMirror  belongs to XOP/NOP package and it is distributed within NOP.
;	PLEASE REFER TO THE XOP/NOP COPYRIGHT NOTICE BEFORE USING IT.
;
; CREDITS:
;	Published calculations made with XOP/NOP should refer:
;
;	  L. Alianelli, M. Sanchez del Rio and R. Felici
;	  "NOP: A new software tool for neutron optics"
;	  To be published in Physica B at the proceedings of the 
;	  ECNS 2003 conference.
;
;
;	  M. Sanchez del Rio and R. J. Dejus "XOP: Recent Developments"
;	  SPIE proceedings vol. 3448, pp.340-345, 1998.
;	  
;
; LAST MODIFICATION: srio@esrf.fr 2003-04-07
;
;
;-
; =========================================================================
;	MODIFICATION HISTORY:
;       by  Manuel Sanchez del Rio. ESRF. September 1996.
;		96/11/22 MSR adds scanning vs theta and scanning vs
;		theta and wavelength options.
;		96/11/25 MSR Implements the Mixture options.
;		97/10/15 MSR makes some changes for xop 1.9.
;		98/12/15 MSR adapts for XOP2.0. Version 1.1
;		01/11/15 C. Boisseau adapting for NOP1.0
;		2003/04/07 srio@esrf.fr adapts for first NOP release.
;
;
Function nMirror_version
return,'1.0'
end
;
;=====================================================================
;
PRO nMirror_event,event

;Catch, error_status
;IF error_status NE 0 THEN BEGIN
;   Message,/Info,'error caught: '+!err_string
;   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
;     'nMirror_EVENT: error caught: '+!err_string)
;   Catch, /Cancel
;   If Type(stateid) EQ 3 THEN $
;     If Widget_Info(stateid,/Valid_Id) AND N_Elements(state) NE 0 THEN $
;     Widget_Control,stateid,Set_UValue=state  ;, /No_Copy
;   On_Error,2
;   RETURN
;ENDIF

Widget_Control, event.id, get_UValue=eventUValue

if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''

stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state   ; , /No_Copy

case eventuvalue of

  'FILEINPUT': BEGIN
    action=''
    Widget_Control,event.id, Get_Value=action
    CASE action OF
      'Load from file...': BEGIN
        if sdep() EQ 'UNIX' then filter='*.nop' else filter=0
        str_par = Xop_Input_Load(Title=$
        'Select nMirror input file...',$
        /NoConf,Filter=filter,Group=event.top)
        IF Type(str_par) EQ 8 THEN BEGIN
          tmp = state.str.parameters
          Copy_Structure,str_par, tmp, Group=event.top , /OnlyFirstField
          state.str.parameters = tmp
        ENDIF
      END
      'Save to file...': BEGIN
        str_par = state.str.parameters
        Xop_Input_Save,str_par,File='nmirror.nop',$
          /Write, Group=event.top, Comment='; nop/nmirror(v'+$
        nmirror_version()+') input file on '+SysTime()
      END
      'Save as default': BEGIN
        str_par = state.str.parameters
        Xop_Input_Save,str_par,Group=event.top, $
          Default='nmirror.nop',Comment='; nop/nmirror(v'+$
        nmirror_version()+') input file on '+SysTime()
      END
    ENDCASE
  END
  'QUIT':begin
	widget_control,/destroy,event.top
	return
	end
  'HELP': xhelp,'nmirror',GROUP=event.top


  'SETDEF': BEGIN
		itmp = Dialog_Message(Dialog_Parent=event.top,$
		/Question,['This option initializes the',$
		'nmirror parameters to their default values.',$
		'Then you must click Set_parameters to run the program.',$
		'Please confirm.'],title='nMirror')
		if itmp eq 'No' then goto,out
		state.str.parameters = state.str_defaults
	END
  'SET':begin
	str = state.str.parameters
	ds = sdep(/ds)
	hfile = Xop_GetEnv('XOP_HOME')+ds+'extensions'+ds+'nop'+ds+'doc'+ds+'nmirror_par.txt'
        helpcmd="xdisplayfilenative,'"+hfile+"'"
        ;helpdir = Xop_GetEnv('DABAX_HELP')
        ;xdisplayfile1,'"+helpdir+$
        ;if helpdir EQ '' then helpcmd=0 else begin
        ;          helpcmd="xdisplayfile1,'"+helpdir+$
        ;         sdep(/ds)+"nmirror.par',group=event.top,/NoMenuBar"
        ;endelse
  	XScrMenu,str,/Interp,/NoType,action=action,Ncol=2, $
		titles=state.str.titles, $
		flags=state.str.flags,help=helpcmd, $
		wtitle='nMirror input parameters'
  	if  action EQ 'DONT' then goto,out
  	widget_control,/hourglass

	state.str.parameters = str

	;
	;mat_flag : 0=element, 1=mixture(form) 2=mixture(table)
	;
	mat_flag = fix(str.mat_flag(0))
	case mat_flag of
	0: begin
	    iname = 0
	    descriptor = strcompress(str.descriptor,/rem)
            if strlen(descriptor) EQ 1 then descriptor=descriptor+' '
	    imix = 0
	   end
	1: begin
	    iname = 0
            descriptor = strcompress(str.descriptor,/rem)
            if strlen(descriptor) EQ 1 then descriptor=descriptor+' '
	    density = str.density
	    imix = 1
	   end
	2: begin
	    iname = 1 & descriptor = str.mat_list(fix(str.mat_list(0))+1)
	    imix = 1
	   end
	endcase

	;
	; 
	;
	polar = Fix(str.polar[0])
	;calc = strcompress(str.calculate[1+Fix(str.calculate(0))],/Rem)

	;
	; define an array with the selected datasets
	;
	nsl_file = strcompress(str.datasets[Fix(str.datasets(0))+1],/Rem)
	if nsl_file EQ 'all' then $
		datasets = str.datasets(2:N_elements(str.datasets)-1) else $
		datasets = nsl_file

	; 
	; wavelength
	;
	case strcompress(str.grid(0),/rem) of
	  '0': wavelength = [str.gridstart]    ; single wavelength
	  '1': begin
		if str.gridn EQ 1 then wavelength = [str.gridstart] else $
		wavelength = MakeArray1(str.gridn,str.gridstart,str.gridend)
		end
	endcase



	;
	; icalc is the magnitude array for calculations (F indices in 
	; nsl_calc.pro )
	; put in f1f2_calc,...,ret=icalc
	icalc = fix(str.calculate(0))
	if icalc eq 0 then icalc=indgen(11)+1

	;
	;flag to know the reflectivity case.
	;
	; itheta=0 no reflectivity calculations
	; itheta=1 reflectivity calculations, single theta value
	; itheta=2 reflectivity calculations, multiple theta value
	itheta = 0
	if n_elements(icalc) GT 1 or icalc(0) GE 11 then begin
	  thetagrid = strcompress(str.thetagrid(0),/rem)
	  if thetagrid EQ '0' then begin
	    theta = str.theta1
	    itheta = 1
	  endif else begin
	    theta = MakeArray1(str.thetan,str.theta1,str.theta2)
	    itheta = 2
	  endelse
	endif

	; refuse to calculate multiple datasets and multiple angles and
	; multiple energies
	IF N_Elements(datasets) GT 1 AND itheta EQ 2 $
	  and N_Elements(wavelength) GT 1 THEN BEGIN
	  itmp = Dialog_Message(dialog_parent=event.top,$
	     /error,['Please do not select ALL datasets '+$
	     'with the reflectivity angular "user defined" option.',$
	     'Too long calculations.'])
	  goto,out
	ENDIF

	; DISPLAY RESULTS 
	; 
	; 3d plot (refl vs wavelength and angle)
	; 
	if (itheta EQ 2 and n_elements(wavelength) GT 1) OR $
	   (itheta EQ 2 and n_elements(wavelength) EQ 0) then begin
	  if imix then $
	    fout = nsl_calc_mix(datasets(0),descriptor,wavelength,ret=icalc(0),$
	    theta=theta,rough=rough,name=iname,density=density,$
	    group=event.top,polar=polar) else $
	    fout = nsl_calc(datasets(0),descriptor,wavelength,ret=icalc(0),$
	    theta=theta,rough=rough,group=event.top,polar=polar)
	  tmp = !p.charsize
	  !p.charsize = 2.0

	  xsurface1,fout,theta,wavelength,xtitle='theta',ytitle='wavelength',$
		group=event.top
	  !p.charsize=tmp
	  goto,out
	endif

	;
	;
	;
	units = [' Real(b) [fm] ', $
	' Imag(b) [fm] ', $
	' Scat. cross sec. [barns] ',$
	' Incoh. cross sec. [barns] ',$
	' Coh. cross sec. [barns] ',$
	' Absorption. cross sec. [barns] ',$
	' delta ', $
	' beta ', $
	' mu [cm^-1] ',$
	' mu [cm^2/g] ',$
	' Reflectivity ']

	;
	; 2d plot (vs wavelength)
	;
	rough = str.rough
	IF ITHETA NE 2 THEN BEGIN ; no reflect or refl with single theta value
	  FOR I = 0,n_elements(datasets)-1 DO BEGIN ; starts main loop
	    my_descriptor=descriptor
	    FOR J = 0,n_elements(icalc)-1 DO BEGIN ; starts loop on magnitudes
	      if imix then begin
	        fout = nsl_calc_mix(datasets(i),descriptor,wavelength,ret=icalc(j),$
	          theta=theta,rough=rough,name=iname,density=density,$
	          group=event.top,polar=polar)
	      endif else begin 
	        fout = nsl_calc(datasets(i),my_descriptor,wavelength,ret=icalc(j),$
	          theta=theta,rough=rough,group=event.top,polar=polar,new_descriptor=new_descriptor)
		my_descriptor=new_descriptor
	      endelse
    	      title= units(icalc(j)-1)

              titlemore = ''
	      if icalc(j) GE 11 then titlemore = $
		  strcompress(theta,/rem)+' mrad; rough rms='+$
	          strcompress(rough,/rem)+' Ang.'

	      if n_elements(labelsY) eq 0 then labelsY = datasets(i)+':'+title $
	          else labelsY = [labelsY,datasets(i)+':'+title]

	      if I EQ 0 AND J EQ 0 then begin
	          out = fltarr(1+n_elements(datasets)*N_elements(icalc),$
	            n_elements(wavelength))

	          out(0,*) = reform(wavelength)
	          out(1,*) = reform(fout)
                  icount = 1
              endif else begin
		  icount = icount + 1
                  out(icount,*) = reform(fout)
              endelse

	      ENDFOR ; ends magnitudes loop
	    ENDFOR ; ends main loop
          coltitles=['Wavelength[A]',labelsY]
          if (size(out))(0) EQ 1 then begin
            tmp = strarr(n_elements(out))
            for ii=0,n_elements(out)-1 do tmp(ii)=coltitles(ii)+'  = '+$
                strcompress(out(ii))
            xdisplayfile1,text=tmp,group=event.top,title='nMirror results'
          endif else begin
	    title = 'Material: '+descriptor+' '
	    if icalc(0) eq 9 then $
	    xplot,out,coltitles=coltitles,Ycol=2,xtitle='-1',ytitle='-1', $
	    group = event.top,title=title+titlemore,$
	    wtitle='nMirror results',no_block=state.no_block  else $
	    xplot,out,coltitles=coltitles,xtitle='-1',ytitle='-1', $
	    group = event.top,title=title+titlemore,$
	    wtitle='nMirror results',no_block=state.no_block
          endelse
	ENDIF ELSE BEGIN ; vs theta
	  FOR I = 0,n_elements(datasets)-1 DO BEGIN ; starts main loop
	    if n_elements(icalc) gt 1 then begin
	      itmp = widget_message(dialog_parent=event.top,$
	      /error,['When a Grazing angle scan is defined, only reflectivity calculations are allowed'])
	      goto,out
	     endif
	    if imix then $
	    fout = nsl_calc_mix(datasets(i),descriptor,wavelength,ret=icalc(0),$
	      theta=theta,rough=rough,name=iname,density=density,$
	      group=event.top,polar=polar) else $
	      fout = nsl_calc(datasets(i),descriptor,wavelength[0],ret=icalc(0),$
	      theta=theta,rough=rough,group=event.top,polar=polar)

	    title= units(icalc(0)-1)
	    titlemore = ''
	    titlemore = 'Wavelength: '+strcompress(wavelength(0),/rem)+' A; rough rms='+$
		strcompress(rough,/rem)+' Ang.'

	    if n_elements(labelsY) eq 0 then labelsY = datasets(i)+':'+title $
	      else labelsY = [labelsY,datasets(i)+':'+title]

	    if I EQ 0 then begin
	      out = fltarr(n_elements(datasets)+1,n_elements(theta))
	      out(0,*) = reform(theta)
	      out(1,*) = reform(fout)
	    endif else out(1+i,*) = reform(fout)
	  ENDFOR ; ends main loop
          coltitles=['Theta[mrad]',labelsY]
          if (size(out))(0) EQ 1 then begin
              tmp = strarr(n_elements(out))
              for ii=0,n_elements(out)-1 do tmp(ii)=coltitles(ii)+'  = '+$
              strcompress(out(ii))
              xdisplayfile1,text=tmp,group=event.top
          endif else begin
	          title = 'Material: '+descriptor+' '
	          xplot,out,coltitles=coltitles,xtitle='-1',ytitle='-1', $
	          group = event.top,title=title+titlemore,no_block=state.no_block
          endelse
	ENDELSE
	; 
	; ----------------- IMD file
	;
	IF icalc[0] EQ 7 OR icalc[0] EQ 8 THEN BEGIN
	  itmp = Dialog_Message(/Question,'Write file for IMD?', $
		Dialog_Parent=event.top)
	  IF itmp EQ 'No' THEN GoTo,out
	ENDIF ELSE GoTo,out

	file = Dialog_PickFile(Title='Write file for IMD', $
	  File='nmirror4imd.nk',Dialog_Parent=event.top )
	IF file EQ '' THEN GoTo,out
	IF checkFIle(file) EQ 1 THEN BEGIN
	  itmp = Dialog_Message(/Question, $
		['Write exists: ',file,'Overwrite it? '], $
		Dialog_Parent=event.top)
	  IF itmp EQ 'No' THEN GoTo,out
	ENDIF

	out = dblarr(3,n_elements(wavelength))
	my_descriptor=descriptor
	if imix then begin
	  fout7 = nsl_calc_mix(datasets[0],descriptor,wavelength,ret=7,$
	  name=iname,density=density,$
	  group=event.top,polar=polar)
	  my_descriptor=new_descriptor
	  fout8 = nsl_calc_mix(datasets[0],descriptor,wavelength,ret=8,$
	  name=iname,density=density,$
	  group=event.top,polar=polar)
	endif else begin 
	  fout7 = nsl_calc(datasets[0],my_descriptor,wavelength,ret=7,$
	  group=event.top,polar=polar, $
	  new_descriptor=new_descriptor)
	  my_descriptor=new_descriptor
	  fout8 = nsl_calc(datasets[0],my_descriptor,wavelength,ret=8,$
	  group=event.top,polar=polar, $
	  new_descriptor=new_descriptor)
	endelse
	out[0,*] = reform(wavelength)
	out[1,*] = reform(fout7)*(-1.0D0)+1.0D0
	out[2,*] = reform(fout8)

	OpenW,unit,file,/Get_Lun
	PrintF,unit,'; file for IMD created by XOP/NOP on: '+SysTime()
	PrintF,unit,'; descriptor: '+descriptor
	PrintF,unit,';        Lambda(A)          n          k '
        PrintF,unit,';--------------------------------------    '
	FOR i=0,N_Elements(wavelength)-1 DO PrintF,unit,out[*,i],Format='(3G19.8)'
	Free_Lun,unit
	Message,/Info,'File written to disk: '+file
	XDisplayFile1,file
	; 
	; ----------------- END IMD file
	;
	end
  else:
endcase

out:
Widget_Control, stateid, set_UValue=state   ; , /No_Copy
END
;
;=======================================================================
;
PRO nMirror,GROUP=group, InputFile=inputFile, No_Block=no_Block
;
Forward_Function nop_defaults
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'NMIRROR: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF
;
wbase = widget_base(/COLUMN,TITLE='NMIRROR '+nmirror_version(),$
        MBAR=wMenuBar)

;
;
; the blocks box
;
Bbox=widget_base(wbase,/Column) ; also to store state
IF N_Elements(no_block) EQ 0 THEN no_block=1


;Menu bar
wFile =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)

  wtmp0 = widget_button(wFile,VALUE='nMirror input parameters', /Menu)
    wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='QUIT',/SEPARATOR)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='SET')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')


wHelpMenu = WIDGET_BUTTON(wMenuBar, VALUE='Help', /HELP)
  wtmp = WIDGET_BUTTON(wHelpMenu, VALUE='nmirror', UVALUE='HELP')

tmp = widget_button(Bbox,VALUE='Set Parameters',UVALUE='SET')


if sdep() EQ 'WINDOWS' then $
font = 'VERDANA*BOLD*ITALIC*24' else $
font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'

junk = WIDGET_LABEL( Bbox, FONT=font, VALUE='nMirror')
junk = WIDGET_LABEL( Bbox, FONT=font, $
  VALUE='Neutron reflectivity')
junk = WIDGET_LABEL( Bbox, FONT=font, $
  VALUE='scattering lengths and cross sections')


str = nop_defaults('nmirror',group=wbase)
str_defaults = str.parameters

IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = str.parameters
    Copy_Structure,str_par, tmp, Group=group, /OnlyFirstField
    str.parameters = tmp
  ENDIF
ENDIF

wids = {dummy:0L}
state = { wids:wids, str:str, str_defaults:str_defaults, no_block:no_block }

widget_control,Widget_Info(wbase,/Child),set_uvalue=state
widget_control,wbase,/REALIZE
xmanager,'nmirror',wbase,GROUP=group, No_Block=no_Block
;
end
