function nbragg_calc,inp,group=group

;+
; NAME:
;       NBRAGG_CALC
;
; PURPOSE:
;	Preprocessor for Structure Factor (FH) calculations.
;	It calculates the basic ingredients of FH.
;
; CATEGORY:
;       Neutron optics.
;
; CALLING SEQUENCE:
;	out = nbragg_calc(inp)
; INPUTS:
;	inp: a structure with the input data. This structure may
;		be created by either daban_defaults('nbragg') or
;		nbragg_inp() and modified by nbragg_inp(ask={1,2}).
;
; KEYWORDS:
;
; OUTPUT:
;	out: a structure with the output data. This structure is the
;		input of other programs (as crystal_fh) and can be
;		dumped to disk by using nbragg_out.
;
;	group = The widget ID of the widget that calls the routine.
;		This is passed to widget_message in its dialog_parent
;		keyword to center the window message.
;
; PROCEDURE:
;	Reads the crystal structure and scattering lengths from the
;	DABAX library and returns some crystal parameters, the
;	geometrical part of FH and the atomic part.
;	It has been adapted from bragg_calc.pro (for x-rays)
;	It uses the following DABAX files:
;	  CrystalStructures.dat, CrystalCell.dat, Neutron_SLCS_*.dat
;
; EXAMPLE:
;	inp = nbragg_inp()
;	out = nbragg_calc(inp)
;
; SEE ALSO:
;       nbragg_inp, nbragg_out, nop_defaults.
;
; MODIFICATION HISTORY:
;	2003-04-07 srio@esrf.fr cleanup version of Boisseau.
;
;-
;on_error,2

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'NBRAGG_CALC: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return,0
endif

if sdep(/w) then ask=2 else ask=1
if n_params() NE 1 then begin
  inp = nbragg_inp(ask=ask)
endif


ifilef0 = inp.parameters.filef0(0) + 1
filef0 = inp.parameters.filef0(ifilef0)
ipolar = inp.parameters.polar(0) + 1
polar = inp.parameters.polar(ipolar)
ilattice = inp.parameters.ilattice(0) + 1
hmiller=inp.parameters.hmiller
kmiller=inp.parameters.kmiller
lmiller=inp.parameters.lmiller
sig_tds = inp.parameters.sig_tds
DWf = inp.parameters.DWf
EXPANS = inp.parameters.expans
;;srio Alpha = inp.parameters.alpha
EMIN = inp.parameters.emin
EMAX = inp.parameters.emax
ESTEP = inp.parameters.estep
OUTFIL = inp.parameters.outfil

filestr='CrystalStructures.dat'
filecell='CrystalCell.dat'

hstr = dabax_access(filestr)
hcell = dabax_access(filecell)
;
; load crystal structure and cell parameters from files
;
index = ilattice
str = double(spec_data(hstr,index,/index))
cell = double(spec_data(hcell,index,/index))

if spec_name(hcell,index,/index) NE spec_name(hstr,index,/index) then $
	message,'Crystal structures are different in files '+$
	filestr+' ('+spec_name(hcell,index,/index)+') and '+$
	filecell+' ('+spec_name(hstr,index,/index)+').'
;
; Calculate the volume of the unit cell and interlattice distance d
; following, for example, "X-ray diffraction", B. E. Warren, Dover, 1990
;
	M_REF = double([0,hmiller,kmiller,lmiller])
	D_LATT_A = cell(0)*1.0D-8
	D_LATT_B = cell(1)*1.0D-8
	D_LATT_C = cell(2)*1.0D-8
	D_LATT_ALPHA  = cell(3)*!dpi/180.
	D_LATT_BETA  =  cell(4)*!dpi/180.
	D_LATT_GAMMA  = cell(5)*!dpi/180.



	d_latt_volume = D_LATT_A*D_LATT_B*D_LATT_C * SQRT(1+ $
	    2.0D0*cos(D_LATT_ALPHA)*cos(D_LATT_BETA)*cos(D_LATT_GAMMA) $
	    - cos(D_LATT_ALPHA)^2 - cos(D_LATT_BETA)^2 $
	    - cos(D_LATT_GAMMA)^2               )

	oneoverd2 = 1.0D0 + 2.0D0*cos(D_LATT_ALPHA)*cos(D_LATT_BETA)* $
		  cos(D_LATT_GAMMA) - cos(D_LATT_ALPHA)^2 - cos(D_LATT_BETA)^2 $
	    - cos(D_LATT_GAMMA)^2
	oneoverd2 = 1.0D0/oneoverd2
	oneoverd2 = oneoverd2  * ( $
	 (M_REF(1)*sin(D_LATT_ALPHA)/D_LATT_A)^2 + $
	 (M_REF(2)*sin(D_LATT_BETA)/D_LATT_B)^2 + $
	 (M_REF(3)*sin(D_LATT_GAMMA)/D_LATT_C)^2 + $
	 2.0D0*M_REF(1)*M_REF(2)/D_LATT_A/D_LATT_B* $
	     (cos(D_LATT_ALPHA)*cos(D_LATT_BETA)-cos(D_LATT_GAMMA)) + $
	 2.0D0*M_REF(2)*M_REF(3)/D_LATT_B/D_LATT_C* $
	     (cos(D_LATT_BETA)*cos(D_LATT_GAMMA)-cos(D_LATT_ALPHA)) + $
	 2.0D0*M_REF(3)*M_REF(1)/D_LATT_A/D_LATT_C* $
	     (cos(D_LATT_GAMMA)*cos(D_LATT_ALPHA)-cos(D_LATT_BETA)) $
		  			  )
	SP_HKL = 1.0D0/SQRT(oneoverd2)
	R_NATOM = 1.0D0/d_latt_volume

	message,/info,'Interplanar distances (Angs) :' +$
		strcompress(SP_HKL*1.0D8,/rem)
	message,/info,'d_latt_volume (cm^3) : ' +$
		strcompress(d_latt_volume,/rem)
	message,/info,'1/V=r_natom (cm^-3) :' +$
		strcompress(R_NATOM,/rem)

	E2_MC2  = physical_constants('RE')  ; 2.81794028D-13  e^2/mc^2 classical e- radius in cm
	RN = E2_MC2*R_NATOM


	zcol = str(0,*)
	fcol = str(1,*)
	batom =  zcol(uniq(zcol,sort(zcol)) )
	nbatom = n_elements( batom )

;
; Retrieve the scattering lengths
;
;
        NPOINT  = (EMAX - EMIN)/ESTEP + 1
	energy = dblarr(npoint)
	for i=0,npoint-1 do energy(i) = emin+estep*i

	fp = dblarr(nbatom,npoint)
	fpp = dblarr(nbatom,npoint)
        for ii=0,nbatom-1 do begin
	  sym = atomic_symbols(batom(ii))
	  f1f2 = nsl_calc(filef0,sym,energy,group=group,polar=Fix(polar))
	  if ((n_elements(f1f2) EQ 1) and (f1f2(0) EQ 0)) then begin
	    if ask EQ 2 then begin
	      itmp = widget_message(dialog_parent=group,/Question,$
	      ['Error calculating br and bi values for '+sym,'Continue?'])
	    endif else begin
	      itmp=0
	      message,/info,'Error calculating br and bi values for '+sym+'.'
	      read,'>>> Continue? [1=Yes,0=No] ',itmp
	      if itmp EQ 1 then itmp='Yes' else itmp = 'No'
	    endelse
	    if itmp EQ 'No' then return,0
	    fp(ii,*) = 0.0+BATOM(ii)
	    fpp(ii,*) = 0.0
	  endif else begin
	    fp(ii,*) = f1f2(0,*)  ; br value in fm (fermi)
	    fp(ii,*) = fp(ii,*)*1.d-13/physical_constants('RE')   ; conversion in f1-like parameter
	    fpp(ii,*) = f1f2(1,*) ; bi value in fm (fermi)
	    fpp(ii,*) = fpp(ii,*)*1.d-13/physical_constants('RE') ; conversion in f2-like parameter
	  endelse
        endfor
;
; Compute the geometrical part G's of the structure factor
;

; Get absorption cross section [barn] at 1.798 A and number of
; molecules per unit cell for the considered element
ener=1.798d0
NsigA=nsl_calc(filef0,sym,ener,Ret=12,group=group,polar=Fix(polar),$
  molecules_per_cc=molecules_per_cc)
noverV = molecules_per_cc*1.0D6

	CI = dcomplex(0.0,1.0)
	G = replicate(dcomplex(0.0,0.0),nbatom)
	G_BAR = G
        for i = 0, nbatom-1 do begin
		ibatom = batom(i)
		tmp = where(zcol EQ ibatom)
		for j=0,n_elements(tmp)-1 do begin
			X = str(2,tmp(j))
			Y = str(3,tmp(j))
			Z = str(4,tmp(j))
			ttt = exp(ci*2*!dpi*  $
			  (X*M_REF(1) + Y*M_REF(2) + Z*M_REF(3)))
                	G(i) = G(i) + ttt
		endfor
        G_BAR(i) = CONJ(G(i))
	endfor
;
; Now prepare the output structure
;

ATNUM= fix(batom)

FRACT = dblarr(nbatom)
for i=0,nbatom-1 do FRACT(i)=(fcol( where(Zcol EQ batom(i))))(0)

G_0 = dblarr(nbatom)
for i=0,nbatom-1 do G_0(i)=n_elements( where(Zcol EQ batom(i)))

NPOINT = long(NPOINT)
; batom(j) has not to be substracted in the case of neutrons !!
; so f2 is simply equal to fp ...
;F1 = dblarr(nbatom,npoint)
;for j=0,nbatom-1 do F1(j,*) = FP(j,*)-BATOM(j)
F1 = FP
F2 = FPP



;;srio DWf = replicate(DWf,nbatom)
;
; temperature parameter
;
temper1 = DWf
IF StrCompress(temper1,/Rem) EQ '?' THEN BEGIN
  temper1 = -300.0
  editFlag=1
ENDIF ELSE BEGIN
  temper1 = Float(temper1)
  editFlag=0
ENDELSE

IF temper1 LT 0.0 THEN BEGIN
  ratio = 1D/ (2D*SP_HKL*1.0D8)
  temper=DebyeWaller(batom,Ratio=ratio, Temper=Abs(temper1), Edit=editFlag )
ENDIF ELSE BEGIN
  TEMPER = replicate(TEMPER1,nbatom)
ENDELSE
DWf = TEMPER




D_SPACING = sp_hkl


out = { outfil:outfil, RN:rn, d_spacing:d_spacing, $
 noverV:noverV, NsigA:NsigA, nbatom:nbatom, $
 atnum:atnum, fract:fract, sig_tds:sig_tds, DWf:DWf, EXPANS:expans, $
;;srio Alpha:alpha, g_0:g_0, G:G, G_BAR:G_BAR, $
 g_0:g_0, G:G, G_BAR:G_BAR, $
 NPOINT:NPOINT, energy:ENERGY, f1:f1, f2:f2 }

return,out

end

