pro loopsh,nloops, keep=keep, append=append, nolost=nolost, $
only_source=only_source, only_trace=only_trace
;+
; NAME:
;	LOOPSH 
; PURPOSE:
;	Runs shadow (source+trace) a number of times with the currently
;	available system. It changes the source seed in the process 
;	and allows the user to keep files for analysis.
; CATEGORY:
;	SHADOW's utilities.
; CALLING SEQUENCE:
;	loopsh, N, keyword_parameters
; INPUTS:
;	N   Number of times that shadow has to be run.
; KEYWORD PARAMETERS:
;	KEEP Array of strings with the files names to be kept.
;		LOOPSH renames the files to the same name with the 
;		"-i" extension, being i the run indes (from 1 to N).
;		Use the keyword NFILES (in READSH, PLOTXY...) to load the
;		resulting files.
;	APPEND This keyword permits to avoid keeping disk files
;		and allows the user to accumulate the result of the
;		loop calculations in a structute containing a number
;		of shadow/idl structures as n_elements(KEEP).
;		Usage: Define a named variable with 1 or with 
;		the structure where to append the result of the 
;		calculation. Call LOOPSH with the keywords
;		KEEP and APPEND switched on.
;		See examples for more details
;	NOLOST This keyword only works with the APPEND keyword.
;		The effect is the same as the NOLOST keyword in
;		the READSH function.
;	ONLY_SOURCE When set then runs only shadow source (default=
;		source+trace)
;	ONLY_TRACE When set then runs only shadow trace (default=
;		source+trace)
; SIDE EFFECTS:
; 	Acces the UNIX shell. Create files.
; RESTRICTIONS:
;	When using the APPEND keyword, the maximum number of
;	elements allowed for KEEP is 10.
; PROCEDURE:
;	Acces to the UNIX shell using spawn.
; 	Runs SHADOW's seed, gen_source and trace (N times).
;	Rename file names.
; EXAMPLES
;	1) Make a shadow loop and save the files
;	   start.03 and screen.0103 of each iteration
;	   in start.03-i and screen.0103-i where 
;	   the index i goes from 0 to NLOOPS
;	loopsh,5,keep=['star.03','screen.0103']
;	plotxy,'star.03',1,3,NFILES=5 ; example of analysis
;
;	2) Like 1) but instead saving files it does save the 
;	   result in an idl structure
;	app = 1 ; initialize variable where to keep data
;	loopsh,5,keep=['star.03','screen.0103'],append=app
;	   The resulting structure app contains 2 tags: the 
;	   first one corresponds to the files 'star.03', and
;	   the second one to 'screen.0103'.
;	plotxy,app.(0),1,3 ; example of analysis eqivalent
;		to the previous example.
;	   A help on the result may be:
;	 for i=0,n_tags(app)-1 do help,/str,app.(i)
;
;	3) After running 2), if we think the statistics is
;	   not good, we can add to the structure app the result 
;	   of other 2 shadow loops:
;	loopsh,2,keep=['star.03','screen.0103'],append=app
;
; MODOFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble, 24 May 1996.
;	96/06/24 MSR (srio@esrf.fr) adds APPEND, NOLOST and 
;		ONLY_(SOURCE,TRACE) keywords.
;
;-
on_error,2
; 
; runs the current shadow system nloops times

if n_params() EQ 0 then begin
  message,"Usage: loopsh,ntimes,keep=['star.01','screen.0101']"
  return
endif
if not(keyword_set(keep)) then keep=['star.01']
if not(keyword_set(nolost)) then nolost=0

if keyword_set(append) then begin
  s0 = 1 & s1 = 1 & s2 = 1 & s3 = 1 & s4 = 1
  s5 = 1 & s6 = 1 & s7 = 1 & s8 = 1 & s9 = 1
  if type(append) EQ 8 then begin
    for ii=0,n_elements(keep)-1 do begin
      iistring=strcompress(ii,/rem)
      command = 's'+iistring+' = append.('+iistring+')'
      message,/info,command  &  itmp = execute(command)
    endfor
  endif
endif


for i=0L,nloops-1 do begin
  message,/info,'>>>>>>>>>>>>>>>>>>> loop '+strcompress(i+1,/rem)+$
	' over '+strcompress(nloops,/rem)
  command = 'seed start.00'
  message,/info,command & spawn,command

  if not(keyword_set(only_trace)) then begin
    command = 'gen_source start.00'
    message,/info,command & spawn,command
  endif
  ;
  ;
  if not(keyword_set(only_source)) then begin
    command = 'echo 0 | trace -m menu'
    message,/info,command & spawn,command
  endif

  if not(keyword_set(append)) then begin
    for jj=0,n_elements(keep)-1 do begin
      command = 'mv '+keep(jj)+' '+keep(jj)+'-'+strcompress(i+1,/rem)
      message,/info,command & spawn,command
    endfor
  endif else begin
    for jj=0,n_elements(keep)-1 do begin
      jjs = strcompress(jj,/rem)
      file = strcompress(keep(jj),/REM)
      command = 's'+jjs+' = readsh('+"'"+file+"'"+',append = s'+$
	jjs+',nolost='+strcompress(nolost,/rem)+')'
      message,/info,command &  itmp = execute(command)
    endfor
  endelse
endfor

if keyword_set(append) then begin
  command = 'append = { s0:s0'
  for ii=1,n_elements(keep)-1 do command = command+', s'+$
    strcompress(ii,/rem)+':'+'s'+strcompress(ii,/rem)
  command = command+'}'
  message,/info, command  &  itmp = execute(command)
endif

end

