//
// reader.h: reader/manipulator for MENU files.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

#ifndef reader_h
#define reader_h

// ====================================================================== //

#include <string>
#include <fstream>

#include "xrtypes.h"

// ====================================================================== //
//
// MenuReader is responsible for parsing/loading MENU description files
// and populating the Menu class. Currently the only reader supplied is
// for the SHADOW menu file format; in the future this will be a more 
// readable format, using a free-form grammar.
//
// ====================================================================== //

class Menu;
class ToolMgr;

class MenuReader {
public:
    MenuReader(ToolMgr& toolmgr, Menu* menu, bool debug = false);
    virtual ~MenuReader();

    int load(const std::string& file);

private:
    static bool is_eos(const char* buf);
    static bool is_comment(const char* buf);

private:
    int load_synonyms(std::ifstream& ifp);
    int load_defaults(std::ifstream& ifp);
    int load_pseudo_defaults(std::ifstream& ifp);
    int load_commands(std::ifstream& ifp);
    int load_pages(std::ifstream& ifp);
    int load_variables(std::ifstream& ifp);
    int load_tool_aliases(std::ifstream& ifp);
    int skip_sections(std::ifstream& ifp, int nskip);
    int goto_section(std::ifstream& ifp, int nsection);
    const char* get_menu_statement(std::ifstream& ifp);

private:
    ToolMgr& toolmgr_;
    Menu* menu_;
    bool debug_;
    char* menu_buf_;
    int line_no_;
};

inline bool MenuReader::is_comment(const char* buf) {
    return buf[0] == ' ' || buf[0] == '\0' || buf[0] == '*' || buf[0] == '#';
}

inline bool MenuReader::is_eos(const char* buf) {
    return buf[0] == '=';
}

// ====================================================================== //

#endif/*reader_h*/
