//
// variable.cc: variables in EXCON
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

/***************************************************************************/

#include <cstdio>
#include <cctype>
#include <cstdlib>
#include <cstring>

#include "utils.h"
#include "value.h"
#include "variable.h"

#if NEED_DECLARATION_STRCASECMP
extern "C" int strcasecmp(const char*, const char*);
#endif
#if NEED_DECLARATION_STRNCASECMP
extern "C" int strncasecmp(const char*, const char*, std::size_t);
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

/***************************************************************************/

static char* tolower_with_alloc(const char* str) {
    char* newstr = nil;
    if (str) {
	char* q = newstr = new char[strlen(str) + 1];
	for(char const* p = str; p && *p; ++p, ++q) {
	    *q = tolower(*p);
	}
	*q = '\0';
    }
    return newstr;
}

/***************************************************************************/

Var::Var(const char* name, const char* type, Value* val) {
    // WARNING: COPIES ARE MADE OF name AND type; NOT FOR val
    name_ = tolower_with_alloc(name);
    type_ = strcpy_with_alloc(type);
    val_ = val; 
    properties_ = new Stq;
}

Var::~Var () {
    delete[] name_; name_ = nil;
    delete[] type_; type_ = nil;
    delete val_; val_ = nil;

    // pop()'ing deletes the StqItem in the list. 
    unsigned size = properties_->size();
    while(size--) {
	char* prop = (char*)properties_->pop();
	delete[] prop; prop = nil;
    }
    delete properties_; properties_ = nil;
}

// CREATE A AND RETURN A VALUE OF THE RIGHT TYPE
Value* Var::genval() const {
    Value* newval = nil;
    switch (type_[0]) {
	case 'i' : 
	    newval = new IntegerValue (0);
	    break;
	case 'r' : 
	    newval = new RealValue (0.0);
	    break;
	case 'f' : 
	    newval = new FilenameValue ("/tmp/xxx");
	    break;
	case 't' : 
	    newval = new TextValue ("INIT!");
	    break;
	case 'b' : 
	    newval = new BooleanValue (0);
	    break;
	case 'e' : 
	    newval = new EnumValue(((EnumValue*)val_)->getenumlist());
	    break;
	default:
	    assume(0);
	    break;
    }
    return newval;
}

void Var::addprop(const char* name) {
    char* prop = strcpy_with_alloc(name);
    properties_->enq (prop);
}

// RETURN true (1) IF THE VARIABLE HAS THIS PROPERTY...OTHERWISE RETURN false
bool Var::hasprop(const char* name) const {
    bool hasproperty = false;
    for (int i=1;i<=properties_->size();i++) {
	char* prop = (char*)properties_->cycle();
	if (strcasecmp(prop,name)==0) 
	    hasproperty = true;
    }
    return hasproperty;
}

Var* Var::clone (const char* str) const {
    // RETURN nil IF str CANNOT IS OF THE WRONG TYPE
    Value* newval = val_->clone (str);
    if (newval == nil) 
	return nil;
    Var* newvar = new Var (name_, type_, newval);
    // COPY PROPERTIES
    for (int i=1;i<=properties_->size();i++) {
	char* prop = (char*)properties_->cycle();
	newvar->addprop (prop);
    }
    return newvar;
}

Var* Var::clone() const {
    Value* newval = val_->clone();
    Var* newvar = new Var (name_, type_, newval);
    // COPY PROPERTIES
    for (int i=1;i<=properties_->size();i++) {
	char* prop = (char*)properties_->cycle();
	newvar->addprop (prop);
    }
    return newvar;
}


void Var::setval(Value* val) { 
    delete val_;  
    val_ = val; 
}

/***************************************************************************/
