//
// pmu.cc: General purpose pattern matching unit
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

/****************************************************************************
 *
 *  INCLUDES
 *
 ****************************************************************************/

#include <cassert>
#include <cctype>
#include <cstdio>
#include <cstring>

#include "pmu.h"
#include "utils.h"

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

/***************************************************************************/

Pmu::Pmu () {
   OTHER_SYM = 0;  SEP_SYM = 1;  EOLN_SYM = 2;
   for (int i=0;i<256;i++) ctype[i] = OTHER_SYM;
   ctype[ 0] = EOLN_SYM; // NUL   
   ctype[ 9] = SEP_SYM;  // HT
   ctype[10] = EOLN_SYM; // LF
   ctype[32] = SEP_SYM;  // SPACE
   debug = 0;
   scanline = nil;
}

Pmu::~Pmu() {
    delete[] scanline;
}

void Pmu::dump () {
   printf ("PMU STATE: \n");
   printf ("   line = '%s'\n",scanline);
   for (int i=0;i<=this->nfields;i++) {
      printf ("   Field %d [%d,%d]\n",i,fieldstart[i],fieldend[i]); 
   }
}     
void Pmu::scaninit (const char* line) {
   delete[] scanline;
   scanline = strcpy_with_alloc(line);
   scanpos = 0;
   nfields = 0;
   nmatches = 0;
}
void Pmu::gettextfield (int ith, char* buf) {
   assume (ith <= this->nfields);
   int n = this->getfieldlen (ith);
   strncpy (buf,&scanline[fieldstart[ith-1]],n);
   buf[n] = '\0';
}
int Pmu::getcodefield (int ith) {
   assume (ith <= this->nfields);
   return fieldnum[ith-1];
}
int Pmu::getintfield (int ith) {
   assume (ith <= this->nfields);
   int pos = fieldstart[ith-1];
   int sign = (scanline[pos] == '-') ? -1 : 1;
   if (scanline[pos] == '+' || scanline[pos] == '-') pos++;
   assume (isdigit(scanline[pos]));
   int val = 0;
   while (isdigit(scanline[pos])) {
      val = 10 * val + (scanline[pos]-'0');
      pos++;
   }
   return sign * val;
}
long Pmu::getlongfield (int ith) {
   assume (ith <= this->nfields);
   int pos = fieldstart[ith-1];
   int sign = (scanline[pos] == '-') ? -1 : 1;
   if (scanline[pos] == '+' || scanline[pos] == '-') pos++;
   assume (isdigit(scanline[pos]));
   long val = 0;
   while (isdigit(scanline[pos])) {
      val = 10 * val + (scanline[pos]-'0');
      pos++;
   }
   return sign * val;
}
double Pmu::getrealfield (int ith) {
   assume (ith <= this->nfields);
#if 0
   int pos = fieldstart[ith-1];
   int sign = (scanline[pos] == '-') ? -1 : 1;
   if (scanline[pos] == '+' || scanline[pos] == '-') pos++;
   double val = 0.0;
   while (isdigit(scanline[pos])) {
      val = 10 * val + (scanline[pos]-'0');
      pos++;  scanline[pos] = scanline[pos];
   }
   if (scanline[pos] == '.') pos++;
   double power = 1.0;
   while (isdigit(scanline[pos])) {
      val = 10 * val + (scanline[pos]-'0');
      power *= 10.0;
      pos++;  scanline[pos] = scanline[pos];
   }
   return sign * (val / power);
#endif
   char buf[1024];
   int n = getfieldlen (ith);
   strncpy (buf,&scanline[fieldstart[ith-1]],n);
   buf[n] = '\0';
   double realvalue;
   sscanf(buf, "%lf", &realvalue);
   return realvalue;
}
void Pmu::skipblanks () {
   while (ctype[scanline[scanpos]] == SEP_SYM) scanpos++;
}
// RETURN 1 IF THE LINE IS A COMMENT LINE...OTHERWISE RETURN 0
// <comment> = <empty line> | <all blank or tabs> | <all blank or tab> "#" <anything>
int Pmu::isacomment (const char* line) {
   int len = strlen(line);
   if (len == 0) return 1;  int pos = 0;
   // SKIP BLANKS AND TABS
   while (ctype[line[pos]] == SEP_SYM) pos++;
   if (ctype[line[pos]] == EOLN_SYM) return 1;
   // COMMENT?
#if 0
   if (line[pos] == '*') return 1;
#endif
   if (line[pos] == '#') return 1;
   return 0;
}
int Pmu::cistrcmp (const char* key1,const char* key2) {
   const char* k1 = key1;  const char* k2 = key2;
   for (;toupper(*k1) == toupper(*k2);k1++,k2++) if (*k1 == '\0') return 1;
   if (*k1 == '\0') return 1;
   return 0;
}
// toupper() IS VERY SLOW...USER THIS VERSION WHERE POSSIBLE 
int Pmu::strcmp (const char* key1,const char* key2) {
   const char* k1 = key1;  const char* k2 = key2;
   for (;*k1 == *k2;k1++,k2++) if (*k1 == '\0') return 1;
   if (*k1 == '\0') return 1;
   return 0;
}
// <nothing> = <blank or tabs to end of line>
int Pmu::scannothing () {
   char ch = scanline[scanpos];
   if (ctype[ch] == EOLN_SYM) return 1;
   fieldstart[nfields] = scanpos;
   int lastnon = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (ctype[ch] != SEP_SYM) break;
   }
   if (ctype[ch] == EOLN_SYM) {
       fieldend[nfields] = lastnon-1;  nfields++;
       return 1;   
   }
   else
       return 0;
}
// CASE INSENSITIVE STRING COMPARE
int Pmu::scancmp (const char* pattern) {
   const char* k1 = pattern;
   const char* k2 = &scanline[scanpos];
   for (;toupper(*k1)==toupper(*k2);k1++,k2++,scanpos++) if (*k1 == '\0') return 1;
   if (*k1 == '\0') return 1;
   return 0;
}
// CASE INSENSITIVE LITERAL SCAN
int Pmu::scanliteral (const char* pattern) {
   fieldstart[nfields] = scanpos;
   if (scancmp (pattern)) {
      fieldend[nfields] = scanpos-1;  nfields++;
      return 1;   
   }
   return 0;
}
// <boolean> = "true" | "false" | "yes" | "no" | "y" | "n"
int Pmu::scanboolean () {
   char ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   if (!isalpha(ch)) return 0;
   fieldstart[nfields] = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (!isalpha(ch)) break;
   }
   fieldnum[nfields] = -1;
   if (cistrcmp ("true" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 1;
   if (cistrcmp ("yes" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 1;
   if (cistrcmp ("y" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 1;

   if (cistrcmp ("false",&scanline[fieldstart[nfields]])) fieldnum[nfields] = 0;
   if (cistrcmp ("no" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 0;
   if (cistrcmp ("n" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 0;
   if (fieldnum[nfields] == -1) return 0;
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}
// <yesno> = "yes" | "no"
int Pmu::scanyesno () {
   char ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   if (!isalpha(ch)) return 0;
   fieldstart[nfields] = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (!isalpha(ch)) break;
   }
   fieldnum[nfields] = -1;
   if (cistrcmp ("yes",&scanline[fieldstart[nfields]])) fieldnum[nfields] = 1;   
   if (cistrcmp ("no" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 0;
   if (fieldnum[nfields] == -1) return 0;
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}
// <yn> = "y" | "n"
int Pmu::scanyn () {
   char ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   if (!isalpha(ch)) return 0;
   fieldstart[nfields] = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (!isalpha(ch)) break;
   }
   fieldnum[nfields] = -1;
   if (cistrcmp ("y",&scanline[fieldstart[nfields]])) fieldnum[nfields] = 1;   
   if (cistrcmp ("n" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 0;
   if (fieldnum[nfields] == -1) return 0;
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}
// <onoff> = "on" | "off"
int Pmu::scanonoff () {
   char ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   if (!isalpha(ch)) return 0;
   fieldstart[nfields] = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (!isalpha(ch)) break;
   }
   fieldnum[nfields] = -1;
   if (cistrcmp ("on" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 1;
   if (cistrcmp ("off",&scanline[fieldstart[nfields]])) fieldnum[nfields] = 0;
   if (fieldnum[nfields] == -1) return 0;
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}
// <relop> = "EQ" | "NE" | "AND" | "OR"
int Pmu::scanrelop () {
   char ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   if (!isalpha(ch)) return 0;
   fieldstart[nfields] = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (!isalpha(ch)) break;
   }
   fieldnum[nfields] = -1;
   if (cistrcmp ("eq" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 0;  
   if (cistrcmp ("ne" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 1;
   if (cistrcmp ("and",&scanline[fieldstart[nfields]])) fieldnum[nfields] = 2;
   if (cistrcmp ("or" ,&scanline[fieldstart[nfields]])) fieldnum[nfields] = 3;
   if (fieldnum[nfields] == -1) return 0;
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}
// <name> = [<anything but comma or blank or tab>] 
//		{<anything but comma or blank or tab>}
int Pmu::scanname () {
   char ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   fieldstart[nfields] = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (ctype[ch] != OTHER_SYM || ch == ',') break;
   }
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}


// "$" added to support variable names. MK
// <variable> = [ letters digits "_" "$" ] { letters digits "_" "(" ")" }
int Pmu::scanvariable () {
   char ch = scanline[scanpos];
   if (!(isalpha(ch)||isdigit(ch)||ch == '_'||ch == '$'||ch == '('||
       ch == ')')) return 0;
   fieldstart[nfields] = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (!(isalpha(ch)||isdigit(ch)||ch == '_'||ch == '('||ch == ')')) break;
   }
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}

// <ident> = { letters digits "(" "_" ")" }
int Pmu::scanident () {
   char ch = scanline[scanpos];
   if (!(isalpha(ch)||isdigit(ch)||ch == '_'||ch == '('||ch == ')')) return 0;
   fieldstart[nfields] = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (!(isalpha(ch)||isdigit(ch)||ch == '_'||ch == '('||ch == ')')) break;
   }
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}
// <integer> = ["+"|"-"] [digit] {digit} 
int Pmu::scaninteger () {
   fieldstart[nfields] = scanpos;
   if (scanline[scanpos] == '+' || scanline[scanpos] == '-') scanpos++;
   if (!isdigit(scanline[scanpos])) return 0;
   while (isdigit(scanline[scanpos])) scanpos++; 
   fieldend[nfields] = scanpos-1;
   nfields++;
   return 1;
}      
// <long> = ["+"|"-"] [digit] {digit} 
int Pmu::scanlong () {
   return scaninteger();
}      
// <real> = ["+"|"-"] {digit} ["." {digit} ] ["e"|"E" ["+"|"-"] {digit}]
int Pmu::scanreal () {
   if (scanline[scanpos] != '+' && scanline[scanpos] != '-' && 
       scanline[scanpos] != '.' && !isdigit(scanline[scanpos])) return 0;
   fieldstart[nfields] = scanpos;
   if (scanline[scanpos] == '+' || scanline[scanpos] == '-') scanpos++;
   while (isdigit(scanline[scanpos])) scanpos++;
   if (scanline[scanpos] == '.') scanpos++;
   while (isdigit(scanline[scanpos])) scanpos++;
   if (scanline[scanpos] == 'e' || scanline[scanpos] == 'E') scanpos++;
   if (scanline[scanpos] == '+' || scanline[scanpos] == '-') scanpos++;
   while (isdigit(scanline[scanpos])) scanpos++;
   fieldend[nfields] = scanpos-1;
   nfields++;
   return 1;
}
// <text> = <first non-tab or blank> <to eoln-trailing blanks or tabs>
int Pmu::scantext () {
   char ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   fieldstart[nfields] = scanpos;
   int lastnon = scanpos;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (ctype[ch] != SEP_SYM) lastnon = scanpos;
      if (ctype[ch] == EOLN_SYM) break;
   }
   fieldend[nfields] = lastnon-1;  nfields++;
   return 1;   
}
// <type> = "i" | "r" | "y" | "e" | "a"
int Pmu::scantype () {
   char ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   if ((ch != 'i') && (ch != 'r') && (ch != 'y') && (ch != 'e') && (ch != 'a') &&
       (ch != 'I') && (ch != 'R') && (ch != 'Y') && (ch != 'E') && (ch != 'A')) return 0;
   fieldstart[nfields] = scanpos;
   fieldend[nfields]   = scanpos;  nfields++;
   return 1;   
}
// <field:#> = where # is any number of digits
int Pmu::scanfield (const char* pattern) {
   // GET FIELD LENGTH (pattern = "<field:##...>"
   if (strlen(pattern) < 9)  return 0;
   if (!isdigit(pattern[7])) return 0;
   int pos = 7; int val = 0; 
   while (isdigit(pattern[pos])) { val = 10 * val + (pattern[pos]-'0'); pos++; }
   char ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   fieldstart[nfields] = scanpos;
   pos = 1;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (ctype[ch] == EOLN_SYM) break;
      pos++;  if (pos > val) break;
   }
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}
// <field#:#> = where # is any number of digits
int Pmu::scanfield2 (const char* pattern) {
   // GET FIELD RANGE
   if (strlen(pattern) < 10)  return 0;
   if (!isdigit(pattern[6])) return 0;
   int pos = 6; int start = 0; 
   while (isdigit(pattern[pos])) { start = 10 * start + (pattern[pos]-'0'); pos++; }
   if (pattern[pos]!=':') return 0; pos++;
   int stop = 0;
   while (isdigit(pattern[pos])) { stop = 10 * stop + (pattern[pos]-'0'); pos++; }
   // IF WE ARE PAST START OF RANGE MAKE SURE THEY ARE ALL BLANKS
   if (scanpos >= start) {
      pos = start;
      while (1) {
         if (ctype[scanline[pos]] != SEP_SYM) return 0;
         pos++;  if (pos >= scanpos) break;
      }
   }
   // SKIP TO START OF RANGE
   char ch;
   while (1) {
      scanpos++;  ch = scanline[scanpos];
      if (ctype[ch] == EOLN_SYM) break;
      if (scanpos >= start) break;
   }
   // COLLECT ALL CHARS IN RANGE
   ch = scanline[scanpos];
   if (ctype[ch] != OTHER_SYM) return 0;
   fieldstart[nfields] = scanpos;
   fieldend[nfields] = scanpos-1;  nfields++;
   return 1;   
}
      
int Pmu::match (const char* line,const char* f1,const char* f2,
			const char* f3,const char* f4, const char* f5,
			const char* f6,const char* f7,const char* f8,
                        const char* f9,const char* f10,const char* f11,
			const char* f12) {
// RETURNS THE NEXT CHARACTER LOCATION IN line WHERE YOU CAN RESUME SCANNING OR
// RETURNS 0 IF NO MATCH
   scaninit (line);
   int ret = 1;  // ASSUME SUCCESS
   const char* f[12];
   f[0] = f1; f[1] = f2; f[2] = f3; f[3] = f4;
   f[4] = f5; f[5] = f6; f[6] = f7; f[7] = f8; 
   f[8] = f9; f[9] = f10; f[10] = f11; f[11] = f12; 
   nmatches = 0;
   for (int i=0;i<12;i++) {
      const char* pattern = f[i];  
      if (pattern == nil) 
	  break;  
      skipblanks();
      /*
      if (ctype[scanline[scanpos]] != OTHER_SYM) { ret = 0;  break; }
      */
      if (pattern[0] == '<') switch (pattern[1]) {
         case 'b' : if (strcmp ("<boolean>",pattern)) { 
            if (!scanboolean()) { ret = 0;  break; } nmatches++;  continue; }
            break;
         case 'i' : 
            if (strcmp ("<integer>",pattern)) { 
               if (!scaninteger()) { ret = 0;  break; } nmatches++;  continue; }
            if (strcmp ("<ident>",pattern)) { 
               if (!scanident()) { ret = 0;  break; } nmatches++;  continue; }
            break;
         case 'l' : 
            if (strcmp ("<long>",pattern)) { 
               if (!scanlong()) { ret = 0;  break; } nmatches++;  continue; }
	    break;
         case 'n' : 
	    if (strcmp ("<name>",pattern)) { 
		if (!scanname()) { ret = 0;  break; } nmatches++;  continue; }
	    if (strcmp ("<nothing>",pattern)) { 
		if (!scannothing()) { ret = 0;  break; } nmatches++;  continue; }
            break;
         case 'r' : 
            if (strcmp ("<real>",pattern)) { 
               if (!scanreal()) { ret = 0;  break; } nmatches++;  continue; }
            if (strcmp ("<relop>",pattern)) { 
               if (!scanrelop()) { ret = 0;  break; } nmatches++;  continue; }
            break;
         case 'y' : 
            if (strcmp ("<yesno>",pattern)) { 
               if (!scanyesno()) { ret = 0;  break; } nmatches++;  continue; }
            if (strcmp ("<yn>",pattern)) { 
               if (!scanyn()) { ret = 0;  break; } nmatches++;  continue; }
            break;
         case 'o' : if (strcmp ("<onoff>",pattern)) { 
            if (!scanonoff()) { ret = 0;  break; } nmatches++;  continue; }
            break;
         case 't' : 
            if (strcmp ("<text>",pattern)) { 
               if (!scantext()) { ret = 0;  break; } nmatches++;  continue; }
            if (strcmp ("<type>",pattern)) { 
               if (!scantype()) { ret = 0;  break; } nmatches++;  continue; }
            break;
         case 'f' : 
            if (strcmp ("<field:",pattern)) { 
               if (!scanfield (pattern)) { ret = 0;  break; } nmatches++;  continue; }
            if (strcmp ("<field",pattern)) { 
               if (!scanfield2 (pattern)) { ret = 0;  break; } nmatches++;  continue; }
            break;
         case 'v' : if (strcmp ("<variable>",pattern)) { 
            if (!scanvariable()) { ret = 0;  break; } nmatches++;  continue; }
            break;
      }
      // IF WE ARE HERE SEARCH FOR LITERAL PATTERN
      if (!scanliteral (pattern)) { ret = 0; break; }
      nmatches++; 
   }
   if (debug) if (ret == 0) {
      printf ("SCAN [nmatches=%d] '%s'.\n",nmatches,scanline);
   }
   // RETURN FAILURE OR POSITION FOR CONTINUING TO MATCH
   return (ret == 0) ? 0 : scanpos;
}
