//
// shadow-nml.cc: SHADOW namelist implementation
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

// =================================================================== //

#ifdef __BORLANDC__
# include <assert>
# include <stdlib>
# include <ctype>
# include <string>
# include <iostream>
#else
# include <cassert>
# include <cstdlib>
# include <cctype>
# include <string>
# include <iostream>
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

#include "global.h"
#include "shadow-nml.h"
#include "value.h"
#include "variable.h"

// =================================================================== //


static string string_tolower(const string& str) {
    string local_str(str);
    string::iterator it = local_str.begin();
    for(; it != local_str.end(); ++it) {
	int lowercase = tolower(*it);
	*it = lowercase;
    }
    return local_str;
}

// =================================================================== //

ShadowNamelist::ShadowNamelist(const ShadowNamelist* parent) 
    : parent_(parent)
{ }

//
// copy constructor: Copy only the instance variables, and set the pointer
// to the parent Namelist object.
//
ShadowNamelist::ShadowNamelist(const ShadowNamelist& nml)
    : parent_(nml.parent_)
{ 
    const VariableList& from_variables = nml.variables();
    VariableList::const_iterator it = from_variables.begin();
    for(; it != from_variables.end(); ++it) {
	const string& varname = *it;
        const Var* var = nml.get(varname);
	assert(var != nil);
	Var* newvar = var->clone();
	string local_name(string_tolower(varname));
	var_list_.push_back(varname);
	var_map_[local_name] = newvar;
    }
}

ShadowNamelist::~ShadowNamelist() {
    VariableMap::iterator it = var_map_.begin();
    for(; it != var_map_.end(); ++it) {
        Var* var = static_cast<Var*>((*it).second);
	delete var;
    }
}

ShadowNamelist* ShadowNamelist::clone() const { 
    return new ShadowNamelist(*this);
}

//
// better have the same parent!!
//
// DO NOT COPY VARIABLE MAPS, since the variable pointers are now in
// multiple places, hence will get destroyed > 1 time.
//
// eg., the second statement of the following 2
//      var_list_ = from.var_list_;
//      var_map_ = from.var_map_;
// is plain wrong.
//
int ShadowNamelist::copy(const ShadowNamelist& from) {
    if (this == &from)
	return 0;

    const VariableList& variables = from.variables();
    VariableList::const_iterator it = variables.begin();
    for(; it != variables.end(); ++it) {
        const Var* var = from.get(*it);
	assert(var != nil);
	Var* newvar = var->clone();
	if (exists(var->getname())) {
	    replace(newvar);
	} else {
	    add(newvar);
	}
    }
    return 0;
}
    
bool ShadowNamelist::exists(const string& name) const {
    string local_name(string_tolower(name));
    VariableMap::const_iterator it = var_map_.find(local_name);
    return (it != var_map_.end()) ? 
        true : ((parent_) ? parent_->exists(local_name) : false);
}

const Var* ShadowNamelist::get(const string& name) const {
    string local_name(string_tolower(name));
    VariableMap::const_iterator it = var_map_.find(local_name);
    const Var* var = (it != var_map_.end()) ? 
        reinterpret_cast<const Var*>((*it).second) : 
	((parent_) ? parent_->get(local_name) : nil);
    return var;
}

//
//
// ShadowNamelist::add: Adds a *new* variable to the namelist. If the
// name already exists in the namespace of the current namelist or in 
// any of the ancestors, return 1; otherwise add/return 0.
//
// note the distinction between varname and local_name. The list of
// variable names maintain the original case, but the Map only enters
// the lower-case version.
//
int ShadowNamelist::add(Var* var) {
    const string varname(var->getname());
    string local_name(string_tolower(varname));

    if (exists(local_name))
	return -1;
    var_map_[local_name] = var;
    var_list_.push_back(varname);
    return 0;
}

//
// ShadowNamelist::replace: Replaces an *existing* variable. By existence,
// it implies that the variable has to exist in the namespace of the current
// or in any of the ancestors. The new variable is added to the current
// namelist/return 0. If it doesn't exist, return -1.
//
int ShadowNamelist::replace(Var* var) {
    const string varname(var->getname());
    string local_name(string_tolower(varname));
    if (!exists(local_name))
	return -1;

    VariableMap::iterator it = var_map_.find(local_name);
    if (it != var_map_.end()) {
	Var* oldvar = static_cast<Var*>(var_map_[local_name]);
	delete oldvar;
    } else {				// add the name to instance list
	var_list_.push_back(varname);
    }
    var_map_[local_name] = var;
    return 0;
}

int ShadowNamelist::change(const string& varname, Value* value) {
    string local_name(string_tolower(varname));

    if (!exists(local_name))
	return -1;
    

    Var* var = 0;
    VariableMap::iterator it = var_map_.find(local_name);
    if (it == var_map_.end()) {
        const Var* default_var = get(local_name);
	var = default_var->clone();
	var_map_[local_name] = var;
	var_list_.push_back(varname);
    } else {
        var = static_cast<Var*>((*it).second);
    }
    var->setval(value);
    return 0;
}

const ShadowNamelist::VariableList& ShadowNamelist::variables() const { 
    return var_list_; 
}

void ShadowNamelist::dump() const {
    cerr << "dumping namelist ..." << endl;
    VariableMap::const_iterator it = var_map_.begin();
    for(; it != var_map_.end(); ++it) {
        const Var* var = reinterpret_cast<const Var*>((*it).second);
	cerr << "\t" << var->getname() << " = " 
	    << var->getval()->getstrval() << endl;
    }
    if (parent_)
        parent_->dump();
    cerr << "done." << endl;
}

// =================================================================== //
