//
// relation.cc: manage SET relation in SHADOW MENU description file
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

/***************************************************************************/

#ifdef __BORLANDC__
# include <assert>
# include <ctype>
# include <stdio>
# include <stdlib>
# include <string>
#else
# include <cassert>
# include <cctype>
# include <cstdio>
# include <cstdlib>
# include <cstring>
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

#include "relation.h"
#include "synonym.h"
#include "beamline.h"
#include "utils.h"
#include "value.h"
#include "variable.h"

/***********************************************************************/

static const int FLAG_RELATION = 1;
static const int BOOLEAN_RELATION = 2;
static const int ENUM_RELATION = 3;

Relation::Relation() {
    flag_values_ = new bool[26];
}

Relation::~Relation() {
    // CHECK/FIXME: delete all Relation items;
    delete[] flag_values_;
}

int Relation::add(RelationItem* item) {
    item_list_.push_back(item);
    item_names_.push_back(item->flag());
    return 0;
}

void Relation::reset_flags(bool def_value) {
    for(unsigned i = 0; i < 26; i++) {
	flag_values_[i] = def_value;
    }
}

bool Relation::get_flag_value(const char* flag) const {
    assert(isalpha(flag[0]));
    int flag_index = toupper(flag[0]) - 'A';
    assert(flag_index >= 0 && flag_index < 26);
    return flag_values_[flag_index];
}

void Relation::set_flag_value(const char* flag, bool flag_value) {
    assert(isalpha(flag[0]));
    int flag_index = toupper(flag[0]) - 'A';
    assert(flag_index >= 0 && flag_index < 26);
    flag_values_[flag_index] = flag_value;
}

int Relation::evaluate(const Beamline* bline, const SynonymTable* syn_table) {
    reset_flags();
    list<void*>::const_iterator it = item_list_.begin();
#if BC5_ITERATOR_BUG
    for(; !(it == item_list_.end()); ++it) {
#else
    for(; it != item_list_.end(); ++it) {
#endif
	const RelationItem* item = reinterpret_cast<const RelationItem*>(*it);
	eval_1_flag(bline, syn_table, item);
    }
    return 0;
}

int Relation::eval_1_flag(const Beamline* bline, const SynonymTable* syn_table,
    const RelationItem* rel_item
) {
    const char* flag = rel_item->flag();
    const char* rec1 = rel_item->rec1();
    const char* name1 = rel_item->name1();
    const char* rel = rel_item->rel();
    const char* rec2 = rel_item->rec2();
    const char* name2 = rel_item->name2();

    int lhs = 0;
    Value* lhs_value = nil;

    if (rec1[0] == '%') {
	bool val1 = get_flag_value(rec1 + 1);
	lhs = FLAG_RELATION;
	lhs_value = new BooleanValue(val1);
    } 
    else if (rec1[0] != '\0') {
	const Var* var = bline->get_var(rec1, name1);
	assert(var != nil);
	const Value* value1 = var->getval();
	assert(value1 != nil);
	if (value1->isaboolean()) {
	    bool val1 = value1->getboolean();
	    lhs = BOOLEAN_RELATION;
	    lhs_value = new BooleanValue(val1);
	}
	else if (value1->isaenum()) {
	    const char* enum_str = value1->getenum();
	    SynonymTable::MatchType match;
	    int val1 = syn_table->translate(name1, enum_str, match);
	    if (match == SynonymTable::NONE) {
		printf(
		    "ERROR: SET item 1 \"%s/%s\" has wrong enum val %s.\n",
		    rec1, name1, enum_str
		);
	    }
	    else if (match == SynonymTable::AMBIGUOUS) {
		printf(
		    "ERROR: SET item 1 \"%s/%s\" has ambiguous enum val %s.\n",
		    rec1, name1, enum_str
		);
	    }
	    lhs = ENUM_RELATION;
	    lhs_value = new IntegerValue(val1);
	}
	else {
	    printf("ERROR: SET item 1 \"%s/%s\" must be Y or E type\n",
		rec1, name1
	    );
	    assume(lhs != 0);
	}
    }

    Value* rhs_value = nil;
    int rhs = 0;

    if (rec2[0] == '%') {
	bool val2 = get_flag_value(rec2 + 1);
	rhs = FLAG_RELATION;
	rhs_value = new BooleanValue(val2);
    }
    else if (rec2[0] == '\0') {
	if (lhs == BOOLEAN_RELATION) {
	    bool val2 = (name2[0] == 'Y' || name2[0] == 'y');
	    rhs = BOOLEAN_RELATION;
	    rhs_value = new BooleanValue(val2);
	}
	else if (lhs == ENUM_RELATION) {
	    SynonymTable::MatchType match;
	    int val2 = syn_table->translate(name1, name2, match);
	    if (match == SynonymTable::NONE) {
		printf(
		    "ERROR: SET item 2 \"%s/%s\" has wrong enum val \"%s\".\n",
		    rec1, name1, name2
		);
	    }
	    else if (match == SynonymTable::AMBIGUOUS) {
		printf(
		    "ERROR: SET item 1 \"%s/%s\" has ambiguous enum val %s.\n",
		    rec1, name1, name2
		);
	    }
	    rhs = ENUM_RELATION;
	    rhs_value = new IntegerValue(val2);
	}
	else {
	    printf("BAD 2nd record in SET statement.\n");
	    assume(0);
	}
    }
    else if (rec2[0] != '\0') {
	const Var* var = bline->get_var(rec2, name2);
	assert(var != nil);
	const Value* value2 = var->getval();
	assert(value2 != nil);
	if (value2->isaboolean()) {
	    bool val2 = value2->getboolean();
	    rhs_value = new IntegerValue(val2);
	}
	else {
	    printf("ERROR: SET item 2 \"%s/%s\" must be Y/N type\n",
		rec2, name2
	    );
	    rhs_value = new BooleanValue(false);
	}
    }

    bool result = false;
    if (strcmp(rel, "EQ") == 0) {
	result = rhs_value->isequal(lhs_value);
    }
    else if (strcmp(rel, "NE") == 0) {
	result = !rhs_value->isequal(lhs_value);
    }
    else if (strcmp(rel, "AND") == 0) {
	assume(lhs_value->isaboolean() && rhs_value->isaboolean());
	result = lhs_value->getboolean() && rhs_value->getboolean();
    }
    else if (strcmp(rel, "OR") == 0) {
	assume(lhs_value->isaboolean() && rhs_value->isaboolean());
	result = lhs_value->getboolean() || rhs_value->getboolean();
    }
    else
	printf("eval_flags: bad rel_item \"%s\".\n", rel);
    
    set_flag_value(flag + 1, result);
    delete lhs_value;
    delete rhs_value;
    return 0;
}

void Relation::dump() const {
    printf ("DUMPING relation... \n");
    list<void*>::const_iterator it = item_list_.begin();
    for(; it != item_list_.end(); ++it) {
	const RelationItem* item = reinterpret_cast<const RelationItem*>(*it);
	item->dump();
    }
    printf ("done DUMPING relation.\n");
}

/***********************************************************************/

RelationItem::RelationItem(const char* flag,
    const char* rec1, const char* name1, 
    const char* rel, 
    const char* rec2, const char* name2,
    bool def_value
) {
    flag_ = strcpy_with_alloc(flag);
    rec1_ = strcpy_with_alloc(rec1);
    name1_ = strcpy_with_alloc(name1);
    rel_ = strcpy_with_alloc(rel);
    rec2_ = strcpy_with_alloc(rec2);
    name2_ = strcpy_with_alloc(name2);
    value_ = def_value;
}

RelationItem::~RelationItem() {
    delete[] flag_;
    delete[] rec1_;
    delete[] name1_;
    delete[] rel_;
    delete[] name2_;
    delete[] rec2_;
}

const char* RelationItem::flag() const { return flag_; }
const char* RelationItem::rec1() const { return rec1_; }
const char* RelationItem::name1() const { return name1_; }
const char* RelationItem::rel() const { return rel_; }
const char* RelationItem::rec2() const { return rec2_; }
const char* RelationItem::name2() const { return name2_; }
bool RelationItem::value() const { return value_; }
void RelationItem::value(bool newval) { value_ = newval; }

void RelationItem::dump() const {
    printf("DUMP RELATION: %s = (%s) %s .%s. (%s) %s.\n", 
	flag(), rec1(), name1(), rel(), rec2(), name2()
    );
}

/***********************************************************************/
