/*
 * shdata_io.c: Routines for reading/writing SHADOW image files.
 *
 * ------------------------------------------------
 * Mumit Khan <khan@xraylith.wisc.edu>
 * Center for X-ray Lithography
 * University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 * Copyright (c) 1996 Mumit Khan
 */

/* ======================================================================= */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include "shdata_io.h"
#include "blt.h"

/* ======================================================================= */

static inline double *allocate_space (int ncol, int npoints) {
    /* allocate the RAY[N_DIM][NCOL] array as 1 long vector. */
    double *ray = 0;
    size_t nmemb = npoints * ncol;
    if (nmemb == 0)
        nmemb = 1;			/* for empty structures */
    ray = (double*) calloc (nmemb, sizeof(double));
    return ray;
}

static inline int check_column (Tcl_Interp* interp, 
    const ShadowData* shdata, int column
) {
    if (column < 1 || column > shdata->ncol) {
	char buf[256];
	sprintf(buf, "%d", column);
	Tcl_AppendResult(interp, "Invalid column index: ", buf, ".", 0);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static inline int check_ray (Tcl_Interp* interp, 
    const ShadowData* shdata, int ray
) {
    if (ray < 1 || ray > shdata->npoints) {
	char buf[256];
	sprintf(buf, "%d", ray);
	Tcl_AppendResult(interp, "Invalid ray index: ", buf, ".", 0);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * get_BLT_vector: Gets the BLT vector named "vecname" and assign the 
 * data to it. The vector is created if it doesn't exist and then Reset 
 * to match the supplied data. BLT vector manages the storage from now on.
 */
static int get_BLT_vector (Tcl_Interp *interp, char *vecname,
    double* data, size_t len
) {
    /* create only if it doesn't exist yet. */
    if (!Blt_VectorExists(interp, vecname)) {
        if (Blt_CreateVector(interp, vecname, 0, 0) != TCL_OK) {
	    Tcl_AppendResult(interp, "Error creating vector \"",
	        vecname, "\".", 0
	    );
	    return TCL_ERROR;
	}
    }

    /* Get the vector */
    Blt_Vector vec_info;
    if (Blt_GetVector(interp, vecname, &vec_info) != TCL_OK) {
	Tcl_AppendResult(interp, "Error creating vector \"",
	    vecname, "\".", 0
	);
	return TCL_ERROR;
    }

    vec_info.numValues = len;
    vec_info.valueArr = data;

    /* 
     * Reset the vector. Clients get notified. BLT deletes the data
     * when done with it.
     */
    if (Blt_ResetVector(interp, vecname, &vec_info, TCL_DYNAMIC) != TCL_OK) {
	Tcl_AppendResult(interp, "Error resetting vector \"",
	    vecname, "\".", 0
	);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/* ======================================================================= */

/*
 * create_shadow_data: Creates a new structure with shadow data info
 * in it. Allow for potentially illegal data, such as initializing with
 * 0 in all fields.
 */
ShadowData *create_shadow_data (Tcl_Interp *interp, int ncol, 
    int npoints, int flag
) {
    ShadowData *shadow_data = 0;
    shadow_data = (ShadowData*) malloc (sizeof (ShadowData));
    if (shadow_data == 0) {
	return NULL;
    }
    memset(shadow_data, 0, sizeof(ShadowData));
    shadow_data->interp = interp;
    shadow_data->ncol = ncol;
    shadow_data->npoints = npoints;
    shadow_data->flag = flag;
    shadow_data->ray = allocate_space(ncol, npoints);
    if (shadow_data->ray == 0) {
        free(shadow_data);
	return NULL;
    }
    return shadow_data;
}

int delete_shadow_data (ShadowData *shadow_data) {
    free(shadow_data->ray);
    free(shadow_data);
    return 0;
}

int reset_shadow_data (ShadowData *shadow_data, int ncol, int npoints, 
    int flag, int good_rays, double *ray
) {
    shadow_data->ncol = ncol;
    shadow_data->npoints = npoints;
    shadow_data->flag = flag;
    shadow_data->good_rays = good_rays;
    if (shadow_data->ray)
        free(shadow_data->ray);
    shadow_data->ray = ray;
    return 0;
}

/*
 * read_shadow_data: Reads a SHADOW ray file and returns the info in a
 * ShadowData structure. Returns TCL_OK if everything goes ok, otherwise
 * returns TCL_ERROR with the appropriate errorInfo. 
 */

int read_shadow_data (Tcl_Interp *interp,
    const char *imagefile, ShadowData *shadow_data
) {
    FILE *fp;
    int i, j;

    int io_dummy[5];
    int reclen;
    int ncol, npoints, flag;
    int nlost = 0;
    double *ray = 0;

    fp = fopen(imagefile, "rb");
    if (!fp) {
	Tcl_AppendResult(interp, "RBEAM18:: Error opening ray file \"",
            imagefile, "\".", 0
	);
        return TCL_ERROR;
    }
    if (fread(io_dummy, sizeof(int), 5, fp) != 5) {
	Tcl_AppendResult(interp, "RBEAM18:: Error reading ray file \"",
	    imagefile, "\".", 0
	);
	fclose(fp);
        return TCL_ERROR;
    }
    /*
     * io_dummy[0] and io_dummy[4] are the F77 record lengths, so ignore.
     */
    ncol = io_dummy[1];
    npoints = io_dummy[2];
    flag = io_dummy[3];

    if (ncol != 12 && ncol != 13 && ncol != 18) {
	char buf[256];
	sprintf(buf, "%d", ncol);
	Tcl_AppendResult(interp, "RBEAM18:: Invalid number of columns ",
	    buf, " in image file \"", imagefile, "\".", 0
	);
	fclose(fp);
	return TCL_ERROR;
    }

    ray = allocate_space(ncol, npoints);
    if (!ray) {
	Tcl_AppendResult(interp, 
	    "RBEAM18:: Not enough memory to read in SHADOW ray data.", 0
        );
	fclose(fp);
	return TCL_ERROR;
    }

    reclen = ncol * sizeof(double);
    /*
     * read the data, but not in the same order as done in the data file.
     * we store by columns, instead of by points, which makes it easier
     * to manage the data by columns needed for analyses. Not as efficient
     * for reading and writing, but that's typically done lot less often
     * as extracting by columns in the analysis and plotting.
     */
    for (i = 0; i < npoints; ++i) {
	fread(&reclen, sizeof(int), 1, fp);
	for (j = 0; j < ncol; ++j) {
	    size_t offset = npoints * j + i;
	    fread(&ray[offset], sizeof(double), 1, fp);
	    if (j == RAY_GOOD-1 && ray[offset] < 0.0)
	        ++nlost;
	}
	fread(&reclen, sizeof(int), 1, fp);
    }
    fclose(fp);

    reset_shadow_data(shadow_data, ncol, npoints, flag, npoints-nlost, ray);
    return TCL_OK;
}

/*
 * write_shadow_data: Reads a SHADOW ray file. This may or may not work
 * since F77 unformatted files are god knows in what format.
 * Returns TCL_OK if everything goes ok, otherwise returns TCL_ERROR with 
 * the appropriate errorInfo. 
 */

int write_shadow_data (Tcl_Interp *interp,
    const char *imagefile, const ShadowData *shadow_data
) {
    FILE *fp;
    int i, j;

    int io_dummy[3];
    int reclen;

    if (shadow_data->ncol != 12 && shadow_data->ncol != 13 &&
	shadow_data->ncol != 18
    ) {
	char buf[256];
	sprintf(buf, "%d", shadow_data->ncol);
	Tcl_AppendResult(interp, "WRITEOFF:: Invalid number of columns: ",
	    buf, ".", 0
	);
	return TCL_ERROR;
    }

    fp = fopen(imagefile, "wb");
    if (!fp) {
	Tcl_AppendResult(interp, "WRITEOFF:: Error opening output file \"",
            imagefile, "\".", 0
	);
        return TCL_ERROR;
    }
    io_dummy[0] = shadow_data->ncol;
    io_dummy[1] = shadow_data->npoints;
    io_dummy[2] = shadow_data->flag;
    reclen = sizeof(io_dummy);
    fwrite(&reclen, sizeof(int), 1, fp);
    fwrite(io_dummy, sizeof(io_dummy), 1, fp);
    fwrite(&reclen, sizeof(int), 1, fp);
    reclen = shadow_data->ncol * sizeof(double);
    /*
     * write the data, but reverse the order in the RAY array. We store
     * in the other order for efficient extraction (see commens in the
     * read_shadow_data routine).
     */
    for (i = 0; i < shadow_data->npoints; ++i) {
	fwrite(&reclen, sizeof(int), 1, fp);
	for (j = 0; j < shadow_data->ncol; ++j) {
	    size_t offset = shadow_data->npoints * j + i;
	    fwrite(&shadow_data->ray[offset], sizeof(double), 1, fp);
	}
	fwrite(&reclen, sizeof(int), 1, fp);
    }
    fclose(fp);
    return TCL_OK;
}

/*
 * extract_shadow_raw_column: Extract the raw data corresponding to the
 * column. The data is read-only, so don't touch.
 */
int extract_shadow_raw_column (Tcl_Interp *interp, 
    const ShadowData *shadow_data, int column, double **coldata
) {
    if (check_column (interp, shadow_data, column) != TCL_OK)
        return TCL_ERROR;

    size_t offset = shadow_data->npoints * (column - 1);
    *coldata = &shadow_data->ray[offset];
    return TCL_OK;
}

/*
 * extract_shadow_raw_ray: Extract the raw data corresponding to the
 * ray. The data is read-only, so don't touch.
 */
int extract_shadow_raw_ray (Tcl_Interp *interp, const ShadowData *shadow_data,
    int ray, double **raydata
) {
    static double *data = 0;
    static int alloc_len = 0;

    if (check_ray (interp, shadow_data, ray) != TCL_OK)
        return TCL_ERROR;

    if (!data) {
	data = (double*) malloc (sizeof(double) * shadow_data->ncol);
	if (!data) {
	    Tcl_AppendResult(interp, 
	        "Error creating data (Out of memory?)", 0
	    );
	    alloc_len = 0;
	    return TCL_ERROR;
	}
	alloc_len = shadow_data->ncol;
    } else if (shadow_data->ncol > alloc_len) {
        data = (double*) realloc(data, shadow_data->ncol * sizeof(double));
	if (!data) {
	    Tcl_AppendResult(interp,
	        "Error reallocating data (Out of memory?)", 0
	    );
	    alloc_len = 0;
	    return TCL_ERROR;
	}
	alloc_len = shadow_data->ncol;
    }

    for (int i = 0; i < shadow_data->ncol; ++i) {
        size_t offset = i * shadow_data->npoints + ray;
	data[i] = shadow_data->ray[offset];
    }

    *raydata = data;
    return TCL_OK;
}

/*
 * extract_shadow_column: Make a BLT vector out a particular column in
 * SHADOW data. The vector is created if it doesn't exist and then
 * Reset to match the column data.
 */
int extract_shadow_column (Tcl_Interp *interp, const ShadowData *shadow_data,
    int column, char *vecname, int which
) {
    if (check_column (interp, shadow_data, column) != TCL_OK)
        return TCL_ERROR;

    int npoints;
    if (which == GOOD_RAYS)
	npoints = shadow_data->good_rays;
    else if (which == LOST_RAYS)
	npoints = shadow_data->npoints - shadow_data->good_rays;
    else 
	npoints = shadow_data->npoints;

    double* data = (double*) malloc (sizeof(double) * npoints);
    if (!data && npoints) {
	Tcl_AppendResult(interp, "Error creating vector \"",
	    vecname, "\". (Out of memory?)", 0
	);
	return TCL_ERROR;
    }
    size_t offset = shadow_data->npoints * (column - 1);
    if (which == ALL_RAYS) {			/* Optimize */
	memcpy(data, &shadow_data->ray[offset], 
	    shadow_data->npoints * sizeof(double));
    } else {
	const double* ray_data = &shadow_data->ray[offset];
	size_t npcol = 0;
	for (int pt = 0; pt < shadow_data->npoints; ++pt) {
	    bool lost = (column_of_point(shadow_data, pt, RAY_GOOD-1) < 0.0);
	    if (which == GOOD_RAYS && !lost ||
		which == LOST_RAYS && lost  ||
		which == ALL_RAYS
	    ) {
		const double& ptval = ray_data[pt];
		data[npcol] = ptval;
		++npcol;
	    }
	}
	assert (npcol == npoints);
    }
    return get_BLT_vector(interp, vecname, data, npoints);
}

/*
 * extract_shadow_column: Make a BLT vector out a particular column in
 * SHADOW data. The vector is created if it doesn't exist and then
 * Reset to match the column data.
 */
int extract_shadow_column (Tcl_Interp *interp, const ShadowData *shadow_data,
    int column, char *vecname_good, char* vecname_lost
) {
    return extract_shadow_column(
	    interp, shadow_data, column, vecname_good, GOOD_RAYS
    ) || extract_shadow_column(
        interp, shadow_data, column, vecname_lost, LOST_RAYS
    );
}

/*
 * extract_shadow_ray: Make a BLT vector out a particular ray in
 * SHADOW data. The vector is created if it doesn't exist and then
 * Reset to match the ray data.
 */
int extract_shadow_ray (Tcl_Interp *interp, const ShadowData *shadow_data,
    int ray, char *vecname
) {
    if (check_ray (interp, shadow_data, ray) != TCL_OK)
        return TCL_ERROR;

    double* data = (double*) malloc (sizeof(double) * shadow_data->ncol);
    if (!data) {
	Tcl_AppendResult(interp, "Error creating vector \"",
	    vecname, "\". (Out of memory?)", 0
	);
	return TCL_ERROR;
    }
    for (int i = 0; i < shadow_data->ncol; ++i) {
        size_t offset = i * shadow_data->npoints + ray;
	data[i] = shadow_data->ray[offset];
    }
    return get_BLT_vector (interp, vecname, data, shadow_data->ncol);
}

/*
 * extract_shadow_histogram: Make a BLT vector out the histogram of a
 * given column data. The vector is created if it doesn't exist and then
 * Reset to match the column data.
 */
int extract_shadow_histogram (Tcl_Interp *interp, 
    const ShadowData *shadow_data, int column, 
    char *xvecname, char* yvecname, int which,
    double min, double max, int bins, double scale
) {
    if (check_column (interp, shadow_data, column) != TCL_OK)
        return TCL_ERROR;

    double* coldata;
    if (extract_shadow_raw_column (interp, shadow_data, column, &coldata)
        != TCL_OK
    ) {
	Tcl_AppendResult(interp, "Error extracting column for histogram", 0);
	return TCL_ERROR;
    }

    double* data = (double*) calloc (bins, sizeof(double));
    double* xvec = (double*) calloc (bins, sizeof(double));
    if ((!data || !xvec) && bins > 0) {
	Tcl_AppendResult(interp, 
	    "Error creating histogram vectors. (Out of memory?)", 0
	);
	return TCL_ERROR;
    }

    const double width = max - min;
    const double half_width = width/2.0;
    const double center = (max + min)/2.0;
    const double step = width/(bins - 1);
    const double half_step = step/2.0;
    const double start = center - half_width - half_step;

    for (size_t i = 0; i < bins; ++i) {
        xvec[i] = start + step * i + half_step;
    }

    for (size_t pt = 0; pt < shadow_data->npoints; ++pt) {
        const double pos = coldata[pt];
	int bin = static_cast<int>((pos - start)/step);

	//
	// CHECK/FIXME:
	// we can go out of bounds since the user specified limits may
	// be a subset of the data space, so ignore out-of-bounds instead
	// of dying for it.
	//
	if (bin < 0 || bin >= bins) {
	    continue;
	}

	bool lost = (column_of_point(shadow_data, pt, RAY_GOOD-1) < 0.0);
	if (which == ALL_RAYS ||
	    which == GOOD_RAYS && !lost ||
	    which == LOST_RAYS && lost
	) 
	    data[bin] += scale;
    }

    bool all_ok = 
        get_BLT_vector (interp, xvecname, xvec, bins) == TCL_OK &&
        get_BLT_vector (interp, yvecname, data, bins) == TCL_OK;
    return (all_ok) ? TCL_OK : TCL_ERROR;
}

/* ======================================================================= */
