//
// menu-item.h: Items in a MENU 
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

#ifndef lib_menu_item_h
#define lib_menu_item_h

// =================================================================== //

#include "xrtypes.h"
#include <string>

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

// =================================================================== //
//
// MenuItem defines a unique entry in a MenuPage. Each item can be of
// the following types: Data, Text, Skip. Depending on the type of item,
// each MenuItem has additional data/info, and each subtype is implemented
// as a derived type from the base MenuItem class.
//
// MenuItem's can be shared among pages, but currently it is upto the
// clients to decide that. Item's do not encode which page(s) it belongs
// to.
//
// =================================================================== //

//
// class MenuItem: base class for MENU items. Don't use RTTI until it
// becomes widely available on various compilers.
//

class MenuItem {
public:
    enum {TEXT_ITEM, SKIP_ITEM, DATA_ITEM};
    typedef int MenuItemType;
public:
    virtual ~MenuItem();
    virtual const string& name() const;
    virtual const MenuItemType type() const;
    virtual const string& submenu() const;

    virtual void dump() const;
protected:
    MenuItem(
	const string& name, const MenuItemType type, 
        const string& submenu = ""
    );
private:
    string name_;
    MenuItemType type_;
    string submenu_;
};

inline const string& MenuItem::name() const { return name_; }
inline const MenuItem::MenuItemType MenuItem::type() const { return type_; }
inline const string& MenuItem::submenu() const { return submenu_; }

class TextMenuItem : public MenuItem {
public:
    TextMenuItem(
	const string& name, const string& text, const string& submenu = ""
    );
    virtual ~TextMenuItem();
    virtual const string& text() const;

    virtual void dump() const;
private:
    string text_;
};

inline const string& TextMenuItem::text() const { return text_; }

class SkipMenuItem : public MenuItem {
public:
    SkipMenuItem(const string& name, const string& submenu = "");
    virtual ~SkipMenuItem();

    virtual void dump() const;
};

class DataMenuItem : public MenuItem {
public:
    enum {READ_ONLY = 0x01, FORCE_UPDATE = 0x02} ItemFlags;
public:
    DataMenuItem(
	const string& name, const string& record, const string& prompt, 
	const string& submenu, const char* flag, unsigned i_flags, 
	int owner1, int owner2
    );
    virtual ~DataMenuItem();

    virtual const string& record() const;
    virtual const string& prompt() const;
    virtual const char* flag() const;
    virtual unsigned flags() const;
    virtual int owner1() const;
    virtual int owner2() const;

    virtual void dump() const;
private:
    string record_;
    string prompt_;
    char* flag_;
    unsigned i_flags_;
    int owner1_;
    int owner2_;
};

inline const string& DataMenuItem::record() const { return record_; }
inline const string& DataMenuItem::prompt() const { return prompt_; }
inline const char* DataMenuItem::flag() const { return flag_; }
inline unsigned DataMenuItem::flags() const { return i_flags_; }
inline int DataMenuItem::owner1() const { return owner1_; }
inline int DataMenuItem::owner2() const { return owner2_; }

// =================================================================== //

#endif/*lib_menu_item_h*/
