//
// beamline.cc: SHADOW beamline system definition
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1994-1996 Mumit Khan
//

// =================================================================== //

#ifdef __GNUG__
# include <cassert>
# include <cstdio>
# include <cstdlib>
# include <cstring>
# include <fstream.h>
#else
# include <assert>
# include <stdio>
# include <stdlib>
# include <string>
# include <fstream>
#endif

#if !CXX_NO_NAMESPACE
using namespace std;
#endif

#if defined(NO_STRNCASECMP_PROTO)
extern "C" int strcasecmp(const char*, const char*);
extern "C" int strncasecmp(const char*, const char*, int);
#endif

#include "global.h"
#include "beamline.h"
#include "pmu.h"
#include "synonym.h"
#include "value.h"
#include "variable.h"

// =================================================================== //

static char* trim(char* str) {
    if (str == nil)
	return nil;
    else if (!*str)
	return str;
    char* p = str;
    for(; *p && (*p == ' ' || *p == '\t'); ++p)
	;
    char* q = &str[strlen(p)-1];
    q = (q > p) ? q : p;
    for(; q >= p && *q && (*q == ' ' || *q == '\t'); --q)
	;
    *(++q) = '\0';
    return p;
}

static char* trimright(char* str) {
    if (str == nil)
	return nil;
    else if (!*str)
	return str;
    char* p = str;
    char* q = &str[strlen(p)-1];
    q = (q > p) ? q : p;
    for(; q >= p && *q && (*q == ' ' || *q == '\t'); --q)
	;
    *(++q) = '\0';
    return p;
}

// =================================================================== //

//
// class BLToolInstance: used to  store an instance of a beamline tool.
// This is really only for lack of usable RTTI on various platforms.
//
BLToolInstance::BLToolInstance (
    const Tool* tool, int instance, int tool_type
) : ToolInstance(tool, instance), tool_type_(tool_type) { }

BLToolInstance::BLToolInstance(const BLToolInstance& bl_ti) :
    ToolInstance(bl_ti),
    tool_type_(bl_ti.tool_type_)
{ }

BLToolInstance::~BLToolInstance() { }

BLToolInstance* BLToolInstance::clone() const {
    return new BLToolInstance(*this);
}

int BLToolInstance::copy(const BLToolInstance& from) {
    if (this == &from)
	return 0;

    int retcode = 0;
    if (tool_type_ != from.tool_type_) {
        cerr << "BLToolInstance::copy: Incompatible types!" << endl;
	retcode = 1;
    } else {
	retcode = ToolInstance::copy(from);
    }
    return retcode;
}

int BLToolInstance::tool_type() const { return tool_type_; }

bool BLToolInstance::is_source() const { return tool_type_ == source; }
bool BLToolInstance::is_oe() const { return tool_type_ == oe; }
bool BLToolInstance::is_screen() const { return tool_type_ == screen; }
bool BLToolInstance::is_inspector() const { return tool_type_ == inspector; }
bool BLToolInstance::is_unspecified() const { return tool_type_ == unspecified; }

void BLToolInstance::dump() const { ToolInstance::dump(); }

// =================================================================== //

Inspector::Inspector (
    const Tool* tool, int id, const BLToolInstance* target
) : BLToolInstance(tool, id, BLToolInstance::inspector), 
    id_(id),
    target_(target)
{ }

Inspector::Inspector (const Inspector& inspector) :
    BLToolInstance(inspector),
    id_(inspector.get_id()),
    target_(inspector.get_target())
{ }
    
Inspector::~Inspector() { }

Inspector* Inspector::clone() const {
    return new Inspector(*this);
}

int Inspector::copy(const BLToolInstance& from) {
    if (this == &from)
	return 0;
    
    if (!from.is_inspector())
	return 1;
    
    const Inspector& inspector = static_cast<const Inspector&>(from);

    int	retcode = BLToolInstance::copy(inspector);
    id_ = inspector.get_id();
    target_ = inspector.get_target();
    return retcode;
}


const BLToolInstance* Inspector::get_target() const {
    return target_;
}

int Inspector::get_id() const { return id_; }
void Inspector::set_id(int id) { id_ = id; }

void Inspector::dump() const { 
    cerr << "dumping INSPECTOR ..." << endl;
    BLToolInstance::dump();
    cerr << "dumping INSPECTOR ... done." << endl;
}

// =================================================================== //

InspectorMgr::InspectorMgr() { }

InspectorMgr::InspectorMgr(const InspectorMgr& from) {
    for (int i = 1; i <= from.num_inspectors(); ++i) {
	const Inspector* from_i = from.get_inspector(i);
	assert(from_i != nil);
	Inspector* inspector = from_i->clone();
	add_inspector(inspector, i);
    }
}

void InspectorMgr::clear() {
    deque<void*>::iterator it = inspectors_.begin();
    for(; it != inspectors_.end(); ++it) {
	Inspector* inspector = static_cast<Inspector*>(*it);
	assert (inspector != nil);
	delete inspector;
    }
    inspectors_.erase(inspectors_.begin(), inspectors_.end());
}

InspectorMgr::~InspectorMgr() { 
    clear();
}

InspectorMgr* InspectorMgr::clone() const {
    return new InspectorMgr(*this);
}

int InspectorMgr::copy(const InspectorMgr& from) {
    if (this == &from)
	return 0;
    
    clear();
    for (int i = 1; i <= from.num_inspectors(); ++i) {
	const Inspector* from_i = from.get_inspector(i);
	assert(from_i != nil);
	Inspector* inspector = from_i->clone();
	assert(add_inspector(inspector, i) == 0);
    }
    return 0;
}

int InspectorMgr::num_inspectors() const { 
    return inspectors_.size(); 
}

int InspectorMgr::add_inspector(Inspector* inspector, int id) {
    if (id < 1 || id > inspectors_.size() + 1)
        return 1;
    deque<void*>::iterator it = inspectors_.begin() + id - 1;
    inspectors_.insert(it, inspector);
    renumber_inspectors();
    return 0; 
}

int InspectorMgr::remove_inspector(int id) {
    if (id < 1 || id > inspectors_.size())
        return 1;
    deque<void*>::iterator it = inspectors_.begin() + id - 1;
    Inspector* inspector = static_cast<Inspector*>(*it);
    assert(inspector != nil);
    delete inspector;
    inspectors_.erase(it);
    renumber_inspectors();
    return 0;
}

Inspector* InspectorMgr::get_inspector(int index) const {
    return (index < 1 || index > inspectors_.size()) ? 
        nil : static_cast<Inspector*>(inspectors_[index-1]);
}

Inspector* InspectorMgr::get_inspector(const string& toolname, int id) const {
    Inspector* inspector = nil;
    deque<void*>::const_iterator it = inspectors_.begin();
    for(; it != inspectors_.end(); ++it) {
	const Inspector* this_inspector = static_cast<Inspector*>(*it);
	if (toolname == this_inspector->name() && 
	    id == this_inspector->get_id()
	) {
	    inspector = const_cast<Inspector*>(this_inspector);
	    break;
	}
    }
    return inspector;
}

int InspectorMgr::renumber_inspectors() {
    for(int i = 1; i <= inspectors_.size(); ++i) {
	Inspector* inspector = static_cast<Inspector*>(inspectors_[i-1]);
	inspector->set_id(i);
    }
    return 0;
}

void InspectorMgr::dump() const {
    for(int i = 1; i <= inspectors_.size(); ++i) {
	Inspector* inspector = static_cast<Inspector*>(inspectors_[i-1]);
	inspector->dump();
    }
}


// =================================================================== //

Source::Source(const Tool* tool) : 
    BLToolInstance(tool, 0, BLToolInstance::source),
    InspectorMgr()
{ }

Source::Source(const Source& source) :
    BLToolInstance(source),
    InspectorMgr(source)
{ }

Source::~Source() { }

Source* Source::clone() const {
    return new Source(*this);
}

int Source::copy(const BLToolInstance& from) {
    if (this == &from)
	return 0;

    if (!from.is_source())
	return 1;
    
    const Source& source = static_cast<const Source&>(from);

    if (BLToolInstance::copy(source) || InspectorMgr::copy(source))
	return 1;
    
    return 0;
}

void Source::dump() const {
    cerr << "dumping SOURCE ..." << endl;
    BLToolInstance::dump();
    InspectorMgr::dump();
    cerr << "done dumping SOURCE" << endl;
}

// =================================================================== //

OE::OE(const Tool* tool, int id) : 
    BLToolInstance(tool, id, BLToolInstance::oe), 
    InspectorMgr(),
    id_(id)
{ }

OE::OE(const OE& oe) : 
    BLToolInstance(oe), 
    InspectorMgr(oe),
    id_(oe.id_)
{ 
    //
    // now copy all the screens as well. what a pain!
    //
    for(int i = 1; i <= oe.num_scr(); ++i) {
	const Screen* screen = oe.get_screen(i);
	assert(screen != nil);
	Screen* new_screen = screen->clone();
	new_screen->set_id(id_, i);
	//
	// now insert the screen in the deque in the right position.
	//
	deque<void*>::iterator it = scr_list_.begin() + i - 1;
	scr_list_.insert(it, new_screen);
    }
    reset_scr_variables();
}

int OE::clear() {
    int num_screen = num_scr();
    //
    // do *not* use num_scr() in the condition below since it changes with
    // every call to remove_scr().
    //
    for(int i = 0; i <= num_screen; ++i) {
	remove_scr(1);		// note that we *always* remove scr 1!
    }
    reset_scr_variables();
    return 0;
}

OE::~OE() { 
    clear();
}

OE* OE::clone() const {
    return new OE(*this);
}

int OE::copy(const BLToolInstance& from) {
    if (this == &from)
	return 0;

    if (!from.is_oe())
	return 1;
    
    const OE& oe = static_cast<const OE&>(from);


    if (BLToolInstance::copy(oe) || InspectorMgr::copy(oe))
	return 1;
    
    //
    // FIXME/CHECK: Is it kosher to copy the OE ID as well?? Does that
    // screw up the sequenc'ing. The caller better renumber the sequence,
    // things stay consistent. 
    //

    id_ = oe.get_id();

    //
    // now copy all the screens as well. what a pain!
    //
    clear();
    for(int i = 1; i <= oe.num_scr(); ++i) {
	const Screen* screen = oe.get_screen(i);
	assert(screen != nil);
	Screen* new_screen = screen->clone();
	new_screen->set_id(id_, i);
	//
	// now insert the screen in the deque in the right position.
	//
	deque<void*>::iterator it = scr_list_.begin() + i - 1;
	scr_list_.insert(it, new_screen);
    }
    reset_scr_variables();
    return 0;
}

int OE::get_id() const { return id_; }

void OE::set_id(int id) {
    id_ = id;
    deque<void*>::iterator it = scr_list_.begin();
    for(int scr = 1; it != scr_list_.end(); ++it, ++scr) {
        Screen* screen = static_cast<Screen*>(*it);
	assert(screen != nil);
	screen->set_id(id_, scr);
    }
}

int OE::num_scr() const { return scr_list_.size(); }

int OE::reset_scr_variables() {
    //
    // set the f_screen and n_screen variables here. This is all a real
    // kludge because of the way SHADOW defines Screens inside OEs.
    //
    int numscr = num_scr();
    const Value* f_screen = get_varval("f_screen");
    assert(f_screen != nil);
    Value* new_f_screen = f_screen->clone((numscr) ? "ON" : "OFF");
    assert(new_f_screen != nil);
    set_varval("f_screen", new_f_screen);

    const Value* n_screen = get_varval("n_screen");
    assert(n_screen != nil);

    char buf[32];
    sprintf(buf, "%d", numscr);
    Value* new_n_screen = n_screen->clone(buf);
    assert(new_n_screen != nil);
    set_varval("n_screen", new_n_screen);

    return 0;
}

int OE::add_scr(const Tool* scr_tool, int req, int& actual) {
    if (req < 1) {
	actual = 0;
	return -1;
    }
    if (req > num_scr()) {
	actual = num_scr() + 1;
    } else {
	actual = req;
    }
    Screen* screen = new Screen(scr_tool, id_, actual);
    //
    // now insert the screen in the deque in the right position.
    //
    deque<void*>::iterator it = scr_list_.begin() + actual - 1;
    scr_list_.insert(it, screen);

    //
    // reset the f_screen and n_screen internal variables now.
    //
    reset_scr_variables();

    return 0; 
}

int OE::remove_scr(int scr) { 
    int retcode = 0;
    if (scr < 1 || scr > num_scr()) {
	retcode = -1;
    } else {
	deque<void*>::iterator it = scr_list_.begin() + scr - 1;
	Screen* screen = static_cast<Screen*>(*it);
	assert(screen != nil);
	delete screen;
	scr_list_.erase(it);
	//
	// reset the f_screen and n_screen internal variables now.
	//
	reset_scr_variables();
    }
    return retcode;
}

Screen* OE::get_screen(int scr) const {
    Screen* scrn = nil;
    if (scr >= 1 && scr <= num_scr()) {
	scrn = static_cast<Screen*>(scr_list_[scr-1]);
	assert(scrn != nil);
    }
    return scrn;
}
    
void OE::dump() const {
    cerr << "dumping OE " << id_ << " ..." << endl;
    BLToolInstance::dump();
    cerr << "number of screens: " << scr_list_.size() << endl;
    deque<void*>::const_iterator it = scr_list_.begin();
    for(; it != scr_list_.end(); ++it) {
	const Screen* screen = reinterpret_cast<const Screen*>(*it);
	screen->dump();
    }
    InspectorMgr::dump();
    cerr << "done dumping OE " << id_ << endl;
}

// =================================================================== //

Screen::Screen(const Tool* tool, int oe, int scr) : 
    BLToolInstance(tool, scr, BLToolInstance::screen), 
    InspectorMgr(),
    oe_(oe), 
    scr_(scr) 
{ }

Screen::Screen(const Screen& screen) : 
    BLToolInstance(screen),
    InspectorMgr(screen),
    oe_(screen.oe_), 
    scr_(screen.scr_) 
{ }

Screen::~Screen() { }

Screen* Screen::clone() const {
    return new Screen(*this);
}

int Screen::copy(const BLToolInstance& from) {
    if (this == &from)
	return 0;

    if (!from.is_screen())
	return 1;
    
    const Screen& screen = static_cast<const Screen&>(from);

    if (BLToolInstance::copy(screen) || InspectorMgr::copy(screen))
	return 1;

    screen.get_id(oe_, scr_);
    return 0;
}

void Screen::get_id(int& oe, int& scr) const { 
    oe = oe_;
    scr = scr_;
}

void Screen::set_id(int oe, int scr) { 
    oe_ = oe;
    scr_ = scr;
}

void Screen::dump() const {
    cerr << "dumping SCREEN " << scr_ << " of OE " << oe_ << "..." << endl;
    BLToolInstance::dump();
    InspectorMgr::dump();
    cerr << "done SCREEN " << scr_ << " of OE " << oe_ << "..." << endl;
}

// =================================================================== //

Beamline::Beamline(const ToolMgr& toolmgr, const SynonymTable& syn_table) : 
    toolmgr_(toolmgr), 
    syn_table_(syn_table),
    src_(nil),
    cur_oe_(0),
    cur_scr_(0),
    cur_inspector_(0),
    cur_selection_(0),
    cur_editing_(0),
    clipboard_(0),
    saved_state_(0)
{ }

//
// Can we actually initialize the InspectorMgr this way, or do we have
// to wait till the ToolMgr is initialized in the current class? Wait
// and see I guess.
// 
Beamline::Beamline(const Beamline& from) : 
    InspectorMgr(from),
    toolmgr_(from.toolmgr_), 
    syn_table_(from.syn_table_),
    src_((from.src_) ? from.src_->clone() : nil),
    cur_oe_(from.cur_oe_),
    cur_scr_(from.cur_scr_),
    cur_inspector_(from.cur_inspector_),
    cur_selection_(from.cur_selection_),
    cur_editing_(from.cur_editing_),
    clipboard_((from.clipboard_) ? from.clipboard_->clone() : nil),
    saved_state_((from.saved_state_) ? from.saved_state_->clone() : nil)
{ 
    for (
        deque<void*>::const_iterator it = from.oe_list_.begin();
	it != from.oe_list_.end(); 
	++it
    ) {
        const OE* oelem = reinterpret_cast<const OE*>(*it);
	assert(oelem != nil);
	oe_list_.push_back(oelem->clone());
    }
    //
    // now the global Tool map. These are tool instances not attached to
    // any particular OE or SCR.
    //
    map<string, void*, less<string> >::const_iterator it2 = 
        from.tool_map_.begin(); 
    for (; it2 != from.tool_map_.end(); ++it2) {
	const string& toolname = (*it2).first;
	const BLToolInstance* ti = 
	    reinterpret_cast<const BLToolInstance*>((*it2).second);
	assert(ti != nil);
	tool_map_[toolname] = ti->clone();
    }
}

int Beamline::clear(bool clear_src) {
    if (clear_src) {
	delete src_;
	src_ = nil;
    }
    clear_clipboard();

    int numoe = num_oe();
    //
    // do *not* use num_oe() in the condition below since it changes with
    // every call to remove_oe().
    //
    for(int i = 0; i <= numoe; ++i) {
	remove_oe(1);		// note that we *always* remove scr 1!
    }
    //
    // now the global Tool map. These are tool instances not attached to
    // any particular OE or SCR.
    //
    map<string, void*, less<string> >::iterator it2 = tool_map_.begin(); 
    for (; it2 != tool_map_.end(); ++it2) {
	BLToolInstance* ti = 
	    reinterpret_cast<BLToolInstance*>((*it2).second);
	assert(ti != nil);
	delete ti;
    }
    tool_map_.erase(tool_map_.begin(), tool_map_.end());
    //
    // reset the rest just in case.
    //
    cur_oe_ = 0;
    cur_scr_ = 0;
    cur_inspector_ = 0;
    cur_selection_ = nil;
    cur_editing_ = nil;
    clipboard_ = nil;

    //
    // note: DO NOT delete the saved state, since that'll defeat the
    // purpose!
    //

    return 0;
}

Beamline::~Beamline() {
    clear();
    //
    // note that this is not cleared in clear().
    //
    delete saved_state_;
}

Beamline* Beamline::clone() const {
    return new Beamline(*this);
}

int Beamline::copy(const Beamline& from) {
    if (this == &from)
	return 0;

    clear();
    if (InspectorMgr::copy(from))
        return 1;
    src_ = (from.src_) ? from.src_->clone() : nil;
    cur_oe_ = from.cur_oe_;
    cur_scr_ = from.cur_scr_;
    cur_inspector_ = from.cur_inspector_;
    cur_selection_ = from.cur_selection_;
    cur_editing_ = from.cur_editing_;
    clipboard_ = (from.clipboard_) ? from.clipboard_->clone() : nil;

    for (
        deque<void*>::const_iterator it = from.oe_list_.begin();
	it != from.oe_list_.end(); 
	++it
    ) {
        const OE* oelem = reinterpret_cast<const OE*>(*it);
	assert(oelem != nil);
	oe_list_.push_back(oelem->clone());
    }
    //
    // now the global Tool map. These are tool instances not attached to
    // any particular OE or SCR.
    //
    map<string, void*, less<string> >::const_iterator it2 = 
        from.tool_map_.begin(); 
    for (; it2 != from.tool_map_.end(); ++it2) {
	const string& toolname = (*it2).first;
	const BLToolInstance* ti = 
	    reinterpret_cast<const BLToolInstance*>((*it2).second);
	assert(ti != nil);
	tool_map_[toolname] = ti->clone();
    }
    return 0;
}

Source* Beamline::get_source() const { return src_; }

OE* Beamline::get_oe(int oe) const {
    return (oe >= 1 && oe <= num_oe()) ? 
        static_cast<OE*>(oe_list_[oe-1]) : nil;
}

Screen* Beamline::get_scr(int oe, int scr) const {
    const OE* oelem = get_oe(oe);
    return (oelem) ? oelem->get_screen(scr) : nil;
}

int Beamline::add_source() {
    if (src_)
	return 1;

    const Tool* src_tool = toolmgr_.find("SRC");
    assert(src_tool != nil);

    src_ = new Source(src_tool);
    return 0;
}

int Beamline::add_oe(int req_id, int& actual_id) {
    if (req_id < 1) {
	actual_id = 0;
	return -1;
    }
    if (req_id > num_oe()) {
	actual_id = num_oe() + 1;
    } else {
	actual_id = req_id;
    }
    const Tool* oe_tool = toolmgr_.find("OE");
    assert(oe_tool != nil);

    OE* oelem = new OE(oe_tool, actual_id);
    oe_list_.insert(oe_list_.begin() + actual_id - 1, oelem);

    renumber_all_oe();
    return 0;
}

int Beamline::add_scr(int oe, int req_id, int& actual_id) {
    OE* oelem = get_oe(oe);
    if (oelem == nil)
	return -1;

    const Tool* scr_tool = toolmgr_.find("SCR");
    assert(scr_tool != nil);

    int retcode = oelem->add_scr(scr_tool, req_id, actual_id);
    if (retcode == 0)
	renumber_all_oe();
    return retcode;
}

int Beamline::add_tool(const string& toolname) {
    const Tool* tool = toolmgr_.find(toolname);
    if (tool == nil) {
        return -1;
    }
    if (!get_tool(toolname)) {
	BLToolInstance* ti = new BLToolInstance(tool, 0);
	tool_map_[toolname] = ti;
    }
    return 0;
}

int Beamline::remove_source() {
    delete src_;
    src_ = nil;
    return 0;
}

int Beamline::remove_oe(int oe) {
    int retcode = 0;
    if (oe < 1 || oe > num_oe()) {
	retcode = -1;
    } else {
	deque<void*>::iterator it = oe_list_.begin() + oe - 1;
	OE* oelem = static_cast<OE*>(*it);
	assert(oelem != nil);
	unsigned nscrn = oelem->num_scr();
	for(unsigned i = 1; i <= nscrn; i++) {
	    if (oelem->remove_scr(1) == -1)
		retcode = -1;
	}
	delete oelem;
	oe_list_.erase(it);
	//
	// CHECK/FIXME: Should OE number decrement if cur_oe > oe?
	//
	if (cur_oe() == oe)
	    cur_oe(0);
    }
    renumber_all_oe();
    return retcode;
}

//
// replace_oe: Replace the OE at given position with this one. Change
// the id's here too just to make sure.
//
int Beamline::replace_oe(int oe, OE* with_this) {
    if (oe < 1 || oe > num_oe()) {
	return -1;
    } else {
	OE* old_oe = static_cast<OE*>(oe_list_[oe-1]);
	assert(old_oe != nil);
	delete old_oe;
	oe_list_[oe-1] = with_this;
	//
	// fix the id's just in case the client forgot.
	//
	with_this->set_id(oe);
    }
    return 0;
}


int Beamline::remove_scr(int oe, int scr) {
    int retcode = 0;
    if (oe < 1 || oe > num_oe()) {
	retcode = -1;
    } else {
	OE* oelem = get_oe(oe);
	retcode = oelem->remove_scr(scr);
    }
    renumber_all_oe();
    return retcode;
}

int Beamline::remove_tool(const string& toolname) {
    map<string, void*, less<string> >::iterator it = 
        tool_map_.find(toolname);
    if (it == tool_map_.end())
        return -1;
    
    BLToolInstance* ti = static_cast<BLToolInstance*>((*it).second);
    tool_map_.erase(it);
    delete ti;
    return 0;
}

int Beamline::renumber_all_oe() {
    unsigned numoe = num_oe();
    for(unsigned oe = 1; oe <= numoe; oe++) {
	OE* oelem = get_oe(oe);
	assert(oelem != nil);
	oelem->set_id(oe);
    }
    return 0;
}

Inspector* Beamline::get_inspector (int id) const {
    return InspectorMgr::get_inspector(id);
}

Inspector* Beamline::get_source_inspector (int id) const {
    return (src_) ? src_->get_inspector(id) : nil;
}

Inspector* Beamline::get_oe_inspector(int oe_id, int id) const {
    const OE* oe = get_oe(oe_id);
    return (oe) ? oe->get_inspector(id) : nil;
}

Inspector* Beamline::get_scr_inspector(int oe_id, int scr_id, int id) const {
    const Screen* screen = get_scr(oe_id, scr_id);
    return (screen) ? screen->get_inspector(id) : nil;
}

BLToolInstance* Beamline::get_tool(const string& toolname) const {
    map<string, void*, less<string> >::const_iterator it = 
        tool_map_.find(toolname);
    return (it != tool_map_.end()) ? 
        static_cast<BLToolInstance*>((*it).second) : nil;
}

int Beamline::add_inspector(
    const string& toolname, int req_id, int& actual_id
) { 
    if (req_id < 1) {
	actual_id = 0;
	return -1;
    }
    if (req_id > num_inspectors()) {
	actual_id = num_inspectors() + 1;
    } else {
	actual_id = req_id;
    }
    const Tool* inspector_tool = toolmgr_.find(toolname);
    assert(inspector_tool != nil);

    Inspector* inspector = new Inspector(inspector_tool, actual_id, nil);
    return InspectorMgr::add_inspector(inspector, actual_id);
}

int Beamline::add_source_inspector(
    const string& toolname, int req_id, int& actual_id
) { 
    if (req_id < 1) {
	actual_id = 0;
	return -1;
    }
    Source* source = get_source();
    assert(source != nil);
    if (req_id > source->num_inspectors()) {
	actual_id = source->num_inspectors() + 1;
    } else {
	actual_id = req_id;
    }
    const Tool* inspector_tool = toolmgr_.find(toolname);
    assert(inspector_tool != nil);

    Inspector* inspector = new Inspector(inspector_tool, actual_id, source);
    return source->add_inspector(inspector, actual_id);
}

int Beamline::add_oe_inspector(
    const string& toolname, int oe_id, int req_id, int& actual_id
) { 
    if (req_id < 1) {
	actual_id = 0;
	return -1;
    }
    OE* oelem = get_oe(oe_id);
    assert(oelem != nil);
    if (req_id > oelem->num_inspectors()) {
	actual_id = oelem->num_inspectors() + 1;
    } else {
	actual_id = req_id;
    }
    const Tool* inspector_tool = toolmgr_.find(toolname);
    assert(inspector_tool != nil);

    Inspector* inspector = new Inspector(inspector_tool, actual_id, oelem);
    return oelem->add_inspector(inspector, actual_id);
}

int Beamline::add_scr_inspector(
    const string& toolname, int oe_id, int scr_id, 
    int req_id, int& actual_id
) { 
    if (req_id < 1) {
	actual_id = 0;
	return -1;
    }
    Screen* screen = get_scr(oe_id, scr_id);
    assert(screen != nil);
    if (req_id > screen->num_inspectors()) {
	actual_id = screen->num_inspectors() + 1;
    } else {
	actual_id = req_id;
    }
    const Tool* inspector_tool = toolmgr_.find(toolname);
    assert(inspector_tool != nil);

    Inspector* inspector = new Inspector(inspector_tool, actual_id, screen);
    return screen->add_inspector(inspector, actual_id);
}

int Beamline::remove_inspector(int id) { 
    return InspectorMgr::remove_inspector(id);
}

int Beamline::remove_source_inspector(int id) { 
    return (src_) ? src_->remove_inspector(id) : -1;
}

int Beamline::remove_oe_inspector(int oe_id, int id) {
    OE* oelem = get_oe(oe_id);
    return (oelem) ? oelem->remove_inspector(id) : -1;
}

int Beamline::remove_scr_inspector(int oe_id, int scr_id, int id) {
    Screen* screen = get_scr(oe_id, scr_id);
    return (screen) ? screen->remove_inspector(id) : -1;
}

int Beamline::cur_oe() const {
    return cur_oe_;
}

int Beamline::cur_scr() const {
    return cur_scr_;
}

int Beamline::cur_inspector() const {
    return cur_inspector_;
}

const BLToolInstance* Beamline::cur_selection() const {
    return cur_selection_;
}

const BLToolInstance* Beamline::cur_editing() const {
    return cur_editing_;
}

void Beamline::cur_oe(int cur_oe) {
    cur_oe_ = cur_oe;
}

void Beamline::cur_scr(int cur_scr) {
    cur_scr_ = cur_scr;
}

void Beamline::cur_inspector(int cur_inspector) {
    cur_inspector_ = cur_inspector;
}

void Beamline::cur_selection(const BLToolInstance* ti) {
    cur_selection_ = const_cast<BLToolInstance*>(ti);
}

void Beamline::cur_editing(const BLToolInstance* ti) {
    cur_editing_ = const_cast<BLToolInstance*>(ti);
}

int Beamline::num_oe() const {
    return oe_list_.size();
}

int Beamline::num_scr(int oe) const {
    int num = -1;
    if (oe >= 1 && oe <= num_oe()) {
	const OE* oelem = get_oe(oe);
	assert(oelem != nil);
	num = oelem->num_scr();
    }
    return num;
}
int Beamline::num_source_inspectors() const {
    return (src_) ? src_->num_inspectors() : -1;
}

int Beamline::num_oe_inspectors(int oe) const {
    const OE* oelem = get_oe(oe);
    return (oelem) ? oelem->num_inspectors() : -1;
}

int Beamline::num_scr_inspectors(int oe, int scr) const {
    const Screen* screen = get_scr(oe, scr);
    return (screen) ? screen->num_inspectors() : -1;
}

//
// the following get_var and set_var should *ONLY* be called from
// the GUI, since there is the concept of current selection is
// involved here. The back-end code should directly get or set 
// using the BLToolInstance* for that object.
//
const Var* Beamline::get_var(const char* tool, const char* name) const {
    string toolname(tool);
    const BLToolInstance* const ti = (cur_editing_) ? 
        cur_editing_ : get_tool(toolname);

    string namestr(name);
    const Var* var = (ti) ? ti->get_var(namestr) : nil;
    return var;
}

const Var* Beamline::get_var(const string& tool, const string& name) const {
    return get_var(tool.c_str(), name.c_str());
}

int Beamline::set_var(const char* tool, const char* name, const char* value) {
    string toolstr(tool);
    BLToolInstance* const ti = (cur_editing_) ? 
        cur_editing_ : get_tool(toolstr);
    if (ti == nil)
        return -1;
    return set_value(ti, name, value);
}

int Beamline::set_var(
    const string& tool, const string& name, const string& value
) {
    return set_var(tool.c_str(), name.c_str(), value.c_str());
}

int Beamline::set_value(
    BLToolInstance* ti, const string& varname, const char* value
) {
    const Value* curvalue = ti->get_varval(varname);
    if (curvalue == nil) {
	return Beamline::BAD_VARIABLE;
    }
    int retcode = OK;
    Value* newval = curvalue->clone();
    if (!newval->setval(value)) {
	delete newval;
	retcode = BAD_VALUE;
    }
    else {
	ti->set_varval(varname, newval);
    }
    return retcode;
}

int Beamline::set_source_value(const char* varname, const char* value) {
    if (src_ == nil)
        return -1;
    
    return set_value(src_, varname, value);
}

int Beamline::set_oe_value(int oe, const char* varname, const char* value) {
    BLToolInstance* ti = get_oe(oe);
    if (ti == nil)
        return -1;
    return set_value(ti, varname, value);
}

int Beamline::set_scr_value(
    int oe, int scr, const char* varname, const char* value
) {
    BLToolInstance* ti = get_scr(oe, scr);
    if (ti == nil)
        return -1;
    return set_value(ti, varname, value);
}

int Beamline::set_tool_value(
    const char* toolname, const char* varname, const char* value
) {
    BLToolInstance* ti = get_tool(toolname);
    if (ti == nil)
        return -1;
    return set_value(ti, varname, value);
}

const char* Beamline::get_value(
    const BLToolInstance* ti, const string& varname
) const {
    const Value* value = ti->get_varval(varname);
    return (value) ? value->getstrval() : nil;
}

const char* Beamline::get_source_value(const char* varname) const {
    string varnamestr(varname);
    return (src_ == nil) ? nil : get_value(src_, varnamestr);
}

const char* Beamline::get_oe_value(int oe, const char* varname) const {
    const BLToolInstance* ti = get_oe(oe);
    //
    // BUG/CHECK/FIXME (BC++ 5.0)
    //
    // borland c++ 5.0 can't handle the following (something about the
    // the conditional and the string conversion that screws it up), so 
    // we have to break it up.
    // return (ti == nil) ? nil : get_value(ti, varname);
    //
    if (ti == nil)
        return nil;
    return get_value(ti, varname);
}

const char* Beamline::get_scr_value(
    int oe, int scr, const char* varname
) const {
    const BLToolInstance* ti = get_scr(oe, scr);
    //
    // BUG/CHECK/FIXME (BC++ 5.0)
    //
    // borland c++ 5.0 can't handle the following (something about the
    // the conditional and the string conversion that screws it up), so 
    // we have to break it up.
    // return (ti == nil) ? nil : get_value(ti, varname);
    //
    if (ti == nil)
        return nil;
    return get_value(ti, varname);
}

const char* Beamline::get_inspector_value (int id, const char* varname) const {
    const Inspector* inspector = get_inspector(id);
    //
    // BUG/CHECK/FIXME (BC++ 5.0)
    //
    // borland c++ 5.0 can't handle the following (something about the
    // the conditional and the string conversion that screws it up), so 
    // we have to break it up.
    // return (inspector == nil) ? nil : get_value(inspector, varname);
    //
    if (inspector== nil)
        return nil;
    return get_value(inspector, varname);
}

const char* Beamline::get_source_inspector_value (
    int id, const char* varname
) const {
    const Inspector* inspector = get_source_inspector(id);
    if (inspector == nil)
        return nil;
    return get_value(inspector, varname);
}

const char* Beamline::get_oe_inspector_value (
    int oe, int id, const char* varname
) const {
    const Inspector* inspector = get_oe_inspector(oe, id);
    if (inspector == nil)
        return nil;
    return get_value(inspector, varname);
}

const char* Beamline::get_scr_inspector_value (
    int oe, int scr, int id, const char* varname
) const {
    const Inspector* inspector = get_scr_inspector(oe, scr, id);
    //
    // the following chokes BC5 for some reason.
    // return (inspector == nil) ? nil : get_value(inspector, varname);
    //
    if (inspector == nil)
        return nil;
    return get_value(inspector, varname);
}

int Beamline::set_inspector_value (
    int id, const char* varname, const char* value
) {
    Inspector* inspector = get_inspector(id);
    if (inspector == nil)
        return -1;
    return set_value(inspector, varname, value);
}

int Beamline::set_source_inspector_value (
    int id, const char* varname, const char* value
) {
    Inspector* inspector = get_source_inspector(id);
    if (inspector == nil)
        return -1;
    return set_value(inspector, varname, value);
}

int Beamline::set_oe_inspector_value (
    int oe, int id, const char* varname, const char* value
) {
    Inspector* inspector = get_oe_inspector(oe, id);
    if (inspector == nil)
        return -1;
    return set_value(inspector, varname, value);
}

int Beamline::set_scr_inspector_value (
    int oe, int scr, int id, const char* varname, const char* value
) {
    Inspector* inspector = get_scr_inspector(oe, scr, id);
    if (inspector == nil)
        return -1;
    return set_value(inspector, varname, value);
}

int Beamline::load_nml(const string& file, BLToolInstance* ti) {
    ifstream ifp(file.c_str());
    if (!ifp) {
	return -1;
    }

    int errors = 0;
    char buf[BUFSIZ];
    char varname[BUFSIZ];  
    char value[BUFSIZ]; 
    Pmu* pmu = new Pmu;

    for(unsigned line = 0; ifp.getline(buf, sizeof(buf)); ++line) {
	if (pmu->isacomment(buf)) 
	    continue;

	if (pmu->match (buf, "<ident>", "=", "<text>")) {
	    pmu->gettextfield (3, value);
	    trimright(value);
	} else if (pmu->match (buf, "<ident>", "=", "<nothing>")) {
	    strcpy(value, "");
	} else {
	    // unrecognizable pattern 
	    cerr << "Beamline::load_nml (" << file << ":" << line 
	        << "): Unable to understand \""
	        << buf << "\"." << endl;
	    errors++;  
	    if (errors > 3) {
		// IF WE ARE HERE TOO MAY ERRORS
		cerr << "Beamline::load_nml: Too many errors in gfile \""
		    << file << "\"." << endl;
		return -1;
	    }
	}

	//
	// we are ok if we got here.
	//
	pmu->gettextfield (1, varname);

	const Var* var = ti->get_var(varname);
	if (var == nil) {
	    cerr << "Ignoring variable \"" << varname << "\" in gfile \""
	        << file << "\"." << endl;
	    continue;
	}
	Value* newval = var->genval();
	if (newval->isaenum()) {
	    int enum_int = atoi(value);
	    if (!syn_table_.exists(varname, enum_int)) {
		cerr << "ERROR: Bad value \"" << value << "\" for ENUM \""
		    << varname << "\"." << endl;
		delete newval;
		errors++;
		continue;
	    }
	    else {
		strcpy(value, syn_table_.translate(varname, enum_int));
	    }
	}
	if (!newval->setval(value)) {
	    cerr << "ERROR: Bad value \"" << value 
	        << "\" for ShadowNamelist item \"" << varname 
		<< "\"." << endl;
	    delete newval;
	    errors++;
	    continue;
	}
	else {
	    ti->set_varval(varname, newval);
	}
    }
    delete pmu;
    return 0;
}

int Beamline::load_source(const string& file) {
    remove_source();
    add_source();
    return load_nml(file, get_source());
}

int Beamline::load_scr_from_oe(const OE* oelem, Screen* screen) {
    int oe, scr;
    screen->get_id(oe, scr);
    assert(oelem->get_id() == oe);

    char varname[BUFSIZ];
    char value[BUFSIZ];

    //
    // populate the SCR variables from OE.
    //
    const Tool* scr_tool = screen->tool();
    const ShadowNamelist& namelist = scr_tool->namelist();
    const Tool::VariableList& varnames = namelist.variables();

    Tool::VariableList::const_iterator it = varnames.begin();
    for(; it != varnames.end(); ++it) {
	const string& svname = *it;
	memset(varname, 0, BUFSIZ);
	sprintf(varname, "%s(%d)", svname.c_str(), scr);
	const char* ovalue = get_oe_value(oe, varname);
	if (ovalue == nil) {
	    cerr 
		<< "ERROR: SCREEN variable (" << varname
		<< ") missing from OE namelist." << endl
		<< "Are you sure you have the right MENU file?"
		<< endl;
	    return -1;
	}
	strcpy(value, ovalue);

	//
	// now the big hack. The variables in SCREEN and the
	// corresponding maps in OE might be defined to be of 
	// different types.
	// eg., I_SCREEN is ENUM in SCR, but I_SCREEN_[1-10] are
	// are INTEGERs in OE.
	//
	// so need to inverse-translate using synonym table.
	//

	// const Var* var = get_var("SCR", svname, oe, scr);
	const Var* var = screen->get_var(svname);
	if (var == nil) {
	    cerr << "ERROR: cannot find screen variable \"" << svname
	        << "\" for OE/SCR " << oe << "/" << scr << "." << endl;
	    continue;
	}
	if (var->getval()->isaenum()) {
	    int int_value = -1;
	    if (sscanf(value, "%d", &int_value) != 1) {
		cerr << "ERROR: OE variable \""
		    << varname << "\" has wrong value \""
		    << value << "\"." << endl;
		return -1;
	    }
	    if (!syn_table_.exists(svname.c_str(), int_value)) {
		cerr << "Bad value for ENUM variable: "
		    << svname << " = " << value << endl;
	    }
	    else {
		SynonymTable::MatchType match;
		const char* enum_str = syn_table_.translate(
		    svname.c_str(), int_value, match
		);
		if (match == SynonymTable::AMBIGUOUS) {
		    //
		    // it's ok to be ambiguous here. some data items
		    // have conflicting types in the MENU file, and
		    // that causes this problem (esp Y/N variables like
		    // scr/i_slit). Basically, we end up with the first
		    // enum string that matches the integer value, which
		    // is ok.
		    //
		    // NEW COMMENT: It's not ok anymore. It's better to
		    // fix the SHADOW menu file that deal with this.
		    //
		    cerr << "Ambiguous value for ENUM variable: "
			<< svname << " = " << value << endl;
		}
		strcpy(value, enum_str);
	    }
	}
	//
	// now ``value'' has the appropriate translated string, so 
	// simply set it in the OE.
	//
	set_scr_value(oe, scr, svname.c_str(), value);
    }
    return 0;
}

// 
// fix_modified_surface_read_flags: When reading a new OE namelist, turn 
// on FDUMMY (see ERROR page in shadow menu) when any one of the modified 
// surface flags are turned on.
//
// Necessary due to the inability of SHADOW's menu format to change the
// the value of one variable when some other value changes.
//
// Also see fix_modified_surface_write_flags() for the converse.
//
static int fix_modified_surface_read_flags(Beamline* bline, int oe) {
    const char* dummy_value = bline->get_oe_value(oe, "FDUMMY");
    if (
	strcasecmp(bline->get_oe_value(oe, "F_RIPPLE"),   "YES") == 0 ||
	strcasecmp(bline->get_oe_value(oe, "F_FACET"),    "YES") == 0 ||
	strcasecmp(bline->get_oe_value(oe, "F_ROUGNESS"), "YES") == 0 ||
	strcasecmp(bline->get_oe_value(oe, "F_KOMA"),     "YES") == 0 ||
	strcasecmp(bline->get_oe_value(oe, "F_SEGMENT"),  "YES") == 0
    ) {
	bline->set_oe_value(oe, "FDUMMY", "YES");
    }
    return 0;
}

int Beamline::load_oe(const string& file, int oe) {
    remove_oe(oe);
    add_oe(oe, oe);
    OE* oelem = get_oe(oe);
    if (load_nml(file, oelem) == -1)
	return -1;

    //
    // Now we check the FDUMMY that controls "Modified Surface", and
    // turn off all the modified surface attributes if FDUMMY is false.
    // UNCONST
    fix_modified_surface_read_flags(this, oe);

    //
    // trivial case: no screens.
    //
    SynonymTable::MatchType match;
    const char* f_screen = get_oe_value(oe, "F_SCREEN");
    int has_screen = syn_table_.translate("F_SCREEN", f_screen, match);
    if (match != SynonymTable::UNIQUE) {
	cerr << "ERROR in NAMELIST file \"" << file << "\"."
	    << "Bad ENUM value for F_SCREEN (0/1)." << endl;
	return -1;
    } else if (has_screen == 0)
	return 0;

    int numscrn = atoi(get_oe_value(oe, "N_SCREEN"));
    if (numscrn <= 0) {
	cerr << "ERROR in NAMELIST file \"" << file << "\"." 
	    << "Inconsistent SCREEN info." << endl;
	return -1;
    }

    //
    // ASSUME: all screens are in order. (no gaps)
    //

    //
    // now add all the Screen values and set the corresponding SCR vars.
    //
    for (int scr = 1; scr <= numscrn; scr++) {
	int actual = 0;
	add_scr(oe, scr, actual);
	if (actual != scr) {
	    cerr << "ERROR: error adding screen in order "
		<< "from NAMELIST \"" << file << "\" for OE "
		<< oe << "." << endl;
	    return -1;
	}
	Screen* screen = get_scr(oe, scr);
	assert(screen != nil);

	if (load_scr_from_oe(oelem, screen)) {
	    cerr << "ERROR: error getting SCREEN values from OE "
		<< "from NAMELIST \"" << file << "\" for OE "
		<< oe << "." << endl;
	    return -1;
	}
    }
    return 0;
}

int Beamline::load_scr(const string& /*file*/, int /*oe*/, int /*scr*/) {
    //
    // meaningless in this context, since SCR is not really a "tool",
    // rather it exists only as a part of a particular OE.
    //
    return 0;
}

int Beamline::load_system(const string& file) {
    ifstream ifp(file.c_str());
    if (!ifp)
	return -1;
    
    clear(false);			// don't clear SOURCE, just OEs
    
    string startfile;
    for(int oe = 1; ifp >> startfile; ++oe) {
	if (load_oe(startfile, oe)) {
	    cerr << "Error loading OE from file " << startfile << endl;
	    return -1;
	}
    }
    return 0;
}

int Beamline::load_tool(const string& toolname, const string& file) {
    add_tool(toolname);
    return load_nml(file, get_tool(toolname));
}

int Beamline::store_nml(const string& file, const BLToolInstance* ti) const {
    ofstream ofp(file.c_str());
    if (!ofp)
	return -1;

    const Tool* tool = ti->tool();
    const ShadowNamelist& namelist = tool->namelist();
    const Tool::VariableList& varnames = namelist.variables();
    
    char strval[BUFSIZ];

    Tool::VariableList::const_iterator it = varnames.begin();
    for(; it != varnames.end(); ++it) {
	const string& varname = *it;
	const Var* var = ti->get_var(varname);
	assert(var != nil);
	strcpy(strval, var->getval()->getstrval());
	if (var->getval()->isaenum()) {
	    if (!syn_table_.exists(varname.c_str(), strval)) {
		cerr << "Bad value for ENUM variable: "
		    << varname << " = " << strval << endl;
	    }
	    else {
		SynonymTable::MatchType match;
		int value = syn_table_.translate(
		    varname.c_str(), strval, match
		);
		if (match == SynonymTable::AMBIGUOUS) {
		    cerr << "Ambiguous value for ENUM variable: "
		        << varname << " = " << strval << endl;
		}
		sprintf(strval, "%d", value);
	    }
	}
	ofp << varname << " = " << strval << endl;
    }
    return 0;
}

int Beamline::store_source(const string& file) const {
    return (src_) ? store_nml(file, get_source()) : -1;
}

//
// fix_modified_surface_write_flags: When writing a OE namelist, turn off 
// ALL the modified surface flags (eg., F_SEGMENT) when FDUMMY (see ERROR 
// page in shadow menu) is turned off. 
//
// Necessary due to the inability of SHADOW's menu format to change the
// the value of one variable when some other value changes.
//
// Also see fix_modified_surface_read_flags() for the converse.
//
static int fix_modified_surface_write_flags(Beamline* bline, int oe) {
    const char* dummy_value = bline->get_oe_value(oe, "FDUMMY");
    if (strcasecmp(dummy_value, "NO") == 0) {
	bline->set_oe_value(oe, "F_RIPPLE",   "NO");
	bline->set_oe_value(oe, "F_FACET",    "NO");
	bline->set_oe_value(oe, "F_ROUGNESS", "NO");
	bline->set_oe_value(oe, "F_KOMA",     "NO");
	bline->set_oe_value(oe, "F_SEGMENT",  "NO");
    }
    return 0;
}

int Beamline::store_oe(const string& file, int oe) const {
    const OE* oelem = get_oe(oe);
    if (oe == nil)
        return -1;
    //
    // trivial case: no screens.
    //
    int numscrn = num_scr(oe);
    //
    // this is too complicated to explain here. Basically, the only
    // time the OE's SCREEN part gets initialized is when we write
    // out the SYSTEM files and that's here. If the user then deletes
    // the screens in the interface, the OE's namelist still has the
    // old value, so we fix it from the OEs interface state.
    //
    char varname[BUFSIZ];
    char value[BUFSIZ];

    Beamline* this_bline = const_cast<Beamline*>(this); // UNCONST
    this_bline->set_oe_value(oe, "F_SCREEN", (numscrn) ? "ON" : "OFF");
    sprintf(value, "%d", numscrn);
    this_bline->set_oe_value(oe, "N_SCREEN", value);

    //
    // Now we check the FDUMMY that controls "Modified Surface", and
    // turn off all the modified surface attributes if FDUMMY is false.
    //
    fix_modified_surface_write_flags(this_bline, oe);


    if (numscrn == 0) {
        return store_nml(file, oelem);
    }

    //
    // CHECK/FIXME: Why do I initialize I_SCREEN(i) to 1!! This is the
    // variable the defines the position (before/after) of the screen
    // in the OE. Leave it alone!
    //
    sprintf(value, "%d", 1);
    int scr;
    for (scr = 1; scr <= numscrn; scr++) {
	memset(varname, 0, BUFSIZ);
	sprintf(varname, "i_screen(%d)", scr);
	this_bline->set_oe_value(oe, varname, value);
    }

    //
    // now get all the Screen values and set the corresponding OE vars.
    //
    for (scr = 1; scr <= numscrn; scr++) {
	Screen* screen = get_scr(oe, scr);
	assert(screen != nil);

	const Tool* scr_tool = screen->tool();
	const ShadowNamelist& namelist = scr_tool->namelist();
	const Tool::VariableList& varnames = namelist.variables();
	
	Tool::VariableList::const_iterator it = varnames.begin();
	for(; it != varnames.end(); ++it) {
	    const string& svname = *it;
	    const char* svalue = get_scr_value(oe, scr, svname.c_str());
	    strcpy(value, svalue);
	    memset(varname, 0, BUFSIZ);
	    sprintf(varname, "%s(%d)", svname.c_str(), scr);

	    //
	    // now the big hack. The variables in SCREEN and the
	    // corresponding maps in OE might be defined to be of 
	    // different types.
	    // eg., I_SCREEN is ENUM in SCR, but I_SCREEN_[1-10] are
	    // are INTEGERs in OE.
	    //
	    // so need to translate using synonym table.
	    //

	    // const Var* var = get_var("SCR", svname, oe, scr);
	    const Var* var = screen->get_var(svname);
	    if (var->getval()->isaenum()) {
		if (!syn_table_.exists(svname.c_str(), value)) {
		    cerr << "Bad value for ENUM variable: "
		        << svname << " = " << value << endl;
		}
		else {
		    SynonymTable::MatchType match;
		    int int_val = syn_table_.translate(
			svname.c_str(), value, match
		    );
		    if (match == SynonymTable::AMBIGUOUS) {
			cerr << "Ambiguous value for ENUM variable: "
			    << svname << " = " << value << endl;
		    }
		    sprintf(value, "%d", int_val);
		}
	    }

	    //
	    // now ``value'' has the appropriate translated string, so 
	    // simply set it in the OE.
	    //
	    this_bline->set_oe_value(oe, varname, value);
	}
    }
    return store_nml(file, oelem);
}

int Beamline::store_scr(const string& /*file*/, int /*oe*/, int /*scr*/) const {
    //
    // meaningless in this context, since SCR is not really a "tool",
    // rather it exists only as a part of a particular OE.
    //
    return 0;
}

int Beamline::store_system(
    const string& file, const string& startprefix
) const {
    ofstream ofp(file.c_str());
    if (!ofp)
	return -1;
    
    char buf[1024];
    deque<void*>::const_iterator it = oe_list_.begin();
    for(; it != oe_list_.end(); ++it) {
        const OE* oelem = reinterpret_cast<const OE*>(*it);
	sprintf(buf, "%s.%.2d", startprefix.c_str(), oelem->get_id());
	ofp << buf << endl;
	if (store_oe(buf, oelem->get_id())) {
	    cerr << "Error saving NAMELIST for OE " << oelem->get_id()
	        << " to file " << buf << endl;
	    return -1;
	} 
    }
    return 0;
}

int Beamline::store_tool(const string& toolname, const string& file) const {
    const BLToolInstance* ti = get_tool(toolname);
    return (ti) ? store_nml(file, ti) : -1;
}

int Beamline::copy_selection_to_clipboard() const {
    if (cur_selection_ == nil)
	return -1;
    Beamline* this_bline = const_cast<Beamline*>(this); // UNCONST
    delete this_bline->clipboard_;
    this_bline->clipboard_ = cur_selection_->clone();

    return 0;
}

void Beamline::clear_clipboard() const {
    Beamline* this_bline = const_cast<Beamline*>(this); // UNCONST
    delete this_bline->clipboard_;
    this_bline->clipboard_ = nil;
}

//
// algorithm:
//    1. match types. If not ok, exit
//    2. copy.
//
// returns:
//          0: ok
//         -1: either nothing selected or clipboard is empty
//         -2: Incompatible types
//
// Caveat: can't prevent copying over itself since I don't know if the
// clipboard object is the same as the selected object (not stored as
// pointers, but cloned object).
//
int Beamline::paste_selection_from_clipboard() { 
    if (cur_selection_ == nil || clipboard_ == nil)
	return -1; 
    
    if (cur_selection_->tool() != clipboard_->tool())
	return -2;
    
    int retcode = cur_selection_->copy(*clipboard_);

    //
    // must renumber the OE's, since some of the OEs or Screens might
    // have incorrect sequence numbers due to copying.
    //
    renumber_all_oe();

    return retcode;
}

const BLToolInstance* Beamline::clipboard() const { return clipboard_; }

void Beamline::save_state() const {
    Beamline* this_bline = const_cast<Beamline*>(this); // UNCONST
    delete this_bline->saved_state_;
    this_bline->saved_state_ = clone();
}
    
int Beamline::restore_state() {
    if (saved_state_) {
	copy(*saved_state_);
	delete saved_state_;
	saved_state_ = nil;
    }
    return 0;
}

const Beamline* Beamline::saved_state() const {
    return saved_state_;
}

void Beamline::dump() const {
    cerr << "Dumping SYSTEM ..." << endl;
    if (src_)
	src_->dump();
    else
	cerr << "NO SOURCE defined." << endl;
    for(int oe = 1; oe <= num_oe(); ++oe) {
	const OE* oelem = get_oe(oe);
	oelem->dump();
    }
    InspectorMgr::dump();
    cerr << "Done dumping SYSTEM." << endl;
}

// =================================================================== //
