#
# run-shadow.tcl: Run various shadow programs
#
# ------------------------------------------------
# Mumit Khan <khan@xraylith.wisc.edu>
# Center for X-ray Lithography
# University of Wisconsin-Madison
# 3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Copyright (c) 1994-1996 Mumit Khan
#
#


set RunTool(delay)	idle
set RunTool(tmpfiles)	{}
set RunTool(w)		.tool_runner
set RunTool(waitvar)	""

#----------------------------------------------------------------------#
# Running programs
#----------------------------------------------------------------------#


proc show_run_window {} {
    global gvars RunTool
    make_run_window

    busy_window release .
    busy_window release $RunTool(w)
    #busy_window release $gvars(w:text)

    set w $RunTool(w)
    grab release $w
    $gvars(w:cancelBtn) config -text Close
}

proc make_run_window {} {
    global gvars RunTool

    set w $RunTool(w)


    if [winfo exists $w] {
	grab $w
	wm deiconify $w
	raise $w
	busy_window hold .
	busy_window hold $RunTool(w)
        #busy_window hold $gvars(w:text)
	update idletasks
	return
    }

    set RunTool(lines) 0
    set RunTool(tmpfile) {}
 
    toplevel $w
    wm title $w "SHADOW Run Tool"
    wm iconname $w "Run SHADOW"
 
    set f [frame $w.top -relief raised -bd 1]
    set f1 [frame $f.f1]
    set t  [text $f1.t -font -*-Courier-*-R-Normal--*-120-*-*-*-*-*-*]

    set v  [scrollbar $f1.v -orient vertical]
    pack $v -side right -fill y
    pack $t -side top -expand yes -fill both
 
    set f2 [frame $f.f2]
    set h  [scrollbar $f2.h -orient horizontal]
    set x  [frame $f2.x -width [winfo reqwidth $v]]
    pack $x -side right -fill y
    pack $h -expand yes -fill both
 
    pack $f2 -side bottom -fill x
    pack $f1 -expand yes -fill both
 
    $t config -xscrollcommand "$h set" -yscrollcommand "$v set"
    $h config -command "$t xview"
    $v config -command "$t yview"
 
    set bot [frame $w.bot -relief raised -bd 1]
    set b [button $bot.b -text Cancel -command cancel_run_window -width 7]
    set b2 [button $bot.b2 -text Save -command save_run_window -width 7]
    set b3 [button $bot.b3 -text Clear -command clear_run_window -width 7]
    set b4 [button $bot.b4 -text "Insert File" \
    	-command insert_into_run_window -width 7]
    pack $b -side left -expand yes -padx 10 -pady 15
    pack $b2 -side left -expand yes -padx 10 -pady 15
    pack $b3 -side left -expand yes -padx 10 -pady 15
    pack $b4 -side left -expand yes -padx 10 -pady 15
    pack $bot -side bottom -fill both
    pack $f -expand yes -fill both

    #
    # do the text fonts and such things.
    #
    $t tag configure bold \
	-font -*-Courier-Bold-O-Normal--*-120-*-*-*-*-*-*
    $t tag configure big \
	-font -*-Courier-Bold-R-Normal--*-140-*-*-*-*-*-*
    $t tag configure verybig \
	-font -*-Helvetica-Bold-R-Normal--*-240-*-*-*-*-*-*

    if {[winfo depth $w] > 1} {
	$t tag configure color1 -background #a0b7ce
	$t tag configure color2 -foreground red
	$t tag configure raised -relief raised -borderwidth 1
	$t tag configure sunken -relief sunken -borderwidth 1
    } else {
	$t tag configure color1 -background black -foreground white
	$t tag configure color2 -background black -foreground white
	$t tag configure raised -background white -relief raised \
		-borderwidth 1
	$t tag configure sunken -background white -relief sunken \
		-borderwidth 1
    }
 
    set gvars(w:text) $t
    set gvars(w:cancelBtn) $b
    set gvars(w:dialog) $w

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w
    busy_window hold .
    busy_window hold $RunTool(w)
    #busy_window hold $gvars(w:text)
    update idletasks
    grab $w
}

proc cancel_run_window {} {
    global gvars RunTool
    if {[$gvars(w:cancelBtn) cget -text] == "Close"} {
        # destroy $gvars(w:dialog)
        wm withdraw $gvars(w:dialog)
        #
        # also time to delete temp files.
        #
        foreach file $RunTool(tmpfiles) {
	   catch "remove_file $file"
	}
    } else {
        $gvars(w:cancelBtn) config -text Close -width 7
        catch {
            close $gvars(pipe)
        }
	set RunTool(waitvar) ""
	busy_window release .
	busy_window release $RunTool(w)
	#busy_window release $gvars(w:text)
    }
}

proc save_run_window {{file {}}} {
    global gvars RunTool
    if {$file != ""} {
	if {[catch "open $file w" f] != 0} {
	    dialog .save {}  \
		"Cannot open output file \"$file\" to save" \
		error 0 Dismiss
	    set_msg "cancelled"
	    raise $RunTool(w)
	    return 1
	}
	puts $f [$gvars(w:text) get 1.0 end]
	close $f
	raise $RunTool(w)
	return 0
    }
    #
    # We only get there when called w/out an argument, eg., by user.
    #

    # FIXME: For newer tk releases >= 4.2
    global tk_version
    if {$tk_version < 4.2} {
	set dialog [tix filedialog]
	set fsbox [$dialog subwidget fsbox]
	$fsbox configure -pattern "*.txt" -value "report.txt"
	$dialog config -title "Save the window contents to text file" \
	    -command "save_run_window"
	$dialog popup
    } else {
	global tcl_platform
	set types {
	    {{Text files}		{.txt}}
	    {{Misc Data Files}		{.dat}}
	    {{All Files}		*}
	}
	set file [tk_getSaveFile \
	    -filetypes $types \
	    -defaultextension .txt \
	    -initialfile report.txt \
	    -title "Save the window contents to text file"
	]
	if {$file != ""} {
	    save_run_window $file
	}
    }
    raise $RunTool(w)
    return 0
}

proc insert_into_run_window {{file {}}} {
    global gvars RunTool
    if {$file != ""} {
	if {[catch "open $file r" f] != 0} {
	    dialog .insert {}  \
		"Cannot open input file \"$file\" to read from" \
		error 0 Dismiss
	    set_msg "cancelled"
	    return 1
	}
	set data [read $f]
	$gvars(w:text) insert end "Inserting file $file\n\n" {big color2}
	$gvars(w:text) insert end "$data"
	$gvars(w:text) insert end "  ----- Done -------\n" {verybig color2}
	$gvars(w:text) see end
	close $f
	raise $RunTool(w)
	return 0
    }
    #
    # We only get there when called w/out an argument, eg., by user.
    #

    # FIXME: For newer tk releases >= 4.2
    global tk_version
    if {$tk_version < 4.2} {
	set dialog [tix filedialog]
	set fsbox [$dialog subwidget fsbox]
	$fsbox configure -pattern "*.txt" -value "report.txt"
	$dialog config -title "Enter filename to insert into this window" \
	    -command "insert_into_run_window"
	$dialog popup
    } else {
	global tcl_platform
	set types {
	    {{Text files}		{.txt}}
	    {{Misc Data Files}		{.dat}}
	    {{All Files}		*}
	}
	set file [tk_getOpenFile \
	    -filetypes $types \
	    -title "Enter filename to insert into this window"
	]
	if {$file != ""} {
	    insert_into_run_window $file
	}
    }
    raise $RunTool(w)
    return 0
}

proc clear_run_window {} {
    global gvars
    $gvars(w:text) delete 1.0 end
}

proc get_run_output {{hook {}}} {
    global gvars RunTool
 
    if {[$gvars(w:text) dlineinfo insert] != {}} {
        set setend 1
    } else {
        set setend 0
    }
 
    if {[gets $gvars(pipe) line] >= 0} {
        incr RunTool(lines)
        $gvars(w:text) insert end "$line\n"
	update idletasks
	fileevent $gvars(pipe) readable "get_run_output [list $hook]"
    } else {
        catch {
            close $gvars(pipe)
        }
	#
	# if there is a hook, run it now, but only if the user hasn't
	# killed the job using the Cancel button.
	# TODO/FIXME/CHECK:
	#
	if {$hook == {}} {
	    grab release $gvars(w:dialog)
	    $gvars(w:text) insert end " \n"
	    $gvars(w:text) insert end "  ----- Done -------\n" {verybig color2}
	    $gvars(w:text) insert end " you can choose to save the output now\n"
	    $gvars(w:cancelBtn) config -text Close
	    busy_window release .
	    busy_window release $RunTool(w)
	    #busy_window release $gvars(w:text)
	    set RunTool(waitvar) ""
	} else {
	    eval $hook
	}
    }
 
    #
    # HACK/FIXME: Should fix the setend thingie correctly instead of brute
    # force.
    #
    if {1 || $setend} {
        $gvars(w:text) see end
    }
}

proc run_error {program msg} {
    global gvars RunTool
    dialog .run_error {Run Error} \
	"ERROR running \"$program\". Not installed? ($msg)" \
	"error" 0 Dismiss
    set_msg "Error running program \"$program\"... ERROR"
    busy_window release .
    busy_window release $RunTool(w)
    #busy_window release $gvars(w:text)
    set RunTool(waitvar) ""
    grab release $RunTool(w)
}

proc run_program {mode progname args} {
    global gvars RunTool
 
    if {[tix platform] == "unix"} {
	set program "| sh -c \"$progname $args 2>&1\""
    } else {
	set program "|$progname $args"
    }

    make_run_window
 
    if [catch {eval set gvars(pipe) [open $program  "$mode"]} errmsg] {
        run_error $progname $errmsg
	return
    }
        
    after $RunTool(delay) [list catch [list fileevent $gvars(pipe) \
        readable get_run_output]]
 
    $gvars(w:text) insert end "executing $program\n\n" {big color2}
}

proc run_source {progname startfile} {
    global gvars RunTool
 
    if {[tix platform] == "unix"} {
	set program "| sh -c \"$progname $startfile 2>&1\""
    } else {
	set program "|$progname $startfile"
    }

    make_run_window
 
    if [catch {eval set gvars(pipe) [open $program  "r"]} errmsg] {
        run_error $progname $errmsg
	return
    }
    after $RunTool(delay) [list catch [list fileevent $gvars(pipe) \
        readable get_run_output]]
 
    $gvars(w:text) insert end "executing $program\n\n" {big color2}
}

#
# Run the whole system as a sequence of OEs. Not used right now. Note
# the use of tkwait.
#
proc run_system_chained {num_oe} {
    global gvars RunTool
    for {set oe 1} {$oe <= $num_oe} {incr oe} {
        run_oe_cmd $oe
	tkwait variable RunTool(waitvar)
    }
}

#
# Run the whole system via a script on Unix or via run_system_chained
# on WIN32 where there is no viable shell. Not used right now.
#
proc run_system_old {progname num_oe startprefix} {
    global gvars RunTool
 
    if {[tix platform] == "unix"} {
	set program "| sh -c \"$progname -n $num_oe -f $startprefix 2>&1\""
    } else {
	return [run_system_chained $num_oe]
    }

    make_run_window
 
    if [catch {eval set gvars(pipe) [open $program  "r"]} errmsg] {
        run_error $progname $errmsg
	return
    }
    after $RunTool(delay) [list catch [list fileevent $gvars(pipe) \
        readable get_run_output]]
 
    $gvars(w:text) insert end "executing $program\n\n" {big color2}
}

proc run_system {progname num_oe startprefix} {
    global gvars RunTool

    make_run_window
    if {[tix platform] == "unix"} {
	set program "|sh -c \"$progname -m batch 2>&1\""
    } else {
	set program "|$progname -m batch"
    }
    if [catch {eval set gvars(pipe) [open $program "r+"]} errmsg] {
        run_error $progname $errmsg
	return
    }
    set fid $gvars(pipe)
    after $RunTool(delay) [list catch [list fileevent $gvars(pipe)  \
	readable get_run_output]]
    $gvars(w:text) insert end "executing $program\n\n" {big color2}

    for {set oe 1} {$oe <= $num_oe} {incr oe} {
	puts $fid 0			;# start anew
	set gfile [format "${startprefix}.%.2d" $oe]
	puts $fid $gfile		;# input file?
	flush $fid
    }
    puts $fid 0
    puts $fid exit
    flush $fid
}

proc run_oe {progname oe gfile prev_image} {
    global gvars RunTool

    make_run_window
    if {[tix platform] == "unix"} {
	set program "|sh -c \"$progname -m batch 2>&1\""
    } else {
	set program "|$progname -m batch"
    }
    if [catch {eval set gvars(pipe) [open $program "r+"]} errmsg] {
        run_error $progname $errmsg
	return
    }
    set fid $gvars(pipe)
    after $RunTool(delay) [list catch [list fileevent $gvars(pipe)  \
	readable get_run_output]]
    $gvars(w:text) insert end "executing $program\n\n" {big color2}

    if {$oe == 1} {
        puts $fid 0			;# start anew
	puts $fid $gfile		;# input file?
    } else {
	set prev_oe [expr $oe - 1]
	puts $fid 1			;# start from particular OE
	puts $fid $prev_oe		;#   prev OE number
	puts $fid $prev_image  		;#     and prev OE image
	puts $fid 0			;# change mode?
	puts $fid $gfile		;# input file?
    }
    puts $fid 0				;# want to change mode?
    puts $fid exit			;# done.

    flush $fid
}

proc read_tool_gfile {file a} {
    upvar 1 $a array
    if {[catch "open $file r" fin] != 0} {
	puts stderr "cannot open gfile \"$file\". Going ahead anyway."
	return 1
    }
    while {[gets $fin line] >= 0} {
	# is it a comment or empty line?
	if {[regexp {^[ \t]*#} $line] == 0 &&
	    [regexp {^[ \t]*$} $line] == 0} {
	    set line [split $line]
	    if {[lindex $line 1] != "="} {
		puts stderr \
		    "Illegal syntax in gfile \"$file\". wrong format file?"
		close $fin
		return 1
	    }
	    # use case-indepent variable names.
	    set lname [string tolower [lindex $line 0]]
	    set lvalue [lrange $line 2 end]
	    #
	    # now simply add a variable in gvars array with the same
	    # name as the one in the gfile and the corresponding value.
	    set array($lname) $lvalue
	}
    }
    return 0
}

#
# obj_info says what object we're inspecting here.
#    Examples: {source}
#            : {oe <oe#>}
#            : {screen <oe#> <scr#>}
#
proc run_plotxy {obj_info progname gfile} {
    global gvars RunTool
    set plotxy(DUMMY) ""			;# force to be array

    read_tool_gfile $gfile plotxy
    if {$plotxy(plx_device) == 2} {		;# postscript
	set ps_device 1
    } else {
	set ps_device 0
    }

    make_run_window
    if {[tix platform] == "unix"} {
	set program "|sh -c \"$progname 2>&1\""
    } else {
	set program "|$progname -no-run-primvs"
    }
    set program_hook ""
    if $ps_device {
	set program_hook \
	    "run_program r echo 'Postscript output is in plotxy.ps'"
    }
    #
    # there is a problem with running X-based programs using the usual
    # method, so we might have to resort exec'ing in the background
    # with Tcl's exec primitive.
    #eval exec $program &
    #
    if [catch {eval set gvars(pipe) [open $program "r+"]} errmsg] {
        run_error $progname $errmsg
	return
    }

    set fid $gvars(pipe)
    after $RunTool(delay) [list catch [list fileevent $gvars(pipe)  \
	readable "get_run_output [list $program_hook]"]]
    $gvars(w:text) insert end "executing $program\n\n" {big color2}

    #
    # shove the data into the program now.
    #
    set imagefile ""
    if {$plotxy(plx_spec_file) == 1} {		;# user specifid filename?
	set imagefile "$plotxy(plx_infile)"
    } else {
	switch $plotxy(plx_plot_what) {
	    0 {					;# plotting screen here.
		if {[get_source] != ""} {
		    set imagefile [beamline vget source FILE_SOURCE]
		} else {
		    set imagefile "begin.dat"	;# hope it exists!
		}
	    }
	    1 {					;# plotting OE here
		set oe [lindex $obj_info 1]
		if {$oe == ""} {
		    error "run_plotxy (Internal): No OE number supplied"
		}
		switch $plotxy(plx_image) {
		    0 {				;# at cont. plane
			set imagefile "[format star.%.2d $oe]"
		    }
		    1 {				;# at mirr plane
			set imagefile "[format mirr.%.2d $oe]"
		    }
		}
	    }
	    2 {
		set oe [lindex $obj_info 1]
		set scr [lindex $obj_info 2]
		if {$oe == "" || $scr == ""} {
		    error "run_plotxy (Internal): No SCR/OE number supplied"
		}
		set imagefile "[format screen.%.2d%.2d $oe $scr]"
	    }
	}

	if {$imagefile == ""} {
	    error "Internal Error in run_plotxy: Check image/plot type"
	}
    }
    puts $fid "$imagefile"

    puts $fid "$plotxy(plx_rayopts)"
    puts $fid "$plotxy(plx_comments)"
    puts $fid "$plotxy(plx_row_horiz)"
    puts $fid "$plotxy(plx_row_vert)"

    if {$plotxy(plx_row_horiz) == 11 || $plotxy(plx_row_vert) == 11} { ;# eV
        puts $fid "$plotxy(plx_unit)"
    }

    puts $fid "$plotxy(plx_scale_opts)"

    if {$plotxy(plx_scale_opts) == 2} {		;# EXTERNAL limits
        puts $fid "$plotxy(plx_ext_hmin)"
        puts $fid "$plotxy(plx_ext_hmax)"
        puts $fid "$plotxy(plx_ext_vmin)"
        puts $fid "$plotxy(plx_ext_vmax)"
    }

    puts $fid "$plotxy(plx_plot_type)"
    puts $fid "$plotxy(plx_hairline)"
    puts $fid "$plotxy(plx_ovl_mirror)"

    #
    # histograms now.
    # 
    # if do_hist == 1, that means that PLOTXY expects 0
    #

    if {$plotxy(plx_do_hist) == 0} {
	puts $fid "-1"
    } elseif {$plotxy(plx_do_hist) == 1} {
	puts $fid "0"
	puts $fid "$plotxy(plx_hist_xbins)"
	puts $fid "$plotxy(plx_hist_ybins)"
	if {$plotxy(plx_hist_lim) == 1} {
	    puts $fid "$plotxy(plx_hist_xctr)"
	    puts $fid "$plotxy(plx_hist_xwid)"
	    puts $fid "$plotxy(plx_hist_yctr)"
	    puts $fid "$plotxy(plx_hist_ywid)"
	}
    }
    puts $fid "$plotxy(plx_device)"
    flush $fid
    if {[tix platform] == "windows" && $ps_device} {
	tkwait variable RunTool(waitvar)
	$gvars(w:text) insert end {
------------
Due to a bug in SHADOW under Win32 ('95 and NT), we have to now run 
PRIMVS to get the Postscript output. This may take a while. The output 
file is "plotxy.ps" in current working directory.
------------
} {big color2}
        $gvars(w:text) see end
	run_program r primvs -x -i plotxy.prm
	update idletasks
    }
}

#
# NOT USED right now.
#
proc run_source_plotxy {progname gfile} {
    global gvars RunTool
    set plotxy(DUMMY) ""			;# force to be array

    read_tool_gfile $gfile plotxy
    if {$plotxy(pl1_device) == 2} {		;# postscript
	set ps_device 1
    } else {
	set ps_device 0
    }

    make_run_window
    set program "|sh -c \"$progname 2>&1"
    if $ps_device {
	append program "; echo 'Postscript output is in plotxy.ps'"
    }
    append program "\""
    #
    # there is a problem with running X-based programs using the usual
    # method, so we might have to resort exec'ing in the background
    # with Tcl's exec primitive.
    #eval exec $program &
    #
    if [catch {eval set gvars(pipe) [open $program "r+"]} errmsg] {
        run_error $progname $errmsg
	return
    }

    set fid $gvars(pipe)

    #
    # shove the data into the program now.
    #
    puts $fid "$plotxy(pl1_infile)"
    puts $fid "$plotxy(pl1_rayopts)"
    puts $fid "$plotxy(pl1_comments)"
    puts $fid "$plotxy(pl1_row_horiz)"
    puts $fid "$plotxy(pl1_row_vert)"

    if {$plotxy(pl1_row_horiz) == 11 || $plotxy(pl1_row_vert) == 11} { ;# eV
        puts $fid "$plotxy(pl1_unit)"
    }


    puts $fid "$plotxy(pl1_scale_opts)"

    if {$plotxy(pl1_scale_opts) == 2} {		;# EXTERNAL limits
        puts $fid "$plotxy(pl1_ext_hmin)"
        puts $fid "$plotxy(pl1_ext_hmax)"
        puts $fid "$plotxy(pl1_ext_vmin)"
        puts $fid "$plotxy(pl1_ext_vmax)"
    }

    puts $fid "$plotxy(pl1_plot_type)"
    puts $fid "$plotxy(pl1_hairline)"
    puts $fid "$plotxy(pl1_ovl_mirror)"

    #
    # histograms now.
    # 
    # if do_hist == 1, that means that PLOTXY expects 0
    #

    if {$plotxy(pl1_do_hist) == 0} {
	puts $fid "-1"
    } elseif {$plotxy(pl1_do_hist) == 1} {
	puts $fid "0"
	puts $fid "$plotxy(pl1_hist_xbins)"
	puts $fid "$plotxy(pl1_hist_ybins)"
	if {$plotxy(pl1_hist_lim) == 1} {
	    puts $fid "$plotxy(pl1_hist_xctr)"
	    puts $fid "$plotxy(pl1_hist_xwid)"
	    puts $fid "$plotxy(pl1_hist_yctr)"
	    puts $fid "$plotxy(pl1_hist_ywid)"
	}
    }
    puts $fid "$plotxy(pl1_device)"
    flush $fid

    after $RunTool(delay) [list catch [list fileevent $gvars(pipe)  \
	readable get_run_output]]
    $gvars(w:text) insert end "executing $program\n\n" {big color2}
}

proc run_mirinfo {obj_info progname gfile} {
    global gvars RunTool
    set mirinfo(DUMMY) ""			;# force to be array

    if {[string toupper [lindex $obj_info 0]] != "OE"} {
	dialog .run {}  \
	    "Mirinfo can inspect only OEs, not Source or Screens." \
	    error 0 Dismiss
	set_msg "Run cancelled."
	return
    }

    read_tool_gfile $gfile mirinfo

    make_run_window
    if {[tix platform] == "unix"} {
	set program "|sh -c \"$progname 2>&1 && cat $mirinfo(mir_outfile)\""
	set program_hook ""
    } else {
	set program "|$progname"
	set program_hook "run_program r cat $mirinfo(mir_outfile)"
    }
    if [catch {eval set gvars(pipe) [open $program "r+"]} errmsg] {
        run_error $progname $errmsg
	return
    }
    after $RunTool(delay) [list catch [list fileevent $gvars(pipe)  \
	readable "get_run_output [list $program_hook]"]]
    $gvars(w:text) insert end "executing $program\n\n" {big color2}

    #
    # shove the data into the program now.
    #
    puts $gvars(pipe) "$mirinfo(mir_infile)"
    puts $gvars(pipe) "$mirinfo(mir_title)"
    puts $gvars(pipe) "$mirinfo(mir_comments)"
    puts $gvars(pipe) "$mirinfo(mir_outfile)"
    flush $gvars(pipe)

}

proc run_srcinfo {obj_info progname gfile} {
    global gvars RunTool
    set srcinfo(DUMMY) ""			;# force to be array

    if {[string toupper [lindex $obj_info 0]] != "SOURCE"} {
	dialog .run {}  \
	    "srcinfo can inspect only SOURCE, not OEs or Screens." \
	    error 0 Dismiss
	set_msg "Run cancelled."
	return
    }

    read_tool_gfile $gfile srcinfo

    make_run_window 
    if {[tix platform] == "unix"} {
	set program "|sh -c \"$progname 2>&1 && cat $srcinfo(sri_outfile)\""
	set program_hook ""
    } else {
	set program "|$progname"
	set program_hook "run_program r cat $srcinfo(sri_outfile)"
    }
    if [catch {eval set gvars(pipe) [open $program "r+"]} errmsg] {
        run_error $progname $errmsg
	return
    }
    after $RunTool(delay) [list catch [list fileevent $gvars(pipe)  \
	readable "get_run_output [list $program_hook]"]]
    $gvars(w:text) insert end "executing $program\n\n" {big color2}

    #
    # shove the data into the program now.
    #
    puts $gvars(pipe) "$srcinfo(sri_infile)"
    puts $gvars(pipe) "$srcinfo(sri_title)"
    puts $gvars(pipe) "$srcinfo(sri_comments)"
    puts $gvars(pipe) "$srcinfo(sri_outfile)"
    flush $gvars(pipe)

}

proc run_minmax {obj_info progname gfile} {
    global gvars RunTool
    set minmax(DUMMY) ""			;# force to be array

    read_tool_gfile $gfile minmax

    make_run_window 
    if {[tix platform] == "unix"} {
	set program "|sh -c \"$progname 2>&1\""
    } else {
	set program "|$progname"
    }
    if [catch {eval set gvars(pipe) [open $program "r+"]} errmsg] {
        run_error $progname $errmsg
	return
    }
    after $RunTool(delay) [list catch [list fileevent $gvars(pipe)  \
	readable get_run_output]]
    $gvars(w:text) insert end "executing $program\n\n" {big color2}

    #
    # shove the data into the program now.
    #
    puts $gvars(pipe) "$minmax(min_infile)"
    puts $gvars(pipe) "$minmax(min_rayopts)"

    if {$minmax(min_rayopts) == 1} {		;# include loss at OE <oe>
	set flagval [expr $minmax(min_oe_loss) * -11000]
	puts $gvars(pipe) $flagval
    }

    puts $gvars(pipe) "$minmax(min_comments)"
    flush $gvars(pipe)

}
