/*****************************************************************************
 *                                                                           *
 *  PRIMVS - Graphic Interpreter for X-Ray Lithography
 *                                                                           *
 ***************************************************************************** 
 *
 *  Author:   Luigi Capodieci
 *  Project:  PLPLOT based graphic interpreter
 *
 *  $Source: 
 *  $Revision:
 *  $Date: 
 *  $Log: graph.c,v $
 *  Revision 1.1.1.1  1996/07/11 13:52:44  khan
 *  Initial checkin of SHADOW 2.1 distribution sources
 *
 * Revision 1.1.1.1  1996/01/02  17:07:38  khan
 * Toolset 2.0 release
 *
 *****************************************************************************/ 

#include	"allinclude.h"


/*** GLOBAL VARIABLES ********************************************************/ 

char	device_name[MAX_WORD_LENGTH];

int	plot_page,plot_win;
int	curr_page,curr_win;

char	function_filename[MAX_WORD_LENGTH]="";
FILE	*functionfile;

int		curr_win;
int		cnx,cny;

int		screen_active=FALSE;
int		text_active=FALSE;
int		device_ok=FALSE;
int		window_ok=FALSE;
int		multiwindow_ok=FALSE;
int		charset=NORMAL_CHARSET;
int		orientation=LANDSCAPE;

/*** Variables for REGIONP,REGIONM,XYRANGE,REGIONR ***/
double		pposxmin,pposxmax,pposymin,pposymax; 
double		mposxmin,mposxmax,mposymin,mposymax; 
double		pratio;
double		cxmin,cxmax,cymin,cymax;

/*** Variables for XYZRANGE ***/
double		czmin,czmax;

/*** Variables for VIEWBOX, VIEW ***/
double		basex,basey,height,alt,azi;

/*** Variables for PLOT3D ***/
int		opt3d;

/*** Variables for PLOTCONT ***/
int		nlevel;

/*** Variables for BOX ***/
char		xoptions[MAX_WORD_LENGTH],yoptions[MAX_WORD_LENGTH];
double		xtickint,ytickint;
int		xticksub,yticksub;

/*** Variables for BOX3D ***/
char		zoptions[MAX_WORD_LENGTH];
double		ztickint;


/*** Variables for COLOR ***/
int		color;

/*** Variables for SETCOLOR ***/
int		color_flag=1;

/*** Variables for BACKGROUND ***/
int		background=16;

/*** Variables for SCALECHR, SCALESYM ***/

double		scalechr, scalesym;

/*** Variables for LABEL ***/
char		xlabel[MAX_WORD_LENGTH]="";
char		ylabel[MAX_WORD_LENGTH]="";
char		tlabel[MAX_WORD_LENGTH]="";
char		zlabel[MAX_WORD_LENGTH]="";

/*** Variables for TEXT, GTEXT ***/
char            side[MAX_WORD_LENGTH];
double          ref_ypos,ref_xpos,ref_just;
char            text[MAX_WORD_LENGTH];
double          txpos,typos,tdxpos,tdypos;


/*** Variables for LINESTYLE, LINEWIDTH ***/ 
int		linestyle;
int		linewidth;

/*** Variables for PLOTL,PLOTP,PLOTVECT,MAGNIVECT ***/
float		xx[MAX_X],yy[MAX_Y];
float		dx[MAX_X],dy[MAX_Y];
double		magx=1.0;
double		magy=1.0;
float		cc[MAX_C];
int		point_symbol;
int		np;
int		datain=FALSE; /* flag dat has been read */

/*** Variables for PLOTGREY, PLOTCOLOR ***/
float 		*xxx, *yyy, **zzz;
float		xxxmax,yyymax,xxxmin,yyymin,zzzmax,zzzmin;
int		alloc3d=FALSE;
int		nx,ny;
float		resol=1.0;

/*** Variables for LINE ***/
double		lx1,ly1,lx2,ly2;

/*** Variables for CIRCLE,PARCIRCLE,WAFER ***/
double		xc,yc,radius;
double		arc;
int		wafertype;

/*****************************************************************************/ 

int	total_subpages(nx,ny)
	int	nx,ny;
{
	if (((nx==1)&&(ny==2))||((nx==2)&&(ny==1)))
		return(3);
	  else 
		return(nx*ny);
}

void	exec_subpage()
{
	if (multiwindow_ok) {
	   if (temp_subpage>total_subpages(cnx,cny)) {
	      error_msg_nl("(Subpage number exceeding total subpages)");
	      if (atcommand) parse_error=TRUE;
	      return;
	      }
	    else {
	      curr_page=temp_subpage;
	      pladv((PLINT) curr_page);
	      return;
	      }
	   }
	  else {
	    error_msg_nl("(Execute initmultipage before using SUBPAGE)");
	    if (atcommand) parse_error=TRUE;
	    return;
	    }
}

/*****************************************************************************/ 

void	exec_initpage()

{

/*** TO BE REVISED AFTER FEEDBACK FROM USERS *********************************/
/*
	if ((screen_active && (strcmp(temp_device_name,C_XWIN)==0)) ||
	    (screen_active && (strcmp(temp_device_name,C_XTERM)==0)) ||
	    (text_active && (strcmp(temp_device_name,C_PS)==0)) ||
	    (text_active && (strcmp(temp_device_name,C_XFIG)==0))) {
		error_msg_nl("(Device already initialized)");
		if (atcommand) parse_error=TRUE;
		return;
		}
*/
/*****************************************************************************/

	if ((screen_active)||(text_active)) {
		error_msg_nl("(Device already initialized)");
		if (atcommand) parse_error=TRUE;
		return;
		}

	if ((strcmp(temp_device_name,C_XWIN)==0) || 
	    (strcmp(temp_device_name,C_XTERM)==0) ||
	    (strcmp(temp_device_name,C_TEKT)==0)) {
		screen_active=TRUE;
		}
	if ((strcmp(temp_device_name,C_PS)==0) || 
	    (strcmp(temp_device_name,C_XFIG)==0) ||
	    (strcmp(temp_device_name,C_TEKF)==0)) {
		text_active=TRUE;
		if ((graphfile=fopen(graph_filename,"w"))==NULL) {
                             error_msg_nl("(Cannot open graph file)");
			     if (atcommand) parse_error=TRUE;
			     return;
                             }
		  else {
	  	   /* plsfnam(graph_filename);*/
	  	   plsfile(graphfile);
		   }
	  	}

/*****************************************************************************/

#if defined DEBUG
	printf("Executing Initialization - single page\n");
#endif

/*****************************************************************************/

	    strcpy(device_name,temp_device_name);
	    plscolor((PLINT)color_flag);
	    temp_color=background; exec_background();
    	    plstart(device_name,(PLINT) 1,(PLINT) 1); 
	    pladv(0);
	    plfontld(charset);
	    device_ok=TRUE;
 	    window_ok=FALSE;

}

/*****************************************************************************/ 

void	exec_initmultipage()

{

	if ((screen_active)||(text_active)) {
		error_msg_nl("(Device already initialized)");
		if (atcommand) parse_error=TRUE;
		return;
		}

	if ((strcmp(temp_device_name,C_XWIN)==0) || 
	    (strcmp(temp_device_name,C_XTERM)==0) ||
	    (strcmp(temp_device_name,C_TEKT)==0)) {
		screen_active=TRUE;
		}
	if ((strcmp(temp_device_name,C_PS)==0) || 
	    (strcmp(temp_device_name,C_XFIG)==0) ||
	    (strcmp(temp_device_name,C_TEKF)==0)) {
		text_active=TRUE;
		if ((graphfile=fopen(graph_filename,"w"))==NULL) {
                             error_msg_nl("(Cannot open graph file)");
			     if (atcommand) parse_error=TRUE;
			     return;
                             }
		  else {
	  	   /* plsfnam(graph_filename);*/
	  	   plsfile(graphfile);
		   }
	  	}

/*****************************************************************************/

#if defined DEBUG
	printf("Executing Initialization - multipage\n");
#endif

/*****************************************************************************/

	    cnx=temp_cnx;
	    cny=temp_cny;
	    strcpy(device_name,temp_device_name);
	    plscolor((PLINT)color_flag);
	    temp_color=background; exec_background();
    	    plstart(device_name,(PLINT) cnx,(PLINT) cny); 
	    pladv(0);
	    plfontld(charset);
	    device_ok=TRUE;
 	    window_ok=FALSE;
	    multiwindow_ok=TRUE;

}


/*****************************************************************************/ 

void	exec_regionp()

{
	if (!device_ok) {
	   error_msg_nl("(Initialize device before using REGIONP)");
	   if (atcommand) parse_error=TRUE;
	   return;
	   }
	if ((temp_xmin < 0.0) || (temp_xmin > 1.0)) {
		error_msg_nl("(Illegal Value for XMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_xmax < 0.0) || (temp_xmax > 1.0)) {
		error_msg_nl("(Illegal Value for XMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymin < 0.0) || (temp_ymin > 1.0)) {
		error_msg_nl("(Illegal Value for YMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymax < 0.0) || (temp_ymax > 1.0)) {
		error_msg_nl("(Illegal Value for YMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	pposxmin = temp_xmin;
	pposxmax = temp_xmax;
	pposymin = temp_ymin;
	pposymax = temp_ymax;

#if defined DEBUG
	printf("Executing regionp at %lf,%lf,%lf,%lf\n",
		pposxmin,pposxmax,pposymin,pposymax);
#endif

	plvpor((PLFLT) pposxmin,(PLFLT) pposxmax,
	       (PLFLT) pposymin,(PLFLT) pposymax);

/*** Perform a default initialization of x-y range ***************************/

	plwind((PLFLT) cxmin,(PLFLT) cxmax,(PLFLT) cymin,(PLFLT) cymax);

	window_ok=TRUE;

	return;
}

/*****************************************************************************/ 

void	exec_regionm()

{
	if (!device_ok) {
	   error_msg_nl("(Initialize device before using REGIONM)");
	   if (atcommand) parse_error=TRUE;
	   return;
	   }
	if ((temp_xmin < 0.0) || (temp_xmin > 220.0)) {
		error_msg_nl("(Illegal Value for XMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_xmax < 0.0) || (temp_xmax > 220.0)) {
		error_msg_nl("(Illegal Value for XMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymin < 0.0) || (temp_ymin > 290.0)) {
		error_msg_nl("(Illegal Value for YMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymax < 0.0) || (temp_ymax > 290.0)) {
		error_msg_nl("(Illegal Value for YMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	mposxmin = temp_xmin;
	mposxmax = temp_xmax;
	mposymin = temp_ymin;
	mposymax = temp_ymax;

#if defined DEBUG
	printf("Executing regionm at %lf,%lf,%lf,%lf\n",
		mposxmin,mposxmax,mposymin,mposymax);
#endif

	plsvpa((PLFLT) mposxmin,(PLFLT) mposxmax,
	       (PLFLT) mposymin,(PLFLT) mposymax);

/*** Perform a default initialization of x-y range ***************************/

	plwind((PLFLT) cxmin,(PLFLT) cxmax,(PLFLT) cymin,(PLFLT) cymax);

	window_ok=TRUE;

	return;
}

/*****************************************************************************/ 

void	exec_regionr()

{
	if (!device_ok) {
	   error_msg_nl("(Initialize device before using REGIONR)");
	   if (atcommand) parse_error=TRUE;
	   return;
	   }
	if ((temp_xmin < 0.0) || (temp_xmin > 1.0)) {
		error_msg_nl("(Illegal Value for XMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_xmax < 0.0) || (temp_xmax > 1.0)) {
		error_msg_nl("(Illegal Value for XMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymin < 0.0) || (temp_ymin > 1.0)) {
		error_msg_nl("(Illegal Value for YMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymax < 0.0) || (temp_ymax > 1.0)) {
		error_msg_nl("(Illegal Value for YMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ratio < 0.0) || (temp_ratio > 100)) {
		error_msg_nl("(Illegal Value for RATIO)");
		if (atcommand) parse_error=TRUE;
		return;
		}

	pposxmin = temp_xmin;
	pposxmax = temp_xmax;
	pposymin = temp_ymin;
	pposymax = temp_ymax;
	pratio = temp_ratio;

#if defined DEBUG
	printf("Executing regionr at %lf,%lf,%lf,%lf,%lf\n",
		pposxmin,pposxmax,pposymin,pposymax,pratio);
#endif

	plvpas((PLFLT) pposxmin,(PLFLT) pposxmax,
	       (PLFLT) pposymin,(PLFLT) pposymax,(PLFLT) pratio); 

/***	plvasp((PLFLT) pratio); ***/

/*** Perform a default initialization of x-y range ***************************/

	plwind((PLFLT) cxmin,(PLFLT) cxmax,(PLFLT) cymin,(PLFLT) cymax);

	window_ok=TRUE;

	return;
}

/*****************************************************************************/ 

void	exec_xyrange()

{
#if defined DEBUG
	printf("Executing xyrange at %lf,%lf,%lf,%lf\n",
                temp_xmin,temp_xmax,temp_ymin,temp_ymax);
#endif


	if (window_ok) {
	   if (!scaleflag) {
	      cxmin = temp_xmin;
	      cxmax = temp_xmax;
	      cymin = temp_ymin;
	      cymax = temp_ymax;
	      }


	   plwind((PLFLT) cxmin,(PLFLT) cxmax,(PLFLT) cymin,(PLFLT) cymax);
	   return;
	   }
	 else {
	   error_msg_nl("(Set region before using XYRANGE)");
	   if (atcommand) parse_error=TRUE;
	   }
}

/*****************************************************************************/ 

void	exec_xyzrange()

{

#if defined DEBUG
	printf("Executing xyzrange at %lf,%lf,%lf,%lf,%lf,%lf\n",
                temp_xmin,temp_xmax,temp_ymin,temp_ymax,temp_zmin,temp_zmax);
#endif


	if (window_ok) {
	   if (!scaleflag) {
	      cxmin = temp_xmin;
	      cxmax = temp_xmax;
	      cymin = temp_ymin;
	      cymax = temp_ymax;
	      czmin = temp_zmin;
	      czmax = temp_zmax;
	      }

	   plwind((PLFLT) cxmin,(PLFLT) cxmax,(PLFLT) cymin,(PLFLT) cymax);
	   return;
	   }
	 else {
	   error_msg_nl("(Set region before using XYZRANGE)");
	   if (atcommand) parse_error=TRUE;
	   }
}


/*** Execute the BOX command *************************************************/ 

void	exec_box()
{

#if defined DEBUG
	printf("Executing BOX command\n");
	printf("%s,%lf,%d,%s,%lf,%d\n",temp_xoptions,temp_xtickint,
		temp_xticksub,temp_yoptions,temp_ytickint,temp_yticksub);
#endif

	if (window_ok) {
		strcpy(xoptions,temp_xoptions);
		xtickint=temp_xtickint;
		xticksub=temp_xticksub;
		strcpy(yoptions,temp_yoptions);
		ytickint=temp_ytickint;
		yticksub=temp_yticksub;

		plbox((char *) xoptions,(PLFLT) xtickint,(PLINT) xticksub,
		      (char *) yoptions,(PLFLT) ytickint,(PLINT) yticksub);
		plsmaj(0.0,1.0);
		}
	   else {
		error_msg_nl("(Set region and x-y range before using BOX)");
		if (atcommand) parse_error=TRUE;
		}

}

/*** Execute the BOX3D command ***********************************************/ 

void	exec_box3d()
{

#if defined DEBUG
	printf("Executing BOX3D command\n");
	printf("%s,%lf,%s,%lf,%s,%lf\n",temp_xoptions,temp_xtickint,
		temp_yoptions,temp_ytickint,temp_zoptions,temp_ztickint);
#endif

	if (window_ok) {
		strcpy(xoptions,temp_xoptions);
		xtickint=temp_xtickint;
		strcpy(yoptions,temp_yoptions);
		ytickint=temp_ytickint;
		strcpy(zoptions,temp_zoptions);
		ztickint=temp_ztickint;

		plbox3((char *) xoptions, xlabel, (PLFLT) xtickint,(PLINT) 0,
		       (char *) yoptions, ylabel, (PLFLT) ytickint,(PLINT) 0,
		       (char *) zoptions, zlabel, (PLFLT) ztickint,(PLINT) 0);
		}
	   else {
		error_msg_nl("(Set region and xyz range before using BOX3D)");
		if (atcommand) parse_error=TRUE;
		}

}

/*** Execute the VIEWBOX Command *********************************************/ 

void	exec_viewbox()
{

#if defined DEBUG
	printf("Executing viewbox with %lf,%lf,%lf\n",
                temp_basex,temp_basey,temp_height);
#endif

	if (window_ok) {
	 if (!scaleflag) {
	   basex = temp_basex;
	   basey = temp_basey;
     	   height = temp_height;
  	   }
	  return;
	  }

	  else {
	    error_msg_nl("(Set region before using VIEWBOX)");
	    if (atcommand) parse_error=TRUE;
	    }
}

/*** Execute the VIEW Command ************************************************/ 

void	exec_view()
{

#if defined DEBUG
	printf("Executing view with %lf,%lf\n",
                temp_alt,temp_azi);
#endif

	if (window_ok) {
	   alt = temp_alt;
	   azi = temp_azi;
  	   }
	  else {
	    error_msg_nl("(Set region before using VIEW)");
	    if (atcommand) parse_error=TRUE;
	    }
}

/*** Execute the Color command ***********************************************/ 

void	exec_color()
{

	if (window_ok) {
	
#if defined DEBUG
	printf("Executing COLOR command\n");
	printf("Color: %d\n",temp_color);
#endif
		color=temp_color;
		if ((temp_color<=15)&&(temp_color>=0)) {
			plcol((PLINT) color);
			return;
			}
		if (temp_color==16) {
			plrgb(0.0,0.0,0.0);
			return;
			}
		}
	   else {
		error_msg_nl("(Set region and x-y range before using COLOR)");
		if (atcommand) parse_error=TRUE;
		}
}

/*** Execute the setcolor command ********************************************/ 
void	exec_setcolor()
{
	if ((!window_ok)&&(!device_ok)) {

#if defined DEBUG
	printf("Executing SETCOLOR command\n");
	printf("Switch: %d\n",temp_color);
#endif
	  if ((temp_color!=1)&&(temp_color!=0)) {
	     error_msg_nl("(Invalid Switch for SETCOLOR)");
	     return;
	     }

	   color_flag=temp_color;
	   }
	 else {
		error_msg_nl("(Use SETCOLOR before initializing the page)");
		if (atcommand) parse_error=TRUE;
		}
}

/*** Execute the background command *****************************************/ 

/* #include <plstream.h> */

void	exec_background()
{
	PLColor		bgcol;

	if ((!window_ok)&&(!device_ok)) {
	
#if defined DEBUG
	printf("Executing BACKGROUND command\n");
	printf("Color: %d\n",temp_color);
#endif
		background=temp_color;
		switch(background) {
		 case 0: bgcol.r=255;bgcol.g=114;bgcol.b=86;break; /* coral */
		 case 1: bgcol.r=255;bgcol.g=0;bgcol.b=0;break;    /* red */
		 case 2: bgcol.r=255;bgcol.g=255;bgcol.b=0;break;  /* yellow */
		 case 3: bgcol.r=0;bgcol.g=255;bgcol.b=0;break;    /* green  */
		 case 4: bgcol.r=50;bgcol.g=191;bgcol.b=193;break; /* aquama */
		 case 5: bgcol.r=255;bgcol.g=181;bgcol.b=197;break;/* pink */
		 case 6: bgcol.r=245;bgcol.g=222;bgcol.b=179;break;/* wheat */
		 case 7: bgcol.r=126;bgcol.g=126;bgcol.b=126;break;/* grey */
		 case 8: bgcol.r=165;bgcol.g=42;bgcol.b=42;break;  /* brown */
		 case 9: bgcol.r=0;bgcol.g=0;bgcol.b=255;break;    /* blue */
		 case 10: bgcol.r=138;bgcol.g=43;bgcol.b=226;break;/* bluevio */
		 case 11: bgcol.r=0;bgcol.g=255;bgcol.b=255;break; /* cyan */
		 case 12: bgcol.r=25;bgcol.g=204;bgcol.b=223;break;/* turquoi */
		 case 13: bgcol.r=255;bgcol.g=0;bgcol.b=255;break; /* magenta */
		 case 14: bgcol.r=233;bgcol.g=150;bgcol.b=122;break;/*salmon */
		 case 15: bgcol.r=255;bgcol.g=255;bgcol.b=255;break;/*white */
		 case 16:
		 default:
		          bgcol.r=0;bgcol.g=0;bgcol.b=0;break;      /* black */

		}

		plscolbg((PLINT)bgcol.r,(PLINT)bgcol.g,(PLINT)bgcol.b);
		}
	   else {
		error_msg_nl("(Use BACKGROUND before initializing the page)");
		if (atcommand) parse_error=TRUE;
		}
}

/*** Execute the SCALECHR command ********************************************/ 

void	exec_scalechr()
{

	if (window_ok) {
	
#if defined DEBUG
	printf("Executing SCALECHR command\n");
	printf("Scaling factor: %d\n",temp_scalechr);
#endif
		scalechr = temp_scalechr;
		plschr((PLFLT) 0.0, (PLFLT) scalechr);
		}
	   else {
		error_msg_nl("(Set region and x-y range before using SCALECHR)");
		if (atcommand) parse_error=TRUE;
		}
}

/*** Execute the SCALESYM command ********************************************/ 

void	exec_scalesym()
{

	if (window_ok) {
	
#if defined DEBUG
	printf("Executing SCALESYM command\n");
	printf("Scaling factor: %d\n",temp_scalesym);
#endif
		scalesym = temp_scalesym;
		plssym((PLFLT) 0.0, (PLFLT) scalesym);
		}
	   else {
		error_msg_nl("(Set region and x-y range before using SCALESYM)");
		if (atcommand) parse_error=TRUE;
		}
}

/*** Execute the Orientation command *****************************************/ 

void	exec_orientation()
{

	if (! window_ok) {
	
#if defined DEBUG
	printf("Executing ORIENTATION command\n");
	printf("Orientation: %d\n",temp_orientation);
#endif
		orientation=temp_orientation;
		plsori((PLINT) orientation);
		}
	   else {
		error_msg_nl("(ORIENTATION must be set before initializing the page)");
		if (atcommand) parse_error=TRUE;
		}
}

/*** Execute the LABEL command ***********************************************/ 

void	exec_label()
{

	if (window_ok) {

#if defined DEBUG
	printf("Executing LABEL command\n");
	printf("X-label %s\n",temp_xlabel);
	printf("Y-label %s\n",temp_ylabel);
	printf("T-label %s\n",temp_tlabel);
#endif
		strcpy(xlabel,temp_xlabel);
		strcpy(ylabel,temp_ylabel);
		strcpy(tlabel,temp_tlabel);
		pllab(xlabel,ylabel,tlabel);
		}
	   else {
		error_msg_nl("(Set region and x-y range before using LABEL)");
		if (atcommand) parse_error=TRUE;
		}
}

/*** Execute the LABEL3D command *********************************************/ 

void	exec_label3d()
{

	if (window_ok) {

#if defined DEBUG
	printf("Executing LABEL3D command\n");
	printf("X-label %s\n",temp_xlabel);
	printf("Y-label %s\n",temp_ylabel);
	printf("Z-label %s\n",temp_zlabel);
#endif
		strcpy(xlabel,temp_xlabel);
		strcpy(ylabel,temp_ylabel);
		strcpy(zlabel,temp_zlabel);
		}
	   else {
		error_msg_nl("(Set region and xyz range before using LABEL3D)");
		if (atcommand) parse_error=TRUE;
		}
}

/*** Execute the TEXT command ************************************************/ 

void	exec_text()
{

	if (window_ok) {

#if defined DEBUG
	printf("Executing TEXT command\n");
	printf("Side  %s\n",temp_side);
	printf("Y-pos %lf\n",temp_ref_ypos);
	printf("X-pos %lf\n",temp_ref_xpos);
	printf("Just  %lf\n",temp_ref_just);
	printf("Text  %s\n",temp_text);
#endif

	   if ((temp_ref_xpos>1.0)||(temp_ref_xpos<0.0)) {
		error_msg_nl("(X position of reference point must be between 0 and 1)");
		if (atcommand) parse_error=TRUE;
		return;
		}

	   if ((temp_ref_just>1.0)||(temp_ref_just<0.0)) {
		error_msg_nl("(Justification of reference point must be between 0 and 1)");
		if (atcommand) parse_error=TRUE;
		return;
		}

	   strcpy(side,temp_side);
	   ref_ypos=temp_ref_ypos;
	   ref_xpos=temp_ref_xpos;
	   ref_just=temp_ref_just;
	   strcpy(text,temp_text);

	   plmtex(side,(PLFLT) ref_ypos,(PLFLT) ref_xpos,(PLFLT) ref_just,text);
	   }
	 else {
		error_msg_nl("(Set region before using TEXT)");
		if (atcommand) parse_error=TRUE;
		}
	
}


/*** Execute the GTEXT command ***********************************************/ 

void	exec_gtext()
{

	if (window_ok) {

#if defined DEBUG
	printf("Executing GTEXT command\n");
	printf("X-pos  %lf\n",temp_txpos);
	printf("Y-pos  %lf\n",temp_typos);
	printf("DX-pos %lf\n",temp_tdxpos);
	printf("DY-pos %lf\n",temp_tdypos);
	printf("Just   %lf\n",temp_ref_just);
	printf("Text   %s\n",temp_text);
#endif

	   if ((temp_ref_just>1.0)||(temp_ref_just<0.0)) {
		error_msg_nl("(Justification of reference point must be between 0 and 1)");
		if (atcommand) parse_error=TRUE;
		return;
		}

	   txpos=temp_txpos;
	   typos=temp_typos;
	   tdxpos=temp_tdxpos;
	   tdypos=temp_tdypos;
	   ref_just=temp_ref_just;
	   strcpy(text,temp_text);

	   plptex((PLFLT) txpos,(PLFLT) typos,(PLFLT) tdxpos,(PLFLT) tdypos,
			(PLFLT) ref_just,text);
	   }
	 else {
		error_msg_nl("(Set region before using GTEXT)");
		if (atcommand) parse_error=TRUE;
		}
	
}


/*** Execute the LINESTYLE command *******************************************/

void	exec_linestyle()
{

	if (window_ok) {
	
#if defined DEBUG
	printf("Executing LINESTYLE command\n");
	printf("Linestyle: %d\n",temp_linestyle);
#endif
	  if ((temp_linestyle>0)&&(temp_linestyle<=8)) {
		linestyle=temp_linestyle;
		pllsty((PLINT) linestyle);
		}
	   else {
		error_msg_nl("(Incorrect LINESTYLE value)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	  }
	  else {
		error_msg_nl("(Set region and x-y range before using LINESTYLE)");
		if (atcommand) parse_error=TRUE;
		}
}


/*** Execute the LINEWIDTH command *******************************************/

void	exec_linewidth()
{

	if (window_ok) {
	
#if defined DEBUG 
	printf("Executing LINEWIDTH command\n");
	printf("Linewidth: %d\n",temp_linewidth);
#endif 
		linewidth=temp_linewidth;
		plwid((PLINT) linewidth);
		}
	   else {
		error_msg_nl("(Set region and x-y range before using LINEWIDTH)");
		if (atcommand) parse_error=TRUE;
		}
}

/*** Execute the RESOLUTION command ******************************************/

void	exec_resolution()
{
#if defined DEBUG 
	printf("Executing RESOLUTION command\n");
	printf("Resolution: %f\n",temp_resol);
#endif 
	if (temp_resol>0.0)
		resol=temp_resol;
}

/*** Execute the PLOTL command ***********************************************/ 

void	exec_plotl()
{

if (window_ok) {

	if (!datain) {
	  np = read_xyc(xcol,ycol,ccol,bnp,fnp); /* Read the data points */
	  }

	if (scaleflag) {
	  cxmin=xxxmin; cxmax=xxxmax;
	  cymin=yyymin; cymax=yyymax;
	  exec_xyrange();
	  }


	/* pladv((PLINT) curr_page);*/	/* Set current page number */

	plgra();		/* Switch to graphic mode */
	plline(np,xx,yy);	/* Plot the data points */
	pltext(); 		/* Switch to text mode */
	}
   else {
	error_msg_nl("(Set region and x-y range before using PLOTL)");
	if (atcommand) parse_error=TRUE;
	}
}


/*** Execute the PLOTP command ***********************************************/ 

void	exec_plotp()
{

if (window_ok) {

	if (!datain) {
	  np = read_xyc(xcol,ycol,ccol,bnp,fnp);
	  }

	if (scaleflag) {
	  cxmin=xxxmin; cxmax=xxxmax;
	  cymin=yyymin; cymax=yyymax;
	  exec_xyrange();
	  }


	/* pladv((PLINT) curr_page);*/	/* Set current page number */

	plgra();			/* Switch to graphic mode */
	plpoin(np,xx,yy,point_symbol);  /* Plot the data points (symbols) */
	pltext(); 			/* Switch to text mode */
	}
   else {
	error_msg_nl("(Set region and x-y range before using PLOTP)");
	if (atcommand) parse_error=TRUE;
	}
}


/*** Execute the PLOTGREY command ********************************************/ 
void	exec_plotgrey()
{
	int	i,j;
	float	xval[4],yval[4];
	int	inc[1],del[1];
	float	ccc,deltaz,dx,dy;

	int	icc=0;	/* X Window Patch */

if (window_ok) {

	if (!datain) {
	  np = read_image();
	  }

	if (scaleflag) {
	  cxmin=xxxmin; cxmax=xxxmax;
	  cymin=yyymin; cymax=yyymax;
	  czmin=zzzmin; czmax=zzzmax;
	  exec_xyrange();
	  }

	/* pladv((PLINT) curr_page);*/	/* Set current page number */

	plgra();			/* Switch to graphic mode */


/* Execute the greyscale plot */
	 deltaz=zzzmax-zzzmin;
	 dx=(xxxmax-xxxmin)/((float)(nx-1.0)*2.0);
	 dy=(yyymax-yyymin)/((float)(ny-1.0)*2.0);
	 inc[0]=0;del[0]=(PLINT) floor(100.0*resol);
	 plpat(1,(PLINT *)inc,(PLINT *)del);
	 gresc((PLINT)200,NULL); /* Set  Greyscale Plot */
	 for (i=0;i<nx;i++) {
	  for (j=0;j<ny;j++) {
		ccc = (float) (((zzz[i][j])-zzzmin)/(deltaz));
		icc = (int) floor(ccc*(float)GREYLEVELS);
/* printf("%d\n",icc); */
		gresc((PLINT)icc,NULL);
		xval[0]=xxx[i]-dx;yval[0]=yyy[j]-dy;
		xval[1]=xxx[i]+dx;yval[1]=yyy[j]-dy;
		xval[2]=xxx[i]+dx;yval[2]=yyy[j]+dy;
		xval[3]=xxx[i]-dx;yval[3]=yyy[j]+dy;
		plfill(4,xval,yval);
		}
	   }

	pltext(); 			/* Switch to text mode */
	}
   else {
	error_msg_nl("(Set region and x-y range before using PLOTGREY)");
	if (atcommand) parse_error=TRUE;
	}
}

/*** Execute the PLOTCOLOR command *******************************************/ 
#define	COLORLEVELS	(94)
void	exec_plotcolor()
{
	int	i,j;
	float	xval[4],yval[4];
	int	inc[1],del[1];
	float	ccc,deltaz,dx,dy;

	int	icc=0;	/* X Window Patch */

if (window_ok) {

	if (!datain) {
	  np = read_image();
	  }

	if (scaleflag) {
	  cxmin=xxxmin; cxmax=xxxmax;
	  cymin=yyymin; cymax=yyymax;
	  czmin=zzzmin; czmax=zzzmax;
	  exec_xyrange();
	  }

	/* pladv((PLINT) curr_page);*/	/* Set current page number */

	plgra();			/* Switch to graphic mode */

/* Execute the colorscale plot */
	 deltaz=zzzmax-zzzmin;
	 dx=(xxxmax-xxxmin)/((float)(nx-1.0)*2.0);
	 dy=(yyymax-yyymin)/((float)(ny-1.0)*2.0);
	 inc[0]=0;del[0]=(PLINT) floor(100.0*resol);
	 plpat(1,(PLINT *)inc,(PLINT *)del);
	 gresc((PLINT)201,NULL); /* Set Color Plot */
	 for (i=0;i<nx;i++) {
	  for (j=0;j<ny;j++) {
		ccc = (float) (((zzz[i][j])-zzzmin)/(deltaz));
		icc = (int) floor(ccc*(float)COLORLEVELS);
		gresc((PLINT)icc,NULL);
		xval[0]=xxx[i]-dx;yval[0]=yyy[j]-dy;
		xval[1]=xxx[i]+dx;yval[1]=yyy[j]-dy;
		xval[2]=xxx[i]+dx;yval[2]=yyy[j]+dy;
		xval[3]=xxx[i]-dx;yval[3]=yyy[j]+dy;
		plfill(4,xval,yval);
		}
	   }

	pltext(); 			/* Switch to text mode */
	}
   else {
	error_msg_nl("(Set region and x-y range before using PLOTCOLOR)");
	if (atcommand) parse_error=TRUE;
	}
}


/*****************************************************************************/ 

void	exec_colorbar()

{
	float	x,y,dx,dy;
	int	level;

	if (!window_ok) {
	   error_msg_nl("(Set region before using COLORBAR)");
	   if (atcommand) parse_error=TRUE;
	   return;
	   }
	if ((temp_xmin < 0.0) || (temp_xmin > 1.0)) {
		error_msg_nl("(Illegal Value for XMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_xmax < 0.0) || (temp_xmax > 1.0)) {
		error_msg_nl("(Illegal Value for XMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymin < 0.0) || (temp_ymin > 1.0)) {
		error_msg_nl("(Illegal Value for YMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymax < 0.0) || (temp_ymax > 1.0)) {
		error_msg_nl("(Illegal Value for YMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	pposxmin = temp_xmin;
	pposxmax = temp_xmax;
	pposymin = temp_ymin;
	pposymax = temp_ymax;

#if defined DEBUG
	printf("Executing colorbar at %lf,%lf,%lf,%lf\n",
		pposxmin,pposxmax,pposymin,pposymax);
#endif

	plvpor((PLFLT) pposxmin,(PLFLT) pposxmax,
	       (PLFLT) pposymin,(PLFLT) pposymax);
	plwind((PLFLT) 0,(PLFLT) 1,(PLFLT) 0,(PLFLT) COLORLEVELS);

/* Draw the color levels */
	 gresc((PLINT)201,NULL); /* Set Color Plot */
	 dy=(float) 1.0/20.0;
	 level=0;
	 for (y=0;y<COLORLEVELS;y=y+dy){
	   if (y>level) {
	      level++;
	      gresc((PLINT)level,NULL);
	      }
	   pljoin(0.0,y,1.0,y);
	   }
	plwind((PLFLT) 0,(PLFLT) 1,(PLFLT) czmin,(PLFLT) czmax);
	plsmaj(0.0,0.0);

	return;
}

/*****************************************************************************/ 

void	exec_greybar()

{
	float	x,y,dx,dy;
	int	level;

	if (!window_ok) {
	   error_msg_nl("(Set region before using GREYBAR)");
	   if (atcommand) parse_error=TRUE;
	   return;
	   }
	if ((temp_xmin < 0.0) || (temp_xmin > 1.0)) {
		error_msg_nl("(Illegal Value for XMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_xmax < 0.0) || (temp_xmax > 1.0)) {
		error_msg_nl("(Illegal Value for XMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymin < 0.0) || (temp_ymin > 1.0)) {
		error_msg_nl("(Illegal Value for YMIN)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	if ((temp_ymax < 0.0) || (temp_ymax > 1.0)) {
		error_msg_nl("(Illegal Value for YMAX)");
		if (atcommand) parse_error=TRUE;
		return;
		}
	pposxmin = temp_xmin;
	pposxmax = temp_xmax;
	pposymin = temp_ymin;
	pposymax = temp_ymax;

#if defined DEBUG
	printf("Executing greybar at %lf,%lf,%lf,%lf\n",
		pposxmin,pposxmax,pposymin,pposymax);
#endif

	plvpor((PLFLT) pposxmin,(PLFLT) pposxmax,
	       (PLFLT) pposymin,(PLFLT) pposymax);
	plwind((PLFLT) 0,(PLFLT) 1,(PLFLT) 0,(PLFLT) GREYLEVELS);

/* Draw the grey levels */
	 gresc((PLINT)200,NULL); /* Set Grey Plot */
	 dy=(float) 1.0/20.0;
	 level=0;
	 for (y=0;y<GREYLEVELS;y=y+dy){
	   if (y>level) {
	      level++;
	      gresc((PLINT)level,NULL);
	      }
	   pljoin(0.0,y,1.0,y);
	   }
	plwind((PLFLT) 0,(PLFLT) 1,(PLFLT) czmin,(PLFLT) czmax);
	plsmaj(0.0,0.0);

	return;
}

/*** Execute the PLOT3D command **********************************************/ 

void	exec_plot3d()
{

	double		x2d,y2d;

if (window_ok) {

	opt3d = temp_opt3d;

	if (!datain) {
	  np = read_image();
	  }

	if (scaleflag) {
	  cxmin=xxxmin; cxmax=xxxmax;
	  cymin=yyymin; cymax=yyymax;
	  czmin=zzzmin; czmax=zzzmax;

	  basex=cxmax-cxmin;
	  basey=cymax-cymin;
	  height=czmax-czmin;
	  }

/*** Set up window ***/
	
	x2d = sqrt((basex*basex)+(basey*basey));
	y2d = sqrt((basex*basex)+(basey*basey)+(height*height));

/*** Increase size of 20% to account for 3D box ***/
	x2d = x2d + (0.20*x2d);
	y2d = y2d + (0.20*y2d);

	if (alt<70) 
	    plwind((PLFLT) (-x2d/2.0),(PLFLT) (x2d/2.0),
	           (PLFLT) (-y2d/3.0),(PLFLT) (2.0*y2d/3.0));
	  else
	    plwind((PLFLT) (-x2d/2.0),(PLFLT) (x2d/2.0),
	           (PLFLT) (-y2d/2.0),(PLFLT) (y2d/2.0));

	plw3d((PLFLT) basex,(PLFLT) basey,(PLFLT) height,
	      (PLFLT) cxmin,(PLFLT) cxmax,
	      (PLFLT) cymin,(PLFLT) cymax,
	      (PLFLT) czmin,(PLFLT) czmax,
	      (PLFLT) alt, (PLFLT) azi);

	plgra();			/* Switch to graphic mode */

	plmesh(xxx,yyy,zzz,nx,ny,opt3d); 

	pltext(); 			/* Switch to text mode */
	}

   else {
	error_msg_nl("(Set region and viewbox  before using PLOT3D)");
	if (atcommand) parse_error=TRUE;
	}
}

/*** Execute the PLOTCONT command ********************************************/ 

/*** Transformation routine for contour plots ***/

void	mypltr(PLFLT x,PLFLT y,PLFLT *tx,PLFLT *ty,void *pltr_data)
{

	*tx=xxx[(int)floor(x)];
	*ty=yyy[(int)floor(y)];

}

void	exec_plotcont()
{
	float	*clevel;
	float	dlevel,contour;
	int	k;

if (window_ok) {

	nlevel = temp_nlevel;

	if (!datain) {
	  np = read_image();
	  }

	if (scaleflag) {
	  cxmin=xxxmin; cxmax=xxxmax;
	  cymin=yyymin; cymax=yyymax;
	  }

	czmin=zzzmin; czmax=zzzmax;
	plwind((PLFLT)cxmin,(PLFLT)cxmax,(PLFLT)cymin,(PLFLT)cymax);

/*** Generate the contour levels ***/

	clevel=(float *)malloc(nlevel*sizeof(float));
	dlevel=(czmax-czmin)/nlevel;
	contour=czmin;
	for (k=0;k<nlevel;k++) {
		clevel[k]=contour;
		contour=contour+dlevel;
		}

/*** Pot the contour ***/

	plgra();			/* Switch to graphic mode */

	plcont(zzz,nx,ny,1,nx,1,ny,clevel,nlevel,mypltr,NULL); 

	pltext(); 			/* Switch to text mode */
	}

   else {
	error_msg_nl("(Set region and xyrange, before using PLOTCONT)");
	if (atcommand) parse_error=TRUE;
	}
}


/*** Execute the PLOTVECT command ********************************************/ 

void	exec_plotvect()
{
	int	i;
	double	x1,y1,x2,y2;
	float	px,py;
	double	alfa,beta,f,d;
	double	xrt1,xrt2,yrt1,yrt2,xa1,xa2,ya1,ya2;

if (window_ok) {

	if (!datain) {
	   np=read_vect();
	   }

	if (scaleflag) {
	  cxmin=xxxmin; cxmax=xxxmax;
	  cymin=yyymin; cymax=yyymax;
	  exec_xyrange();
	  }


	/* pladv((PLINT) curr_page);*/	/* Set current page number */

	plgra();		/* Switch to graphic mode */
	for(i=0;i<np;i++) {
		x1=xx[i];
		y1=yy[i];
		x2=x1+dx[i]*magx;
		y2=y1+dy[i]*magy;
		pljoin(x1,y1,x2,y2);
		px=(float)x2;
		py=(float)y2;
		/*** Draw an arrow if mark is zero ***/
		if (temp_mark==0) {
		   if ((dx[i]==0)&&(dy[i]==0)) continue;
		   alfa=(20.0/180.0)*3.1415;
		   if ((dx[i]*magx)==0) dx[i]=0.0;
		   beta=atan((dy[i]*magy)/(dx[i]*magx));
		   d=sqrt(dx[i]*dx[i]*magx*magx+dy[i]*dy[i]*magy*magy);
		   f=0.25*d;
		   xrt1=d-f*cos(alfa);
		   yrt1=f*sin(alfa);
		   xrt2=d-f*cos(alfa);
		   yrt2=-f*sin(alfa);
		   /*** RotoTranslation ***/
		   if ((dx[i]*magx)<0) {
			xrt1=-xrt1;
			yrt1=-yrt1;
			xrt2=-xrt2;
			yrt2=-yrt2;
			}
		   xa1=xrt1*cos(beta)-yrt1*sin(beta)+x1;
		   ya1=xrt1*sin(beta)+yrt1*cos(beta)+y1;
		   xa2=xrt2*cos(beta)-yrt2*sin(beta)+x1;
		   ya2=xrt2*sin(beta)+yrt2*cos(beta)+y1;
		   pljoin(xa1,ya1,x2,y2);
		   pljoin(xa2,ya2,x2,y2);
		   }
		  else 
		   plpoin(1,(PLFLT *)&px,(PLFLT *)&py,(PLINT)temp_mark);
		}
/***	printf("%d PLOTTING VECTOR FIELD\n",np); ***/
	pltext(); 		/* Switch to text mode */
	}
   else {
	error_msg_nl("(Set region and x-y range before using PLOTVECT)");
	if (atcommand) parse_error=TRUE;
	}
}


/*** Execute the BREAKPAGE command *******************************************/ 

void	exec_breakpage()

{	if (window_ok) {
		plpage();
		plclr();
		}
   	  else {
	    error_msg_nl("(Set region and x-y range before using BREAKPAGE)");
	    if (atcommand) parse_error=TRUE;
	    }
}

/*** Execute the CLOSEPAGE command *******************************************/ 

void	exec_closepage()

{
	if (screen_active) screen_active=FALSE;
	if (text_active) text_active=FALSE;
	device_ok=FALSE;
	window_ok=FALSE;
	multiwindow_ok=FALSE;
	plend(); 
}


/*** Execute the LINE command ************************************************/ 

void	exec_line()

{
if (window_ok) {

	lx1=temp_x1;
	ly1=temp_y1;
	lx2=temp_x2;
	ly2=temp_y2;

	/* pladv((PLINT) curr_page);*/	/* Set current page number */

	plgra(); 			/* Switch to graphic mode */
	pljoin(lx1,ly1,lx2,ly2); 	/* Draw the line */
	pltext(); 			/* Switch to text mode */
	}
   else {
	error_msg_nl("(Set region and x-y range before using LINE)");
	if (atcommand) parse_error=TRUE;
	}
}

/*** Execute the CIRCLE command *********************************************/


void	exec_circle()

{
	double	phi,x1,y1,x2,y2;
	double	delta=0.1;

if (window_ok) {

	xc=temp_xc;
	yc=temp_yc;
	radius=temp_radius;

	/* pladv((PLINT) curr_page);*/	/* Set current page number */

	plgra(); 			/* Switch to graphic mode */
	for(phi=0;phi<=6.2830;phi=phi+delta) {

		x1=xc+(radius*cos(phi));
		y1=yc+(radius*sin(phi));
		x2=xc+(radius*cos(phi+delta));
		y2=yc+(radius*sin(phi+delta));
		
		pljoin(x1,y1,x2,y2);
		}
	pltext(); 			/* Switch to text mode */
	}
   else {
	error_msg_nl("(Set region and x-y range before using CIRCLE)");
	if (atcommand) parse_error=TRUE;
	}
}


/*** Execute the WAFER command **********************************************/


void	exec_wafer()

{
	double	phi,rho,x1,y1,x2,y2; /*** phi in degrees, rho in radiants ***/
	double	delta=1.0;	     /*** delta in degrees ***/
	double	ratio=6.2830/360;
	double	cut1,cut2;		     /* in degrees */

if (window_ok) {

	xc=temp_xc;
	yc=temp_yc;
	radius=temp_radius;
	wafertype=temp_wafertype;

	cut1=0-(90-wafertype/2);
	cut2=180+(90-wafertype/2);

	/* pladv((PLINT) curr_page);*/	/* Set current page number */

	plgra(); 	/* Switch to graphic mode */

	phi=cut1;
	rho=phi*ratio;
	x1=xc+(radius*cos(rho));
	y1=yc+(radius*sin(rho));
	phi=cut2;
	rho=(phi)*ratio;
	x2=xc+(radius*cos(rho));
	y2=yc+(radius*sin(rho));
	pljoin(x1,y1,x2,y2);

	for(phi=cut1;phi<cut2;phi=phi+delta) {
		
		rho=phi*ratio;
		x1=xc+(radius*cos(rho));
		y1=yc+(radius*sin(rho));
		rho=(phi+delta)*ratio;
		x2=xc+(radius*cos(rho));
		y2=yc+(radius*sin(rho));
		
		pljoin(x1,y1,x2,y2);
		}
	pltext(); 			/* Switch to text mode */
	}
   else {
	error_msg_nl("(Set region and x-y range before using WAFER)");
	if (atcommand) parse_error=TRUE;
	}
}

/*** Execute the MARK command **********************************************/

void	exec_mark()

{

	double	*x,*y;

if (window_ok) {


	x=&temp_xm;
	y=&temp_ym;

	/* pladv((PLINT) curr_page);*/	/* Set current page number */

	plgra(); 			/* Switch to graphic mode */
	plpoin(1,(PLFLT *)x,(PLFLT *)y,(PLINT)temp_mark);
	pltext(); 			/* Switch to text mode */
	}
   else {
	error_msg_nl("(Set region and x-y range before using MARK)");
	if (atcommand) parse_error=TRUE;
	}

}

/***************************************************************************/
