//
// value.cc: various variable value types
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#include <cmath>
#include <cstdio>
#include <cstring>

//
// On Linux/glibc2, gettext macro causes problems with Value::gettext()
// in value.h, so we just undefine it here.
//
#ifdef gettext
# undef gettext
#endif

#include "pmu.h"
#include "utils.h"
#include "value.h"

using namespace std;

Pmu* Value::pmu_ = new Pmu;

Value::Value() { strvalbuf_ = 0; }
Value::~Value() { delete[] strvalbuf_; strvalbuf_ = 0; }

bool Value::isequal(const Value*) const { return false; }
bool Value::isgt(const Value*) const { return false; }
bool Value::isainteger() const { return false; }
bool Value::isareal() const { return false; }
bool Value::isatext() const { return false; }
bool Value::isaboolean() const { return false; }
bool Value::isaenum() const { return false; }
bool Value::isarange() const { return false; }

Value* Value::clone() const { return 0; }
Value* Value::clone(const char* /*str*/) const { return 0;  }
int Value::getinteger() const { return 0; }
double Value::getreal() const { return 0.0; }
const char* Value::gettext() const { return 0; }
bool Value::getboolean() const { return false; }
const char* Value::getenum() const { return 0; }
const Value* Value::getmin() const { return 0; }
const Value* Value::getmax() const { return 0; }
const Value* Value::getinc() const { return 0; }
const char* Value::getstrval() const { return 0; }
int Value::setval(const char* /*str*/) { return 0; }

const Value* Value::add(const Value*) { return 0; }
const Value* Value::subtract(const Value*) { return 0; }
const Value* Value::mul(const Value*) { return 0; }
const Value* Value::div(const Value*) { return 0; }

void Value::dump() const { printf ("Generic value.\n"); }

IntegerValue::IntegerValue(int val) { val_ = val; }
IntegerValue::~IntegerValue() { }
Value* IntegerValue::clone() const { 
    return (new IntegerValue(val_)); 
}
bool IntegerValue::isequal(const Value* val) const { 
    bool equal = false;
    if (val->isainteger()) {
	equal = val_ == val->getinteger();
    }
    return equal;
}

bool IntegerValue::isgt(const Value* val) const { 
    bool gt = false;
    if (val->isainteger()) {
	gt = val_ > val->getinteger();
    }
    return gt;
}


bool IntegerValue::isainteger() const { return true; }
int IntegerValue::getinteger() const { return val_; }
const char* IntegerValue::getstrval() const {
    if (strvalbuf_ == 0) {
	((IntegerValue*)this)->strvalbuf_ = new char[256];	// UNCONST
    }
    sprintf (strvalbuf_, "%d", val_);  
    return strvalbuf_; 
}

const Value* IntegerValue::add(const Value* val) {
    bool sametype = val->isainteger();
    if (sametype) {
	val_ += val->getinteger();
    }
    return (sametype) ? this : 0;
}

const Value* IntegerValue::subtract(const Value* val) {
    bool sametype = val->isainteger();
    if (sametype) {
	val_ -= val->getinteger();
    }
    return (sametype) ? this : 0;
}

const Value* IntegerValue::mul(const Value* val) {
    bool sametype = val->isainteger();
    if (sametype) {
	val_ *= val->getinteger();
    }
    return (sametype) ? this : 0;
}

const Value* IntegerValue::div(const Value* val) {
    bool sametype = val->isainteger();
    if (sametype) {
	val_ /= val->getinteger();
    }
    return (sametype) ? this : 0;
}

void IntegerValue::dump() const { printf("Integer value '%d'.\n",val_); }


Value *IntegerValue::clone(const char* str) const {
   if (!pmu_->match(str,"<integer>")) return 0; 
   Value *newval = new IntegerValue (pmu_->getintfield(1));
   return newval;
}

int IntegerValue::setval(const char* str) {
   if (!pmu_->match(str,"<integer>")) return 0;
   val_ = pmu_->getintfield(1);
   return 1;
}

RealValue::RealValue(double val) { val_ = val; }
RealValue::~RealValue() { }
Value* RealValue::clone() const{ return (new RealValue(val_)); }

bool RealValue::isequal(const Value* val) const { 
    bool equal = false;
    if (val->isareal()) {
	double difference = val_ - val->getreal();
	difference = (difference > 0) ? difference : -difference;
	equal = difference < 1.0e-28L;
	// equal = abs(val_ - val->getreal()) < 1.0e-28L;
    }
    return equal;
}

bool RealValue::isgt(const Value* val) const { 
    bool gt = false;
    if (val->isareal()) {
	gt = val_ > val->getreal();
    }
    return gt;
}


bool RealValue::isareal() const{ return true; }
double RealValue::getreal() const { return val_; }
const char* RealValue::getstrval() const { 
    if (strvalbuf_ == 0) {
	((RealValue*)this)->strvalbuf_ = new char[256];	// UNCONST
    }
    sprintf (strvalbuf_, "%.17g", val_);  
    return strvalbuf_; 
}

const Value* RealValue::add(const Value* val) {
    bool sametype = val->isareal();
    if (sametype) {
	val_ += val->getreal();
    }
    return (sametype) ? this : 0;
}

const Value* RealValue::subtract(const Value* val) {
    bool sametype = val->isareal();
    if (sametype) {
	val_ -= val->getreal();
    }
    return (sametype) ? this : 0;
}

const Value* RealValue::mul(const Value* val) {
    bool sametype = val->isareal();
    if (sametype) {
	val_ *= val->getreal();
    }
    return (sametype) ? this : 0;
}

const Value* RealValue::div(const Value* val) {
    bool sametype = val->isareal();
    if (sametype) {
	val_ /= val->getreal();
    }
    return (sametype) ? this : 0;
}

void RealValue::dump() const { printf ("Real value `%.17g'.\n", val_); }

Value *RealValue::clone(const char* str) const {
   if (!pmu_->match(str,"<real>")) return 0; 
   Value *newval = new RealValue (pmu_->getrealfield(1));
   return newval;
}

int RealValue::setval(const char* str) {
   if (!pmu_->match(str,"<real>")) return 0;
   val_ = pmu_->getrealfield(1);
   return 1;
}

Value* TextValue::clone() const { return (new TextValue(val_)); }
bool TextValue::isatext() const { return true; }
const char* TextValue::gettext() const { return val_; }
const char* TextValue::getstrval() const { 
    if (strvalbuf_ == 0) {
	((TextValue*)this)->strvalbuf_ = new char[256];	// UNCONST
    }
    strcpy(strvalbuf_, val_); 
    return strvalbuf_; 
}
void TextValue::dump() const { printf ("Text value '%s'.\n", val_); }


TextValue::TextValue(const char* text) { 
    val_ = strcpy_with_alloc(text);
}
TextValue::~TextValue() { delete[] val_; val_ = 0; }

bool TextValue::isequal(const Value* val) const { 
    bool equal = false;
    if (val->isatext()) {
	equal = strcmp(val_, val->gettext()) == 0;
    }
    return equal;
}

// CHECK: this is garbage!
bool TextValue::isgt(const Value* val) const { 
    bool gt = false;
    if (val->isatext()) {
	gt = strcmp(val_, val->gettext()) > 0;
    }
    return gt;
}


int TextValue::setval(const char* str) {
    delete[] val_;
    val_ = strcpy_with_alloc(str);
    return 1;
}

Value *TextValue::clone(const char* str) const {
   Value *newval = new TextValue (str);
   return newval;
}

BooleanValue::BooleanValue(bool val) { val_ = val; }
BooleanValue::~BooleanValue() { }
Value* BooleanValue::clone() const { return (new BooleanValue(val_)); }
bool BooleanValue::isaboolean() const { return true; }
bool BooleanValue::getboolean() const { return val_; }
const char* BooleanValue::getstrval() const { 
    if (strvalbuf_ == 0) {
	((BooleanValue*)this)->strvalbuf_ = new char[256];	// UNCONST
    }
    strcpy(strvalbuf_, (val_ == true) ? "true" : "false"); 
    return strvalbuf_; 
}

bool BooleanValue::isequal(const Value* val) const { 
    bool equal = false;
    if (val->isaboolean()) {
	equal = val_ == val->getboolean();
    }
    return equal;
}

void BooleanValue::dump() const { 
    printf ("Boolean value '%d'.\n", (int)val_);
}

Value *BooleanValue::clone(const char* str) const {
   int b = -1;
   if (pmu_->match(str,"<boolean>"))    b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<yesno>")) b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<yn>"))    b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<onoff>")) b = pmu_->getcodefield(1);
   if (b == -1) return 0;
   Value *newval = new BooleanValue (b);
   return newval;
}


int BooleanValue::setval(const char* str) {
   int b = -1;
   if (pmu_->match(str,"<boolean>"))    b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<yesno>")) b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<yn>"))    b = pmu_->getcodefield(1);
   else if (pmu_->match(str,"<onoff>")) b = pmu_->getcodefield(1);
   if (b == -1) return 0;
   val_ = b;
   return 1;
}

EnumValue::~EnumValue() { }
bool EnumValue::isaenum() const { return true; }
const char* EnumValue::getenum() const{ return val_; }

RangeValue::~RangeValue() { }
bool RangeValue::isainteger() const { return min_->isainteger(); }
bool RangeValue::isareal() const { return min_->isareal(); }
const Value* RangeValue::getmin() const { return min_; }
const Value* RangeValue::getmax() const { return max_; }
const Value* RangeValue::getinc() const { return inc_; }
