#include "command.h"
#ifdef decstation
#include <cursesX.h>
#else
#include <curses.h>
#endif

#define QUOTE_CHAR	'\\'
#define CNTRL(ch)	((ch) ^ 0100)

typedef struct CmdTableRec {
    char 	*full_cmd;
    int 	binding;
    int 	(*func)();
    char 	*synopsis;
} CmdTableRec, *CmdTable;



CmdTableRec cmd_table[] = {
    {	"up",		KEY_UP,			cmd_up,
		"move selection up one line on screen"		},
    {	"up",		'k',			cmd_up,
		"move selection up one line on screen"		},
    {	"down",		KEY_DOWN,		cmd_down,
		"move selection down one line on screen"	},
    {	"down",		'j',			cmd_down,
		"move selection down one line on screen"	},
    {	"edit",		KEY_ENTER,		cmd_edit,
		"edit value of variable"			},
    {	"edit",		'\n',			cmd_edit,
		"edit value of variable"			},
    {	"edit",		'e',			cmd_edit,
		"edit value of variable"			},
    {	"search",	'/',			cmd_search,
		"search a named variable"			},
    {	"go",	'g',				cmd_go,
		"go to a particular line"			},
    {	"go",	'G',				cmd_go,
		"go to a particular line"			},
    {	"halfup",	CNTRL('U'),		cmd_halfup,
		"move up half a screen"				},
    {	"halfdown",	CNTRL('D'),		cmd_halfdown,
		"move window down half a screen"		},
    {	"fullup",	CNTRL('B'),		cmd_fullup,
		"move window up to the top"			},
    {	"fulldown",	CNTRL('F'),		cmd_fulldown,
		"move window down to the bottom"		},
    {	"redraw",	'\f',			cmd_redraw,
		"redraw screen"					},
    {	"quit",		'q',			cmd_quit,
		"quit without saving the namelist"		},
    {	"save",		's',			cmd_save,
		"save the current namelist"			},
    {	"exit",		'x',			cmd_exit,
		"save the current namelist and quit"		},
    {	"default",	'd',			cmd_default,
		"reset default value" 				},
    {	"help",		'?',			cmd_help,
		"print this message"				},
    {	NULL,		'\0',			NULL,
		""						}
};


static char buf[BUFSIZ];


#define NCOMMANDS	(sizeof(cmd_table)/sizeof(cmd_table[0]))

int do_command (cmd)
int cmd;
{
    int i = 0;
    int ret = CMD_ERROR;
    int found = FALSE;

    for (; i < NCOMMANDS; i++) {
	if (cmd == cmd_table[i].binding) {
	    sprintf (buf, " %%%% Command entered: %s", cmd_table[i].full_cmd);
	    put_msg (buf);
	    ret = (*cmd_table[i].func) ();
	    found = TRUE;
	}
    }
    if (!found) {
	sprintf (buf, " %%%% Command unknown: (%c)", cmd);
	put_msg (buf);
    }
    return ret;
}


int cmd_up () {

    move_sel (-1);
    return CMD_OK;
}


int cmd_down () {

    move_sel (1);
    return CMD_OK;
}

int cmd_default () {
    put_msg (" %%%% Not available yet");
}

int cmd_help () {
    put_msg (" %%%% No help available yet");
}

int cmd_edit () {
    
    char *value;
    int index = get_cursel ();
    char *get_user_input ();

    if (!allow_change (index)) {
	sprintf (buf, " %%%% this parameter is not user-modifiable!");
	beep ();
	put_msg (buf);

	return CMD_OK;
    }

    display_prompt (" Enter new value > ");
#if defined(sun)
    echo ();
#endif
    value = get_user_input ();

    clear_prompt ();

    if (*value) {			/* empty input?? */
	if (update_val (index, value) == -1) {
	    sprintf (buf, " %%%%%%%% Error: Illegal value! Ignored");
	    beep ();
	    put_msg (buf);
	}
	else {
	    update_list (index, value);
	}
    }
    else {
	sprintf (buf, " %%%%%%%% No value given! Left unchanged");
	beep ();
	put_msg (buf);
    }

#if defined(sun)
    noecho ();
#endif

    return CMD_OK;
}

int cmd_search () {
    
    char *value;
    char *get_user_input ();

    display_prompt (" Enter search string > ");
#if defined(sun)
    echo ();
#endif
    value = get_user_input ();

    clear_prompt ();

    if (*value) {			/* empty input?? */
	int index = get_search_item (value);
	if (index == -1) {
	    sprintf (buf, " %%%% search string not found!");
	    put_msg (buf);
	    beep ();
	}
	else {
	    unhighlight_item (get_cursel ());
	    set_cursel (index);
	    make_sel_visible ();
	}
    }
    else {
	sprintf (buf, " %%%% No string given! search canceled");
	put_msg (buf);
	beep ();
    }

#if defined(sun)
    noecho ();
#endif

    return CMD_OK;
}


int cmd_quit () {

    return CMD_DONE;
}

int cmd_save () {

    write_nml_file ();
    return CMD_OK;
}

int cmd_exit () {

    return cmd_save(), cmd_quit ();
}


int cmd_halfup() {

    move_sel (-8);
    return CMD_OK;
}

int cmd_halfdown() {

    move_sel (8);
    return CMD_OK;
}

int cmd_fullup() {

    move_sel (-16);
    return CMD_OK;
}

int cmd_fulldown() {

    move_sel (16);
    return CMD_OK;
}

int cmd_redraw() {

    display_list ();
    return CMD_OK;
}

int cmd_go () {

    char *value;
    char *get_user_input ();

    display_prompt (" Go to what line [0: first]? ");
#if defined(sun)
    echo ();
#endif
    value = get_user_input ();

    clear_prompt ();

    if (*value) {			/* empty input?? */
	int index = 0;
	if (get_integer (value, &index)) {
	    sprintf (buf, " %%%% line # \"%s\" not valid!", value);
	    put_msg (buf);
	    beep ();
	}
	else {
	    unhighlight_item (get_cursel ());
	    set_cursel (bound_item_index (index));
	    make_sel_visible ();
	}
    }
    else {
	sprintf (buf, " %%%% No line # given! canceled");
	put_msg (buf);
	beep ();
    }

#if defined(sun)
    noecho ();
#endif

    return CMD_OK;
}
    

#if NOTDEF
static char *helpArgTypes = "\
An argument that specifies an object can be:\n\
    The external ID of an object in the current facet\n\
    The character `.' for the current object\n\
    The character `/' for the current facet\n\
    A name that was associated with an object by the name command\n\
\n\
An argument that specifies a set of object types (the generator mask)\n\
is a string of letters, like the ones in the top right corner of the\n\
display (case is ignored):\n\
    f - Facet			p - Property\n\
    i - Instance		b - Bag\n\
    t - Terminal		l - Layer\n\
    n - Net			g - All geometric objects\n\
		a - All object types\n\
\n\
";


int cmd_help() {

    CmdTableRec *cTabPtr;
    char line[1000];

    io_clearScreen();
    io_home();
    io_puts("Key  Command         Action\n");
    for (cTabPtr = cmd_table; cTabPtr->binding != '\0'; cTabPtr++) {
	if (iscntrl(cTabPtr->binding)) {
	    io_putchar('^');
	    io_putchar(0100 ^ cTabPtr->binding);
	} else {
	    io_putchar(' ');
	    io_putchar(cTabPtr->binding);
	}
	(void) sprintf(line, "    %-15s %s\n",
		    cTabPtr->fullCommand, cTabPtr->synopsis);
	io_puts(line);
    }
    io_botHome();
    io_puts("\t<any character to continue>");
    io_refresh();
    (void) io_getchar();

    io_forceClear();
    io_home();
    io_puts(helpArgTypes);
    io_botHome();
    io_puts("\t<any character to continue>");
    io_refresh();
    (void) io_getchar();

    return(OCT_OK);
}


int cmd_help () {
    
    fprintf (stderr, "helping ...\n");
    return CMD_OK;
}



char *
get_cmd_string()
{
    static char string [BUFSIZ];
    static char stop_list [] = "  \t\n\r";
    int stop_ch;

    stop_list[0] = line_kill_char;

    stop_ch = get_string(string, string, stop_list);
    /*
    if (stop_ch == line_kill_char) errRaise(optProgName, 0, "");
    */
    return string;
}

int
get_string(start, c_ptr, stop_list)
char *start; 
char *c_ptr;
char *stop_list;
{
    int ch;

    while ((ch = io_getchar()) != EOF) {
	if (strchr(stop_list, ch)) {
	    io_putchar(' ');
	    *c_ptr = '\0';
	    io_refresh();
	    return(ch);
	} else if (ch == erase_char) {
	    if (c_ptr > start) {
		io_puts("\b \b");
		c_ptr--;
	    }
	} else if (ch == word_erase_char) {
	    while (c_ptr > start && isspace(*c_ptr)) {
		io_puts("\b \b");
		c_ptr--;
	    }
	    while (c_ptr > start && !isspace(*c_ptr)) {
		io_puts("\b \b");
		c_ptr--;
	    }
	} else if (ch == line_kill_char) {
	    while (c_ptr > start) {
		io_puts("\b \b");
		c_ptr--;
	    }
	} else if (ch == QUOTE_CHAR) {
	    io_putchar(QUOTE_CHAR);
	    io_putchar('\b');
	    io_refresh();
	    ch = io_getchar();
	    io_putchar_see(ch);
	    *c_ptr++ = ch;
	} else {
	    io_putchar_see(ch);
	    *c_ptr++ = ch;
	}
	io_refresh();
    }
    *c_ptr = '\0';
    return(ch);
}



str_up_case(str_ptr)
char *str_ptr;
{
    while (*str_ptr != '\0') {
	if (islower(*str_ptr)) *str_ptr = toupper(*str_ptr);
	if (*str_ptr == '-') *str_ptr = '_';
	str_ptr++;
    }
}

#endif
