//
// logger.h: An ostream that logs to multiple streams.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//
// ====================================================================== //

#ifndef excon_logger_h
#define excon_logger_h

#include <iostream>

// ====================================================================== //

//
// simple macro to do the logging depending on current log level.
//
// Example:
//
//    LOGGER(logger, Logger::run) << "Running program ... " << endl;
//    LOGGER(logger, Logger::run | Logger::report) << "blah ... " << endl;
//
// where logger is an existing Logger object.
//
// If you never use OR'd combos in LOG macro, use the following:
//
//    LOGGER2(logger, run) << "Running program ..." << endl;
//

#define LOGGER(logger,lvl) \
    if (!(logger.cur_loglevel() & (lvl))); else logger

#define LOGGER2(logger,lvl) \
    if (!(logger.cur_loglevel() & Logger::lvl)); else logger


// ====================================================================== //
//
// class Logger: Logger allows streaming to multiple streambuffers at
// the same time and hence is very useful to logging to cerr and to a
// file for example.
//
// Idea lifted from an article in comp.lang.c++ back in Oct '94 by 
// Bernd Eggink <admin@rzaix13.rrz.uni-hamburg.de>
//

class Logger : public std::ostream {
public:
    static const unsigned none = 0x0;
    static const unsigned minimal = 0x1;
    static const unsigned load = 0x2;
    static const unsigned nested_load = 0x4;
    static const unsigned parse = 0x8;
    static const unsigned report = 0x10;
    static const unsigned resolve = 0x20;
    static const unsigned run = 0x40;
    static const unsigned collect = 0x80;
    static const unsigned debug = 0x100;
    static const unsigned job = 0x200;
    static const unsigned all = ~none;

public:
    //
    // Create a new Logger object. Initially w/out any streams attached,
    // so it's a null stream.
    //
    Logger(unsigned loglevel = none);
    //
    // Create a new Logger object and initialize it with an already
    // created ostream object (eg., cout or cerr). 
    //
    Logger(std::ostream& os, unsigned loglevel = none);
    ~Logger();

    //
    // Add a ostream object to the list of output streams. Be very careful
    // about the lifetime of the ostream!
    //
    Logger& add(std::ostream& os);
    //
    // Add a streambuf to the list of output streams. This is used when
    // using a log file.
    //    ofstream("xyz.log");
    //    logger.add(cerr).add(ofstream);
    // Be very careful about the lifetime of the fstream!
    //
    Logger& add(std::streambuf* stb);

    //
    // get the current logging level. Returns the OR'd set.
    //
    unsigned cur_loglevel() const;
    //
    // set the current logging level. Returns the existing one.
    //
    unsigned set_loglevel(unsigned);

private:
    class LogBuf* logbufp_;
    int loglevel_;
};

inline unsigned Logger::cur_loglevel() const {
    return loglevel_;
}

inline unsigned Logger::set_loglevel(unsigned new_loglevel) {
    unsigned old_loglevel = loglevel_;
    loglevel_ = new_loglevel;
    return old_loglevel;
}

// ====================================================================== //

#endif/*excon_logger_h*/
