/*
 * fstr.c: string manipulation routines.
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * Source: src/utils/io/fstr.c
 *
 * ------------------------------------------------
 * Mumit Khan
 * Center for X-ray Lithography
 * University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 * Log:	fstr.c
 * Revision 1.3  90/05/09  12:01:05  khan
 * Cleanup
 * 
 * Revision 1.2  90/05/01  14:31:00  khan
 * routine
 * 
 * Revision 1.1  90/04/12  21:35:37  khan
 * Initial revision
 * 
 *
 */

/* 
 * contains routines for string manipulation. Especially handly when
 * dealing with strings passed in from FORTRAN routines.
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "libcxrl.h"

#ifdef VMS 
#include <stdlib.h>
#define malloc VAXC$MALLOC_OPT
#define free VAXC$FREE_OPT
#endif

#define STATIC static
#define PRIVATE static 
#define PUBLIC 

/*
 * pad_fort_str:
 *
 * FORTRAN associates a length with each string, and there is not end
 * marker. To pass back a string from C, we clean up the terminating
 * '\0' and also pad it with BLANK spaces (mainly for aesthetics).
 *
 */

PUBLIC
char *pad_fort_str (fstr, start, len)
char *fstr;                     /* The string to be padded with blanks */
unsigned int start;             /* Starting position */
unsigned int len;               /* Length of string */
{

    while (start < len) {
	*(fstr + start++) = ' ';
    }

    return fstr;
}


PUBLIC fix_fort_str (c_str, fort_str, len)
char *c_str;
char *fort_str;
unsigned int len;
{
    register unsigned int i = 0;
    register char *p = c_str;

    for (; *p ; *(fort_str + i++) = *(p++))
	;
    for (; i < len; *(fort_str + i++) = ' ')
	;			/* Note how the '\0' is plowed over */
}

/*
 * fort_str:
 * 
 * Make a fortran string somewhat palatable to the rest of the world.
 * This involves stripping the white space from the back of the string
 * (The white space in the beginning is a different issue), and put
 * '\0' at the right place.
 *
 */

PUBLIC 
char *fort_str (str, len) 
char *str;
unsigned long int len;
{

    char *ret = (char *) xmalloc (len + 1);

    strncpy (ret, str, len);
    *(ret + len) = '\0';
    return trim_trailing_white (ret);
}



PUBLIC char *trim_leading_white (str)
char *str;
{
    register char *p = str;

    for (; *p == ' ' || *p == '\t'; p++)
	;
    return p;
}

PUBLIC char *trim_trailing_white (str)
char *str;
{
    register char *p = str + strlen (str);

    for (; *p == ' ' || *p == '\t' || !*p || !isprint(*p); p--)
	;
    *(p+1) = '\0';
    return str;
}

PUBLIC char *lo_case (str) 
char *str;
{
    register char *p = str;

/* the following code tickes a bug in the IBM c compiler. beware */

#ifdef WEIRD
    for (; *p ; *p = (char) (isupper (*p)) ? tolower ((int) *p) : *p, p++)
	;
#else
    for (; *p ; p++) {
	*p = (isupper (*p)) ? tolower (*p) : *p;
    }
#endif
    return str;
}

