/*
 * nml-to-excon.c: Convert Namelist to EXCON file
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * ------------------------------------------------
 *             SHADOW
 *    Center for X-ray Lithography
 *   University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "common.h"
#include "namelist.h"

int parse_args (int argc, char** argv, const char* prog_name, 
    char** infile, char** outfile, NmlType* type
);
void usage(const char* prog_name);

static char* data_types[] = {
    "integer", "real", "real", "filename"
};

int main (int argc, char** argv) {
    FILE* fp = nil;
    int i = 0, size = 0;
    Namelist* namelist = nil;
    char* prog_name = nil;
    char* nml_file = nil;
    char* outfile = nil;
    NmlType type;

    char *ptr = strrchr(argv[0], '/');

    if (ptr) 
	prog_name = ptr + 1;
    else 
	prog_name = argv[0];

    (void) parse_args(argc, argv, prog_name, &nml_file, &outfile, &type);
 
    namelist = read_namelist(nml_file, type);
    if (namelist == nil) {
	fprintf(stderr, "ERROR: Cannot read namelist file `%s'\n", nml_file);
	exit(1);
    }

    if (strcmp(outfile, "-") == 0) {
	fp = stdout;
    } else {
	if ((fp = fopen(outfile, "w")) == nil) {
	    fprintf(stderr, "Open failed for output file ``%s''.\n", outfile);
	    exit(1);
	}
    }
    size = namelist->size;
    for(i = 0; i < size; i++) {
	NamelistData* data = &namelist->nml_data[i];
	fprintf (fp, "%s:%s:%s\n", data->name, data_types[data->data_type],
	    sprint_value(data)
	);
    }
    if (fp != stdout)
	fclose(fp);

    exit (0);
}

int parse_args (int argc, char** argv, const char* prog_name, char** infile, 
    char** outfile, NmlType* type
) {
    extern int optind;
    extern char *optarg;

    boolean type_supplied = FALSE;
    int c;

    *infile = nil;
    *outfile = nil;

    while ((c = getopt (argc, argv, "ht:o:")) != EOF) {
	switch (c) {
	    case 't':
		if (optarg[0] == 's' || optarg[0] == 'S') {
		    *type = NML_source;
		}
		else if (optarg[0] == 'o' || optarg[0] == 'O') {
		    *type = NML_oe;
		}
		else {
		    fprintf(stderr, "Illegal namelist type --  %s\n", 
			optarg
		    );
		    usage(prog_name);
		    exit(1);
		}
		type_supplied = TRUE;
		break;

	    case 'o':
		*outfile = strcpy (malloc (strlen (optarg) + 1), optarg);
		break;
	    
	    case 'h':
		usage (prog_name);
		exit (0);
		break;
	    
	    default:
		usage (prog_name);
		exit (1);
		break;
	}
    }

    if (argc == optind+1) {
	*infile = strcpy (malloc (strlen (argv[optind]) + 1), argv[optind]);
    }

    if (*infile == nil || *outfile == nil || type_supplied == FALSE) {
	fprintf(stderr, "Must supply all required parameters.\n");
	usage(prog_name);
	exit(1);
    }
    return 0;
}


void usage (const char* prog_name) {
    fprintf (stderr, 
    "Usage:%s [-h] -o EXCON_TOOLS_FILE -t NamelistType NamelistFile\n\
    -h:			print this info\n\
    -o TOOLS_FILE:	EXCON tools file format output (stdout == - )\n\
    -t NamelistType:	\"source\" or \"oe\" (also accepts \"s\" or \"o\")\n\
    NamelistFile:	SHADOW namelist (eg., start.00, start.01, etc)\n\n",
    prog_name);
    return;
}
