/*
 *
 * nml-extract.c:  Extract the named parameter value from F77 namelist
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * ------------------------------------------------
 *              SHADOW
 *    Center for X-ray Lithography
 *   University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "common.h"
#include "namelist.h"

int parse_args (int argc, char** argv, const char* prog_name, 
    char** param, char** infile, NmlType* type
);
void usage(const char* prog_name);

int main (int argc, char *argv[]) {
    Namelist* namelist = nil;
    char* prog_name = nil;
    char* nml_file = nil;
    char* param = nil;
    NmlType type;
    NamelistData* data = nil;
    int retval = 0;

    char *ptr = strrchr(argv[0], '/');

    if (ptr) 
	prog_name = ptr + 1;
    else 
	prog_name = argv[0];

    (void) parse_args(argc, argv, prog_name, &param, &nml_file, &type);
 
    namelist = read_namelist(nml_file, type);
    if (namelist == nil) {
	fprintf(stderr, "ERROR: Cannot reade namelist file `%s'\n", nml_file);
	exit(1);
    }
    data = get_namelist_data(param, namelist);
    if (data == nil) {
	fprintf(stderr, "No such paramter ``%s'' in namelist ``%s''\n", 
	    param, nml_file
	);
	exit(1);
    }
    printf("%s\n", sprint_value(data));
    exit (0);
}

int parse_args (int argc, char** argv, const char* prog_name, char** param, 
    char** infile, NmlType* type
) {
    extern int optind;
    extern char *optarg;

    boolean type_supplied = FALSE;
    int c;

    *param = nil;
    *infile = nil;

    while ((c = getopt (argc, argv, "ht:p:")) != EOF) {
	switch (c) {
	    case 't':
		if (optarg[0] == 's' || optarg[0] == 'S') {
		    *type = NML_source;
		}
		else if (optarg[0] == 'o' || optarg[0] == 'O') {
		    *type = NML_oe;
		}
		else {
		    fprintf(stderr, "Illegal namelist type --  %s\n", 
			optarg
		    );
		    usage(prog_name);
		    exit(1);
		}
		type_supplied = TRUE;
		break;

	    case 'p':
		*param = strcpy (malloc (strlen (optarg) + 1), optarg);
		break;
	    
	    case 'h':
		usage (prog_name);
		exit (0);
		break;
	    
	    default:
		usage (prog_name);
		exit (1);
		break;
	}
    }

    if (argc == optind+1) {
	*infile = strcpy (malloc (strlen (argv[optind]) + 1), argv[optind]);
    }

    if (*infile == nil || *param == nil || type_supplied == FALSE) {
	fprintf(stderr, "Must supply all required parameters.\n");
	usage(prog_name);
	exit(1);
    }
    return 0;
}


void usage (const char* prog_name) {
    fprintf (stderr, 
    "Usage:%s [-h] -p paramName -t NamelistType NamelistFile\n\
    -h:			print this info\n\
    -p paramName:	Name of parameter to extract\n\
    -t NamelistType:	\"source\" or \"oe\" (also accepts \"s\" or \"o\")\n\
    NamelistFile:	SHADOW namelist (eg., start.00, start.01, etc)\n\n",
    prog_name);
    return;
}
