/*
 * namelist.c: C interface to SHADOW FORTRAN NAMELIST data
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * ------------------------------------------------
 *             SHADOW
 *    Center for X-ray Lithography
 *   University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 */

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>

#include "common.h"
#include "namelist.h"

int OE_NUMBER;
double DUMMY;
int IDUMMY;
int SCR_NUMBER[10];

/*
 * LOCAL static function defs.
 */

/*
 * Only define the following when used inside of SHADOW. Don't need these
 * for the GUI for example, which shares this same file.
 */
#if USE_SHADOW_LIBRARY

static Namelist* make_namelist(int size, NmlType type);
static int get_float(const char *string, float *fltval);
static int get_double(const char *string, double *dblval);
static int get_integer(const char *string, int *intval);
static int do_change_value (NamelistData *data, const char *newval);
static char* str_lo_case(char *str);
static char* str_up_case(char *str);

#endif /*!USE_SHADOW_LIBRARY*/

/*
 * Routines from SHADOW FORTRAN library. The F77_NAME macro is defined
 * common.h, which in turn depens on configuration parameter,
 * NO_FORT_UNDERSCORE.
 *
 * Note that we force reading/writing FORTRAN NAMELIST format by calling
 * rwname_f77 directly, since otherwise the environment variable
 * SHADOW_START_FORMAT may force reading/writing the wrong format.
 */

#if G77
# define RWNAME_F77		G77_HACK_NAME(rwname_f77)
#else
# define RWNAME_F77		F77_NAME(rwname_f77)
#endif

#define RWNAME			F77_NAME(rwname)

extern void RWNAME_F77 (
    const char*, const char*, int*, unsigned long, unsigned long
);
/* procedure rwname(character(*), character(*), integer) */
extern void RWNAME (
    const char*, const char*, int*, unsigned long, unsigned long
);

/*
 * now decide whether to use array variables in x(i) or x_i notation.
 * This is for the various array memebers of SCR_NUMBER, etc.
 *
 * the default is to use the paren notation.
 *
 * G_ARRAY_ELEM(f,5) becomes f(5) or f_5 depending on the choice.
 * MAKE_XSTR(G_ARRAY_ELEM(f,5)) produces "f(5)" or "f_5"
 */

#define MAKE_XSTR(foo) MAKE_STR(foo)
#define MAKE_STR(foo) #foo
#if G_UNDERSCORE_ARRAY_NOTATION
# define G_ARRAY_ELEM(a,i) a ## _ ## i
#else
# define G_ARRAY_ELEM(a,i) a ## ( ## i ## )
#endif

#define ARR_ELEM(a,i) MAKE_XSTR(G_ARRAY_ELEM(a,i))
 
/*
 * Setup the namelists
 */
    

#define	Number(arr)	((unsigned int) (sizeof(arr) / sizeof(arr[0])))

static NamelistData nml_source_list[] = {
    {"Source angle distribution",		TRUE,
	"fdistr", 		DT_int, 	(char *)&FLAGS.fdistr},
    {"Source modelling type",			TRUE,
	"fgrid", 		DT_int, 	(char *)&FLAGS.fgrid},
    {"Source cross-section (X-Z)",		TRUE,
	"fsour", 		DT_int, 	(char *)&FLAGS.fsour},
    {"Source depth (Y)",			TRUE,
	"fsource_depth", 	DT_int, 	(char *)&FLAGS.fsource_depth},
    {"Incoherent (0) or coherent (1)",		TRUE,
	"f_coher", 		DT_int, 	(char *)&FLAGS.f_coher},
    {"Photon Energy Distribution",		TRUE,
	"f_color",		DT_int, 	(char *)&FLAGS.f_color},
    {"Photon Energy (eV (0)/Ang (1))",		TRUE,
	"f_phot", 		DT_int, 	(char *)&FLAGS.f_phot},
    {"Polarization component-S, P, TOT",	TRUE,
	"f_pol", 		DT_int, 	(char *)&FLAGS.f_pol},
    {"Generate A vectors: y/n (1/0)",		TRUE,
	"f_polar", 		DT_int, 	(char *)&FLAGS.f_polar},
    {"Save optical paths: y/n (1/0)",		TRUE,
	"f_opd", 		DT_int, 	(char *)&FLAGS.f_opd},
    {"Source: SR (0), wiggler (1), und (2)",	TRUE,
	"f_wiggler",		DT_int, 	(char *)&FLAGS.f_wiggler},
    {"Optimize the source: y/n (1/0)",		TRUE,
	"f_bound_sour", 	DT_int, 	(char *)&FLAGS.f_bound_sour},
    {"SR: photon (0)/power (1) distrib",	TRUE,
	"f_sr_type",		DT_int, 	(char *)&FLAGS.f_sr_type},
    {"Random number seed, odd",			TRUE,
	"istar1", 		DT_int, 	(char *)&CALC.istar1},
    {"Number of random rays (0-5000)",		TRUE,
	"npoint", 		DT_int, 	(char *)&CALC.npoint},
    {"Internal parameter",			FALSE,
	"ncol", 		DT_int, 	(char *)&CALC.ncol},
    {"Grid points on each circle",		TRUE,
	"n_circle",		DT_int, 	(char *)&CALC.n_circle},
    {"Number of energy lines",			TRUE,
	"n_color", 		DT_int, 	(char *)&CALC.n_color},
    {"Number of concentric circles",		TRUE,
	"n_cone", 		DT_int, 	(char *)&CALC.n_cone},
    {"Points in horizontal angle distrib",	TRUE,
	"ido_vx", 		DT_int, 	(char *)&CALC.ido_vx},
    {"Points in vertical angle distrib",	TRUE,
	"ido_vz", 		DT_int, 	(char *)&CALC.ido_vz},
    {"fsour=1 pts on x, =2 no. ellipses",	TRUE,
	"ido_x_s", 		DT_int, 	(char *)&CALC.ido_x_s},
    {"Points along depth (Y)",			TRUE,
	"ido_y_s", 		DT_int, 	(char *)&CALC.ido_y_s},
    {"fsour=1 pts on z, =2 no. ellipses",	TRUE,
	"ido_z_s", 		DT_int, 	(char *)&CALC.ido_z_s},
    {"No. of sigma levels (X)",			TRUE,
	"ido_xl", 		DT_int, 	(char *)&CALC.ido_xl},
    {"No. of rays/sigma level in X",		TRUE,
	"ido_xn", 		DT_int, 	(char *)&CALC.ido_xn},
    {"No. of sigma levels (Z)",			TRUE,
	"ido_zl", 		DT_int, 	(char *)&CALC.ido_zl},
    {"No. of rays/sigma level in Z",		TRUE,
	"ido_zn", 		DT_int, 	(char *)&CALC.ido_zn},
    {"Value for sigma level 1 (X)",		TRUE,
	"sigxl1", 		DT_dbl, 	(char *)&LIGHT.sigxl1},
    {"Value for sigma level 2 (X)",		TRUE,
	"sigxl2", 		DT_dbl, 	(char *)&LIGHT.sigxl2},
    {"Value for sigma level 3 (X)",		TRUE,
	"sigxl3", 		DT_dbl, 	(char *)&LIGHT.sigxl3},
    {"Value for sigma level 4 (X)",		TRUE,
	"sigxl4", 		DT_dbl, 	(char *)&LIGHT.sigxl4},
    {"Value for sigma level 5 (X)",		TRUE,
	"sigxl5", 		DT_dbl, 	(char *)&LIGHT.sigxl5},
    {"Value for sigma level 6 (X)",		TRUE,
	"sigxl6", 		DT_dbl, 	(char *)&LIGHT.sigxl6},
    {"Value for sigma level 7 (X)",		TRUE,
	"sigxl7", 		DT_dbl, 	(char *)&LIGHT.sigxl7},
    {"Value for sigma level 8 (X)",		TRUE,
	"sigxl8", 		DT_dbl, 	(char *)&LIGHT.sigxl8},
    {"Value for sigma level 9 (X)",		TRUE,
	"sigxl9", 		DT_dbl, 	(char *)&LIGHT.sigxl9},
    {"Value for sigma level 10 (X)",		TRUE,
	"sigxl10",		DT_dbl, 	(char *)&LIGHT.sigxl10},
    {"Value for sigma level 1 (Z)",		TRUE,
	"sigzl1", 		DT_dbl, 	(char *)&LIGHT.sigzl1},
    {"Value for sigma level 2 (Z)",		TRUE,
	"sigzl2", 		DT_dbl, 	(char *)&LIGHT.sigzl2},
    {"Value for sigma level 3 (Z)",		TRUE,
	"sigzl3", 		DT_dbl, 	(char *)&LIGHT.sigzl3},
    {"Value for sigma level 4 (Z)",		TRUE,
	"sigzl4", 		DT_dbl, 	(char *)&LIGHT.sigzl4},
    {"Value for sigma level 5 (Z)",		TRUE,
	"sigzl5", 		DT_dbl, 	(char *)&LIGHT.sigzl5},
    {"Value for sigma level 6 (Z)",		TRUE,
	"sigzl6", 		DT_dbl, 	(char *)&LIGHT.sigzl6},
    {"Value for sigma level 7 (Z)",		TRUE,
	"sigzl7", 		DT_dbl, 	(char *)&LIGHT.sigzl7},
    {"Value for sigma level 8 (Z)",		TRUE,
	"sigzl8", 		DT_dbl, 	(char *)&LIGHT.sigzl8},
    {"Value for sigma level 9 (Z)",		TRUE,
	"sigzl9", 		DT_dbl, 	(char *)&LIGHT.sigzl9},
    {"Value for sigma level 10 (Z)",		TRUE,
	"sigzl10",		DT_dbl, 	(char *)&LIGHT.sigzl10},
    {"From meters to user units",		TRUE,
	"conv_fact", 		DT_dbl, 	(char *)&SOURCE.conv_fact},
    {"fdistr=5; max half divergence",		TRUE,
	"cone_max", 		DT_dbl, 	(char *)&SOURCE.cone_max},
    {"fdistr=5; min half divergence",		TRUE,
	"cone_min", 		DT_dbl, 	(char *)&SOURCE.cone_min},
    {"X dist from waist of emittance",		TRUE,
	"epsi_dx", 		DT_dbl, 	(char *)&ALADDIN.epsi_dx},
    {"Z dist from waist of emittance",		TRUE,
	"epsi_dz", 		DT_dbl, 	(char *)&ALADDIN.epsi_dz},
    {"Beam emitt.-X rad*length units",		TRUE,
	"epsi_x", 		DT_dbl, 	(char *)&ALADDIN.epsi_x},
    {"Beam emitt.-Z rad*length units",		TRUE,
	"epsi_z", 		DT_dbl, 	(char *)&ALADDIN.epsi_z},
    {"Horizontal div. in +X (rads)",		TRUE,
	"hdiv1", 		DT_dbl, 	(char *)&SOURCE.hdiv1},
    {"Horizontal div. in -X (rads)",		TRUE,
	"hdiv2", 		DT_dbl, 	(char *)&SOURCE.hdiv2},
    {"Photon energy: min or 1st line",		TRUE,
	"ph1", 			DT_dbl, 	(char *)&LIGHT.ph1},
    {"Photon energy: max or 2nd line",		TRUE,
	"ph2", 			DT_dbl, 	(char *)&LIGHT.ph2},
    {"Third energy line",			TRUE,
	"ph3", 			DT_dbl, 	(char *)&LIGHT.ph3},
    {"Fourth energy line",			TRUE,
	"ph4", 			DT_dbl, 	(char *)&LIGHT.ph4},
    {"Fifth energy line",			TRUE,
	"ph5", 			DT_dbl, 	(char *)&LIGHT.ph5},
    {"Sixth energy line",			TRUE,
	"ph6", 			DT_dbl, 	(char *)&LIGHT.ph6},
    {"Seventh energy line",			TRUE,
	"ph7", 			DT_dbl, 	(char *)&LIGHT.ph7},
    {"Eigth energy line",			TRUE,
	"ph8", 			DT_dbl, 	(char *)&LIGHT.ph8},
    {"Ninth energy line",			TRUE,
	"ph9", 			DT_dbl, 	(char *)&LIGHT.ph9},
    {"Tenth energy line",			TRUE,
	"ph10", 		DT_dbl, 	(char *)&LIGHT.ph10},
    {"First Relative intensity",		TRUE,
	"rl1", 			DT_dbl, 	(char *)&LIGHT.rl1},
    {"Second Relative intensity",		TRUE,
	"rl2", 			DT_dbl, 	(char *)&LIGHT.rl2},
    {"Third Relative intensity",		TRUE,
	"rl3", 			DT_dbl, 	(char *)&LIGHT.rl3},
    {"Fourth Relative intensity",		TRUE,
	"rl4", 			DT_dbl, 	(char *)&LIGHT.rl4},
    {"Fifth Relative intensity",		TRUE,
	"rl5", 			DT_dbl, 	(char *)&LIGHT.rl5},
    {"Sixth Relative intensity",		TRUE,
	"rl6", 			DT_dbl, 	(char *)&LIGHT.rl6},
    {"Seventh Relative intensity",		TRUE,
	"rl7", 			DT_dbl, 	(char *)&LIGHT.rl7},
    {"Eigth Relative intensity",		TRUE,
	"rl8", 			DT_dbl, 	(char *)&LIGHT.rl8},
    {"Ninth Relative intensity",		TRUE,
	"rl9", 			DT_dbl, 	(char *)&LIGHT.rl9},
    {"Tenth Relative intensity",		TRUE,
	"rl10", 		DT_dbl, 	(char *)&LIGHT.rl10},
    {"Electron beam energy (GeV)",		TRUE,
	"bener", 		DT_dbl, 	(char *)&ALADDIN.bener},
    {"Phase difference in degrees",		TRUE,
	"pol_angle",		DT_dbl, 	(char *)&LIGHT.pol_angle},
    {"Degree of polarization (0<x<1)",		TRUE,
	"pol_deg", 		DT_dbl, 	(char *)&LIGHT.pol_deg},
    {"Bending magnet radius in user units",	TRUE,
	"r_aladdin", 		DT_dbl, 	(char *)&ALADDIN.r_aladdin},
    {"Bending magnet radius (m)",		TRUE,
	"r_magnet", 		DT_dbl, 	(char *)&ALADDIN.r_magnet},
    {"Sigma (rad) for horizontal div.",		TRUE,
	"sigdix", 		DT_dbl, 	(char *)&SOURCE.sigdix},
    {"Sigma (rad) for vertical div.",		TRUE,
	"sigdiz", 		DT_dbl, 	(char *)&SOURCE.sigdiz},
    {"Sigma in X",				TRUE,
	"sigmax", 		DT_dbl, 	(char *)&SOURCE.sigmax},
    {"Sigma in Y",				TRUE,
	"sigmay", 		DT_dbl, 	(char *)&SOURCE.sigmay},
    {"Sigma in Z",				TRUE,
	"sigmaz", 		DT_dbl, 	(char *)&SOURCE.sigmaz},
    {"Vertical div. in +Z (radians)",		TRUE,
	"vdiv1", 		DT_dbl, 	(char *)&SOURCE.vdiv1},
    {"Vertical div. in -Z (radians)",		TRUE,
	"vdiv2", 		DT_dbl, 	(char *)&SOURCE.vdiv2},
    {"Source width (X)",			TRUE,
	"wxsou", 		DT_dbl, 	(char *)&SOURCE.wxsou},
    {"Source depth (Y)",			TRUE,
	"wysou", 		DT_dbl, 	(char *)&SOURCE.wysou},
    {"Source height (Z)",			TRUE,
	"wzsou", 		DT_dbl, 	(char *)&SOURCE.wzsou},
    {"Source cone full opening (radians)",	TRUE,
	"plasma_angle", 	DT_dbl, 	(char *)&SOURCE.plasma_angle},
    {"Filename from make_id",			TRUE,
	"file_traj", 		DT_str, 	(char *)NAMES.file_traj},
    {"Internal parameter",			FALSE,
	"file_source", 		DT_str, 	(char *)NAMES.file_source},
    {"File from reflag and histo3",		TRUE,
	"file_bound", 		DT_str, 	(char *)NAMES.file_bound},
    {"Internal parameter",			FALSE,
	"oe_number",		DT_int, 	(char *)&OE_NUMBER},
    {"Internal parameter",			FALSE,
	"idummy", 		DT_int, 	(char *)&IDUMMY},
    {"Internal parameter",			FALSE,
	"dummy", 		DT_dbl, 	(char *)&DUMMY},
    {"Internal parameter",			FALSE,
	"f_new", 		DT_int, 	(char *)&FLAGS.f_new},
};

    /* end of source namelist block */

static NamelistData nml_oe_list[] = {
    {"Mirror surface, ie., spherical",		TRUE,
	"fmirr", 		DT_int, 	(char *)&FLAGS.fmirr},
    {"fmirr=3; mirror pole location",		TRUE,
	"f_torus",		DT_int, 	(char *)&FLAGS.f_torus},
    {"Cylindrical (1), regular(0)",		TRUE,
	"fcyl", 		DT_int, 	(char *)&FLAGS.fcyl},
    {"Internal (0), external (1) parms",	TRUE,
	"f_ext", 		DT_int, 	(char *)&FLAGS.f_ext},
    {"Move source; yes (1), no (0)",		TRUE,
	"fstat", 		DT_int, 	(char *)&FLAGS.fstat},
    {"Screens; yes (1), no (0)",		TRUE,
	"f_screen",		DT_int, 	(char *)&FLAGS.f_screen},
    {"Im. plates orth to opt axis",		TRUE,
	"f_plate", 		DT_int, 	(char *)&FLAGS.f_plate},
    {"Slit at contin plane",			TRUE,
	"fslit", 		DT_int, 	(char *)&FLAGS.fslit},
    {"Files to save option",			TRUE,
	"fwrite", 		DT_int, 	(char *)&FLAGS.fwrite},
    {"Distorted surface; yes(1), no(0)",	TRUE,
	"f_ripple", 		DT_int, 	(char *)&FLAGS.f_ripple},
    {"Move mirror; yes (1), no (0)",		TRUE,
	"f_move", 		DT_int, 	(char *)&FLAGS.f_move},
    {"Multilayer vary as the cosine (1/0)",	TRUE,
	"f_thick", 		DT_int, 	(char *)&FLAGS.f_thick},
    {"Asymmetric xtal; yes (1), no (0)",	TRUE,
	"f_bragg_a", 		DT_int, 	(char *)&FLAGS.f_bragg_a},
    {"Type of distortion",			TRUE,
	"f_g_s", 		DT_int, 	(char *)&FLAGS.f_g_s},
    {"Internal parameter",			FALSE,
	"f_r_ran", 		DT_int, 	(char *)&FLAGS.f_r_ran},
    {"Grating; yes (1), no (0)",		TRUE,
	"f_grating", 		DT_int, 	(char *)&FLAGS.f_grating},
    {"Mosaic xtal; yes (1), no (0)",		TRUE,
	"f_mosaic", 		DT_int, 	(char *)&FLAGS.f_mosaic},
    {"Johansson Geom; yes(1), no (0)",		TRUE,
	"f_johansson", 		DT_int, 	(char *)&FLAGS.f_johansson},
    {"Focus location; image/source",		TRUE,
	"f_side",		DT_int, 	(char *)&FLAGS.f_side},
    {"Autotuning of grating/xtal",		TRUE,
	"f_central", 		DT_int, 	(char *)&FLAGS.f_central},
    {"Mirror concave (0) or convex (1)",	TRUE,
	"f_convex", 		DT_int, 	(char *)&FLAGS.f_convex},
    {"Reflectivity of surface",			TRUE,
	"f_reflec", 		DT_int, 	(char *)&FLAGS.f_reflec},
    {"Line density absolute/signed",		TRUE,
	"f_rul_abs", 		DT_int, 	(char *)&FLAGS.f_rul_abs},
    {"Ruling type",				TRUE,
	"f_ruling", 		DT_int, 	(char *)&FLAGS.f_ruling},
    {"Recording sources spher/plane",		TRUE,
	"f_pw", 		DT_int, 	(char *)&FLAGS.f_pw},
    {"Recording sources cylindrical",		TRUE,
	"f_pw_c", 		DT_int, 	(char *)&FLAGS.f_pw_c},
    {"Source type real/virtual",		TRUE,
	"f_virtual", 		DT_int, 	(char *)&FLAGS.f_virtual},
    {"Mirror shape",				TRUE,
	"fshape", 		DT_int, 	(char *)&FLAGS.fshape},
    {"Mirror dimensions finite",		TRUE,
	"fhit_c", 		DT_int, 	(char *)&FLAGS.fhit_c},
    {"Monochromator type",			TRUE,
	"f_mono", 		DT_int, 	(char *)&FLAGS.f_mono},
    {"Reflector (0) refractor (1)",		TRUE,
	"f_refrac", 		DT_int, 	(char *)&FLAGS.f_refrac},
    {"Focii at contin plane",			TRUE,
	"f_default", 		DT_int, 	(char *)&FLAGS.f_default},
    {"Source of optical constants",		TRUE,
	"f_refl", 		DT_int, 	(char *)&FLAGS.f_refl},
    {"First(1) or second (2) grating",		TRUE,
	"f_hunt", 		DT_int, 	(char *)&FLAGS.f_hunt},
    {"xtal; yes (1), no (0)",			TRUE,
	"f_crystal", 		DT_int, 	(char *)&FLAGS.f_crystal},
    {"Tune to eV(0) or Angst(1)",		TRUE,
	"f_phot_cent", 		DT_int, 	(char *)&FLAGS.f_phot_cent},
    {"Roughness; y(1), n(0)",			TRUE,
	"f_roughness",		DT_int,		(char *)&FLAGS.f_roughness},
    {"Save inc. angle; y/1,n/0",		TRUE,
	"f_angle",		DT_int,		(char *)&FLAGS.f_angle},
    {"Internal parameter",			FALSE,
	"npoint", 		DT_int, 	(char *)&CALC.npoint},
    {"Internal parameter",			FALSE,
	"ncol", 		DT_int, 	(char *)&CALC.ncol},

    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,1), DT_int, 	(char *)&SCR_NUMBER[0]},
    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,2), DT_int, 	(char *)&SCR_NUMBER[1]},
    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,3), DT_int, 	(char *)&SCR_NUMBER[2]},
    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,4), DT_int, 	(char *)&SCR_NUMBER[3]},
    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,5), DT_int, 	(char *)&SCR_NUMBER[4]},
    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,6), DT_int, 	(char *)&SCR_NUMBER[5]},
    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,7), DT_int, 	(char *)&SCR_NUMBER[6]},
    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,8), DT_int, 	(char *)&SCR_NUMBER[7]},
    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,9), DT_int, 	(char *)&SCR_NUMBER[8]},
    {"Internal parameter",			FALSE,
	ARR_ELEM(scr_number,10), DT_int, 	(char *)&SCR_NUMBER[9]},

    {"Number of screens",			TRUE,
	"n_screen", 		DT_int, 	(char *)&CALC.n_screen},

    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,1), 	DT_int, 	(char *)&CALC.i_screen[0]},
    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,2), 	DT_int, 	(char *)&CALC.i_screen[1]},
    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,3), 	DT_int, 	(char *)&CALC.i_screen[2]},
    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,4), 	DT_int, 	(char *)&CALC.i_screen[3]},
    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,5), 	DT_int, 	(char *)&CALC.i_screen[4]},
    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,6), 	DT_int, 	(char *)&CALC.i_screen[5]},
    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,7), 	DT_int, 	(char *)&CALC.i_screen[6]},
    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,8), 	DT_int, 	(char *)&CALC.i_screen[7]},
    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,9), 	DT_int, 	(char *)&CALC.i_screen[8]},
    {"Screen before/after (1/0) mirror",	TRUE,
	ARR_ELEM(i_screen,10), DT_int, 	(char *)&CALC.i_screen[9]},

    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,1), 	DT_int, 	(char *)&CALC.i_abs[0]},
    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,2), 	DT_int, 	(char *)&CALC.i_abs[1]},
    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,3), 	DT_int, 	(char *)&CALC.i_abs[2]},
    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,4), 	DT_int, 	(char *)&CALC.i_abs[3]},
    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,5), 	DT_int, 	(char *)&CALC.i_abs[4]},
    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,6), 	DT_int, 	(char *)&CALC.i_abs[5]},
    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,7), 	DT_int, 	(char *)&CALC.i_abs[6]},
    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,8), 	DT_int, 	(char *)&CALC.i_abs[7]},
    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,9), 	DT_int, 	(char *)&CALC.i_abs[8]},
    {"Screen incl. absorption (1/0)",		TRUE,
	ARR_ELEM(i_abs,10), 	DT_int, 	(char *)&CALC.i_abs[9]},

    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,1), 	DT_int, 	(char *)&CALC.i_stop[0]},
    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,2), 	DT_int, 	(char *)&CALC.i_stop[1]},
    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,3), 	DT_int, 	(char *)&CALC.i_stop[2]},
    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,4), 	DT_int, 	(char *)&CALC.i_stop[3]},
    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,5), 	DT_int, 	(char *)&CALC.i_stop[4]},
    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,6), 	DT_int, 	(char *)&CALC.i_stop[5]},
    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,7), 	DT_int, 	(char *)&CALC.i_stop[6]},
    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,8), 	DT_int, 	(char *)&CALC.i_stop[7]},
    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,9), 	DT_int, 	(char *)&CALC.i_stop[8]},
    {"Obstruction (0) or aperture (1)",		TRUE,
	ARR_ELEM(i_stop,10), 	DT_int, 	(char *)&CALC.i_stop[9]},

    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,1), 	DT_int, 	(char *)&CALC.i_slit[0]},
    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,2), 	DT_int, 	(char *)&CALC.i_slit[1]},
    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,3), 	DT_int, 	(char *)&CALC.i_slit[2]},
    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,4), 	DT_int, 	(char *)&CALC.i_slit[3]},
    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,5), 	DT_int, 	(char *)&CALC.i_slit[4]},
    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,6), 	DT_int, 	(char *)&CALC.i_slit[5]},
    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,7), 	DT_int, 	(char *)&CALC.i_slit[6]},
    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,8), 	DT_int, 	(char *)&CALC.i_slit[7]},
    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,9), 	DT_int, 	(char *)&CALC.i_slit[8]},
    {"Aperture stop yes/no (1/0)",		TRUE,
	ARR_ELEM(i_slit,10), 	DT_int, 	(char *)&CALC.i_slit[9]},

    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,1), 	DT_int, 	(char *)&CALC.k_slit[0]},
    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,2), 	DT_int, 	(char *)&CALC.k_slit[1]},
    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,3), 	DT_int, 	(char *)&CALC.k_slit[2]},
    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,4), 	DT_int, 	(char *)&CALC.k_slit[3]},
    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,5), 	DT_int, 	(char *)&CALC.k_slit[4]},
    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,6), 	DT_int, 	(char *)&CALC.k_slit[5]},
    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,7), 	DT_int, 	(char *)&CALC.k_slit[6]},
    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,8), 	DT_int, 	(char *)&CALC.k_slit[7]},
    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,9), 	DT_int, 	(char *)&CALC.k_slit[8]},
    {"Stop/slit shape",				TRUE,
	ARR_ELEM(k_slit,10), 	DT_int, 	(char *)&CALC.k_slit[9]},

    {"Internal parameter",			FALSE,
	"istar1", 		DT_int, 	(char *)&CALC.istar1},
    {"Ang (deg) of cyl axis from X",		TRUE,
	"cil_ang", 		DT_dbl, 	(char *)&MIRROR.cil_ang},
    {"Ang of semi-major axis,mirr pole",	TRUE,
	"ell_the", 		DT_dbl, 	(char *)&MIRROR.ell_the},
    {"No. of extra image plates",		TRUE,
	"n_plates", 		DT_int, 	(char *)&CALC.n_plates},
    {"Internal parameters",			FALSE,
	"ig_seed", 		DT_int, 	(char *)&CALC.ig_seed},
    {"Random seed for mosaic xtal", 		TRUE,
	"mosaic_seed", 		DT_int, 	(char *)&CALC.mosaic_seed},
    {"Mirror orientation ang (deg)",		TRUE,
	"alpha", 		DT_dbl, 	(char *)&SYSTEM1.alpha},
    {"Dist of obj focus to mirr pole",		TRUE,
	"ssour", 		DT_dbl, 	(char *)&SYSTEM1.ssour},
    {"Incidence angle (deg)",			TRUE,
	"theta", 		DT_dbl, 	(char *)&SYSTEM1.theta},
    {"Dist from  mirr pole to im. focus",	TRUE,
	"simag", 		DT_dbl, 	(char *)&SYSTEM1.simag},
    {"Source dist from mirr pole",		TRUE,
	"rdsour", 		DT_dbl, 	(char *)&SYSTEM1.rdsour},
    {"Incidence ang (deg)",			TRUE,
	"rtheta", 		DT_dbl, 	(char *)&SYSTEM1.rtheta},
    {"offset in X in MRF",			TRUE,
	"off_soux", 		DT_dbl, 	(char *)&SYSTEM1.off_soux},
    {"offset in Y in MRF",			TRUE,
	"off_souy", 		DT_dbl, 	(char *)&SYSTEM1.off_souy},
    {"offset in Z in MRF",			TRUE,
	"off_souz", 		DT_dbl, 	(char *)&SYSTEM1.off_souz},
    {"Source rotation around Z(CCW +)",		TRUE,
	"alpha_s", 		DT_dbl, 	(char *)&SYSTEM1.alpha_s},
    {"Mirror half-length +Y",			TRUE,
	"rlen1", 		DT_dbl, 	(char *)&MIRROR.rlen1},
    {"Mirror half-length -Y",			TRUE,
	"rlen2", 		DT_dbl, 	(char *)&MIRROR.rlen2},
    {"Spherical mirror radius",			TRUE,
	"rmirr", 		DT_dbl, 	(char *)&MIRROR.rmirr},
    {"Semi-major axis of mirror",		TRUE,
	"axmaj", 		DT_dbl, 	(char *)&MIRROR.axmaj},
    {"Semi-minor axis of mirror",		TRUE,
	"axmin", 		DT_dbl, 	(char *)&MIRROR.axmin},
    {"Cone half opening (deg)",			TRUE,
	"cone_a", 		DT_dbl, 	(char *)&MIRROR.cone_a},
    {"Tangential (major) axis",			TRUE,
	"r_maj", 		DT_dbl, 	(char *)&MIRROR.r_maj},
    {"Sagittal (minor) axis",			TRUE,
	"r_min", 		DT_dbl, 	(char *)&MIRROR.r_min},
    {"Mirror half-width +X",			TRUE,
	"rwidx1", 		DT_dbl, 	(char *)&MIRROR.rwidx1},
    {"Mirror half-width -X",			TRUE,
	"rwidx2", 		DT_dbl, 	(char *)&MIRROR.rwidx2},
    {"Parab parm P of Y^2 = 2 P X",		TRUE,
	"param", 		DT_dbl, 	(char *)&MIRROR.param},
    {"Monochromator height",			TRUE,
	"hunt_h", 		DT_dbl, 	(char *)&GRATING.hunt_h},
    {"Monochromator length",			TRUE,
	"hunt_l", 		DT_dbl, 	(char *)&GRATING.hunt_l},
    {"Blaze length (deg)",			TRUE,
	"blaze", 		DT_dbl, 	(char *)&GRATING.blaze},
    {"Ruling density in 1/cm at origin",	TRUE,
	"ruling", 		DT_dbl, 	(char *)&GRATING.ruling},
    {"Diffraction order, inside < 0",		TRUE,
	"order", 		DT_dbl, 	(char *)&GRATING.order},
    {"Photon energy (eV) to autotune",		TRUE,
	"phot_cent", 		DT_dbl, 	(char *)&GRATING.phot_cent},
    {"Rotation about X (deg, +CW)",		TRUE,
	"x_rot", 		DT_dbl, 	(char *)&ROT_MIR.x_rot},
    {"Internal  parameter",			FALSE,
	"d_spacing", 		DT_dbl, 	(char *)&GRATING.d_spacing},
    {"Ang btwn xtal planes, surface",		TRUE,
	"a_bragg", 		DT_dbl, 	(char *)&GRATING.a_bragg},
    {"Mosaic spread FWHM (deg)",		TRUE,
	"spread_mos", 		DT_dbl, 	(char *)&XTAL.spread_mos},
    {"xtal thickness (cm)",			TRUE,
	"thickness", 		DT_dbl, 	(char *)&XTAL.thickness},
    {"Johansson radius",			TRUE,
	"r_johansson", 		DT_dbl, 	(char *)&XTAL.r_johansson},
    {"Rotation about Y (deg, +CW)",		TRUE,
	"y_rot", 		DT_dbl, 	(char *)&ROT_MIR.y_rot},
    {"Rotation about Z (deg, +CW)",		TRUE,
	"z_rot", 		DT_dbl, 	(char *)&ROT_MIR.z_rot},
    {"Mirror offset in X (MRF)",		TRUE,
	"offx", 		DT_dbl, 	(char *)&ROT_MIR.offx},
    {"Mirror offset in Y (MRF)",		TRUE,
	"offy", 		DT_dbl, 	(char *)&ROT_MIR.offy},
    {"Mirror offset in Z (MRF)",		TRUE,
	"offz", 		DT_dbl, 	(char *)&ROT_MIR.offz},
    {"Slit length (sagitall X)",		TRUE,
	"sllen", 		DT_dbl, 	(char *)&SLIT.sllen},
    {"Slit width (dispersion Z)",		TRUE,
	"slwid", 		DT_dbl, 	(char *)&SLIT.slwid},
    {"Slit tilt (+CCW degress)",		TRUE,
	"sltilt", 		DT_dbl, 	(char *)&SLIT.sltilt},
    {"Codling slit length (sagittal)",		TRUE,
	"cod_len", 		DT_dbl, 	(char *)&SLIT.cod_len},
    {"Codling slit width (tangential)",		TRUE,
	"cod_wid", 		DT_dbl, 	(char *)&SLIT.cod_wid},

    {"Source offset in X (SRF)",		TRUE,
	"x_sour", 		DT_dbl, 	(char *)&SOURCE.x_sour},
    {"Source offset in Y (SRF)",		TRUE,
	"y_sour", 		DT_dbl, 	(char *)&SOURCE.y_sour},
    {"Source offset in Z (SRF)",		TRUE,
	"z_sour", 		DT_dbl, 	(char *)&SOURCE.z_sour},
    {"Source rotation about X (SRF)",		TRUE,
	"x_sour_rot", 		DT_dbl, 	(char *)&SOURCE.x_sour_rot},
    {"Source rotation about Y (SRF)",		TRUE,
	"y_sour_rot", 		DT_dbl, 	(char *)&SOURCE.y_sour_rot},
    {"Source rotation about Z (SRF)",		TRUE,
	"z_sour_rot", 		DT_dbl, 	(char *)&SOURCE.z_sour_rot},
    {"Angstroms to autotune to",		TRUE,
	"r_lambda", 		DT_dbl, 	(char *)&GRATING.r_lambda},
    {"Elevation (deg) of im. plate",		TRUE,
	"theta_i", 		DT_dbl, 	(char *)&IMAGE.theta_i},
    {"Azimuth (deg) of im. plate",		TRUE,
	"alpha_i", 		DT_dbl, 	(char *)&IMAGE.alpha_i},
    {"Incidence angle (deg)",			TRUE,
	"t_incidence", 		DT_dbl, 	(char *)&AXIS.t_incidence},
    {"Source plane distance",			TRUE,
	"t_source", 		DT_dbl, 	(char *)&AXIS.t_source},
    {"Image plane distance",			TRUE,
	"t_image", 		DT_dbl, 	(char *)&AXIS.t_image},
    {"Reflection angle (deg)",			TRUE,
	"t_reflection",		DT_dbl, 	(char *)&AXIS.t_reflection},
    {"Filename of SOURCE array",		TRUE,
	"file_source", 		DT_str, 	(char *)&NAMES.file_source[0]},
    {"File with distortion info",		TRUE,
	"file_rip", 		DT_str, 	(char *)&NAMES.file_rip[0]},
    {"File with optical const - mirr",		TRUE,
	"file_refl", 		DT_str, 	(char *)&NAMES.file_refl[0]},

#if HP_F77_BUGGY_NAMELIST
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,1), 	DT_str, 	(char *)&NAMES.filabs[0][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,2), 	DT_str, 	(char *)&NAMES.filabs[1][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,3), 	DT_str, 	(char *)&NAMES.filabs[2][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,4), 	DT_str, 	(char *)&NAMES.filabs[3][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,5), 	DT_str, 	(char *)&NAMES.filabs[4][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,6), 	DT_str, 	(char *)&NAMES.filabs[5][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,7), 	DT_str, 	(char *)&NAMES.filabs[6][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,8), 	DT_str, 	(char *)&NAMES.filabs[7][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,9), 	DT_str, 	(char *)&NAMES.filabs[8][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,10),	DT_str, 	(char *)&NAMES.filabs[9][0]},

    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,1), DT_str, 	(char *)&NAMES.filscr[0][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,2), DT_str, 	(char *)&NAMES.filscr[1][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,3), DT_str, 	(char *)&NAMES.filscr[2][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,4), DT_str, 	(char *)&NAMES.filscr[3][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,5), DT_str, 	(char *)&NAMES.filscr[4][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,6), DT_str, 	(char *)&NAMES.filscr[5][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,7), DT_str, 	(char *)&NAMES.filscr[6][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,8), DT_str, 	(char *)&NAMES.filscr[7][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,9), DT_str, 	(char *)&NAMES.filscr[8][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,10),DT_str, 	(char *)&NAMES.filscr[9][0]},
#else
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,1), 	DT_str, 	(char *)&NAMES.file_abs[0][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,2), 	DT_str, 	(char *)&NAMES.file_abs[1][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,3), 	DT_str, 	(char *)&NAMES.file_abs[2][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,4), 	DT_str, 	(char *)&NAMES.file_abs[3][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,5), 	DT_str, 	(char *)&NAMES.file_abs[4][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,6), 	DT_str, 	(char *)&NAMES.file_abs[5][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,7), 	DT_str, 	(char *)&NAMES.file_abs[6][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,8), 	DT_str, 	(char *)&NAMES.file_abs[7][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,9), 	DT_str, 	(char *)&NAMES.file_abs[8][0]},
    {"File with optical const - screen",	TRUE,
	ARR_ELEM(file_abs,10),	DT_str, 	(char *)&NAMES.file_abs[9][0]},

    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,1), DT_str, (char *)&NAMES.file_scr_ext[0][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,2), DT_str, (char *)&NAMES.file_scr_ext[1][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,3), DT_str, (char *)&NAMES.file_scr_ext[2][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,4), DT_str, (char *)&NAMES.file_scr_ext[3][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,5), DT_str, (char *)&NAMES.file_scr_ext[4][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,6), DT_str, (char *)&NAMES.file_scr_ext[5][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,7), DT_str, (char *)&NAMES.file_scr_ext[6][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,8), DT_str, (char *)&NAMES.file_scr_ext[7][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,9), DT_str, (char *)&NAMES.file_scr_ext[8][0]},
    {"File with external slit/stop - screen",	TRUE,
	ARR_ELEM(file_scr_ext,10),DT_str, (char *)&NAMES.file_scr_ext[9][0]},
#endif

    {"File with polynomial description",	TRUE,
	"file_mir", 		DT_str, 	(char *)&NAMES.file_mir[0]},
    {"File with roughness profile",		TRUE,
	"file_rough",		DT_str,		(char *)&NAMES.file_rough[0]},

    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,1), 	DT_dbl, 	(char *)&SCREENS.thick[0]},
    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,2), 	DT_dbl, 	(char *)&SCREENS.thick[1]},
    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,3), 	DT_dbl, 	(char *)&SCREENS.thick[2]},
    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,4), 	DT_dbl, 	(char *)&SCREENS.thick[3]},
    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,5), 	DT_dbl, 	(char *)&SCREENS.thick[4]},
    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,6), 	DT_dbl, 	(char *)&SCREENS.thick[5]},
    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,7), 	DT_dbl, 	(char *)&SCREENS.thick[6]},
    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,8), 	DT_dbl, 	(char *)&SCREENS.thick[7]},
    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,9), 	DT_dbl, 	(char *)&SCREENS.thick[8]},
    {"Screen - thickness of file (cm)",		TRUE,
	ARR_ELEM(thick,10), 	DT_dbl, 	(char *)&SCREENS.thick[9]},

    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,1), 	DT_dbl, 	(char *)&SCREENS.rx_slit[0]},
    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,2), 	DT_dbl, 	(char *)&SCREENS.rx_slit[1]},
    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,3), 	DT_dbl, 	(char *)&SCREENS.rx_slit[2]},
    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,4), 	DT_dbl, 	(char *)&SCREENS.rx_slit[3]},
    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,5), 	DT_dbl, 	(char *)&SCREENS.rx_slit[4]},
    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,6), 	DT_dbl, 	(char *)&SCREENS.rx_slit[5]},
    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,7), 	DT_dbl, 	(char *)&SCREENS.rx_slit[6]},
    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,8), 	DT_dbl, 	(char *)&SCREENS.rx_slit[7]},
    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,9), 	DT_dbl, 	(char *)&SCREENS.rx_slit[8]},
    {"Screen - slit/stop dimension in X",	TRUE,
	ARR_ELEM(rx_slit,10), 	DT_dbl, 	(char *)&SCREENS.rx_slit[9]},

    {"Dist from mirr center to plate",		TRUE,
	ARR_ELEM(d_plate,1), 	DT_dbl, 	(char *)&IMAGE.d_plate[0]},
    {"Dist from mirr center to plate",		TRUE,
	ARR_ELEM(d_plate,2), 	DT_dbl, 	(char *)&IMAGE.d_plate[1]},
    {"Dist from mirr center to plate",		TRUE,
	ARR_ELEM(d_plate,3), 	DT_dbl, 	(char *)&IMAGE.d_plate[2]},
    {"Dist from mirr center to plate",		TRUE,
	ARR_ELEM(d_plate,4), 	DT_dbl, 	(char *)&IMAGE.d_plate[3]},
    {"Dist from mirr center to plate",		TRUE,
	ARR_ELEM(d_plate,5), 	DT_dbl, 	(char *)&IMAGE.d_plate[4]},

    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,1), 	DT_dbl, 	(char *)&SCREENS.rz_slit[0]},
    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,2), 	DT_dbl, 	(char *)&SCREENS.rz_slit[1]},
    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,3), 	DT_dbl, 	(char *)&SCREENS.rz_slit[2]},
    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,4), 	DT_dbl, 	(char *)&SCREENS.rz_slit[3]},
    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,5), 	DT_dbl, 	(char *)&SCREENS.rz_slit[4]},
    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,6), 	DT_dbl, 	(char *)&SCREENS.rz_slit[5]},
    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,7), 	DT_dbl, 	(char *)&SCREENS.rz_slit[6]},
    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,8), 	DT_dbl, 	(char *)&SCREENS.rz_slit[7]},
    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,9), 	DT_dbl, 	(char *)&SCREENS.rz_slit[8]},
    {"Screen - slit/stop dimension in Z",	TRUE,
	ARR_ELEM(rz_slit,10), 	DT_dbl, 	(char *)&SCREENS.rz_slit[9]},

    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,1), 	DT_dbl, 	(char *)&SCREENS.cx_slit[0]},
    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,2), 	DT_dbl, 	(char *)&SCREENS.cx_slit[1]},
    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,3), 	DT_dbl, 	(char *)&SCREENS.cx_slit[2]},
    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,4), 	DT_dbl, 	(char *)&SCREENS.cx_slit[3]},
    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,5), 	DT_dbl, 	(char *)&SCREENS.cx_slit[4]},
    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,6), 	DT_dbl, 	(char *)&SCREENS.cx_slit[5]},
    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,7), 	DT_dbl, 	(char *)&SCREENS.cx_slit[6]},
    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,8), 	DT_dbl, 	(char *)&SCREENS.cx_slit[7]},
    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,9), 	DT_dbl, 	(char *)&SCREENS.cx_slit[8]},
    {"Screen - slit/stop center in X",		TRUE,
	ARR_ELEM(cx_slit,10), 	DT_dbl, 	(char *)&SCREENS.cx_slit[9]},

    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,1), 	DT_dbl, 	(char *)&SCREENS.cz_slit[0]},
    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,2), 	DT_dbl, 	(char *)&SCREENS.cz_slit[1]},
    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,3), 	DT_dbl, 	(char *)&SCREENS.cz_slit[2]},
    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,4), 	DT_dbl, 	(char *)&SCREENS.cz_slit[3]},
    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,5), 	DT_dbl, 	(char *)&SCREENS.cz_slit[4]},
    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,6), 	DT_dbl, 	(char *)&SCREENS.cz_slit[5]},
    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,7), 	DT_dbl, 	(char *)&SCREENS.cz_slit[6]},
    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,8), 	DT_dbl, 	(char *)&SCREENS.cz_slit[7]},
    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,9), 	DT_dbl, 	(char *)&SCREENS.cz_slit[8]},
    {"Screen - slit/stop center in Z",		TRUE,
	ARR_ELEM(cz_slit,10), 	DT_dbl, 	(char *)&SCREENS.cz_slit[9]},

    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,1), 	DT_dbl, 	(char *)&SCREENS.sl_dis[0]},
    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,2), 	DT_dbl, 	(char *)&SCREENS.sl_dis[1]},
    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,3), 	DT_dbl, 	(char *)&SCREENS.sl_dis[2]},
    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,4), 	DT_dbl, 	(char *)&SCREENS.sl_dis[3]},
    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,5), 	DT_dbl, 	(char *)&SCREENS.sl_dis[4]},
    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,6), 	DT_dbl, 	(char *)&SCREENS.sl_dis[5]},
    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,7), 	DT_dbl, 	(char *)&SCREENS.sl_dis[6]},
    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,8), 	DT_dbl, 	(char *)&SCREENS.sl_dis[7]},
    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,9), 	DT_dbl, 	(char *)&SCREENS.sl_dis[8]},
    {"Screen - abs. dist from mirror",		TRUE,
	ARR_ELEM(sl_dis,10), 	DT_dbl, 	(char *)&SCREENS.sl_dis[9]},

    {"Fresnel zone plate; yes(1),no(0)",	TRUE,
	"fzp", 			DT_int, 	(char *)&FLAGS.fzp},
    {"Holo: entrance slit dist (cm)",		TRUE,
	"holo_r1", 		DT_dbl, 	(char *)&HOLO.holo_r1},
    {"Holo: exit slit dist (cm)",		TRUE,
	"holo_r2", 		DT_dbl, 	(char *)&HOLO.holo_r2},
    {"Holo: entrance slit incid ang",		TRUE,
	"holo_del", 		DT_dbl, 	(char *)&HOLO.holo_del},
    {"Holo: exit slit incid ang",		TRUE,
	"holo_gam", 		DT_dbl, 	(char *)&HOLO.holo_gam},
    {"Recording wavelength angst",		TRUE,
	"holo_w", 		DT_dbl, 	(char *)&HOLO.holo_w},
    {"Holo: entrance slit rot ang",		TRUE,
	"holo_rt1", 		DT_dbl, 	(char *)&HOLO.holo_rt1},
    {"Holo: exit slit rot ang",			TRUE,
	"holo_rt2", 		DT_dbl, 	(char *)&HOLO.holo_rt2},
    {"Ruling focus on mirror plane",		TRUE,
	"azim_fan", 		DT_dbl, 	(char *)&GRATING.azim_fan},
    {"Distance from grating ctr",		TRUE,
	"dist_fan", 		DT_dbl, 	(char *)&GRATING.dist_fan},
    {"Coma correction factor; 0 if none",	TRUE,
	"coma_fac", 		DT_dbl, 	(char *)&GRATING.coma_fac},
    {"Alpha: epsilon=(1-alpha)+i gamma",	TRUE,
	"alfa", 		DT_dbl, 	(char *)&LIGHT.alfa},
    {"Gamma of the above eqn",			TRUE,
	"gamma", 		DT_dbl, 	(char *)&LIGHT.gamma},
    {"Index of refraction in obj space",	TRUE,
	"r_ind_obj", 		DT_dbl, 	(char *)&LIGHT.r_ind_obj},
    {"Index of refraction in im. space",	TRUE,
	"r_ind_ima", 		DT_dbl, 	(char *)&LIGHT.r_ind_ima},
    {"a1 coefficient of polynomial",		TRUE,
	"rul_a1", 		DT_dbl, 	(char *)&GRATING.rul_a1},
    {"a2 coefficient of polynomial",		TRUE,
	"rul_a2", 		DT_dbl, 	(char *)&GRATING.rul_a2},
    {"a3 coefficient of polynomial",		TRUE,
	"rul_a3", 		DT_dbl, 	(char *)&GRATING.rul_a3},
    {"a4 coefficient of polynomial",		TRUE,
	"rul_a4", 		DT_dbl, 	(char *)&GRATING.rul_a4},
    {"Intercept selected (facet)",		TRUE,
	"f_polsel",		DT_int,		(char *)&FACET.f_polsel},
    {"Faceted mirror y/n",			TRUE,
	"f_facet",		DT_int,		(char *)&FACET.f_facet},
    {"Internal parameter",			FALSE,
	"f_fac_orient",		DT_int,		(char *)&FACET.f_fac_orient},
    {"Internal",				FALSE,
	"f_fac_latt",		DT_int,		(char *)&FACET.f_fac_latt},
    {"Facet width (x)",				TRUE,
	"rfac_lenx",		DT_dbl,		(char *)&FACET.rfac_lenx},
    {"Facet length",				TRUE,
	"rfac_leny",		DT_dbl,		(char *)&FACET.rfac_leny},
    {"Internal",				FALSE,
	"rfac_phax",		DT_dbl,		(char *)&FACET.rfac_phax},
    {"Internal",				FALSE,
	"rfac_phay",		DT_dbl,		(char *)&FACET.rfac_phay},
    {"Internal",				FALSE,
	"rfac_delx1",		DT_dbl,		(char *)&FACET.rfac_delx1},
    {"Internal",				FALSE,
	"rfac_delx2",		DT_dbl,		(char *)&FACET.rfac_delx2},
    {"Internal",				FALSE,
	"rfac_dely1",		DT_dbl,		(char *)&FACET.rfac_dely1},
    {"Internal",				FALSE,
	"rfac_dely2",		DT_dbl,		(char *)&FACET.rfac_dely2},
    {"File w/ facet info",			TRUE,
	"file_fac",		DT_str,		(char *)&NAMES.file_fac[0]},
    {"Segmented mirror y/n?",			TRUE,
	"f_segment",		DT_int,		(char *)&FLAGS.f_segment},
    {"Number of segments (X)",			TRUE,
	"iseg_xnum",		DT_int,		(char *)&SEGMENT.iseg_xnum},
    {"Number of segments (Y)",			TRUE,
	"iseg_ynum",		DT_int,		(char *)&SEGMENT.iseg_ynum},
    {"File w/orientation",			TRUE,
	"file_segment",		DT_str,		(char *)&NAMES.file_segment[0]},
    {"File w/polynomial",			TRUE,
	"file_segp",		DT_str,		(char *)&NAMES.file_segp[0]},
    {"Length of segments (x)",			TRUE,
	"seg_lenx",		DT_dbl,		(char *)&SEGMENT.seg_lenx},
    {"Length of segments (y)",			TRUE,
	"seg_leny",		DT_dbl,		(char *)&SEGMENT.seg_leny},
    {"kumakhov lens? y/n",			TRUE,
	"f_koma",		DT_int,		(char *)&KOMA.f_koma},
    {"File with parameters:",			TRUE,
	"file_koma",		DT_str,		(char *)&NAMES.file_koma[0]},
    {"Internal",				FALSE,
	"f_exit_shape",		DT_int,		(char *)&KOMA.f_exit_shape},
    {"Internal",				FALSE,
	"f_inc_mnor_ang",	DT_int,		(char *)&KOMA.f_inc_mnor_ang},
    {"Internal",				FALSE,
	"zko_length",		DT_dbl,		(char *)&KOMA.zko_length},
    {"Internal",				FALSE,
	"rkoma_cx",		DT_dbl,		(char *)&KOMA.rkoma_cx},
    {"Internal",				FALSE,
	"rkoma_cy",		DT_dbl,		(char *)&KOMA.rkoma_cy},
    {"Specify as r^2 instead of r",		TRUE,
	"f_koma_ca",		DT_int,		(char *)&KOMA.f_koma_ca},
    {"File with parameters (r^2)",		TRUE,
	"file_koma_ca",		DT_str,		(char *)&NAMES.file_koma_ca[0]},
    {"Save bounce/intercept info:",		TRUE,
	"f_koma_bounce",	DT_int,		(char *)&KOMA.f_koma_bounce},
    {"Amplitude of distortion along X",		TRUE,
	"x_rip_amp", 		DT_dbl, 	(char *)&RIPPLE.x_rip_amp},
    {"Wavelength along X axis",			TRUE,
	"x_rip_wav", 		DT_dbl, 	(char *)&RIPPLE.x_rip_wav},
    {"Phase for X axis;0 - max at origin",	TRUE,
	"x_phase", 		DT_dbl, 	(char *)&RIPPLE.x_phase},
    {"Amplitude of distortion along Y",		TRUE,
	"y_rip_amp", 		DT_dbl, 	(char *)&RIPPLE.y_rip_amp},
    {"Wavelength along X axis",			TRUE,
	"y_rip_wav", 		DT_dbl, 	(char *)&RIPPLE.y_rip_wav},
    {"Phase for Y axis;0 - max at origin",	TRUE,
	"y_phase", 		DT_dbl, 	(char *)&RIPPLE.y_phase},
    {"Internal parameter",			FALSE,
	"n_rip", 		DT_int, 	(char *)&RIPP_2.n_rip},
    {"RMS roughness X",				TRUE,
	"rough_x",		DT_dbl,		(char *)&RIPPLE.rough_x},
    {"RMS roughness Y",				TRUE,
	"rough_y",		DT_dbl,		(char *)&RIPPLE.rough_y},
    {"Internal parameter",			FALSE,
	"oe_number", 		DT_int, 	(char *)&OE_NUMBER},
    {"Internal parameter",			FALSE,
	"idummy", 		DT_int, 	(char *)&IDUMMY},
    {"Internal parameter",			FALSE,
	"dummy", 		DT_dbl, 	(char *)&DUMMY},
};


static Namelist nml_lists[] = {
    {NML_source,	Number (nml_source_list),	nml_source_list},
    {NML_oe,		Number (nml_oe_list),		nml_oe_list},
};

#define NML_READ_SO	"R_SOUR"
#define NML_WRITE_SO	"W_SOUR"
#define NML_READ_OE	"R_OE"
#define NML_WRITE_OE	"W_OE"

/*
 * Only define the following when used inside of SHADOW. Don't need these
 * for the GUI for example, which shares this same file.
 */
#if USE_SHADOW_LIBRARY

static char *nml_rd_req[] = {NML_READ_SO, NML_READ_OE};
static char *nml_wr_req[] = {NML_WRITE_SO, NML_WRITE_OE};

static int num_lists = Number (nml_lists);

/*
 * used for error messages.
 */
static char* data_types[] = {
    "integer", "real", "real", "filename"
};

#endif /*!USE_SHADOW_LIBRARY*/

Namelist* get_namelist(NmlType type) {
    Namelist* list = nil;
    if(type == NML_source) 
	list = &nml_lists[NML_source];
    else if (type == NML_oe) 
	list = &nml_lists[NML_oe];
    return list;
}

/*
 * Only define the following when used inside of SHADOW. Don't need these
 * for the GUI for example, which shares this same file.
 */
#if USE_SHADOW_LIBRARY

NamelistData* get_namelist_data (const char* name, const Namelist* namelist) {
    int i = 0;
    NamelistData* nml_data = namelist->nml_data;
    unsigned int size = namelist->size;

    static char local_name[1024];
    strcpy(local_name, name);
    str_lo_case(local_name);
    
    for (i = 0; i < size; i++) {
	if (!strcmp (nml_data[i].name, local_name)) {
	    return &nml_data[i];
	}
    }

    return nil;
}

int change_value(const char* name, const char* newval, Namelist* namelist) {
    NamelistData* data = get_namelist_data(name, namelist);
    int retval = -1;

    if (data != nil)
	retval = do_change_value(data, newval);

    return retval;
}

/*
 * read a namelist file ``nmlfile'' of given type and return the Namelist.
 * return nil on error.
 */
Namelist* read_fnamelist(const char *nml_file, NmlType nml_type) {
    int success;

    RWNAME_F77 (nml_file, nml_rd_req[nml_type], &success, 
	(unsigned long) strlen (nml_file),
	(unsigned long) strlen (nml_rd_req[nml_type]));
	
    return (success == 0) ? get_namelist(nml_type) : nil;
}

/*
 * read a namelist file ``nmlfile'' of given type and return the Namelist.
 * return nil on error.
 */
Namelist* read_namelist(const char *nml_file, NmlType nml_type) {
    int success;

    RWNAME (nml_file, nml_rd_req[nml_type], &success, 
	(unsigned long) strlen (nml_file),
	(unsigned long) strlen (nml_rd_req[nml_type]));
	
    return (success == 0) ? get_namelist(nml_type) : nil;
}

/*
 * read a gfile file ``gfile'' and create a Namelist of the given type..
 * return nil on error.
 */
Namelist* read_gfile(const char* gfile, NmlType nml_type) {
    boolean found_error = FALSE;
    int i = 0, size = 0, retval = 0;
    char buf[1024];
    Namelist* curlist = get_namelist(nml_type);
    if (curlist == nil)
	return nil;
    
    g_file(gfile);
    size = curlist->size;
    for(i = 0; i < size; i++) {
	NamelistData* data = &curlist->nml_data[i];
	if (g_peek(data->name)) {
	    g_string(data->name, data->synopsis, buf);
	    retval = do_change_value(data, buf);
	    if(retval == -1) {
		fprintf (stderr, 
		    "Namelist: error creating namelist from g-file.\n"
		    "\tparameter: %s, type: %s\n"
		    "\tvalue found: `%s'. Wrong type perhaps?\n",
		    data->name, data_types[data->data_type], buf
		);
		found_error = TRUE;
		break;
	    }
	}
    }
    g_close();
    return (found_error) ? nil : curlist;
}

/*
 * read a gfile file ``gfile'' and create a Namelist of the given type..
 * return nil on error.
 */
#if G77
void G77_HACK_NAME(read_gfile_f) (
#else
void F77_NAME(read_gfile_f) (
#endif
    const char* gfilename, int* type, int* status, unsigned long len
) {
    char buf[1024];
    NmlType nml_type = *type;

    strncpy(buf, gfilename, len);
    while(buf[--len] == ' ')
	;
    buf[len+1] = '\0';

    *status = (read_gfile(buf, nml_type) != NULL) ? 0 : -1;
}

/*
 * write a namelist file ``nmlfile'' of given type from the Namelist.
 * return -1 on error, 0 otherwise.
 */
int write_fnamelist(const char* nml_file, NmlType nml_type) {
    int success = 0;
    Namelist* namelist = get_namelist(nml_type);

    RWNAME_F77 (nml_file, nml_wr_req[nml_type], &success, 
	(unsigned long) strlen (nml_file),
	(unsigned long) strlen (nml_wr_req[nml_type])
    );
    return success;
}

/*
 * write a namelist file ``nmlfile'' of given type from the Namelist.
 * return -1 on error, 0 otherwise.
 */
int write_namelist(const char* nml_file, NmlType nml_type) {
    int success = 0;
    Namelist* namelist = get_namelist(nml_type);

    RWNAME (nml_file, nml_wr_req[nml_type], &success, 
	(unsigned long) strlen (nml_file),
	(unsigned long) strlen (nml_wr_req[nml_type])
    );
    return success;
}

/*
 * write a gfile file ``outfile'' from the Namelist.
 * return -1 on error, 0 otherwise.
 */
int write_gfile (const char* outfile, NmlType type) {
    int i = 0;
    unsigned size = 0;
    Namelist* namelist = get_namelist(type);

    FILE* fp = fopen(outfile, "w");
    if(fp == nil || namelist == nil) {
	return -1;
    }
    
    size = namelist->size;
    for (i = 0; i < size; i++) {
	NamelistData* data = &namelist->nml_data[i];
	fprintf (fp, "%s\t=\t%s\n", data->name, sprint_value(data));
    }
    fclose(fp);
    return 0;
}

#if G77
void G77_HACK_NAME(write_gfile_f) (
#else
void F77_NAME(write_gfile_f) (
#endif
    const char* gfilename, int* type, int* status, unsigned long len
) {
    NmlType nml_type = *type;
    char buf[BUFSIZ];

    strncpy(buf, gfilename, len);
    while(buf[--len] == ' ')
	;
    buf[len+1] = '\0';

    *status = write_gfile(buf, nml_type);
}

/*
 * sprint_value:
 * return a TEXT representation of data value.
 * if the datum is not valid, the returned string says it.
 *
 */

char* sprint_value(const NamelistData* data) {
    static char buf[BUFSIZ];
    strcpy(buf, "!! Illegal data item !!");

    switch (data->data_type) {
	case DT_int:
	    sprintf (buf, "%d", *(int *)data->value);
	    break;

	case DT_flt:
	    sprintf (buf, "%-.17g", *(float *)data->value);
	    break;

	case DT_dbl:
	    sprintf (buf, "%-.17g", *(double *)data->value);
	    break;

	case DT_str:
	    strncpy (buf, data->value, FNAME_LEN);
	    buf[FNAME_LEN] = '\0';
	    break;
	
	default:
	    break;
    }

    return buf;
}

static Namelist *make_namelist(int size, NmlType nml_type) {
    Namelist* new_namelist = (Namelist*) calloc (1, sizeof (Namelist));
    new_namelist->size = size;
    new_namelist->nml_type = nml_type;
    new_namelist->nml_data = (size > 0) ?
	(NamelistData*) calloc(size, sizeof(NamelistData)) : nil;
    return new_namelist;
}


Namelist* clone_namelist(const Namelist* namelist_in) {
    int i = 0;
    Namelist* cloned = 0;
    if (namelist_in == 0)
	return cloned;
    
    cloned = make_namelist(namelist_in->size, namelist_in->nml_type);
    for (i = 0; i < namelist_in->size; ++i) {
	int nbytes;
        const NamelistData* namelist_data_in = &namelist_in->nml_data[i];
	NamelistData* new_namelist_data = &cloned->nml_data[i];

	new_namelist_data->synopsis = 
	    (char*) malloc(strlen(namelist_data_in->synopsis) + 1);
	strcpy(new_namelist_data->synopsis, namelist_data_in->synopsis);

	new_namelist_data->allow_change = namelist_data_in->allow_change;

	new_namelist_data->name = 
	    (char*) malloc(strlen(namelist_data_in->name) + 1);
	strcpy(new_namelist_data->name, namelist_data_in->name);

	new_namelist_data->data_type = namelist_data_in->data_type;

	switch (new_namelist_data->data_type) {
	    case DT_int:
		nbytes = sizeof(int);
		break;

	    case DT_flt:
		nbytes = sizeof(float);
		break;

	    case DT_dbl:
		nbytes = sizeof(double);
		break;

	    case DT_str:
		/* CHECK/FIXME: All strings are 80 characters? */
		nbytes = 80;
		break;
	    
	    default:
		assert(0);
		break;
	}
	new_namelist_data->value = calloc(1, nbytes);
	memcpy(new_namelist_data->value, namelist_data_in->value, nbytes);
    }
    return cloned;
}

/*
 * Now comes the local static functions.
 */


/***********************************************************************
 *
 * Check to see if the string "string" is an integer or not. An integer
 * is defined to be a string of digits optionally preceded by a '+\-'.
 * ie., integer = [+-][DIGIT][DIGIT]*.
 *
 * returns:
 *    0: ok
 *   -1: error
 *
 **********************************************************************/

static int get_integer(const char *string, int *intval) {
    int positive = TRUE;
    char const *tmpstring = string;

    /* find the sign of the damn thing */
    switch (*tmpstring) {
	case '-':
	    positive = FALSE;
	    /* FALL THRU */
	case '+':
	    tmpstring++;
	    break;
	default:
	    break;
    }
	
    if (*tmpstring == '\0') 
	return -1;

    while (*tmpstring != '\0') {
	if (!isdigit(*tmpstring++)) 
	    return -1;
    }

    /* now scan the damn thing (REPLACE with my own scanf!!) */
    sscanf (string, "%d", intval);
    return 0;
}


/***********************************************************************
 *
 * Is the string "string" a floating point number? A floating point @
 * is defined to be: [+-][DIGIT]*[.(digits)|(exponent)] 
 *
 * returns:
 *    0: ok
 *   -1: error
 *
 **********************************************************************/

static int get_float(const char *string, float *fltval) {
    int got_period = 0;
    int dig_count = 0;
    char const *tmpstring = string;
    int dummy = 0;

    if (*tmpstring == '-' || *tmpstring == '+') 
	tmpstring++;

    while (*tmpstring != '\0') {
	if (isdigit (*tmpstring)) {
	    dig_count++;
	} 
	else if (*tmpstring == '.' && !got_period) {
	    got_period = 1;
	} 
	else if (*tmpstring == 'e' || *tmpstring == 'E') {
	    if  (!((dig_count > 0) && (get_integer(++tmpstring,&dummy) != -1)))
		return -1;
	    else
		break;		/* tmpstring is now == '\0' */
	} 
	else {
	    return -1;
	}
	tmpstring++;
    }

    if (dig_count > 0) 
	sscanf (string, "%g", fltval);

    return (dig_count > 0) ? 0 : -1;
}


/***********************************************************************
 *
 * Is the string "string" a doubleing point number? A doubleing point @
 * is defined to be: [+-][DIGIT]*[.(digits)|(exponent)] 
 *
 * returns:
 *    0: ok
 *   -1: error
 *
 **********************************************************************/

static int get_double(const char *string, double *dblval) {
    int got_period = 0;
    int dig_count = 0;
    char const *tmpstring = string;
    int dummy = 0;

    if (*tmpstring == '-' || *tmpstring == '+') 
	tmpstring++;

    while (*tmpstring != '\0') {
	if (isdigit (*tmpstring)) {
	    dig_count++;
	} 
	else if (*tmpstring == '.' && !got_period) {
	    got_period = 1;
	} 
	else if (*tmpstring == 'e' || *tmpstring == 'E') {
	    if  (!((dig_count > 0) && (get_integer(++tmpstring,&dummy) != -1)))
		return -1;
	    else
		break;		/* tmpstring is now == '\0' */
	} 
	else {
	    return -1;
	}
	tmpstring++;
    }

    if (dig_count > 0) 
	sscanf (string, "%lg", dblval);

    return (dig_count > 0) ? 0 : -1;
}


static int do_change_value (NamelistData* data, const char* newval) {

    float f = 0;
    double d = 0;
    int i = 0;

    int len = 0;

    int retval = 0;

    switch (data->data_type) {
	case DT_int:
	    if (get_integer (newval, &i) == -1) {
		retval = -1;
	    }
	    else {
		*((int *) data->value) = i;
	    }
	    break;

	case DT_flt:
	    if (get_float (newval, &f) == -1) {
		retval = -1;
	    }
	    else {
		*((float *) data->value) = f;
	    }
	    break;

	case DT_dbl:
	    if (get_double (newval, &d) == -1) {
		retval = -1;
	    }
	    else {
		*((double *) data->value) = d;
	    }
	    break;

	case DT_str:
	    if ((len = strlen (newval)) > FNAME_LEN) {
		retval = -1;
	    }
	    else {
		/* FORTRAN strings have to nil character, so kill it */
		strcpy (data->value, newval);
		for (i = strlen(newval); i < FNAME_LEN; data->value[i++] = ' ')
		    ;
	    }
	    break;

	default:
	    break;
    }

    return retval;
}

static char* str_up_case(char *str) {
    char *str_ptr = str;
    for (; *str_ptr; ++str_ptr) {
	if (islower (*str_ptr)) 
	    *str_ptr = toupper (*str_ptr);
	if (*str_ptr == '-') 
	    *str_ptr = '_';
    }
    return str;
}


static char* str_lo_case(char *str) {
    char *str_ptr = str;
    for (; *str_ptr; ++str_ptr) {
	if (isupper (*str_ptr)) 
	    *str_ptr = tolower (*str_ptr);
	if (*str_ptr == '-') 
	    *str_ptr = '_';
    }
    return str;
}

#endif /*!USE_SHADOW_LIBRARY*/
