/*
 *
 * g-to-nml.c: Convert G file to FORTRAN namelist file
 *
 * Source: src/utils/cnamelist/g-to-nml.c
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * ------------------------------------------------
 *              SHADOW
 *     Center for X-ray Lithography
 *   University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "common.h"
#include "namelist.h"

int parse_args (int argc, char** argv, const char* prog_name, 
    char** infile, char** outfile, NmlType* type
);
void usage(const char* prog_name);

int main (int argc, char** argv) {
    Namelist* namelist = nil;
    char* prog_name = nil;
    char* nml_file = nil;
    char* g_file = nil;
    NmlType type;

    char *ptr = strrchr(argv[0], '/');
    if (ptr) 
	prog_name = ptr + 1;
    else 
	prog_name = argv[0];

    (void) parse_args(argc, argv, prog_name, &g_file, &nml_file, &type);

    namelist = read_gfile(g_file, type);
    if (namelist == nil) {
	fprintf(stderr, "ERROR: Cannot create namelist from `%s'\n", g_file);
	exit(1);
    }
    return write_fnamelist(nml_file, type);
}

int parse_args (int argc, char** argv, const char* prog_name, char** infile, 
    char** outfile, NmlType* type
) {
    extern int optind;
    extern char *optarg;

    boolean type_supplied = FALSE;
    int c;

    *infile = nil;
    *outfile = nil;

    while ((c = getopt (argc, argv, "ht:o:")) != EOF) {
	switch (c) {
	    case 't':
		if (optarg[0] == 's' || optarg[0] == 'S') {
		    *type = NML_source;
		}
		else if (optarg[0] == 'o' || optarg[0] == 'O') {
		    *type = NML_oe;
		}
		else {
		    fprintf(stderr, "Illegal namelist type --  %s\n", 
			optarg
		    );
		    usage(prog_name);
		    exit(1);
		}
		type_supplied = TRUE;
		break;

	    case 'o':
		*outfile = strcpy (malloc (strlen (optarg) + 1), optarg);
		break;
	    
	    case 'h':
		usage (prog_name);
		exit (0);
		break;
	    
	    default:
		usage (prog_name);
		exit (1);
		break;
	}
    }

    if (argc == optind+1) {
	*infile = strcpy (malloc (strlen (argv[optind]) + 1), argv[optind]);
    }

    if (*infile == nil || *outfile == nil || type_supplied == FALSE) {
	fprintf(stderr, "Must supply all required parameters.\n");
	usage(prog_name);
	exit(1);
    }
    return 0;
}


void usage (const char* prog_name) {
    fprintf (stderr, 
    "Usage:%s [-h] -o OutputNamelist -t NamelistType GFile\n\
    -h:			print this info\n\
    -o OutputNamelist:	SHADOW namelist file to convert to gfile\n\
    -t NamelistType:	\"source\" or \"oe\" (also accepts \"s\" or \"o\")\n\
    GFile:		EXCON GFile\n\n",
    prog_name);
    return;
}
