//
// value.h: interfaces various variable value types
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#ifndef excon_batch_value_h
#define excon_batch_value_h

/****************************************************************************
 *
 *  VALUES
 *
 ****************************************************************************/
class Value {
public:
    Value();
    virtual ~Value();

    virtual int setval(const char* str);

    virtual bool isainteger() const;
    virtual bool isareal() const;
    virtual bool isatext() const;
    virtual bool isaboolean() const;
    virtual bool isaenum() const;
    virtual bool isarange() const;

    virtual bool isequal(const Value*) const;
    virtual bool isgt(const Value*) const;

    virtual Value* clone() const;
    virtual Value* clone(const char* str) const;

    virtual int getinteger() const;
    virtual double getreal() const;
    virtual const char* gettext() const;
    virtual bool getboolean() const;
    virtual const char* getenum() const;
    virtual const Value* getmin() const;
    virtual const Value* getmax() const;
    virtual const Value* getinc() const;
    virtual const char* getstrval() const;

    virtual const Value* add(const Value*);
    virtual const Value* subtract(const Value*);
    virtual const Value* mul(const Value*);
    virtual const Value* div(const Value*);

    virtual void dump() const;

protected:
    char* strvalbuf_;  		// getstrval RETURNS POINTER TO THIS
    static class Pmu* pmu_;
};


/***************************************************************************/

class IntegerValue : public Value {
public:
    IntegerValue(int val);
    virtual ~IntegerValue();
    virtual int setval(const char* str);

    virtual bool isequal(const Value*) const;
    virtual bool isgt(const Value*) const;

    virtual Value* clone() const;
    virtual Value* clone(const char* str) const;

    virtual bool isainteger() const;

    virtual int getinteger() const;
    virtual const char* getstrval() const;

    virtual const Value* add(const Value*);
    virtual const Value* subtract(const Value*);
    virtual const Value* mul(const Value*);
    virtual const Value* div(const Value*);

    virtual void dump() const;
private:
    int val_;
};

/***************************************************************************/

class RealValue : public Value {
public:
    RealValue(double val);
    virtual ~RealValue();
    virtual bool isequal(const Value*) const;
    virtual bool isgt(const Value*) const;
    virtual int setval(const char* str);
    virtual Value* clone() const;
    virtual Value* clone(const char* str) const;
    virtual bool isareal() const;
    virtual double getreal() const;
    virtual const char* getstrval() const;
    virtual const Value* add(const Value*);
    virtual const Value* subtract(const Value*);
    virtual const Value* mul(const Value*);
    virtual const Value* div(const Value*);

    virtual void dump() const;
private:
    double val_;
};


/***************************************************************************/

class TextValue : public Value {
public:
    TextValue(const char* text);
    virtual ~TextValue();
    virtual bool isequal(const Value*) const;
    virtual bool isgt(const Value*) const;
    virtual int setval(const char* str);
    virtual Value* clone() const;
    virtual Value* clone(const char* str) const;
    virtual bool isatext() const;
    virtual const char* gettext() const;
    virtual const char* getstrval() const;

    virtual void dump() const;
private:
    char* val_;
};
/***************************************************************************/

class BooleanValue : public Value {
public:
    BooleanValue(bool val);
    virtual ~BooleanValue();
    virtual bool isequal(const Value*) const;
    virtual int setval(const char* str);
    virtual Value* clone() const;
    virtual Value* clone(const char* str) const;
    virtual bool isaboolean() const;
    virtual bool getboolean() const;
    virtual const char* getstrval() const;
    // virtual void add(const Value*);
    // virtual void subtract(const Value*);

    virtual void dump() const;
private:
   bool val_;
};


/***************************************************************************/

class EnumValue : public Value {
public:
    virtual ~EnumValue();
    //virtual bool isequal(const Value*) const;
    virtual bool isaenum() const;
    virtual const char* getenum() const;
private:
    class Stq* vallist;
    char* val_;       // POINTER TO item IN vallist
};

/***************************************************************************/

class RangeValue : public Value {
public:
    virtual ~RangeValue();
    //virtual bool isequal(const Value*) const;
    virtual bool isainteger() const;
    virtual bool isareal() const;
    virtual const Value* getmin() const;
    virtual const Value* getmax() const;
    virtual const Value* getinc() const;
private:
    // NOTE: ALL Value* BELOW ASSUMED TO BE SAME TYPE
    Value* min_;
    Value* max_;
    Value* inc_;  // INCREMENT VALUE
};

/***************************************************************************/

#endif/*excon_batch_value_h*/
