//
// logger.cc: implementation of logger
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1995 Mumit Khan
//

// ====================================================================== //

#include "logger.h"
#include <strstream.h>

// ====================================================================== //

//
// various compile time constants. I use enums instead of defines to
// help debugging.
//
enum {
    //
    // Max number of simultaneous streams per Logger object. Feel free to
    // up the number and recompile this module.
    //
    LOGGER_MAX_STREAMS = 10,
    //
    // Default buffer size used by internal strstream object in our 
    // custom streambuf.
    //
    LOGGER_DEFAULT_BUFSIZ = 2048
};

// ====================================================================== //

//
// private helper class to create the buffer passed to strstream.
//
class MakeLogBuf {
protected:
    MakeLogBuf(int bsize) : buffer_(new char[bsize]) { }

    ~MakeLogBuf() { delete[] buffer_; }
protected:
    char* buffer_;
};

// ====================================================================== //

//
// class LogBuf: The actual streambuf that we use in Logger. It basically
// cycles thru all the streams and outputs to it.
//
class LogBuf : protected MakeLogBuf, public strstreambuf {
public:
    LogBuf(int bufsize = LOGGER_DEFAULT_BUFSIZ);
    ~LogBuf();

    LogBuf& add(streambuf* stb);

protected:
    int sync();
    int overflow(int c);

private:
    int nstbs_;
    streambuf* stbs_[LOGGER_MAX_STREAMS];
};

// ====================================================================== //

LogBuf::LogBuf(int bufsize) : 
    MakeLogBuf(bufsize), 
    strstreambuf(buffer_, bufsize, buffer_),
    nstbs_(0)
{ }

LogBuf::~LogBuf() { 
    sync(); 
}

LogBuf& LogBuf::add(streambuf* stb) { 
    if (nstbs_ < LOGGER_MAX_STREAMS)
	stbs_[nstbs_++] = stb;
    return *this;
}

int LogBuf::sync() {
    for(int i = 0; i < nstbs_; ++i) {
	stbs_[i]->sputn(buffer_, out_waiting());
	stbs_[i]->sync();
    }
    setp(base(), ebuf());
    return 0;
}

int LogBuf::overflow(int c) {
    sync();
    sputc(c); 
    return 1;
}

// ====================================================================== //

//
// Logger implementation.
//
Logger::Logger(unsigned loglevel) : 
    ostream(logbufp_ = new LogBuf(LOGGER_DEFAULT_BUFSIZ)),
    loglevel_(loglevel)
{ }

Logger::Logger(ostream& os, unsigned loglevel) :
    ostream(logbufp_ = new LogBuf(LOGGER_DEFAULT_BUFSIZ)),
    loglevel_(loglevel)
{
    flags(os.flags());
    logbufp_->add(os.rdbuf());
}

Logger::~Logger() { 
    delete logbufp_; 
}

Logger& Logger::add(ostream& os) {
    logbufp_->add(os.rdbuf());
    return *this;
}

Logger& Logger::add(streambuf* stb) {
    logbufp_->add(stb);
    return *this;
}


// ====================================================================== //

