//
// exp.h: Interface to EXCON experiment manager
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//

#ifndef excon_batch_experiment_h
#define excon_batch_experiment_h

#include <sys/types.h>			// for time_t

/**************************************************************************/

class FreeVariable;
class FreeVariableGroup;
class Hashtable;
class Stq;
class ToolInstance;

/**************************************************************************/


/***************************************************************************
 *
 *  EXPERIMENTS
 *
 ***************************************************************************/

class Exp {
public:
    Exp();
    virtual ~Exp();
    virtual  bool expfileok() const;

    virtual  int buildexpseq(const char* exptext);
    virtual ToolInstance* gettoolinstance(
	const char* name, int iteration = 0
    ) const;
    virtual Stq* get_job_sequence(int iteration) const;
    virtual Stq* get_freevar_groups() const;

    virtual int setup_links(const Stq* sequence);
    virtual int resolvelinks(const Stq* sequence);
    virtual int resolveunique(const Stq* sequence);

    // experiment control loop
    virtual int setfvgroup(
       const char* tool, const char* var, const char* others, 
       const char* origline
    );
    virtual int setsetvar(
       const char* tool, const char* var, const char* valueset, 
       const char* origline
    );
    virtual int setloopvar(
       const char* tool, const char* var, const char* fromtext, 
       const char* totext, const char* incrtext, const char* origline
    );
    virtual int setrandomvar(
       const char* tool, const char* var, int nvalues,
       const char* fromtext, const char* totext, bool genseed, long seed,
       bool dosort, const char* origline
    );
    virtual FreeVariable* getfreevar(
	const char* tool, const char* var
    ) const;
    virtual FreeVariableGroup* getfvgroup(
	const char* tool, const char* var
    ) const;
    virtual int setcontroller(const char* tiname, int stopcode,
       const char* origline, bool negate = false
    );
    virtual int run();

    // reports
    virtual void reportresults(int future = 0) const;
    virtual void setoutput(const char* name);
    virtual void unlinkreports();
    virtual void settableheaders(const char* tablename, const char* headers);
    void settableseparator (const char* tablename, const char* separator);
    virtual void settablecolumns(const char* tablename, const char* columns);
    virtual void dump() const;

    virtual Stq* get_next_job(int iteration);
    virtual int getnruns() const;
    virtual int resolvelinks(ToolInstance* ti);

protected:

    virtual int gettoolnum(const char* name, int &pos) const;
    virtual void addtool(const char* name);    

    virtual void delete_job_sequence(int iteration);

    virtual void reportsetup();
    virtual void collectresults(int iteration);
    virtual void progresssetup(int nruns);
    virtual void progressreport(int runs);

    virtual void errormissingtool(const char* name);
    virtual void errormissingvar(const char* name, const char* toolname);
    virtual void errorforkfailed(const char* file);
    virtual void erroraddtool(const char* file);
    virtual void errorrunopen(const char* file);

private:

#ifdef RUN_MULTIPLE
    virtual char* genscript(const Stq* job, int iteration);
#endif /*RUN_MULTIPLE*/

protected:
    // EXPERIMENT
    Stq* expseq_;		// TOOLS IN EXEC SEQUENCE: LIST OF ToolInstance
    Stq* joblist_;		// JOB SEQUENCE: LIST OF (List of ToolInstance)
    // FREE VARIABLES
    Stq* fvgroups_;		// LIST OF FreeVarGroup*
    // Outside control
    char* controller_;		// controlling program (eg., optimizer1)
    int stop_code_;		// stop code
    bool negate_code_;		// equal to or not.
    // PROGRESS
    time_t prstart_;		// TIME EXPERIMENT STARTED
    time_t prnext_;		// TIME OF NEXT RUN
    time_t prestimate_;		// ESTIMATED TIME TOTAL RUN TO COMPLETE
    int prnruns_;		// TOTAL NUMBER OF RUNS
    // REPORTS
    Stq* reports_;		// TABULAR REPORTS IN A LIST
    // STATE DATA
    char* expfile_;		// NAME OF EXPERIMENT FILE
    char* toolsfile_;		// NAME OF TOOLS FILE
};

/***************************************************************************
 *
 *  EXPERIMENT MANAGER
 *
 ***************************************************************************/


class ExpMgr {
public:
    ExpMgr();
    virtual ~ExpMgr();
    virtual int load(const char* path, int nesting = 0); 
    virtual void setoutput(const char* name);
    virtual void unlinkreports();
    virtual int run();
    virtual void reportresults();
    virtual ToolInstance* gettoolinstance(
	const char* tname, int iteration = 0
    ) const;
    virtual Stq* get_job_sequence(int iteration) const;
    virtual Stq* get_freevar_groups() const;
protected:
    virtual void errorfilenotfound(const char* path);
    virtual void setsystemparams(const char* param, const char* value);

protected:
    Exp* exp_;			// THE EXPERIMENT
};

#endif/*excon_batch_experiment_h*/
