/*
 * win32_registry.c: Get win32 registry values for SHADOW.
 */

#ifndef _WIN32
# error win32_registry.c: This file should only be compiled for WIN32 platform.
#endif

#ifndef SHADOW_VERSION
# error win32_registry.c: Must define SHADOW_VERSION string.
#endif

#include <windows.h>
#include <stdlib.h>
#include <string.h>

#define WIN32_GET_REGISTRY_VALUE win32_get_registry_value__

static HKEY key;
static int first = 1;

/*
 * win32_get_registry_value_c: The C interface to SHADOW registry entry.
 * 
 * returns:
 *    0 : all ok
 *    1 : major key not found
 *    2 : string value not found
 *    3 : other error
 */
int win32_get_registry_value_c (const char* var, char* value, 
    size_t max_len) 
{
    DWORD size;

    if (first) {
      const char *major_key = "Software\\CXrL\\SHADOW\\" SHADOW_VERSION;
      if (RegOpenKeyEx (HKEY_LOCAL_MACHINE,
      			major_key,
			0,
			KEY_READ, &key) != ERROR_SUCCESS)
      {
	key = INVALID_HANDLE_VALUE;
        return 1;
      }
      first = 0;
    }
    if (key == INVALID_HANDLE_VALUE)
      return 1;

    if (RegQueryValueEx(key, var, NULL, NULL, NULL, &size) == ERROR_SUCCESS) {
      if (size >= max_len)
	return 3;
      RegQueryValueEx(key, var, NULL, NULL, value, &size);
    } else {
      return 2;
    }

#ifdef __CYGWIN32__
    if (strchr(value, '\\')) {
      char *p;
      for (p = value; p && *p; ++p)
        if (*p == '\\')
	  *p = '/';
    }
#endif

    return 0;
}

/*
 * win32_get_registry_value: The FORTRAN interface to SHADOW registry entry.
 * 
 * returns:
 *    0 : all ok
 *    1 : major key not found
 *    2 : string value not found
 *    3 : other error
 */
int WIN32_GET_REGISTRY_VALUE (const char* var, char* value, 
    unsigned long varlen, unsigned long vallen) 
{
    char *cvalue;
    char *cvar;
    int i;
    int retcode;

    memset(value, ' ', vallen);

    if (!(cvar = (char*)malloc(varlen+1)) || 
        !(cvalue = (char*)malloc(vallen+1))) 
    {
      return 3;
    }
    /* fortran to C string */
    for(i = varlen; var[i-1] == ' '; --i)
	;
    strncpy(cvar, var, i);
    cvar[i] = '\0';

    if ((retcode = win32_get_registry_value_c (cvar, cvalue, vallen)) == 0)
    {
      /* C to fortran string */
      strcpy(value, cvalue);
      i = strlen(cvalue);
      memset(value + i, ' ', vallen - i);
    }

    free(cvar);
    free(cvalue);
    return retcode;
}
