/*
 * screen.c: curses routines for chval
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * ------------------------------------------------
 * Mumit Khan
 * Center for X-ray Lithography
 * University of Wisconsin-Madison
 * 3731 Schneider Dr., Stoughton, WI, 53589
 * ------------------------------------------------
 *
 */

/* 
 * Longer description and stuff.
 */

#ifdef decstation
#include <cursesX.h>
#else
#include <curses.h>
#endif
#include <string.h>
#include <common.h>
#include "chval.h"

#include <libcxrl.h>

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif


#define PROMPT_HEIGHT 	(1)
#define MSG_HEIGHT 	(7)
#define TITLE_HEIGHT 	(1)
#define EMPTY		(2)
#define LIST_HEIGHT	(LINES-TITLE_HEIGHT-MSG_HEIGHT-PROMPT_HEIGHT-EMPTY)


#define TITLE_WIDTH	(COLS)
#define MSG_WIDTH 	(COLS)
#define LIST_WIDTH	(COLS)
#define PROMPT_WIDTH	(COLS)


#define MAX(a,b)	((a) > (b) ? (a) : (b))
#define MIN(a,b)	((a) < (b) ? (a) : (b))


#define SCROLL_DELTA	(6)

#define EQPOSITION	55


WINDOW *title_win = NULL;
WINDOW *list_win = NULL;
WINDOW *msg_win = NULL;
WINDOW *prompt_win = NULL;


static int list_top = 0;
static int list_bot = 0;
static int list_sel = 0;
static int list_height = 0;

static int cur_sel = 0;


static char **list_items = NULL;
static unsigned int list_size = 0;


char *blank = NULL;


static char tmpbuf[BUFSIZ];

int init_screen () {
    
    extern WINDOW *title_win;
    extern WINDOW *list_win;
    extern WINDOW *msg_win;
    extern WINDOW *prompt_win;

    int i = 0;

    extern int LINES, COLS;

    if (initscr () == (WINDOW *) ERR) {
	return -1;
    }
    title_win = newwin (1, TITLE_WIDTH, 0, 0);
    list_win = newwin (LIST_HEIGHT, LIST_WIDTH, TITLE_HEIGHT + 2, 0);
    msg_win = newwin (MSG_HEIGHT, MSG_WIDTH, TITLE_HEIGHT + LIST_HEIGHT + 3, 0);
    prompt_win = newwin (PROMPT_HEIGHT, PROMPT_WIDTH, LINES - 1, 0);
    keypad (prompt_win, TRUE);

#if defined(sun)
    noecho ();
#endif

    list_height = LIST_HEIGHT;

    blank = xmalloc (COLS+1);
    for (i = 0, blank[COLS-1] = '\0'; i < COLS-1; blank[i++] = ' ')
	;
    return (title_win && list_win && msg_win && prompt_win) ? 0 : -1;
}


int end_screen () {

    return endwin ();
}


int init_list (ch_size)
unsigned int ch_size;
{
    extern char **list_items;
    extern unsigned int list_size;

    list_items = (char **) xmalloc (ch_size * sizeof (char *));
    list_size = ch_size;

    return (list_items) ? 0 : -1;
}


int add_item_to_list (index, synopsis, shortname, value)
unsigned int index;
char *synopsis;
char *shortname; 
char *value;
{
    extern char **list_items;
    extern unsigned int list_size;

    static char buf[BUFSIZ];
    char *str = xmalloc (COLS + 1);

    unsigned int len = 0;

    if (index >= list_size || index < 0)
	return -1;
    
    /* I'm assuming the synopsis and shortname are really short */
    sprintf (str, "%2d.[%s] %s ", index+1, shortname, synopsis);
    if ((len = strlen (str)) >= EQPOSITION) {
	str[EQPOSITION-2] = str[EQPOSITION-3] = str[EQPOSITION-4] = '.';
    }
    else {
	int i = len;
	for (; i < EQPOSITION-1; str[i++] = ' ')
	    ;
    }

    str[EQPOSITION-1] = '\0';
    strcat (str, "= ");
    if ((len = strlen (value)) >= COLS - (EQPOSITION+2)) {
	strncpy (buf, value, COLS-(EQPOSITION+2));
	buf[COLS-(EQPOSITION+2)] = '\0';
    }
    else {
	strcpy (buf, value);
    }

    strcat (str, buf);
    for (len = strlen (str), str[COLS-1] = '\0'; len < COLS-1; str[len++] = ' ')
	;
    list_items [index] = str;
    return 0;
}

int display_list () {
    extern char **list_items;
    extern WINDOW *list_win;
    extern unsigned int list_size;
    
    if (!list_items || list_size == 0) {
	return -1;
    }
    else {
	int i = list_top;
	int j = 0;
	for (; i < list_bot; mvwaddstr (list_win,j,0,list_items[i]),i++,j++)
	    ;
    }
    wrefresh (list_win);
}


int init_list_viewport () {
    extern char **list_items;
    extern unsigned int list_size;
    
    if (!list_items || list_size == 0) {
	return -1;
    }
    else {
	list_top = 0;
	list_bot = MIN (LIST_HEIGHT, list_size) - 1;
    }

    return 0;
}


int set_cursel (index)
unsigned int index;
{
    extern int cur_sel;

    cur_sel = index;
    return 0;
}


int get_cursel ()
{
    extern int cur_sel;

    return cur_sel;
}


int highlight_item (index)
unsigned int index;
{
    wstandout (list_win);
    mvwaddstr (list_win, index - list_top, 0, list_items[index]);
    wstandend (list_win);
    wrefresh (list_win);
    return 0;
}
 

int unhighlight_item (index)
unsigned int index;
{
    mvwaddstr (list_win, index - list_top, 0, list_items[index]);
    wrefresh (list_win);
    return 0;
}
    

add_list_item (item, index)
char *item;
unsigned int index;
{
    extern WINDOW *list_win;
    extern char tmpbuf[];

    sprintf (tmpbuf, " %2d. %s", index, item);
    mvwaddstr (list_win, index, 0, item);
    return 0;
}


display_file_header (file, type)
char *file;
char *type;
{
    extern WINDOW *title_win;
    extern char tmpbuf[];

    sprintf (tmpbuf, " Namelist file: %s\t\t\t\tType: %s", file, type);
    mvwaddstr (title_win, 0, 0, tmpbuf);
    wrefresh (title_win);
    return 0;
}


int get_command_char () {
    return wgetch (prompt_win);
}


char *get_user_input () {
    static char buf[BUFSIZ];
    wgetstr (prompt_win, buf);
    (void) trim_leading_white (buf);
    return buf;
}


int display_msg () {
    extern WINDOW *msg_win;

    mvwaddstr (msg_win, 0, 0, "<Enter>: select\t\t</>:search\t\t<?>: help");
    mvwaddstr (msg_win, 1, 0, "<q>: quit\t\t<s>: save\t\t<x>: save and exit");
    mvwaddstr (msg_win, 2, 0, 
	"<^F,^B>: pgdn/pgup\t<^D,^U>: scroll dn/up\t<g>: go to line");
    wrefresh (msg_win);
}

int put_msg (msg) 
char *msg;
{

    extern WINDOW *msg_win;
    extern char *blank;

    mvwaddstr (msg_win, 3, 0, blank);
    mvwaddstr (msg_win, 3, 0, msg);
    wrefresh (msg_win);
}


int display_prompt (prompt) 
char *prompt;
{
    extern WINDOW *prompt_win;

    mvwaddstr (prompt_win, 0, 0, prompt); 
    wrefresh (prompt_win);
}


int clear_prompt () 
{
    extern char *blank;

    mvwaddstr (prompt_win, 0, 0, blank); 
    wrefresh (prompt_win);
}


int update_list (index, value)
int index;
char *value;
{
    extern WINDOW *list_win;
    extern char **list_items;

    char *str = list_items[index];
    char *ptr = strrchr (str, '=');

    int len = 0;

    strcpy (ptr += 2, value);
    for (len = strlen (str), str[COLS-1] = '\0'; len < COLS-1; str[len++] = ' ')
	;
    mvwaddstr (list_win, index-list_top, 0, blank);
    wstandout (list_win);
    mvwaddstr (list_win, index-list_top, 0, str);
    wstandend (list_win);
    wrefresh (list_win);

    return 0;
}


int move_sel (dir)
int dir;		/* +n for down, -n for up by n lines */
{
    extern WINDOW *list_win;

    extern int list_top;
    extern int list_bot;
    extern int list_sel;
    extern int list_height;

    extern int cur_sel;

    int incr = 0;

    if ((dir < 0 && cur_sel == 0) ||
	(dir > 0 && cur_sel == list_size-1))
	return;	
    
    unhighlight_item (cur_sel);
    cur_sel += dir;
    if (cur_sel < 0) {
	dir -= cur_sel;
	cur_sel = 0;
    }
    else if (cur_sel > list_size-1) {
	dir -= cur_sel - (list_size - 1);
	cur_sel = list_size - 1;
    }
    
    if ((cur_sel > list_bot && dir > 0) || 
	(cur_sel  < list_top && dir < 0)) {
	if (dir > 0) {
	    incr = MIN (MAX (SCROLL_DELTA, dir), list_size - list_bot - 1);
	}
	else {
	    incr = -MIN (MAX (SCROLL_DELTA, -dir), list_top + 1);
	}
	list_top += incr;
	list_bot += incr;

	display_list ();
    }
    highlight_item (cur_sel);

    return 0;
}


int make_sel_visible () {
    
    extern int list_top;
    extern int list_bot;

    extern int cur_sel;

    if (cur_sel >= list_top && cur_sel <= list_bot) {
        highlight_item (cur_sel);
    }
    else {
	center_on_cursel ();
    }

    return 0;
}


int center_on_cursel () {
    
    extern int list_top;
    extern int list_bot;
    extern int list_sel;
    extern int list_height;

    extern unsigned int list_size;

    extern int cur_sel;

    int newtop = cur_sel - list_height / 2;
    int newbot = MIN (newtop + list_height, list_size) - 1;

    if (newtop < 0) {
	newtop = 0;
	list_bot = MIN (list_height, list_size) - 1;
    }
    else if (newtop + list_height - 1 >= list_size) {
	newbot = cur_sel;
	newtop = newbot - list_height + 1;
    }
        

    list_top = newtop;
    list_bot = newbot;

    display_list ();
    highlight_item (cur_sel);

    return 0;
}


touch_all_windows () {
    
    extern WINDOW *title_win;
    extern WINDOW *list_win;
    extern WINDOW *msg_win;
    extern WINDOW *prompt_win;

    touchwin (title_win);
    touchwin (list_win);
    touchwin (msg_win);
    touchwin (prompt_win);
}
