/* 
 * groutines - a set of library routines for selectively reading parameters.
 *
 * Author: Mumit Khan <khan@xraylith.wisc.edu>
 *
 * Source: src/utils/io/groutines.c
 *
 * Log:	groutines.c
 * Revision 1.25  90/08/30  18:02:49  khan
 * About to add my own hashing routines.
 * 
 * Revision 1.24  90/05/28  10:17:46  khan
 * Hashing added, but trouble ahead.
 * 
 * Revision 1.1        
 * date: 89/09/30 17:26:05;  author: khan;  state: Exp;  
 * Initial Revision
 *
 */

#include "groutines.h"

STATIC extract_int (); 
STATIC extract_double (); 
STATIC fill_buf ();
STATIC install_node ();
STATIC cleanup ();
STATIC buffer_read ();

/* #define DO_CASE_SENSITIVE */

static int gfile_open = 0;
static char gfile_name[BUFSIZ]; 

/*------------------- FORTRAN INTERFACE BEGINS -------------------*/

PUBLIC F77_NAME(gset) (code) 
unsigned int *code;
{
    unsigned int new_code = *code;
    if (new_code & DEFAULT) {
	new_code ^= CASE;
	new_code |= NOCASE;
    }
    g_set (new_code);
}


PUBLIC F77_NAME(gabort) () {
    return abort_on_error = TRUE;
}

PUBLIC F77_NAME(gnoabort) () {
    return abort_on_error = FALSE;
}

/*
 * Make the query and options in the database case INsensitive. This is
 * the default behaviour for FORTRAN interface. Use should use gnocase ()
 * for the insensitive behaviour.
 */

PUBLIC F77_NAME(gnocase) () {
    return (int) (case_sense = insensitive);
}

/*
 * Make searches case-Sensitive. Default for C, but NOT FORTRAN.
 */

PUBLIC F77_NAME(gcase) () {
    return (int) (case_sense = sensitive);
}

/*
 * Print out the symbol table for debugging info.
 */

PUBLIC F77_NAME(gprint) () {
    return pr_sym_table ();
}

/*
 * Print out the user query and option value returned at each call.
 */

PUBLIC F77_NAME(gdebug) () {
    return (int) (usr_debug = yes);
}

PUBLIC F77_NAME(gnodebug) () {
    return (int) (usr_debug = no);
}

/*
 * Peek into the stored table to see if the parameter exists or not.
 * This is to check if the parameter is specifiec w/out having been
 * prompted for it.
 */

PUBLIC int F77_NAME(gpeek) (opt, opt_len)
char *opt;
unsigned long int opt_len;
{
    char *opt_name = fort_str (opt, opt_len);
    int ret = g_peek (opt_name);

    xfree (opt_name);
    return ret;
}


/*
 * Get a string type parameter from the stored database.
 */

PUBLIC F77_NAME(gstring)(key, uprompt, data, key_len, prompt_len, data_len)
char *key, *uprompt, *data; 
unsigned long key_len, prompt_len, data_len;
{
    register hbucket match;
    register char *user_query = fort_str (key, key_len);

    set_intr_handler ();

    interface = f_lang;          /* What language interface? */

    strcpy (routine, "gstring");

    if (!buffer_read ())
	fill_buf ();

    match = match_opt (user_query);

/*
 * allow null strings for SHADOW variables.
 */
    if (match == (hbucket) NULL) {
	fix_fort_str(prompt(uprompt), data, data_len); 
    } else if (!*match->data) {
	fix_fort_str("", data, data_len);
    } {
	fix_fort_str(match->data, data, data_len);
    }


    if (usr_debug == yes) {
	fprintf (stderr, "(%s)\t%s\t=\t%s\n", 
	    routine, user_query, (match) ? match -> data : data);
    }

    xfree (user_query);
    return;	
}

/*
 *  Get a REAL*8 type parameter from the stored database.
 */

PUBLIC double F77_NAME(gnumber)(key, uprompt, data, key_len, prompt_len)
char *key, *uprompt; 
double *data;
unsigned long key_len, prompt_len;
{
    register hbucket match;
    register char *user_query = fort_str (key, key_len);
    char *val = NULL;
    double value;

    set_intr_handler ();
	
    interface = f_lang;          /* What language interface? */
    strcpy (routine, "gnumber");

    if (!buffer_read ())
	fill_buf ();
 	   
    match = match_opt (user_query);

    if ((match == (hbucket) NULL) || 
	(!extract_double (match -> data, key, &value))) {
	while (!extract_double (val = prompt (uprompt), key, &value)) 
	    ;
	    	/* keep on prompting until user gives correct input */
	update_value (key, val);
    }
    
   
    bcopy ((char *) &value, data, sizeof (double));

    if (usr_debug == yes) {
	fprintf (stderr, "(%s)\t%s\t=\t%f\n", 
	    routine, user_query, value);
    }

    xfree (user_query);
    return value;	
}

PUBLIC int F77_NAME(gint)(key, uprompt, data, key_len, prompt_len)
char *key, *uprompt;
int *data;
unsigned long key_len, prompt_len;
{
    register hbucket match;
    register char *user_query = fort_str (key, key_len);
    char *val = NULL;
    int value;

    set_intr_handler ();
	
    interface = f_lang;          /* What language interface? */
    strcpy (routine, "gint");

    if (!buffer_read ())
	fill_buf ();
 	   
    match = match_opt (user_query);

    if ((match == (hbucket) NULL) || 
	(!extract_int (match -> data, key, &value))) {
	while (!extract_int (val = prompt (uprompt), key, &value)) 
	    ;
	update_value (key, val);
    }


    bcopy ((char *) &value, data, sizeof (int));

    if (usr_debug == yes) {
	fprintf (stderr, "(%s)\t%s\t=\t%d\n", 
	    routine, user_query, value);
    }

    xfree (user_query);
    return value;	
}

PUBLIC F77_NAME(addcmdarg) (argument, len)
char* argument;
unsigned long len;
{
    char* argv[2];
    char *arg = fort_str (argument, len);
    argv[0] = arg;
    argv[1] = NULL;
 
    interface = f_lang;          /* What language interface? */
    strcpy (routine, "addcmdarg");

    g_cmdline (1, argv);
    xfree (arg);
    return;
}


/*
 * do all i/o from file 'fname'.
 */

PUBLIC F77_NAME(gfile) (fname, len)
char *fname;
unsigned long len;
{
    char *file = fort_str (fname, len);
 
    interface = f_lang;          /* What language interface? */
    strcpy (routine, "gfile");

    g_file (file);
    xfree (file);
    return;

}

PUBLIC F77_NAME(gclose) ()
{

    g_close ();
}


PUBLIC F77_NAME(gcleanup) () {
    return cleanup ();
}

/*------------------- FORTRAN INTERFACE ENDS -------------------*/


/*------------------- C INTERFACE BEGINS -------------------*/

PUBLIC int g_set (code) 
unsigned int code;
{
    if (code & CASE) {
	g_case ();
    }
    if (code & NOCASE) {
	g_nocase ();
    }
    if (code & DEBUG) {
	g_debug ();
    }
    if (code & NODEBUG) {
	g_nodebug ();
    }
    if (code & ABORT) {
	g_abort ();
    }
    if (code & NOABORT) {
	g_noabort ();
    }
}


PUBLIC g_abort () {
    return abort_on_error = TRUE;
}

PUBLIC g_noabort () {
    return abort_on_error = FALSE;
}

/*
 * Make the query and options in the database case sensitive. This is
 * the default behaviour for C interface. Use should use gnocase ()
 * for the insensitive behaviour.
 */

PUBLIC g_case () {
    return (int) (case_sense = sensitive);
}

/*
 * Make searches case-insensitive. Default for FORTRAN, but NOT C.
 */

PUBLIC g_nocase () {
    return (int) (case_sense = insensitive);
}

/*
 * Print out the symbol table for debugging info.
 */

PUBLIC g_print () {
    return pr_sym_table ();
}

/*
 * Print out the user query and option value returned at each call.
 */

PUBLIC g_debug () {
    return (int) (usr_debug = yes);
}

PUBLIC g_nodebug () {
    return (int) (usr_debug = no);
}


/* 
 * Peek, but don't squeak. Return a 0 if !found, non-zero otherwise.
 */

PUBLIC int g_peek (query)
char *query;
{
    if (!buffer_read ())
	fill_buf ();
    return (match_opt (query)) ? 1 : 0;
}
    

PUBLIC char *g_string (query, uprompt, result)
char *query, *uprompt, *result;
{
    register hbucket match;

    set_intr_handler ();

    strcpy (routine, "g_string");

    if (!buffer_read ())
	fill_buf ();
 	   
    match = match_opt (query);

/*
 * allow null strings for SHADOW variables.
 */
    if (match == (hbucket) NULL) {
	strcpy(result, prompt(uprompt));
    } else if (!*match->data) {
	strcpy(result, "");
    } else {
	strcpy(result, match->data);
    }
    return result;
}

/*
 *  Get a FLOAT type parameter from the stored database.
 */

PUBLIC double g_number(query, uprompt, result)
char *query, *uprompt;
double *result;
{
    register hbucket match;
    char *val = NULL;
    double value;

    set_intr_handler ();

    strcpy (routine, "g_number");

    if (!buffer_read ())
	fill_buf ();
 	   
    match = match_opt (query);

    if ((match == (hbucket) NULL) || 
	(!extract_double (match -> data, query, &value))) {
	while (!extract_double (val = prompt (uprompt), query, &value)) 
	    ;
	update_value (query, val);
    }
    bcopy ((char *) &value, result, sizeof (double));
   
    return value;
}

PUBLIC int g_int(query, uprompt, result)
char *query, *uprompt;
int *result;
{
    register hbucket match;
    char *val = NULL;
    int value;

    set_intr_handler ();

    strcpy (routine, "g_int");

    if (!buffer_read ())
	fill_buf ();
 	   
    match = match_opt (query);
    if ((match == (hbucket) NULL) || 
	(!extract_int (match -> data, query, &value))) {
	while (!extract_int (val = prompt (uprompt), query, &value)) 
	    ;
	update_value (query, val);
    }
    bcopy ((char *) &value, result, sizeof (int));

    return value;
}


PUBLIC g_cmdline(argc, argv)
int argc;
char** argv;
{
    char* eq = 0;
    int i = 0;
    for (; i < argc; i++) {
        char tmpbuf[1024];
        strcpy (tmpbuf, argv[i]);
        if (eq = strchr (tmpbuf, '=')) {
            char* name = 0;
            char* value = 0;
            hbucket new = (hbucket) NULL;
 
            value = strcpy ((char*)malloc (strlen (eq+1) + 1), eq+1);
            *eq = '\0';
            name = strcpy ((char*)malloc (strlen (tmpbuf) + 1), tmpbuf);
            lo_case (name);
 
            new = (hbucket) xmalloc (sizeof (hbucket_rec));
            new->key = name;
            new->data = value;
            if (!hash_table_inited) {
                hash_table_inited = TRUE;
                init_hash_table (HASH_TABLE_SIZE);
            }
            install_node(new);
        }
    }
}


PUBLIC g_file (fname)
char *fname;
{

    strcpy (routine, "gfile");

    if (gfile_open) {
	error ("g_file: one g_file \"%s\" already open. Use g_close() first",
	    gfile_name);
	exit (1);
    }
    else {
	gfile_open = 1;
    }
    strcpy (gfile_name, fname);
    if ((istream = fopen (fname, "r")) == (FILE *) NULL) {
	perror (fname);
	exit (1);
    }

    if (!buffer_read ())
	fill_buf ();
}

PUBLIC g_close ()
{

    if (!gfile_open) {
	error ("g_close: no files currently open to close");
	exit (1);
    }
    else {
	gfile_open = 0;
    }
    fflush (istream);
    if (close (fileno (istream))) {
	error ("g_close: Error closing g_file");
	exit (-1);
    }

    return cleanup ();
}

PUBLIC int g_cleanup () {
    return cleanup ();
}


/*------------------- C INTERFACE ENDS -------------------*/

/*
 * This routine is called to prompt the user to enter the appropriate
 * value when a parameter being sought is not found in the symbol table.
 * Note that this routine reads from /dev/tty directly since in case the
 * user redirected the input, gets() will return EOF when prompting.
 *
 * Unless you're *familiar* with Unix I/O, leave this routine
 * alone.
 */


STATIC char *prompt (prompt_str)
char *prompt_str;
{
    static char uinput [MAXLINELEN];
    static unsigned int err_cnt = 0;

    if ((tty = open (TTY_DEV, O_RDWR, NULL)) == -1) {
	error ("%s: Can't Open %s\n", routine, TTY_DEV);
	exit (1);
    }
    file_open = TRUE;

    close (0); dup (tty); 
    fprintf (stderr, "%s", prompt_str);
    fflush (stderr);
    while (TRUE) {
	if (!gets (uinput)) {
	    fprintf (stderr, "** Error Reading Input **\n");
	    fprintf (stderr, "%s", prompt_str);
	    fflush (stderr);
	    if (++err_cnt > MAXTRY) {
		error ("Error Reading Input Too Many Times\n");
		close (tty);
		exit (1);
	    }
	    continue;
	}
	else {
	    err_cnt = 0;
	    break;
	}
    }
    close (tty); 
    file_open = FALSE;

    return uinput;
}

/* 
 * Extract the integer value from str and return the pointer in ival.
 * opt_name is provided to report errors.
 */

STATIC extract_int (str, opt_name, ival) 
char *str, *opt_name;
int *ival;
{
    int ret_code = TRUE;

    if (!*str)
	ret_code = FALSE;
    else if (!sscanf (str, "%d", ival)) {
	fprintf (stderr, "** %s : Illegal Integer value **\n", opt_name);
	ret_code = FALSE;
    }
    return ret_code;
}


/* 
 * Extract the double value from str and return the pointer in fval.
 * opt_name is provided to report errors.
 */

STATIC extract_double (str, opt_name, fval) 
char *str, *opt_name;
double *fval;
{
    register int ret_code = TRUE;

    if (!*str)
	ret_code = FALSE;
    else if (!sscanf (str, "%le", fval)) {
	fprintf (stderr, "** %s : Illegal Floating Point Value **\n",
	    opt_name);
	ret_code = FALSE;
    }
    return ret_code;
}

/* 
 * gettokens - parses the tokens out of a character string.
 * The delimiter of the tokens is assumed to SEPARATOR (groutines.h),
 * and if not found, return FALSE, else returns TRUE. 
 *
 */

STATIC gettokens (line, node)
char *line;
hbucket node;
{
    unsigned int len_key, len_data;
    register char *value, *option, *equal;
    char *comment;
    int comment_len = 0;
    int line_len = strlen (line);

    if (!(equal = strchr (line, SEPARATOR)))
	return FALSE;	
    
    if ((comment = strchr (equal+1, COMMENT)) && 
	(*(comment-1) != ESCAPE)) { 
	comment_len = strlen (comment); 	/* in-line comment */
    }

    len_key = line_len - strlen (equal-1) + 1;
    len_data = line_len - len_key - comment_len - 1; 

    option = strncpy ((char *) xmalloc (len_key + 1), line, len_key);
    *(option+len_key) = '\0';
    value = strncpy ((char *) xmalloc (len_data + 1), equal+1, len_data);
    *(value+len_data) = '\0';

    option = trim_trailing_white (option);
    value = trim_leading_white (value);
    if (!*value && usr_debug == yes) {
	fprintf (stderr, "** Warning : No value given for %s **\n", option);
    }

    node -> key = option;
    node -> data = value;

    return TRUE;
}

static int buf_read = FALSE;

STATIC buffer_read () {
    return (buf_read);
}

STATIC reset_buffer_read () {
    return buf_read = FALSE;
}

/*
 * If the input is read from a file or being redirected into, then read
 * in the whole file and build the symbol table. Each successive call to
 * g-routine then simply searches the table and returns the corresponding
 * value. This is explicitly called with a the gfile()_ interface, and
 * implicitly called with the first g-routine.
 */

STATIC fill_buf () {

    register hbucket new = (hbucket) NULL;
				/* The new buffer entry to entered */
    char *line_ptr = &line[0];  /* The line to be read in */
    register int line_no;       /* Current line number in script */
    int do_case_sense = FALSE;  /* Treat as case sensitive? */
    int num_symbols = 0;        /* how many symbols */

    if (!hash_table_inited) {
	hash_table_inited = TRUE;
	init_hash_table (HASH_TABLE_SIZE);
    }

    /* if no redirection, then don't wait for the user */
#if defined(__CYGWIN32__)
    if (isatty (fileno (stdin)))
#else
    if (isatty (fileno (stdin)) && (istream == stdin))
#endif
	return;			
    line_no = 0;

    if (buf_read)
	error ("Buffer already read");
    else
	buf_read = TRUE;


    while (fgets(line_ptr, BUFSIZ - 1, istream )){
	*(line_ptr + strlen (line_ptr) - 1) = '\0';  /* kill the newline */
	line_ptr = trim_leading_white (line_ptr);
	line_no++;
	if (!*line_ptr || *line_ptr == COMMENT) {
	    continue;
	}
	line_ptr = trim_trailing_white (line_ptr);

	new = (hbucket) xmalloc (sizeof (hbucket_rec));
	if (!gettokens (line_ptr, new)) {
	    error ("Error in file %s line %d: No '%c' in assignment", 
		gfile_name, line_no, SEPARATOR);
	    if (abort_on_error) 
		exit (1);
	    else
	        continue;
	}
	
#ifdef DO_CASE_SENSITIVE
	/* Are we "case-sensitive" today? */
	if (case_sense == default_case) {
	    do_case_sense = (interface == c_lang) ? TRUE : FALSE;
	}
	else {
	    do_case_sense = (case_sense == sensitive) ? TRUE : FALSE;
	}
#else
	do_case_sense = FALSE;
#endif

	new -> key = (do_case_sense) ? new -> key: lo_case (new -> key);

	install_node (new);
	num_symbols++;
    }

}

/* The list containing the queries and the corresponding values */
static OPTIONS *head = (OPTIONS *) NULL;


/*
 * Install the node in the symbol table. The symbols are stored in a
 * hash table and the names are also stored in a linked list so they
 * can be retrieved.
 */

STATIC install_node (node)
hbucket node;
{
    
    OPTIONS *option = (OPTIONS *) xmalloc (sizeof (OPTIONS)); 
    hbucket bucket = NULL;

    if (bucket = find_hitem (hash_table, node -> key)) {
	warn ("Multiple definition of %s. Overriding previous value",
	    node -> key);
	bucket -> data = node -> data;
    }
    else {
	if (!add_hitem (hash_table, node -> key, node -> data, 
	    sizeof (node -> data))) {
	    error ("Could not enter into hash table");
	    exit (1);
	}
	option -> key = node -> key;
	option -> next = head;
	head = option;
    }
}

STATIC clear_option_list () {
    head = NULL;
}

update_value (key, data)
char *key;
char *data;
{

    hbucket bucket = (hbucket) xmalloc (sizeof (hbucket_rec));
    static char query[132];
    int do_case_sense = FALSE;

#ifdef DO_CASE_SENSITIVE
    /* Are we "case-sensitive" today? */
    if (case_sense == default_case) {
	do_case_sense = (interface == c_lang) ? TRUE : FALSE;
    }
    else {
	do_case_sense = (case_sense == sensitive) ? TRUE : FALSE;
    }
#else
	do_case_sense = FALSE;
#endif
	
    strcpy (query, key);
    if (!do_case_sense) {
	lo_case (query);
    }

    bucket -> key = query;
    bucket -> data = (char *) 
	strcpy ((char *) xmalloc (strlen (data)+1), data);

    install_node (bucket);
}


/*
 * Create and initialize the hash table.
 */
init_hash_table (num_entries)
unsigned int num_entries;
{

    if (!(hash_table = make_htable (num_entries, ALLOCATE_NONE))) {
	error ("%s: Couldn't allocate room for hash table", routine);
	exit (1);
    }
}


/* 
 * Enumerate the symbol table. This is accessed by the public interface
 * gprint (), supplied for user debugging purposes. 
 *
 * Note the output is written to stderr.
 */
pr_sym_table () {
    
    register OPTIONS *b_ptr = head;
    hbucket bucket = NULL;

    fprintf (stderr, "\nSymbol table contents:\n");
    while (b_ptr) {
	if (!(bucket = find_hitem (hash_table, b_ptr -> key))) {
	    error ("Internal Error: inconsistency between hash table and option list");
	    exit (1);
	}
	fprintf (stderr, "\t%s\t=\t%s\n", 
	    bucket -> key, bucket -> data);
	b_ptr = b_ptr -> next;
    }
    fprintf (stderr, "\n\n");
    fflush (stderr);
}

/*
 * Due to temporary bugs, this is disabled 
 */

STATIC cleanup () {
    
    register OPTIONS *p = head;
    register OPTIONS *q = head;

    /* Clean up the list containing the option names. */
    while (p) {
	if (p -> key)
	    xfree (p -> key);
	q = p;
	p = p -> next;
	xfree (q);
    }

    head = NULL;

    /* and the hash table we don't need anymore */

    delete_htable (hash_table);
    hash_table = NULL;
    reset_buffer_read ();
    hash_table_inited = FALSE;
}


/*
 * Match the user query with the entries in the table. If there is a 
 * match, it returns the symbol table entry, else it returns NULL.
 * The search could be either case sensitive or insensitive.
 */

STATIC hbucket match_opt (opt)
char *opt;
{
    char query [MAXLINELEN];
    int do_case_sense = FALSE;
    hbucket_rec hash_query;

#ifdef DO_CASE_SENSITIVE
    /* Are we "case-sensitive" today? */
    if (case_sense == default_case) {
	do_case_sense = (interface == c_lang) ? TRUE : FALSE; 
    }
    else {
	do_case_sense = (case_sense == sensitive) ? TRUE : FALSE;
    }
#else
	do_case_sense = FALSE;
#endif
	
    strcpy (query, opt);
    if (!do_case_sense) {
	lo_case (query);
    }

    hash_query.key = &query[0];

    
    return find_hitem (hash_table, query);
}

/*
 * Optionally set the interrupt handler for the g-routines. Could be much
 * better, but it's better than nothing.
 */
set_intr_handler () {
#ifdef INT_HANDLER
    init_signal ();
    setjmp (env);
    if (intr) {
        intr = FALSE;
        return;	
    }
#endif /* INT_HANDLER */
}

#ifdef INT_HANDLER
STATIC init_signal () {

    int onintr ();

    if (signal (SIGINT, SIG_IGN) != SIG_IGN)
	signal (SIGINT, onintr);
}

STATIC onintr () {

    char response [81];

    fprintf (stderr, "\nInterrupt in routine %s\n", routine);
    fprintf (stderr, "Quit, Continue, or return to calling routine [qcr] ? ");
    if (!gets (response)) {
	fprintf (stderr, "Response read error\n");
	exit (1);
    }
    switch (response[0]) {
	case 'c' :
	case 'C' :
	    signal (SIGINT, onintr);	/* re-install signal for SYSV */
	    return;
	    break;
	case 'r' :
	case 'R' :
	    intr = TRUE;
	    if (file_open) {
		close (tty);
	    }
	    signal (SIGINT, onintr);	/* re-install signal for SYSV */
	    longjmp (env, 0);
	    break;
	default:
	    signal (SIGINT, SIG_DFL);	/* re-install signal for SYSV */
	    exit (1);
	    break;
    }

}

#endif  /* INT_HANDLER */
