//
// utils.h: simple utility classes (linked list and hashing functions).
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#ifndef excon_batch_utils_h
#define excon_batch_utils_h

/****************************************************************************
 *
 *  INCLUDES
 *
 ****************************************************************************/

#include "xrtypes.h"

/****************************************************************************
 *
 *  ASSERTIONS
 *
 ****************************************************************************/

#ifdef __STDC__
# define assume(EX) if (EX) ; else assume_fail(#EX,__FILE__,__LINE__)
#else
# define assume(EX) if (EX) ; else assume_fail("EX",__FILE__,__LINE__)
#endif

/****************************************************************************
 *
 *  STQ: STORE LINKED LISTS OF ITEMS...ADD/RETRIEVE FROM FRONT OR REAR
 *
 ****************************************************************************/
class Stqitem {
public:
    void* item;
    Stqitem* next;
    Stqitem();
};

inline Stqitem::Stqitem() { item = next = nil; }

class Stq {
public:
    Stq();
    ~Stq();

    int size() const; 
    void* top() const;
    void push(void const* item);
    void* pop();
    void* get(int ith) const;
    void enq(void const* item);
    void insert(int, void const*);
    void* remove(int);
    void* cycle() const; 
    Stq* clone() const;
    void clear();
    void reset();

private:
    int nitems_;
    Stqitem* head_;
    Stqitem* tail_;
    Stqitem* free_;		// CACHE SO THAT SCANNING 
				// (pop followed by enq) HAD NO MEMORY IMPACT
};

inline void Stq::reset() { head_ = tail_ = free_ = nil;  nitems_ = 0; }
inline Stq::Stq() { reset(); }
inline Stq::~Stq() { clear(); }
inline int Stq::size() const { return nitems_; }
inline void* Stq::top() const  { return (head_ != nil) ? head_->item : nil; } 

/****************************************************************************
 *
 *  HASH TABLE
 *
 ****************************************************************************/

typedef	void* COPIER_FUNC_TYP(const void*);
typedef	COPIER_FUNC_TYP *COPIER_FP;

class Hashlink {
public:
    char const* key;
    void const* item;
    Hashlink* next;
    Hashlink(char const* key, void const* item, Hashlink* next = nil);
};

inline Hashlink::Hashlink(char const* k, void const* i, Hashlink* n) :
    key(k), item(i), next(n) { }


class Hashtable {
public:
    // PICK PRIME NUMBER FOR size: 9,17,53,89,107,131,223,383,569,769,941,1223
    Hashtable(int tablesize);
    ~Hashtable();
    int size() const;
    void clear();
    void insert(char const* key, void const* item);
    void* find(char const* key) const;
    void* remove(char const* key);           
    void* get(int ith) const;
    Hashtable* clone(COPIER_FP cloner = nil) const;
protected:
    int hash(char const* key) const;
    bool compare(char const* key1, char const* key2) const;
private:
    int nitems_;
    int tablesize_;
    Hashlink** table_;
};

inline int Hashtable::size() const { return nitems_; }


/*******************************************************************************
*
*  OTHER UTILITIES
*
*******************************************************************************/

extern int strsub (
    const char* replacethis, const char* withthis, const char* inthisline, 
    char* results
);
extern int expandpath(const char* orig_path, char path[]);
extern int search_file(
    const char* orig_path, const Stq* search_path, char path[]
);
extern int find_include_file(
    const char* orig_path, const Stq* search_path, char path[]
);
extern void getunusedpath(char* path, char* prefix, bool create);
extern char* getprogramname(char* argv[]);
extern int getflag(char flag, int argc, char* argv[]);
extern int getflagval(char flag, int argc, char* argv[], char* buf);
extern void promptforval(const char* prompt, char* buf);
extern bool fileexists(const char* path);

extern char* strcpy_with_alloc(const char* str);
extern void assume_fail (const char* x, const char* file, int line);

#endif/*excon_batch_utils_h*/
