//
// freevars.h: Interface to free variable (loops, sets) definition.
//
// ------------------------------------------------
// Mumit Khan <khan@xraylith.wisc.edu>
// Center for X-ray Lithography
// University of Wisconsin-Madison
// 3731 Schneider Dr., Stoughton, WI, 53589
// ------------------------------------------------
//
// Copyright (c) 1991-1996 Mumit Khan
//
//

#ifndef excon_batch_freevars_h
#define excon_batch_freevars_h

#include "xrtypes.h"		// for bool typedef.

/****************************************************************************/

class FreeVariable;
class FreeVariableGroup;
class Stq;
class Value;

/****************************************************************************/

class FreeVariableGroup {
public:
    FreeVariableGroup();
    ~FreeVariableGroup();
    void add(const FreeVariable*);
    void remove(const FreeVariable*);
    const Stq* getgroup() const;

    int getnruns() const;
    bool done() const;
    void increment();
    void reset();
private:
    Stq* group_;
};

inline const Stq* FreeVariableGroup::getgroup() const { return group_; }

/****************************************************************************/

//
//  FREE VARIABLES
//

class FreeVariable { 
public:
    enum FreeVariableType { unknown, loop, set, random };
public:
    virtual ~FreeVariable();
    virtual int getnruns() const;

    virtual const char* gettool() const;
    virtual const char* getvar() const;

    virtual const Value* getcurrent() const;

    virtual bool done() const;
    virtual void increment();
    virtual void reset();

    virtual FreeVariableType type() const = 0;

protected:
    FreeVariable(const char* tool, const char* var);
private:
    char* tool_;
    char* var_;
};

/**************************************************************************/

class LoopVariable : public FreeVariable {
public:
    LoopVariable(const char* tool, const char* var, Value* start, 
	Value* stop, Value* increment
    );
    virtual ~LoopVariable();
    virtual int getnruns() const;

    virtual const Value* getcurrent() const;

    virtual bool done() const;
    virtual void increment();
    virtual void reset();

    virtual FreeVariableType type() const;

    const Value* get_start() const;
    const Value* get_stop() const;
    const Value* get_increment() const;

private:
    Value* start_;
    Value* stop_;
    Value* increment_;
    Value* current_;
    bool done_;
};

inline const Value* LoopVariable::get_start() const { return start_; }
inline const Value* LoopVariable::get_stop() const { return stop_; }
inline const Value* LoopVariable::get_increment() const { return increment_; }

/**************************************************************************/

class SetVariable : public FreeVariable {
public:
    SetVariable(const char* tool, const char* var, Stq* values);
    virtual ~SetVariable();
    virtual int getnruns() const;

    virtual const Value* getcurrent() const;

    virtual bool done() const;
    virtual void increment();
    virtual void reset();

    virtual FreeVariableType type() const;

    const Stq* get_values() const;
private:
    Stq* values_;
    int counter_;
    bool done_;
};

inline const Stq* SetVariable::get_values() const { return values_; }

/**************************************************************************/

class RandomVariable : public FreeVariable {
public:
    RandomVariable(const char* tool, const char* var, long seed,
	Value* start, Value* stop, int nvalues, bool sorted = false
    );
    RandomVariable(const char* tool, const char* var, 
	Value* start, Value* stop, int nvalues, bool sorted = false
    );
    virtual ~RandomVariable();
    virtual int getnruns() const;

    virtual const Value* getcurrent() const;

    virtual bool done() const;
    virtual void increment();
    virtual void reset();

    virtual FreeVariableType type() const;

    const Value* get_start() const;
    const Value* get_stop() const;
    const int get_nvalues() const;

private:
    Value* create_random() const;

private:
    Value* start_;
    Value* stop_;
    int nvalues_;
    long seed_;
    int counter_;
    Value** values_;
    bool done_;
};

inline const Value* RandomVariable::get_start() const { return start_; }
inline const Value* RandomVariable::get_stop() const { return stop_; }
inline const int RandomVariable::get_nvalues() const { return nvalues_; }

/**************************************************************************/


#endif/*excon_batch_freevars_h*/
