'\"
'\" Copyright 1991-1997 by Lucent Technologies, Inc.
'\"
'\" Permission to use, copy, modify, and distribute this software and its
'\" documentation for any purpose and without fee is hereby granted, provided
'\" that the above copyright notice appear in all copies and that both that the
'\" copyright notice and warranty disclaimer appear in supporting documentation,
'\" and that the names of Lucent Technologies any of their entities not be used
'\" in advertising or publicity pertaining to distribution of the software
'\" without specific, written prior permission.
'\"
'\" Lucent Technologies disclaims all warranties with regard to this software,
'\" including all implied warranties of merchantability and fitness.  In no event
'\" shall Lucent Technologies be liable for any special, indirect or
'\" consequential damages or any damages whatsoever resulting from loss of use,
'\" data or profits, whether in an action of contract, negligence or other
'\" tortuous action, arising out of or in connection with the use or performance
'\" of this software.  
'\"
'\" Vector command created by George Howlett.
'\"
'\" The definitions below are for supplemental macros used in Tcl/Tk
'\" manual entries.
'\"
'\" .AP type name in/out ?indent?
'\"	Start paragraph describing an argument to a library procedure.
'\"	type is type of argument (int, etc.), in/out is either "in", "out",
'\"	or "in/out" to describe whether procedure reads or modifies arg,
'\"	and indent is equivalent to second arg of .IP (shouldn't ever be
'\"	needed;  use .AS below instead)
'\"
'\" .AS ?type? ?name?
'\"	Give maximum sizes of arguments for setting tab stops.  Type and
'\"	name are examples of largest possible arguments that will be passed
'\"	to .AP later.  If args are omitted, default tab stops are used.
'\"
'\" .BS
'\"	Start box enclosure.  From here until next .BE, everything will be
'\"	enclosed in one large box.
'\"
'\" .BE
'\"	End of box enclosure.
'\"
'\" .CS
'\"	Begin code excerpt.
'\"
'\" .CE
'\"	End code excerpt.
'\"
'\" .VS ?version? ?br?
'\"	Begin vertical sidebar, for use in marking newly-changed parts
'\"	of man pages.  The first argument is ignored and used for recording
'\"	the version when the .VS was added, so that the sidebars can be
'\"	found and removed when they reach a certain age.  If another argument
'\"	is present, then a line break is forced before starting the sidebar.
'\"
'\" .VE
'\"	End of vertical sidebar.
'\"
'\" .DS
'\"	Begin an indented unfilled display.
'\"
'\" .DE
'\"	End of indented unfilled display.
'\"
'\" .SO
'\"	Start of list of standard options for a Tk widget.  The
'\"	options follow on successive lines, in four columns separated
'\"	by tabs.
'\"
'\" .SE
'\"	End of list of standard options for a Tk widget.
'\"
'\" .OP cmdName dbName dbClass
'\"	Start of description of a specific option.  cmdName gives the
'\"	option's name as specified in the class command, dbName gives
'\"	the option's name in the option database, and dbClass gives
'\"	the option's class in the option database.
'\"
'\" .UL arg1 arg2
'\"	Print arg1 underlined, then print arg2 normally.
'\"
'\" RCS: @(#) $Id: man.macros,v 1.1.1.1 2002/04/02 03:50:11 khan Exp $
'\"
'\"	# Set up traps and other miscellaneous stuff for Tcl/Tk man pages.
.if t .wh -1.3i ^B
.nr ^l \n(.l
.ad b
'\"	# Start an argument description
.de AP
.ie !"\\$4"" .TP \\$4
.el \{\
.   ie !"\\$2"" .TP \\n()Cu
.   el          .TP 15
.\}
.ta \\n()Au \\n()Bu
.ie !"\\$3"" \{\
\&\\$1	\\fI\\$2\\fP	(\\$3)
.\".b
.\}
.el \{\
.br
.ie !"\\$2"" \{\
\&\\$1	\\fI\\$2\\fP
.\}
.el \{\
\&\\fI\\$1\\fP
.\}
.\}
..
'\"	# define tabbing values for .AP
.de AS
.nr )A 10n
.if !"\\$1"" .nr )A \\w'\\$1'u+3n
.nr )B \\n()Au+15n
.\"
.if !"\\$2"" .nr )B \\w'\\$2'u+\\n()Au+3n
.nr )C \\n()Bu+\\w'(in/out)'u+2n
..
.AS Tcl_Interp Tcl_CreateInterp in/out
'\"	# BS - start boxed text
'\"	# ^y = starting y location
'\"	# ^b = 1
.de BS
.br
.mk ^y
.nr ^b 1u
.if n .nf
.if n .ti 0
.if n \l'\\n(.lu\(ul'
.if n .fi
..
'\"	# BE - end boxed text (draw box now)
.de BE
.nf
.ti 0
.mk ^t
.ie n \l'\\n(^lu\(ul'
.el \{\
.\"	Draw four-sided box normally, but don't draw top of
.\"	box if the box started on an earlier page.
.ie !\\n(^b-1 \{\
\h'-1.5n'\L'|\\n(^yu-1v'\l'\\n(^lu+3n\(ul'\L'\\n(^tu+1v-\\n(^yu'\l'|0u-1.5n\(ul'
.\}
.el \}\
\h'-1.5n'\L'|\\n(^yu-1v'\h'\\n(^lu+3n'\L'\\n(^tu+1v-\\n(^yu'\l'|0u-1.5n\(ul'
.\}
.\}
.fi
.br
.nr ^b 0
..
'\"	# VS - start vertical sidebar
'\"	# ^Y = starting y location
'\"	# ^v = 1 (for troff;  for nroff this doesn't matter)
.de VS
.if !"\\$2"" .br
.mk ^Y
.ie n 'mc \s12\(br\s0
.el .nr ^v 1u
..
'\"	# VE - end of vertical sidebar
.de VE
.ie n 'mc
.el \{\
.ev 2
.nf
.ti 0
.mk ^t
\h'|\\n(^lu+3n'\L'|\\n(^Yu-1v\(bv'\v'\\n(^tu+1v-\\n(^Yu'\h'-|\\n(^lu+3n'
.sp -1
.fi
.ev
.\}
.nr ^v 0
..
'\"	# Special macro to handle page bottom:  finish off current
'\"	# box/sidebar if in box/sidebar mode, then invoked standard
'\"	# page bottom macro.
.de ^B
.ev 2
'ti 0
'nf
.mk ^t
.if \\n(^b \{\
.\"	Draw three-sided box if this is the box's first page,
.\"	draw two sides but no top otherwise.
.ie !\\n(^b-1 \h'-1.5n'\L'|\\n(^yu-1v'\l'\\n(^lu+3n\(ul'\L'\\n(^tu+1v-\\n(^yu'\h'|0u'\c
.el \h'-1.5n'\L'|\\n(^yu-1v'\h'\\n(^lu+3n'\L'\\n(^tu+1v-\\n(^yu'\h'|0u'\c
.\}
.if \\n(^v \{\
.nr ^x \\n(^tu+1v-\\n(^Yu
\kx\h'-\\nxu'\h'|\\n(^lu+3n'\ky\L'-\\n(^xu'\v'\\n(^xu'\h'|0u'\c
.\}
.bp
'fi
.ev
.if \\n(^b \{\
.mk ^y
.nr ^b 2
.\}
.if \\n(^v \{\
.mk ^Y
.\}
..
'\"	# DS - begin display
.de DS
.RS
.nf
.sp
..
'\"	# DE - end display
.de DE
.fi
.RE
.sp
..
'\"	# SO - start of list of standard options
.de SO
.SH "STANDARD OPTIONS"
.LP
.nf
.ta 4c 8c 12c
.ft B
..
'\"	# SE - end of list of standard options
.de SE
.fi
.ft R
.LP
See the \\fBoptions\\fR manual entry for details on the standard options.
..
'\"	# OP - start of full description for a single option
.de OP
.LP
.nf
.ta 4c
Command-Line Name:	\\fB\\$1\\fR
Database Name:	\\fB\\$2\\fR
Database Class:	\\fB\\$3\\fR
.fi
.IP
..
'\"	# CS - begin code excerpt
.de CS
.RS
.nf
.ta .25i .5i .75i 1i
.ft CW
.sp
..
'\"	# CE - end code excerpt
.de CE
.fi
.RE
.ft R
.sp
..
.de UL
\\$1\l'|0\(ul'\\$2
..
.TH vector n 2.4 BLT "BLT Built-In Commands"
.BS
'\" Note:  do not modify the .SH NAME line immediately below!
.SH NAME
vector \-  Vector data type for Tcl
.SH SYNOPSIS
\fBvector create \fIvecName \fR?\fIvecName\fR...? ?\fIswitches\fR? 
.sp
\fBvector destroy \fIvecName \fR?\fIvecName\fR...?
.sp
\fBvector expr \fIexpression\fR
.sp
\fBvector names \fR?\fIpattern\fR...?
.BE
.SH DESCRIPTION
The \fBvector\fR command creates a vector of floating point
values.  The vector's components can be manipulated in three ways:
through a Tcl array variable, a Tcl command, or the C API.
.SH INTRODUCTION
A vector is simply an ordered set of numbers.  The components of a
vector are real numbers, indexed by counting numbers.
.PP
Vectors are common data structures for many applications.  For
example, a graph may use two vectors to represent the X-Y
coordinates of the data plotted.  The graph will automatically
be redrawn when the vectors are updated or changed. By using vectors, 
you can separate
data analysis from the graph widget.  This makes it easier, for
example, to add data transformations, such as splines.  It's possible
to plot the same data to in multiple graphs, where each graph presents
a different view or scale of the data.
.PP
You could try to use Tcl's associative arrays as vectors.  Tcl arrays
are easy to use.  You can access individual elements randomly by
specifying the index, or the set the entire array by providing a list
of index and value pairs for each element.  The disadvantages of 
associative arrays as vectors lie in the fact they are implemented as
hash tables.
.TP 2
\(bu 
There's no implied ordering to the associative arrays.  If you used
vectors for plotting, you would want to insure the second component
comes after the first, an so on.  This isn't possible since arrays
are actually hash tables.  For example, you can't get a range of
values between two indices.  Nor can you sort an array.
.TP 2
\(bu
Arrays consume lots of memory when the number of elements becomes
large (tens of thousands).  This is because each element's index and
value are stored as strings in the hash table.
.TP 2
\(bu 
The C programming interface is unwieldy.  Normally with vectors, you
would like to view the Tcl array as you do a C array, as an array of
floats or doubles.  But with hash tables, you must convert both the
index and value to and from decimal strings, just to access
an element in the array.  This makes it cumbersome to perform operations on
the array as a whole.
.PP
The \fBvector\fR command tries to overcome these disadvantages while
still retaining the ease of use of Tcl arrays.  The \fBvector\fR
command creates both a new Tcl command and associate array which are
linked to the vector components.  You can randomly access vector
components though the elements of array.  Not have all indices are
generated for the array, so printing the array (using the \fBparray\fR
procedure) does not print out all the component values.  You can use
the Tcl command to access the array as a whole.  You can copy, append,
or sort vector using its command.  If you need greater performance, or
customized behavior, you can write your own C code to manage vectors.
.SH EXAMPLE
You create vectors using the \fBvector\fR command and its \fBcreate\fR
operation.
.CS
# Create a new vector. 
vector create y(50)
.CE
This creates a new vector named \f(CWy\fR.  It has fifty components, by
default, initialized to \f(CW0.0\fR.  In addition, both a Tcl command
and array variable, both named \f(CWy\fR, are created.  You can use
either the command or variable to query or modify components of the
vector.
.CS
# Set the first value. 
set y(0) 9.25
puts "y has [y length] components"
.CE
The array \f(CWy\fR can be used to read or set individual components of
the vector.  Vector components are indexed from zero.  The array index
must be a number less than the number of components.  For example,
it's an error if you try to set the 51st element of \f(CWy\fR.
.CS
# This is an error. The vector only has 50 components.
set y(50) 0.02
.CE
You can also specify a range of indices using a colon (:) to separate
the first and last indices of the range.
.CS
# Set the first six components of y 
set y(0:5) 25.2
.CE
If you don't include an index, then it will default to the first
and/or last component of the vector.
.CS
# Print out all the components of y 
puts "y = $y(:)"
.CE
There are special non-numeric indices.  The index \f(CWend\fR, specifies
the last component of the vector.  It's an error to use this index if
the vector is empty (length is zero).  The index \f(CW++end\fR can be
used to extend the vector by one component and initialize it to a specific 
value.  You can't read from the array using this index, though.
.CS
# Extend the vector by one component.
set y(++end) 0.02
.CE
The other special indices are \f(CWmin\fR and \f(CWmax\fR.  They return the
current smallest and largest components of the vector.  
.CS
# Print the bounds of the vector
puts "min=$y(min) max=$y(max)"
.CE
To delete components from a vector, simply unset the corresponding
array element. In the following example, the first component of
\f(CWy\fR is deleted.  All the remaining components of \f(CWy\fR will be
moved down by one index as the length of the vector is reduced by
one.
.CS
# Delete the first component
unset y(0)
puts "new first element is $y(0)"
.CE
The vector's Tcl command can also be used to query or set the vector.
.CS
# Create and set the components of a new vector
vector create x
x set { 0.02 0.04 0.06 0.08 0.10 0.12 0.14 0.16 0.18 0.20 }
.CE
Here we've created a vector \f(CWx\fR without a initial length specification.
In this case, the length is zero.  The \fBset\fR operation resets the vector,
extending it and setting values for each new component.  
.PP
There are several operations for vectors.  The \fBrange\fR operation
lists the components of a vector between two indices.
.CS
# List the components 
puts "x = [x range 0 end]"
.CE
You can search for a particular value using the \fBsearch\fR
operation.  It returns a list of indices of the components with the
same value.  If no component has the same value, it returns \f(CW""\fR.
.CS
# Find the index of the biggest component
set indices [x search $x(max)]
.CE
Other operations copy, append, or sort vectors.  You can append
vectors or new values onto an existing vector with the \fBappend\fR
operation.
.CS
# Append assorted vectors and values to x
x append x2 x3 { 2.3 4.5 } x4
.CE
The \fBsort\fR operation sorts the vector.  If any additional vectors
are specified, they are rearranged in the same order as the vector.
For example, you could use it to sort data points represented by x and
y vectors.
.CS
# Sort the data points
x sort y
.CE
The vector \f(CWx\fR is sorted while the components of \f(CWy\fR are 
rearranged so that the original x,y coordinate pairs are retained.
.PP
The \fBexpr\fR operation lets you perform arithmetic on vectors.  
The result is stored in the vector.
.CS
# Add the two vectors and a scalar
x expr { x + y }
x expr { x * 2 }
.CE
When a vector is modified, resized, or deleted, it may trigger
call-backs to notify the clients of the vector.  For example, when a
vector used in the \fBgraph\fR widget is updated, the vector
automatically notifies the widget that it has changed.  The graph can
then redrawn itself at the next idle point.  By default, the
notification occurs when Tk is next idle.  This way you can modify the
vector many times without incurring the penalty of the graph redrawing
itself for each change.  You can change this behavior using the
\fBnotify\fR operation.
.CS
# Make vector x notify after every change
x notify always
	...
# Never notify
x notify never
	...
# Force notification now
x notify now
.CE
To delete a vector, use the \fBvector delete\fR command.  
Both the vector and its corresponding Tcl command are destroyed.
.CS
# Remove vector x
vector destroy x
.CE
.SH SYNTAX
Vectors are created using the \fBvector create\fR operation.  
Th \fBcreate\fR operation can be invoked in one of three forms:
.TP
\fBvector create \fIvecName\fR
This creates a new vector \fIvecName\fR which initially has no components.
.TP
\fBvector create \fIvecName\fR(\fIsize\fR)
This second form creates a new vector which will contain \fIsize\fR
number of components.  The components will be indexed starting from
zero (0). The default value for the components is \f(CW0.0\fR.
.TP
\fBvector create \fIvecName\fR(\fIfirst\fR:\fIlast\fR)
The last form creates a new vector of indexed \fIfirst\fR through
\fIlast\fR.  \fIFirst\fR and \fIlast\fR can be any integer value
so long as \fIfirst\fR is less than \fIlast\fR.
.PP
Vector names must start with a letter and consist of letters, digits,
or underscores.  
.CS
# Error: must start with letter
vector create 1abc
.CE
You can automatically generate vector names using the
"\f(CW#auto\fR" vector name.  The \fBcreate\fR operation will generate a 
unique vector name.
.CS
set vec [vector create #auto]
puts "$vec has [$vec length] components"
.CE
.SS VECTOR INDICES
Vectors are indexed by integers.  You can access the individual vector
components via its array variable or Tcl command.  The string
representing the index can be an integer, a numeric expression, a
range, or a special keyword.
.PP
The index must lie within the current range of the vector, otherwise
an an error message is returned.  Normally the indices of a vector
are start from 0.  But you can use the \fBoffset\fR operation to
change a vector's indices on-the-fly.
.CS
puts $vecName(0)
vecName offset -5
puts $vecName(-5)
.CE
You can also use numeric expressions as indices.  The result
of the expression must be an integer value.  
.CS
set n 21
set vecName($n+3) 50.2
.CE
The following special non-numeric indices are available: \f(CWmin\fR, \f(CWmax\fR, \f(CWend\fR, and
\f(CW++end\fR.  
.CS
puts "min = $vecName($min)"
set vecName(end) -1.2
.CE
The indices \f(CWmin\fR and \f(CWmax\fR will return the minimum and maximum
values of the vector.  The index \f(CWend\fR returns the value of the 
last component in the vector.  The index \f(CW++end\fR is used to append
new value onto the vector.  It automatically extends the vector by
one component and sets its value.
.CS
# Append an new component to the end
set vecName(++end) 3.2
.CE
A range of indices can be indicated by a colon (:).  
.CS
# Set the first six components to 1.0
set vecName(0:5) 1.0
.CE
If no index is supplied the first or last component is assumed.
.CS
# Print the values of all the components
puts $vecName(:)
.CE
.SH VECTOR OPERATIONS
.TP
\fBvector create \fIvecName\fR?(\fIsize\fR)?... \fR?\fIswitches\fR? 
The \fBcreate\fR operation creates a new vector \fIvecName\fR.  Both a
Tcl command and array variable \fIvecName\fR are also created.  The
name \fIvecName\fR must be unique, so another Tcl command or array
variable can not already exist in that scope.  You can access the
components of the vector using its variable.  If you change a value in
the array, or unset an array element, the vector is updated to reflect
the changes.  When the variable \fIvecName\fR is unset, the vector and
its Tcl command are also destroyed.
.sp
The vector has optional switches that affect how the vector is created. They
are as follows:
.RS
.TP
\fB\-variable \fIvarName\fR
Specifies the name of a Tcl variable to be mapped to the vector. If
the variable already exists, it is first deleted, then recreated. 
If \fIvarName\fR is the empty string, then no variable will be mapped.
You can always map a variable back to the vector using the vector's 
\fBvariable\fR operation.
.TP
\fB\-command \fIcmdName\fR
Maps a Tcl command to the vector. The vector can be accessed using 
\fIcmdName\fR and one of the vector instance operations.  
A Tcl command by that name cannot already exist.
If \fIcmdName\fR is the empty string, no command mapping
will be made.
.TP
\fB\-watchunset \fIboolean\fR
Indicates that the vector should automatically delete itself if
the variable associated with the vector is unset.  By default,
the vector will not be deleted.  This is different from previous
releases.  Set \fIboolean\fR to "true" to get the old behavior.
.RE
.TP
\fBvector destroy \fIvecName\fR \fR?\fIvecName...\fR?
.TP
\fBvector expr \fIexpression\fR
.RS
All binary operators take vectors as operands (remember that numbers
are treated as one-component vectors).  The exact action of binary
operators depends upon the length of the second operand.  If the
second operand has only one component, then each element of the first
vector operand is computed by that value.  For example, the expression
"x * 2" multiples all elements of the vector x by 2.  If the second
operand has more than one component, both operands must be the same
length.  Each pair of corresponding elements are computed.  So "x + y"
adds the the first components of x and y together, the second, and so on.
.sp
The valid operators are listed below, grouped in decreasing order
of precedence:
.TP 20
\fB\-\0\0!\fR
Unary minus and logical NOT.  The unary minus flips the sign of each
component in the vector.  The logical not operator returns a vector of
whose values are 0.0 or 1.0.  For each non-zero component 1.0 is returned,
0.0 otherwise.
.TP 20
\fB^\fR
Exponentiation.  
.TP 20
\fB*\0\0/\0\0%\fR
Multiply, divide, remainder.  
.TP 20
\fB+\0\0\-\fR
Add and subtract.  
.TP 20
\fB<<\0\0>>\fR
Left and right shift.  Circularly shifts the values of the vector 
(not implemented yet).
.TP 20
\fB<\0\0>\0\0<=\0\0>=\fR
Boolean less, greater, less than or equal, and greater than or equal.
Each operator returns a vector of ones and zeros.  If the condition is true, 
1.0 is the component value, 0.0 otherwise.
.TP 20
\fB==\0\0!=\fR
Boolean equal and not equal.
Each operator returns a vector of ones and zeros.  If the condition is true, 
1.0 is the component value, 0.0 otherwise.
.TP 20
\fB|\fR
Bit-wise OR.  (Not implemented).
.TP 20
\fB&&\fR
Logical AND.  Produces a 1 result if both operands are non-zero, 0 otherwise.
.TP 20
\fB||\fR
Logical OR.  Produces a 0 result if both operands are zero, 1 otherwise.
.TP 20
\fIx\fB?\fIy\fB:\fIz\fR
If-then-else, as in C.  (Not implemented yet).
.LP
See the C manual for more details on the results produced by each
operator.  All of the binary operators group left-to-right within the
same precedence level.  
.sp
Several mathematical functions are supported for vectors.  Each of
the following functions invokes the math library function of the same name;
see the manual entries for the library functions for details on what
they do.  The operation is applied to all elements of the vector
returning the results. 
.CS
.ta 3c 6c 9c
\fBacos\fR	\fBcos\fR	\fBhypot\fR	\fBsinh\fR 
\fBasin\fR	\fBcosh\fR	\fBlog\fR	\fBsqrt\fR 
\fBatan\fR	\fBexp\fR	\fBlog10\fR	\fBtan\fR  
\fBceil\fR	\fBfloor\fR	\fBsin\fR	\fBtanh\fR 
.CE
Additional functions are:
.TP 1i
\fBabs\fR
Returns the absolute value of each component.
.TP 1i
\fBrandom\fR
Returns a vector of non-negative values uniformly distributed 
between [0.0, 1.0) using \fIdrand48\fR.
The seed comes from the internal clock of the machine or may be 
set manual with the srandom function.
.TP 1i
\fBround\fR
Rounds each component of the vector.
.TP 1i
\fBsrandom\fR
Initializes the random number generator using \fIsrand48\fR.
The high order 32-bits are set using the integral portion of the first 
vector component. All other components are ignored.  The low order 16-bits 
are set to an arbitrary value.
.PP
The following functions return a single value.
.TP 1i
\fBadev\fR 
Returns the average deviation (defined as the sum of the absolute values 
of the differences between component and the mean, divided by the length
of the vector).
.TP 1i
\fBkurtosis\fR
Returns the degree of peakedness (fourth moment) of the vector.
.TP 1i
\fBlength\fR
Returns the number of components in the vector.
.TP 1i
\fBmax\fR
Returns the vector's maximum value.
.TP 1i
\fBmean\fR
Returns the mean value of the vector.
.TP 1i
\fBmedian\fR
Returns the median of the vector.
.TP 1i
\fBmin\fR
Returns the vector's minimum value.
.TP 1i
\fBq1\fR
Returns the first quartile of the vector.
.TP 1i
\fBq3\fR
Returns the third quartile of the vector.
.TP 1i
\fBprod\fR 
Returns the product of the components.
.TP 1i
\fBsdev\fR 
Returns the standard deviation (defined as the square root of the variance)
of the vector.
.TP 1i
\fBskew\fR 
Returns the skewness (or third moment) of the vector.  This characterizes
the degree of asymmetry of the vector about the mean.
.TP 1i
\fBsum\fR 
Returns the sum of the components.
.TP 1i
\fBvar\fR
Returns the variance of the vector. The sum of the squared differences 
between each component and the mean is computed.  The variance is 
the sum divided by the length of the vector minus 1.
.PP
The last set returns a vector of the same length as the argument.
.TP 1i
\fBnorm\fR 
Scales the values of the vector to lie in the range [0.0..1.0].
.TP 1i
\fBsort\fR
Returns the vector components sorted in ascending order.
.RE
.TP
\fBvector names \fR?\fIpattern\fR?
.SH INSTANCE OPERATIONS
You can also use the vector's Tcl command to query or modify it.  The
general form is
.DS
\fIvecName \fIoperation\fR \fR?\fIarg\fR?...
.DE
Both \fIoperation\fR and its arguments determine the exact behavior of
the command.  The operations available for vectors are listed below.
.TP
\fIvecName \fBappend\fR \fIitem\fR ?\fIitem\fR?...
Appends the component values from \fIitem\fR to \fIvecName\fR.
\fIItem\fR can be either the name of a vector or a list of numeric
values.
.TP
\fIvecName \fBclear\fR 
Clears the element indices from the array variable associated with
\fIvecName\fR.  This doesn't affect the components of the vector.  By
default, the number of entries in the Tcl array doesn't match the
number of components in the vector.  This is because its too expensive
to maintain decimal strings for both the index and value for each
component.  Instead, the index and value are saved only when you read
or write an element with a new index.  This command removes the index
and value strings from the array.  This is useful when the vector is
large.
.TP
\fIvecName \fBdelete\fR \fIindex\fR ?\fIindex\fR?...
Deletes the \fIindex\fRth component from the vector \fIvecName\fR.
\fIIndex\fR is the index of the element to be deleted.  This is the
same as unsetting the array variable element \fIindex\fR.  The vector
is compacted after all the indices have been deleted.
.TP
\fIvecName \fBdup\fR \fIdestName\fR 
Copies \fIvecName\fR to \fIdestName\fR. \fIDestName\fR is the name of a
destination vector.  If a vector \fIdestName\fR already exists, it is
overwritten with the components of \fIvecName\fR.  Otherwise a 
new vector is created.
.TP
\fIvecName \fBexpr\fR \fIexpression\fR
Computes the expression and resets the values of the vector accordingly.
Both scalar and vector math operations are allowed.  All values in
expressions are either real numbers or names of vectors.  All numbers
are treated as one component vectors.
.TP
\fIvecName \fBlength\fR ?\fInewSize\fR?
Queries or resets the number of components in \fIvecName\fR.
\fINewSize\fR is a number specifying the new size of the vector.  If
\fInewSize\fR is smaller than the current size of \fIvecName\fR,
\fIvecName\fR is truncated.  If \fInewSize\fR is greater, the vector
is extended and the new components are initialized to \f(CW0.0\fR.  If
no \fInewSize\fR argument is present, the current length of the vector
is returned.
.TP
\fIvecName \fBmerge\fR \fIsrcName\fR ?\fIsrcName\fR?...
Merges the named vectors into a single vector.  The resulting 
vector is formed by merging the components of each source vector 
one index at a time.
.TP
\fIvecName \fBnotify\fR \fIkeyword\fR
Controls how vector clients are notified of changes to the vector.  
The exact behavior is determined by \fIkeyword\fR.
.RS
.TP 0.75i
\f(CWalways\fR 
Indicates that clients are to be notified immediately whenever the
vector is updated.
.TP
\f(CWnever\fR
Indicates that no clients are to be notified.
.TP
\f(CWwhenidle\fR
Indicates that clients are to be notified at the next idle point
whenever the vector is updated.
.TP
\f(CWnow\fR
If any client notifications is currently pending, they are notified
immediately.
.TP
\f(CWcancel\fR
Cancels pending notifications of clients using the vector.
.TP
\f(CWpending\fR
Returns \f(CW1\fR if a client notification is pending, and \f(CW0\fR otherwise.
.RE
.TP
\fIvecName \fBoffset\fR ?\fIvalue\fR?
Shifts the indices of the vector by the amount specified by \fIvalue\fR.
\fIValue\fR is an integer number.  If no \fIvalue\fR argument is 
given, the current offset is returned.
.TP
\fIvecName \fBpopulate\fR \fIdestName\fR ?\fIdensity\fR?
Creates a vector \fIdestName\fR which is a superset of \fIvecName\fR.
\fIDestName\fR will include all the components of \fIvecName\fR, in
addition the interval between each of the original components will
contain a \fIdensity\fR number of new components, whose values are
evenly distributed between the original components values.  This is
useful for generating abscissas to be interpolated along a spline.
.TP
\fIvecName \fBrange\fR \fIfirstIndex\fR ?\fIlastIndex\fR?...
Returns a list of numeric values representing the vector components
between two indices. Both \fIfirstIndex\fR and \fIlastIndex\fR are 
indices representing the range of components to be returned. If 
\fIlastIndex\fR is less than \fIfirstIndex\fR, the components are
listed in reverse order.
.TP
\fIvecName \fBsearch\fR \fIvalue\fR ?\fIvalue\fR?  
Searches for a value or range of values among the components of
\fIvecName\fR.  If one \fIvalue\fR argument is given, a list of
indices of the components which equal \fIvalue\fR is returned.  If a
second \fIvalue\fR is also provided, then the indices of all
components which lie within the range of the two values are returned.
If no components are found, then \f(CW""\fR is returned.
.TP
\fIvecName \fBset\fR \fIitem\fR
Resets the components of the vector to \fIitem\fR. \fIItem\fR can
be either a list of numeric expressions or another vector.
.TP
\fIvecName \fBseq\fR \fIstart\fR ?\fIfinish\fR? ?\fIstep\fR?
Generates a sequence of values starting with the value \fIstart\fR.
\fIFinish\fR indicates the terminating value of the sequence.  
The vector is automatically resized to contain just the sequence.
If three arguments are present, \fIstep\fR designates the interval.  
.sp
With only two arguments (no \fIfinish\fR argument), the sequence will
continue until the vector is filled.  With one argument, the interval 
defaults to 1.0.
.TP
\fIvecName \fBsort\fR ?\fB-reverse\fR? ?\fIargName\fR?...  
Sorts the vector \fIvecName\fR in increasing order.  If the
\fB-reverse\fR flag is present, the vector is sorted in decreasing
order.  If other arguments \fIargName\fR are present, they are the
names of vectors which will be rearranged in the same manner as
\fIvecName\fR.  Each vector must be the same length as \fIvecName\fR.
You could use this to sort the x vector of a graph, while still
retaining the same x,y coordinate pairs in a y vector.
.TP
\fIvecName \fBvariable\fR \fIvarName\fR
Maps a Tcl variable to the vector, creating another means for 
accessing the vector.  The variable \fIvarName\fR can't already 
exist. This overrides any current variable mapping the vector
may have.
.RE
.SH C LANGUAGE API
You can create, modify, and destroy vectors from C code, using 
library routines.  
You need to include the header file \f(CWblt.h\fR. It contains the
definition of the structure \fBBlt_Vector\fR, which represents the
vector.  It appears below.
.CS
\fRtypedef struct {
    double *\fIvalueArr\fR; 
    int \fInumValues\fR;    
    int \fIarraySize\fR;    
    double \fImin\fR, \fImax\fR;  
} \fBBlt_Vector\fR;
.CE
The field \fIvalueArr\fR points to memory holding the vector
components.  The components are stored in a double precision array,
whose size size is represented by \fIarraySize\fR.  \fINumValues\fR is
the length of vector.  The size of the array is always equal to or
larger than the length of the vector.  \fIMin\fR and \fImax\fR are
minimum and maximum component values.
.SH LIBRARY ROUTINES
The following routines are available from C to manage vectors.
Vectors are identified by the vector name.
.PP
\fBBlt_CreateVector\fR 
.RS .25i
.TP 1i
Synopsis:
.CS 
int \fBBlt_CreateVector\fR (\fIinterp\fR, \fIvecName\fR, \fIlength\fR, \fIvecPtrPtr\fR)
.RS 1.25i
Tcl_Interp *\fIinterp\fR;
char *\fIvecName\fR;
int \fIlength\fR;
Blt_Vector **\fIvecPtrPtr\fR;
.RE
.CE
.TP
Description:
Creates a new vector \fIvecName\fR\fR with a length of \fIlength\fR.
\fBBlt_CreateVector\fR creates both a new Tcl command and array 
variable \fIvecName\fR.  Neither a command nor variable named 
\fIvecName\fR can already exist.  A pointer to the vector is 
placed into \fIvecPtrPtr\fR.
.TP
Results:
Returns \f(CWTCL_OK\fR if the vector is successfully created.  If
\fIlength\fR is negative, a Tcl variable or command \fIvecName\fR
already exists, or memory cannot be allocated for the vector, then
\f(CWTCL_ERROR\fR is returned and \fIinterp->result\fR will contain an
error message.
.RE
.sp
.PP
\fBBlt_DeleteVectorByName\fR 
.RS .25i
.TP 1i
Synopsis:
.CS
int \fBBlt_DeleteVectorByName\fR (\fIinterp\fR, \fIvecName\fR)
.RS 1.25i
Tcl_Interp *\fIinterp\fR;
char *\fIvecName\fR;
.RE
.CE
.TP 1i
Description:
Removes the vector \fIvecName\fR.  \fIVecName\fR is the name of a vector
which must already exist.  Both the Tcl command and array variable
\fIvecName\fR are destroyed.  All clients of the vector will be notified
immediately that the vector has been destroyed.
.TP
Results:
Returns \f(CWTCL_OK\fR if the vector is successfully deleted.  If
\fIvecName\fR is not the name a vector, then \f(CWTCL_ERROR\fR is returned
and \fIinterp->result\fR will contain an error message.
.RE
.sp
.PP
\fBBlt_DeleteVector\fR 
.RS .25i
.TP 1i
Synopsis:
.CS
int \fBBlt_DeleteVector\fR (\fIvecPtr\fR)
.RS 1.25i
Blt_Vector *\fIvecPtr\fR;
.RE
.CE
.TP 1i
Description:
Removes the vector pointed to by \fIvecPtr\fR.  \fIVecPtr\fR is a
pointer to a vector, typically set by \fBBlt_GetVector\fR or
\fBBlt_CreateVector\fR.  Both the Tcl command and array variable of
the vector are destroyed.  All clients of the vector will be notified
immediately that the vector has been destroyed.
.TP
Results:
Returns \f(CWTCL_OK\fR if the vector is successfully deleted.  If
\fIvecName\fR is not the name a vector, then \f(CWTCL_ERROR\fR is returned
and \fIinterp->result\fR will contain an error message.
.RE
.sp
.PP
\fBBlt_GetVector\fR 
.RS .25i
.TP 1i
Synopsis:
.CS
int \fBBlt_GetVector\fR (\fIinterp\fR, \fIvecName\fR, \fIvecPtrPtr\fR)
.RS 1.25i
Tcl_Interp *\fIinterp\fR;
char *\fIvecName\fR;
Blt_Vector **\fIvecPtrPtr\fR;
.RE
.CE
.TP 1i
Description:
Retrieves the vector \fIvecName\fR.  \fIVecName\fR is the name of a
vector which must already exist.  \fIVecPtrPtr\fR will point be set to
the address of the vector.
.TP
Results:
Returns \f(CWTCL_OK\fR if the vector is successfully retrieved.  If
\fIvecName\fR is not the name of a vector, then \f(CWTCL_ERROR\fR is
returned and \fIinterp->result\fR will contain an error message.
.RE
.sp
.PP
\fBBlt_ResetVector\fR 
.PP
.RS .25i
.TP 1i
Synopsis:
.CS
int \fBBlt_ResetVector\fR (\fIvecPtr\fR, \fIdataArr\fR, 
	\fInumValues\fR, \fIarraySize\fR, \fIfreeProc\fR)
.RS 1.25i
Blt_Vector *\fIvecPtr\fR;
double *\fIdataArr\fR;
int *\fInumValues\fR;
int *\fIarraySize\fR;
Tcl_FreeProc *\fIfreeProc\fR;
.RE
.CE
.TP
Description: 
Resets the components of the vector pointed to by \fIvecPtr\fR.
Calling \fBBlt_ResetVector\fR will trigger the vector to dispatch
notifications to its clients. \fIDataArr\fR is the array of doubles
which represents the vector data. \fINumValues\fR is the number of
elements in the array. \fIArraySize\fR is the actual size of the array
(the array may be bigger than the number of values stored in
it). \fIFreeProc\fP indicates how the storage for the vector component
array (\fIdataArr\fR) was allocated.  It is used to determine how to
reallocate memory when the vector is resized or destroyed.  It must be
\f(CWTCL_DYNAMIC\fR, \f(CWTCL_STATIC\fR, \f(CWTCL_VOLATILE\fR, or a pointer
to a function to free the memory allocated for the vector array. If
\fIfreeProc\fR is \f(CWTCL_VOLATILE\fR, it indicates that \fIdataArr\fR
must be copied and saved.  If \fIfreeProc\fR is \f(CWTCL_DYNAMIC\fR, it
indicates that \fIdataArr\fR was dynamically allocated and that Tcl
should free \fIdataArr\fR if necessary.  \f(CWStatic\fR indicates that
nothing should be done to release storage for \fIdataArr\fR.
.TP
Results:
Returns \f(CWTCL_OK\fR if the vector is successfully resized.  If
\fInewSize\fR is negative, a vector \fIvecName\fR does not exist, or
memory cannot be allocated for the vector, then \f(CWTCL_ERROR\fR is
returned and \fIinterp->result\fR will contain an error message.
.RE
.sp
.PP
\fBBlt_ResizeVector\fR 
.RS .25i
.TP 1i
Synopsis:
.CS
int \fBBlt_ResizeVector\fR (\fIvecPtr\fR, \fInewSize\fR)
.RS 1.25i
Blt_Vector *\fIvecPtr\fR;
int \fInewSize\fR;
.RE
.CE
.TP
Description:
Resets the length of the vector pointed to by \fIvecPtr\fR to
\fInewSize\fR.  If \fInewSize\fR is smaller than the current size of
the vector, it is truncated.  If \fInewSize\fR is greater, the vector
is extended and the new components are initialized to \f(CW0.0\fR.
Calling \fBBlt_ResetVector\fR will trigger the vector to dispatch
notifications.
.TP
Results:
Returns \f(CWTCL_OK\fR if the vector is successfully resized.  If
\fInewSize\fR is negative or memory can not be allocated for the vector, 
then \f(CWTCL_ERROR\fR is returned and \fIinterp->result\fR will contain 
an error message.
.sp
.PP
\fBBlt_VectorExists\fR 
.RS .25i
.TP 1i
Synopsis:
.CS
int \fBBlt_VectorExists\fR (\fIinterp\fR, \fIvecName\fR)
.RS 1.25i
Tcl_Interp *\fIinterp\fR;
char *\fIvecName\fR;
.RE
.CE
.TP
Description:
Indicates if a vector named \fIvecName\fR exists in \fIinterp\fR.
.TP
Results:
Returns \f(CW1\fR if a vector \fIvecName\fR exists and \f(CW0\fR otherwise.
.RE
.sp
.PP
If your application needs to be notified when a vector changes, it can
allocate a unique \fIclient identifier\fR for itself.  Using this
identifier, you can then register a call-back to be made whenever the
vector is updated or destroyed.  By default, the call-backs are made at
the next idle point.  This can be changed to occur at the time the
vector is modified.  An application can allocate more than one
identifier for any vector.  When the client application is done with
the vector, it should free the identifier.
.PP
The call-back routine must of the following type.
.CS
.RS
.sp
typedef void (\fBBlt_VectorChangedProc\fR) (Tcl_Interp *\fIinterp\fR, 
.RS .25i
ClientData \fIclientData\fR, Blt_VectorNotify \fInotify\fR);
.RE
.sp
.RE
.CE
.fi
\fIClientData\fR is passed to this routine whenever it is called.  You
can use this to pass information to the call-back.  The \fInotify\fR 
argument indicates whether the vector has been updated of destroyed. It
is an enumerated type.
.CS
.RS
.sp
typedef enum {
    \f(CWBLT_VECTOR_NOTIFY_UPDATE\fR=1,
    \f(CWBLT_VECTOR_NOTIFY_DESTROY\fR=2
} \fBBlt_VectorNotify\fR;
.sp
.RE
.CE
.PP
\fBBlt_AllocVectorId\fR
.RS .25i
.TP 1i
Synopsis:
.CS
Blt_VectorId \fBBlt_AllocVectorId\fR (\fIinterp\fR, \fIvecName\fR)
.RS 1.25i
Tcl_Interp *\fIinterp\fR;
char *\fIvecName\fR;
.RE
.CE
.TP
Description:
Allocates an client identifier for with the vector \fIvecName\fR.
This identifier can be used to specify a call-back which is triggered
when the vector is updated or destroyed.
.TP
Results:
Returns a client identifier if successful.  If \fIvecName\fR is not
the name of a vector, then \f(CWNULL\fR is returned and
\fIinterp->result\fR will contain an error message.
.RE
.sp
.PP
\fBBlt_GetVectorById\fR 
.RS .25i
.TP 1i
Synopsis:
.CS
int \fBBlt_GetVector\fR (\fIinterp\fR, \fIclientId\fR, \fIvecPtrPtr\fR)
.RS 1.25i
Tcl_Interp *\fIinterp\fR;
Blt_VectorId \fIclientId\fR;
Blt_Vector **\fIvecPtrPtr\fR;
.RE
.CE
.TP 1i
Description:
Retrieves the vector used by \fIclientId\fR.  \fIClientId\fR is a valid
vector client identifier allocated by \fBBlt_AllocVectorId\fR.
\fIVecPtrPtr\fR will point be set to the address of the vector.
.TP
Results:
Returns \f(CWTCL_OK\fR if the vector is successfully retrieved.  
.RE
.sp
.PP
\fBBlt_SetVectorChangedProc\fR
.RS .25i
.TP 1i
Synopsis:
.CS
void \fBBlt_SetVectorChangedProc\fR (\fIclientId\fR, \fIproc\fR, \fIclientData\fR);
.RS 1.25i
Blt_VectorId \fIclientId\fR;
Blt_VectorChangedProc *\fIproc\fR;
ClientData *\fIclientData\fR;
.RE
.CE
.TP
Description: 
Specifies a call-back routine to be called whenever the vector
associated with \fIclientId\fR is updated or deleted.  \fIProc\fR is a
pointer to call-back routine and must be of the type
\fBBlt_VectorChangedProc\fR.  \fIClientData\fR is a one-word value to
be passed to the routine when it is invoked. If \fIproc\fR is
\f(CWNULL\fR, then the client is not notified.
.TP
Results:
The designated call-back procedure will be invoked when the vector is 
updated or destroyed.
.RE
.sp
.PP
\fBBlt_FreeVectorId\fR
.RS .25i
.TP 1i
Synopsis:
.CS
void \fBBlt_FreeVectorId\fR (\fIclientId\fR);
.RS 1.25i
Blt_VectorId \fIclientId\fR;
.RE
.CE
.TP
Description: 
Frees the client identifier.  Memory allocated for the identifier 
is released.  The client will no longer be notified when the
vector is modified.
.TP
Results:
The designated call-back procedure will be no longer be invoked when
the vector is updated or destroyed.
.RE
.sp
.PP
\fBBlt_NameOfVectorId\fR
.RS .25i
.TP 1i
Synopsis:
.CS
char *\fBBlt_NameOfVectorId\fR (\fIclientId\fR);
.RS 1.25i
Blt_VectorId \fIclientId\fR;
.RE
.CE
.TP
Description: 
Retrieves the name of the vector associated with the client identifier
\fIclientId\fR.  
.TP
Results:
Returns the name of the vector associated with \fIclientId\fR.  If
\fIclientId\fR is not an identifier or the vector has been destroyed, 
\f(CWNULL\fR is returned.
.RE
.sp
.PP
\fBBlt_InstallIndexProc\fR
.RS .25i
.TP 1i
Synopsis:
.CS
void \fBBlt_InstallIndexProc\fR (\fIindexName\fR, \fIprocPtr\fR)
.RS 1.25i
char *\fIindexName\fR;
Blt_VectorIndexProc *\fIprocPtr\fR;
.RE
.CE
.TP
Description: 
Registers a function to be called to retrieved the index \fIindexName\fR
from the vector's array variable.  
.sp
typedef double Blt_VectorIndexProc(Vector *vecPtr);
.sp
The function will be passed a pointer to the vector.  The function must
return a double representing the value at the index.
.TP
Results:
The new index is installed into the vector.
.RE
.RE
.SH C API EXAMPLE
The following example opens a file of binary data and stores it in an
array of doubles. The array size is computed from the size of the
file. If the vector "data" exists, calling \fBBlt_VectorExists\fR,
\fBBlt_GetVector\fR is called to get the pointer to the vector.
Otherwise the routine \fBBlt_CreateVector\fR is called to create a new
vector and returns a pointer to it. Just like the Tcl interface, both
a new Tcl command and array variable are created when a new vector is
created. It doesn't make any difference what the initial size of the
vector is since it will be reset shortly. The vector is updated when
\fBlt_ResetVector\fR is called.  Blt_ResetVector makes the changes
visible to the Tcl interface and other vector clients (such as a graph
widget).
.sp
.CS
#include <tcl.h>
#include <blt.h>				
...
Blt_Vector *vecPtr;
double *newArr;
FILE *f;
struct stat statBuf;
int numBytes, numValues;

f = fopen("binary.dat", "r");
fstat(fileno(f), &statBuf);
numBytes = (int)statBuf.st_size;

/* Allocate an array big enough to hold all the data */
newArr = (double *)malloc(numBytes);
numValues = numBytes / sizeof(double);
fread((void *)newArr, numValues, sizeof(double), f);
fclose(f);

if (Blt_VectorExists(interp, "data"))  {
    if (Blt_GetVector(interp, "data", &vecPtr) != TCL_OK) {
	return TCL_ERROR;
    }
} else {
   if (Blt_CreateVector(interp, "data", 0, &vecPtr) != TCL_OK) {
	return TCL_ERROR;
   }
}
/* 
 * Reset the vector. Clients will be notified when Tk is idle. 
 * TCL_DYNAMIC tells the vector to free the memory allocated 
 * if it needs to reallocate or destroy the vector.
 */
if (Blt_ResetVector(vecPtr, newArr, numValues, numValues, 
	TCL_DYNAMIC) != TCL_OK) {
    return TCL_ERROR;
}
.CE
.SH "INCOMPATIBILITIES"
In previous versions, if the array variable isn't global 
(i.e. local to a Tcl procedure), the vector is automatically 
destroyed when the procedure returns.
.CS
proc doit {} {
    # Temporary vector x
    vector x(10)
    set x(9) 2.0
      ...
}
.CE
.PP
This has changed.  Variables are not automatically destroyed when
their variable is unset.  You can restore the old behavior by
setting the "-watchunset" switch.
.CE
.SH KEYWORDS
vector, graph, widget
