#
# shexport.tcl: Main Tcl script for the SHADOW data exporter
#
# INCOMPLETE -- DOES NOT WORK YET.
#
# ------------------------------------------------
# Mumit Khan <khan@xraylith.wisc.edu>
# Center for X-ray Lithography
# University of Wisconsin-Madison
# 3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Copyright (c) 1996 Mumit Khan
#

######################### Initialize options ##############################

#
# shexport:init:env is really executed ONLY once per interpreter. It has
# checks to avoid reinventing the wheel if we get here by "New Window"
# command.
#

proc shexport:init:env {rwin} {
    global shexport shadow_library
    if [info exists shexport(initialized)] {
        incr shexport(instances)
        incr shexport(total_instances)
	shexport:vputs "instance # $shexport(instances)"
        return
    } else {
        set shexport(instances) 1
        set shexport(total_instances) 1
	shexport:vputs "first instance"
    }
    #
    # make all the globals available here.
    #
    eval global [uplevel \#0 info vars]
    # Handle Tcl library environment variables
    if ![info exists env(SHADOW_GUI_ROOT)] {
        puts stderr "Warning: env(SHADOW_GUI_ROOT) does not exist."
	set env(SHADOW_GUI_ROOT) [pwd]
    }

    if [catch {set env(SHADOW_GUI_ROOT)} shexport(shadow_gui_root)] {
	set shexport(shadow_gui_root) $shexport(shadow_root)
    }

    if ![info exists shadow_library] {
        puts stderr "Warning: shadow_library variable is not set."
	set shadow_library [pwd]
    }
    set shexport(shadow_tcl_library) $shadow_library

    set tcl_precision 17

    # Set-up various paths
    lappend auto_path $shexport(shadow_tcl_library) 
    if [catch {file join x y}] {
	lappend auto_path $shexport(shadow_tcl_library)/shexport
    } else {
	lappend auto_path [file join $shexport(shadow_tcl_library) shexport]
    }
    lappend auto_path [pwd]
    lappend auto_path [pwd]/../tcl
    lappend auto_path $blt_library

    # Check for correct Tcl version
    if {[info tclversion] < 7.0} {
	puts stderr "shexport: Tcl version < 7.0 cannot be used with shexport"
	exit 1
    }

    #
    # Win32 lacks a few standard variables, so fix those here.
    #
    set platform $tcl_platform(platform)
    if {$platform == "windows"} {
	if [catch {set env(HOME)} shexport(homedir)] {
	    if [catch {set env(windir)} shexport(homedir)] {
		set shexport(homedir) [pwd]
	    }
	}
	if [catch {set env(USER)} shexport(username)] {
	    set shexport(username) ""
	}
	if [catch {set env(TEMP)} shexport(tmpdir)] {
	    if [file isdir c:/temp] {
		set shexport(tmpdir) c:/temp
	    } elseif [file isdir c:/tmp] {
		set shexport(tmpdir) c:/tmp
	    } else {
		set shexport(tmpdir) .
	    }
	    # cygwin32 likes TMPDIR env variable; this way we avoid having to
	    # have a /tmp (per P_tmpdir in <stdio.h>
	    if ![info exists env(TMPDIR)] {set env(TMPDIR) $shexport(tmpdir)}
	}
	# sanitize pathnames for C strings.
	set shexport(homedir) [cvt_filename_to_tcl $shexport(homedir)]
	set shexport(tmpdir) [cvt_filename_to_tcl $shexport(tmpdir)]
    } elseif {$platform == "unix"} {
	if [catch {set env(HOME)} shexport(homedir)] {
	    set shexport(homedir) [pwd]
	}
	if [catch {set env(USER)} shexport(username)] {
	    set shexport(username) "unknown"
	}
	if [catch {set env(TMPDIR)} shexport(tmpdir)] {
	    set shexport(tmpdir) "/tmp"
	}
    } else {
	error "Only works on Unix and Windows platforms for now. Sorry"
    }
}

######################### Initialize globals ##############################

# initialization that only happen once per interpreter, like the 
# shexport:init:env, but only for shexport specific globals.

proc shexport:init:_once_globals {} {
    global env shexport

    if [info exists shexport(initialized)] {return}

    # datasets and files
    #
    set shexport(datasets)	""	;#   and list of active datasets

    # multiple toplevels?
    set shexport(toplevels)	""

    #
    # general SHADOW related names.
    #
    set shexport(col_names,1)	X
    set shexport(col_names,2)	Y
    set shexport(col_names,3)	Z
    set shexport(col_names,4)	X'
    set shexport(col_names,5)	Y'
    set shexport(col_names,6)	Z'

    set shexport(verbose)		0
}

proc shexport:init:globals {rwin} {
    global env shexport

    shexport:init:_once_globals

    # datasets and files
    #
    set shexport($rwin,curfile)		""	;# current datafile name
    set shexport($rwin,curdataset) 	""	;#  ... dataset being viewed
    set shexport($rwin,instance)		$shexport(total_instances)

    # interface globals (various window paths typically)
    if {$shexport(total_instances) == 1} {
        set inst ""
    } else {
        set inst "#$shexport(total_instances)"
    }
    set shexport($rwin,win_title)		"SHADOW Exporter$inst"
}

######################### Initialize preferences, startup etc 

#
# initialize user preferences
#
proc shexport:init:preferences {rwin} { }

#
# initialize fonts. Should be called *after* init:preferences
# We should really use "tix resetoptions" and change the scheme used by
# Tix, but Tix only has 12 and 14 point font schemes and we need smaller
# ones esp on laptops.
#
proc shexport:init:fonts {rwin} {
    global shexport shexport_prefs
}

#
# initialize the image to be loaded and such. Called after all other
# intialization is done.
#
proc shexport:init:startup {rwin} {
    global shexport shexport_prefs

    #
    # now do the initial GUI loading if specified.
    #
    if {$shexport($rwin,curfile) != ""} {
	set file $shexport($rwin,curfile)
	if [catch {shexport:_load_image $rwin $file} msg] {
	    dialog .load_error "Load Error" \
		"Error loading SHADOW ray file $file. ($msg)" \
		error 0 Dismiss
	    return
	}
    } else {
	shexport:misc:add_file_to_title $rwin ""
    }
    return 0
}

######################### Utilities #######################################


#----------------------------------------------------------------------#
# Misc front-end Routines 
#----------------------------------------------------------------------#

proc shexport:cleanup_and_exit {{status 0}} {
    global shexport
    # REMOVE any temp files ...
    exit $status
}

# effects - Print usage message and exist
proc shexport:usage {{status 1}} {
    puts stderr {
Usage: shexport [-hv] SHADOW_IMAGE_FILE
 
    -v:         turn on verbose mode (default: off)
    -h:         print this info
    SHADOW_IMAGE:  Loads the shadow image file and plot
 
Examples:

    % shexport                  ;# Start up SHADOW workspace with def options.
    % shexport star.02          ;# Start up and load star.02 (OE 2)

    }
    exit $status
}


# verbose mode puts
proc shexport:vputs {args} {
    global shexport
    if {[info exists shexport(verbose)] && $shexport(verbose)} {
	puts stderr "shexport: [join $args]"
    }
}

# parse the command line arguments and fill up the shexport array elements.
proc shexport:parse_args {rwin} {
    global argv shexport
    shexport:vputs "parsing argv: $argv"
    while {[llength $argv] != 0} {
	set arg [lindex $argv 0]
	set argv [lrange $argv 1 end]
	shexport:vputs "arg = $arg, argv = $argv"

	case $arg in {
	    -h* {
		# don't exit with an error status - not an error
		shexport:usage 0
	    }
	    -v* {
		set shexport(verbose) 1
		shexport:vputs "setting verbose mode to on"
	    }
	    -* {
		puts stderr "shexport: bad option \"$arg\"."
		shexport:usage
	    }
	    * {
		if {$shexport($rwin,curfile) != ""} {
		    puts stderr "shexport: Can only load one image at a time."
		    shexport:usage 
		} else {
		    set shexport($rwin,curfile) "$arg"
		}
	    }
	}
    }

    shexport:vputs "current image file:       $shexport($rwin,curfile)"
}


####################### Plotting routines ################################

proc shexport:graph:make {rwin} {
    global shexport

    set w $shexport($rwin,plot_master_w)

    set plotrelief sunken

    # make scatter plot and the barcharts for histogram
    graph ${w}.scatter -title "" \
	-bufferelements 1 \
	-topmargin 0 -rightmargin 0 \
	-plotrelief $plotrelief \
	;#-width $scatterw \
	;#-height $scatterh
    ${w}.scatter grid configure \
	-mapx both -mapy both \
	-mapped $shexport(plot_grid)

    ${w}.scatter postscript configure -decorations 0

    barchart ${w}.histo_t -title "" \
	-bufferelements 1 \
	-bottommargin 0 \
	-plotrelief $plotrelief

    ${w}.histo_t postscript configure -decorations 0

    # the side histogram needs X/Y axis inversion as well.
    barchart ${w}.histo_s -title "" \
	-invertxy 1 \
	-bufferelements 1 \
	-leftmargin 0 \
	-plotrelief $plotrelief

    ${w}.histo_s postscript configure -decorations 0

    # initialize the common options
    foreach i "scatter histo_t histo_s" {
	set g ${w}.${i}
	$g legend configure -mapped 0
	$g xaxis configure -title "" 
	$g yaxis configure -title "" 
	$g x2axis configure -title "" -mapped true -showticks 0
	$g y2axis configure -title "" -mapped true -showticks 0
    }

    # only allow user to zoom the scatter plot area. 
    # TODO: (un)zooming scatter plot should also correspondingly zoom
    # the histograms.

    shexport:Blt_ZoomStack ${w}.scatter ${w}.histo_t ${w}.histo_s
    Blt_Crosshairs ${w}.scatter
    Blt_Crosshairs ${w}.histo_t
    Blt_Crosshairs ${w}.histo_s

    # make the graph elements right now, and supply the data vectors
    # when needed (in shexport:graph:plot and procs called it).
    foreach ray "good lost" {
	${w}.scatter element create $ray -mapx both -mapy both \
	    -linewidth 0 -symbol circle -color $shexport(color_$ray) \
	    -pixels 1.5 -scalesymbols 1
    }
    foreach histo "histo_t histo_s" {
	${w}.${histo} element create histo -mapx both -mapy both
    }

    # now the various sizes and placements.
    set scatterw [winfo reqwidth ${w}.scatter]
    set scatterh [winfo reqheight ${w}.scatter]
    set histo_t_h [expr round($scatterh / 2.5)]
    set histo_s_w [expr round($scatterw / 2.0)]
    ${w}.histo_t configure -height $histo_t_h
    ${w}.histo_s configure -width $histo_s_w

    set rwidth [winfo reqwidth $w]
    set rheight [winfo reqheight $w]

    set basemargin $shexport(plot_basemargin)

    set shexport(title_llx) 50
    set shexport(title_lly) 0

    set shexport(histo_t_llx) $basemargin
    set shexport(histo_t_lly) [expr {$basemargin + $histo_t_h + 30}]

    set shexport(scatter_llx) $basemargin
    set shexport(scatter_lly) \
        [expr {$shexport(histo_t_lly) + $basemargin + $scatterh}]

    set shexport(histo_s_llx) \
        [expr {$shexport(scatter_llx) + $basemargin + $scatterw}]
    set shexport(histo_s_lly) $shexport(scatter_lly)

    set shexport(info_llx) \
        [expr {$shexport(histo_s_llx) + $basemargin + $histo_s_w}]
    set shexport(info_lly) $shexport(histo_s_lly)

    set shexport(info_height) $scatterh
    set shexport(info_width) [expr {$histo_s_w * 0.85}]
}

proc shexport:graph:set_limits {rwin xmin xmax ymin ymax} {
    global shexport

    set w $shexport($rwin,plot_master_w)
    foreach i "scatter histo_t" {
	set g ${w}.${i}
	$g xaxis configure -min $xmin -max $xmax
	$g x2axis configure -min $xmin -max $xmax
    }
    foreach i "histo_s" {
	set g ${w}.${i}
	$g xaxis configure -min $ymin -max $ymax
	$g x2axis configure -min $ymin -max $ymax
    }
    foreach i "scatter" {
	set g ${w}.${i}
	$g yaxis configure -min $ymin -max $ymax
	$g y2axis configure -min $ymin -max $ymax
    }
    set shexport($rwin,cur_limits) "$xmin $xmax $ymin $ymax"
}

#
# CHECK/FIXME/TODO: MEMORY LEAK in the vectors when these windows are
# closed.
#
proc shexport:graph:show_scatter_plot {rwin} {
    global shexport
    set llx $shexport(scatter_llx)
    set lly $shexport(scatter_lly)
    set w $shexport($rwin,plot_master_w)
    set scatter ${w}.scatter
    $w create window $llx $lly -window $scatter -anchor sw

    set dataset $shexport($rwin,curdataset)
    set col1 $shexport($rwin,col_x)
    set col2 $shexport($rwin,col_y)
    set limit $shexport($rwin,scale)
    global Xgood_vector_$rwin Ygood_vector_$rwin 
    global Xlost_vector_$rwin Ylost_vector_$rwin
    if {$shexport($rwin,rays) == "good"} {
	$dataset column $col1 good Xgood_vector_$rwin
	$dataset column $col2 good Ygood_vector_$rwin
	if [info exists Xlost_vector_$rwin] {
	    Xlost_vector_$rwin set {}
	    Ylost_vector_$rwin set {}
	}
	$scatter element configure good \
	    -xdata Xgood_vector_$rwin \
	    -ydata Ygood_vector_$rwin
	shexport:vputs "scatter: good rays"
	shexport:vputs "x veclen = [Xgood_vector_$rwin length]"
    } elseif {$shexport($rwin,rays) == "lost"} {
	$dataset column $col1 lost Xlost_vector_$rwin
	$dataset column $col2 lost Ylost_vector_$rwin
	if [info exists Xgood_vector_$rwin] {
	    Xgood_vector_$rwin set {}
	    Ygood_vector_$rwin set {}
	}
	$scatter element configure lost \
	    -xdata Xlost_vector_$rwin \
	    -ydata Ylost_vector_$rwin
    } else {
	$dataset column $col1 good Xgood_vector_$rwin lost Xlost_vector_$rwin
	$dataset column $col2 good Ygood_vector_$rwin lost Ylost_vector_$rwin
	$scatter element configure good \
	    -xdata Xgood_vector_$rwin \
	    -ydata Ygood_vector_$rwin
	$scatter element configure lost \
	    -xdata Xlost_vector_$rwin \
	    -ydata Ylost_vector_$rwin
    }
}

#
# CHECK/FIXME/TODO: MEMORY LEAK in the vectors when these windows are
# closed.
#
proc shexport:graph:show_histogram {rwin histo_which col bins} {
    global shexport
    set llx $shexport(${histo_which}_llx)
    set lly $shexport(${histo_which}_lly)
    set w $shexport($rwin,plot_master_w)
    set histo ${w}.${histo_which}
    $w create window $llx $lly -window $histo -anchor sw

    if {$histo_which == "histo_t"} {
	set min [lindex $shexport($rwin,cur_limits) 0]
	set max [lindex $shexport($rwin,cur_limits) 1]
    } else {
	set min [lindex $shexport($rwin,cur_limits) 2]
	set max [lindex $shexport($rwin,cur_limits) 3]
    }
    set dataset $shexport($rwin,curdataset)
    global  ${histo_which}_xvector_$rwin ${histo_which}_vector_$rwin
    $dataset histogram $bins $col \
	${histo_which}_xvector_$rwin ${histo_which}_vector_$rwin \
	$min $max

    $histo element configure histo \
	-xdata ${histo_which}_xvector_$rwin \
	-ydata ${histo_which}_vector_$rwin

    # set the barwidth from min/max info
    set step [expr ($max - $min)/($bins - 1)]
    $histo configure -barwidth [expr $step * 1.0]
}

proc shexport:graph:show_info {rwin} {
    global shexport
    set dataset $shexport($rwin,curdataset)

    set ray_info   [$dataset info rays]
    set total_rays [lindex $ray_info 0]
    set good_rays  [lindex $ray_info 1]
    set lost_rays  [expr $total_rays - $good_rays]

    set horiz_min    [lindex $shexport($rwin,cur_limits) 0]
    set horiz_max    [lindex $shexport($rwin,cur_limits) 1]
    set horiz_length [expr $horiz_max - $horiz_min]
    set horiz_center [expr $horiz_max - $horiz_length/2.0]

    set vert_min     [lindex $shexport($rwin,cur_limits) 2]
    set vert_max     [lindex $shexport($rwin,cur_limits) 3]
    set vert_length  [expr $vert_max - $vert_min]
    set vert_center  [expr $vert_max - $vert_length/2.0]

    set horiz_col [set shexport(col_names,$shexport($rwin,col_x))]
    set vert_col  [set shexport(col_names,$shexport($rwin,col_y))]

    set text ""
    append text "H Length: $horiz_length\n"
    append text "H Center: $horiz_center\n"
    append text "V Length: $vert_length\n"
    append text "V Center: $vert_center\n\n"

    append text "Limits:   $shexport($rwin,scale)\n\n"

    append text "Rays:     $shexport($rwin,rays)\n\n"

    append text "Total:    $total_rays\n"
    append text "Lost:     $lost_rays\n\n"

    append text "Horiz:    $shexport(col_names,$shexport($rwin,col_x))\n"
    append text "Vert:     $shexport(col_names,$shexport($rwin,col_y))\n"

    set w $shexport($rwin,plot_master_w)
    set llx $shexport(info_llx)
    set lly $shexport(info_lly)
    set urx [expr $llx + $shexport(info_width)]
    set ury [expr $lly - $shexport(info_height)]
    $w create rectangle $llx $lly $urx $ury -width 2 ;#-outline red
    set x [expr $llx + 10]
    set y [expr $ury + 10]

    set font -*-fixed-bold-r-normal-*-14-*-*-*-*-*-*-*
    set font -*-courier-medium-r-normal-*-14-*-*-*-*-*-*-*
    set font -*-courier-medium-r-normal-*-*-120-*-*-*-*-*-*

    $w create text $x $y -text $text -anchor nw -font $font
}

proc shexport:graph:show_title {rwin} {
    global shexport
    set file [string trim $shexport($rwin,curfile)]

    set text "Filename : $file\n\n"

    set w $shexport($rwin,plot_master_w)
    set x $shexport(title_llx)
    set y $shexport(title_lly)

    #set font -*-Helvetica-bold-o-normal-*-18-*-*-*-*-*-*-*
    set font -*-Helvetica-bold-o-normal-*-*-240-*-*-*-*-*-*

    $w create text $x $y -text $text -anchor nw -font $font
}

proc shexport:graph:clear {rwin} {
    global shexport
    set w $shexport($rwin,plot_master_w)
    $w delete all
    set shexport($rwin,showing_plots) 0
}

proc shexport:graph:print {rwin {file ""}} {
    global shexport
    set w $shexport($rwin,plot_master_w)
    set file [string trim $file]
    if {[string length $file] == 0} {
	set types {
	    {{Postscript Files}		{.ps}}
	    {{Postscript Files}		{.eps}}
	    {{All Files}		{*}}
	}
	set file [tk_getSaveFile \
	    -parent $rwin \
	    -defaultextension ".ps" \
	    -filetypes $types \
	    -title "Print to Postscript file" \
	    -initialfile "shexport.ps" \
	]
	if {[string length $file] == 0} {
	    return
	}
    }
    set canvas $shexport($rwin,plot_master_w)
    set bbox [$canvas bbox all]
    $shexport($rwin,plot_master_w) postscript -file $file -rotate 1 \
        -x [lindex $bbox 0] \
        -y [lindex $bbox 1] \
        -width [lindex $bbox 2] \
        -height [lindex $bbox 3]
}

proc shexport:graph:show_plots {rwin} {
    global shexport

    set w $shexport($rwin,plot_master_w)
    set scatter ${w}.scatter
    set histo_t ${w}.histo_t
    set histo_s ${w}.histo_s
    set info ${w}.info
    catch [list shexport:graph:show_title $rwin]

    if $shexport($rwin,plot_scatter) {
	catch [list shexport:graph:show_scatter_plot $rwin]
    }
    if $shexport($rwin,plot_histo_t) {
	catch [list shexport:graph:show_histogram $rwin histo_t \
	    $shexport($rwin,col_x) \
	    $shexport($rwin,xbins)]
    }
    if $shexport($rwin,plot_histo_s) {
	catch [list shexport:graph:show_histogram $rwin histo_s \
	    $shexport($rwin,col_y) \
	    $shexport($rwin,ybins)]
    }
    if $shexport($rwin,plot_info) {
	catch [list shexport:graph:show_info $rwin]
    }
    set shexport($rwin,showing_plot) 1
    catch {busy hold $rwin}
    update
    catch {busy release $rwin}
}

proc shexport:graph:plot {rwin} {
    global shexport
    set dataset $shexport($rwin,curdataset)

    if {$dataset == ""} {
	dialog .plot_error "Plot Error" \
	    "Please load a dataset before trying to plot." \
	    error 0 Dismiss
        return
    }

    set col1 $shexport($rwin,col_x)
    set col2 $shexport($rwin,col_y)

    $dataset select $shexport($rwin,rays)
    
    set scale $shexport($rwin,scale)
    if ![string compare $scale "user"] {
        set limits $shexport($rwin,user_limits)
    } else {
	set limits [$dataset compute limits columns $col1 $col2 $scale]
    }
    set xmin [lindex $limits 0]
    set xmax [lindex $limits 1]
    set ymin [lindex $limits 2]
    set ymax [lindex $limits 3]

    shexport:vputs "plotting: limits $limits"
    shexport:vputs "plotting: scale type = $scale"

    shexport:graph:clear $rwin

    # the placements are computed previously in shexport:graph:make

    if {$xmax <= $xmin || $ymax <= $ymin} {
	dialog .plot_error "Plot Error" \
	    [build_msg \
	    "Bad limits (plotting columns with no rays?)\n" \
	    "(xmin,xmax) = ($xmin,$xmax) and (ymin,ymax)=($ymin,$ymax)" \
	    ] \
	    error 0 Dismiss 7i
        return
    }

    shexport:graph:set_limits $rwin $xmin $xmax $ymin $ymax
    shexport:graph:show_plots $rwin
}

# refresh the current options and plot again. If plots are not visible,
# return.
proc shexport:graph:replot {rwin} {
    global shexport

    if {$shexport($rwin,curdataset) != "" && $shexport($rwin,showing_plot)} {
	shexport:graph:plot $rwin
    }
}

####################### Command callbacks ################################

proc shexport:close_cmd {rwin} {
    global shexport
    # release resources. CHECK
    catch {shdata delete $shexport($rwin,curdataset)}
    destroy $rwin
}

proc shexport:exit_cmd {rwin {status 0}} {
    shexport:cleanup_and_exit $status
}

proc shexport:new_window_cmd {} {
    for {set cnt 1} {$cnt <= 50} {incr cnt} {
	set toplevel .shexport#$cnt
	if ![winfo exists $toplevel] {
	    shexport:main $toplevel
	    return
	}
    }
    error "Too many (50) Shadow plot windows open. Please close some"
}

#
# this has to happen for ALL the toplevels, so the rwin parameter (for
# specifying the "this" toplevel) is  not needed here.
# really not used.
proc shexport:show_console_cmd {} {
    global shexport shexport_prefs
    foreach rwin $shexport(toplevels) {
	shexport:vputs "console for toplevel $rwin"
	catch {pack unpack $shexport($rwin,consle_w)}
	if {$shexport_prefs(show_console)} {
	    pack $shexport($rwin,consle_w) \
		-after $shexport($rwin,consle_w_stub) -fill x 
	}
    }
}

proc shexport:_export_histogram {rwin histo_which file} {
    shexport:vputs "_export_histogram: rwin = $rwin, histo = $histo_which"
    if [catch {open $file "w"} fid] {
	dialog .export "Export Error" \
	    "Error opening output file \"$file\"\n\t(Message: $msg)" \
	    error 0 Dismiss 
	return
    }
    global shexport
    global  ${histo_which}_xvector_$rwin ${histo_which}_vector_$rwin
    set npoints [${histo_which}_xvector_$rwin length]
    for {set i 0} {$i < $npoints} {incr i} {
	set var ${histo_which}_xvector_${rwin}($i)
	set xval [set $var]
	set var ${histo_which}_vector_${rwin}($i)
	set yval [set $var]
	puts $fid [format "%12.5E    %12.5E" $xval $yval]
    }
    close $fid
}

proc shexport:export_histogram_cmd {rwin histo_which {file {}}} {
    global shexport
    if {$shexport($rwin,curdataset) == ""} {
	dialog .export "Export Error" \
	    "Must load a dataset before you can export data from it" \
	    error 0 Dismiss 
	return
    }
    if {$file == ""} {
	set types {
	    {{Misc Data Files}	{.dat}}
	    {{All Files}	{*}}
	}
        set file [tk_getSaveFile \
	    -parent $rwin \
	    -title "Export SHADOW histogram data to file" \
	    -initialfile "${histo_which}.dat" \
	    -defaultextension ".dat" \
	    -filetypes $types \
        ]
	set file [string trim $file]
	if {[string length $file] != 0} {
	    shexport:_export_histogram $rwin $histo_which $file
	}
    }
}

proc shexport:_export_scatter {rwin file} {
    if [catch {open $file "w"} fid] {
	dialog .export "Export Error" \
	    "Error opening output file \"$file\"\n\t(Message: $msg)" \
	    error 0 Dismiss 
	return
    }
    global shexport
    catch {busy hold $rwin}
    if {$shexport($rwin,rays) == "all" || $shexport($rwin,rays) == "good"} {
	global Xgood_vector_${rwin} Ygood_vector_${rwin}
	set npoints [Xgood_vector_${rwin} length]
	shexport:vputs "_export_scatter: npoints = $npoints"
	for {set i 0} {$i < $npoints} {incr i} {
	    set var Xgood_vector_${rwin}($i)
	    set xval [set $var]
	    set var Ygood_vector_${rwin}($i)
	    set yval [set $var]
	    puts $fid [format "%12.5E    %12.5E" $xval $yval]
	}
    } 
    if {$shexport($rwin,rays) == "all" || $shexport($rwin,rays) == "lost"} {
	global Xlost_vector_${rwin} Ylost_vector_${rwin}
	set npoints [Xlost_vector_${rwin} length]
	shexport:vputs "_export_scatter: npoints = $npoints"
	for {set i 0} {$i < $npoints} {incr i} {
	    set var Xlost_vector_${rwin}($i)
	    set xval [set $var]
	    set var Ylost_vector_${rwin}($i)
	    set yval [set $var]
	    puts $fid [format "%12.5E    %12.5E" $xval $yval]
	}
    } 
    close $fid
    catch {busy release $rwin}
}

proc shexport:export_scatter_cmd {rwin {file {}}} {
    global shexport
    if {$shexport($rwin,curdataset) == ""} {
	dialog .export "Export Error" \
	    "Must load a dataset before you can export data from it" \
	    error 0 Dismiss 
	return
    }
    if {$file == ""} {
	set types {
	    {{Misc Data Files}	{.dat}}
	    {{All Files}	{*}}
	}
        set file [tk_getSaveFile \
	    -parent $rwin \
	    -title "Export SHADOW scatter data to file" \
	    -initialfile "scatter.dat" \
	    -defaultextension ".dat" \
	    -filetypes $types \
        ]
	set file [string trim $file]
	if {[string length $file] != 0} {
	    shexport:_export_scatter $rwin $file
	}
    }
}

####################### Menubar callbacks ################################

proc shexport:info:make {w dataset which_rays} {
    set headers {
	"NpCol"
	"Parameter"
	"Minimum"
	"Maximum"
	"Center"
	"Std. Dev."
    }

    global shexport
    set data ""
    foreach column "1 2 3 4 5 6 11 20" {
        set coldata [$dataset info column $column $which_rays]
	lappend data $coldata
    }

    # Create the grid
    #
    tixScrolledGrid $w.g -bd 0
    pack $w.g -expand yes -fill both -padx 3 -pady 3

    set grid [$w.g subwidget grid]
    $grid config -formatcmd "shexport:info:gformat $grid"

    # Set the size of the columns
    #
    $grid size col 0 -size auto
    $grid size col 1 -size auto
    $grid size col 2 -size auto
    $grid size col 3 -size auto
    $grid size col 4 -size auto
    $grid size col 5 -size auto
    $grid size col 6 -size auto

    # set the default size of the column and rows. these sizes will be used
    # if the size of a row or column has not be set via the "size col ?"
    # command
    $grid size col default -size 5char
    $grid size row default -size 1.1char -pad0 3

    set margin [tixDisplayStyle text -refwindow $grid   \
	-anchor c -padx 3 -font [tix option get bold_font]]
    set name [tixDisplayStyle text  -refwindow $grid  \
	-anchor w]
    set number [tixDisplayStyle text  -refwindow $grid  \
	-anchor e]

    # Create the headers
    #
    set x 1
    foreach h $headers {
	$grid set $x 0 -itemtype text -text $h -style $margin
	incr x
    }

    set i         1
    foreach line $data {
	set col    [lindex $line 0]
	set npcol  [lindex $line 1]
	set par    [lindex $line 2]
	set min    [lindex $line 3]
	set max    [lindex $line 4]
	set center [lindex $line 5]
	set stddev [lindex $line 6]

	$grid set 0 $i -itemtype text -style $margin -text $col
	$grid set 1 $i -itemtype text -style $number -text $npcol
	$grid set 2 $i -itemtype text -style $name -text $par
	$grid set 3 $i -itemtype text -style $number -text $min
	$grid set 4 $i -itemtype text -style $number -text $max
	$grid set 5 $i -itemtype text -style $number -text $center
	$grid set 6 $i -itemtype text -style $number -text $stddev

	incr i
    }
}

proc shexport:show_info {rwin {dataset {}}} {
    global shexport
    set w $shexport($rwin,info_w)
    catch {destroy $w}
    set dataset [string trim $dataset]
    if {[string length $dataset] == 0} {
        set dataset $shexport($rwin,curdataset)
    }
    set dataset [string trim $dataset]
    if {[string length $dataset] == 0} {
	set info "No dataset loaded."
	dialog .image_info "Image Info" "$info" {} 0 Dismiss
	return
    }

    toplevel $w -class Info
    wm title $w "Image Info"
    shexport:info:make $w $dataset $shexport($rwin,rays)
}

proc shexport:load {rwin file} {
    global shexport

    # for now assume that we only have one dataset loaded.
    # set cnt [expr [llength $shexport(datasets)]]
    # set dataset dataset#${cnt}
    catch {shdata delete $shexport($rwin,curdataset)}
    set dataset dataset,$rwin
    shdata create $dataset -load $file
    $dataset compute minmax
    $dataset select $shexport($rwin,rays)
    set shexport($rwin,curdataset) $dataset
    lappend shexport(datasets) $dataset
    shexport:show_info $rwin $dataset
}

proc shexport:_load_image {rwin file} {
    global shexport
    if [catch {shexport:load $rwin [cvt_filename_to_tcl $file]} msg] {
	error "Error reading SHADOW data file \"$file\" ($msg)?"
    }
    shexport:misc:add_file_to_title $rwin $file
    set shexport($rwin,curfile) $file
}

proc shexport:load_image_cmd {rwin} {
    global shexport
    set types {
	{{SHADOW Image Files}	{begin.dat star.* mirr.* }}
	{{All Files}		*}
    }
    set file [tk_getOpenFile \
	-parent $rwin \
	-title "Load SHADOW Image Data From File" \
	-initialfile "" \
	-filetypes $types \
    ]
    if {$file != ""} {
	shexport:_load_image $rwin $file
    }
}

######################## GUI routines ####################################

proc shexport:GUI:topmenu {rwin menu_name} {
    global shexport shexport_prefs

    lappend menu_names $menu_name.mFile
    # build widget $menu_name.mFile
    menubutton $menu_name.mFile \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mFile.m" \
	-text {File} \
	-underline {0}

    # build widget $menu_name.mFile.m
    menu $menu_name.mFile.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mFile.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command "shexport:load_image_cmd $rwin" \
	-label {Load ...} \
	-underline {-1}

    $menu_name.mFile.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command "shexport:export $rwin" \
	-label {Export ...} \
	-underline {-1}

    $menu_name.mFile.m add separator

    $menu_name.mFile.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command "shexport:new_window_cmd" \
	-label {New Window} \

    $menu_name.mFile.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command "shexport:close_cmd $rwin" \
	-label {Close Window} \
	-underline {1}

    # only toplevel "." gets the exit command. this makes sure that if
    # we're running under SHADOW GUI, we don't exit the entire GUI when
    # closing the Plot window.
    if ![string compare $rwin "."] {
	$menu_name.mFile.m add separator
	$menu_name.mFile.m add command \
	    -activebackground {SteelBlue2} \
	    -background {LightGray} \
	    -command "shexport:exit_cmd $rwin" \
	    -label {Exit} \
	    -underline {1} \
	    -accelerator {Ctrl-x}
    }

    lappend menu_names $menu_name.mOptions
    # build widget $menu_name.mOptions
    menubutton $menu_name.mOptions \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mOptions.m" \
	-text {Options} \
	-underline {0}

    # build widget $menu_name.mOptions.m
    menu $menu_name.mOptions.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mOptions.m add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Verbose Mode} \
	-variable shexport(verbose) \
	-underline {-1}

    global tcl_platform
    if ![string compare $tcl_platform(platform) unix] {
	$menu_name.mOptions.m add checkbutton \
	    -activebackground {SteelBlue2} \
	    -background {LightGray} \
	    -label {Strict Motif} \
	    -variable tk_strictMotif \
	    -onvalue 1 -offvalue 0 \
	    -underline {-1}
    }

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Change directory} \
	-state disabled \
	-command "shexport:chdir_cmd $rwin" \
	-underline {-1}

    $menu_name.mOptions.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Reread Data} \
	-state disabled \
	-command "shexport:reread_data $rwin" \
	-underline {-1}

    $menu_name.mOptions.m add separator

    # build cascade menu for Preferences
    $menu_name.mOptions.m add cascade \
	-menu "$menu_name.mOptions.m.mPreferences" \
	-label {Preferences} \
	-underline {-1}

    # build cascaded menu for $menu_name.mDelete
    menu $menu_name.mOptions.m.mPreferences \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mOptions.m.mPreferences add checkbutton \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Fonts} \
	-state disabled \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add separator

    $menu_name.mOptions.m.mPreferences add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Reload preferences} \
	-state disabled \
	-command "shexport:load_prefs_cmd $rwin" \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Save preferences} \
	-state disabled \
	-command "shexport:save_prefs_cmd $rwin" \
	-underline {-1}

    $menu_name.mOptions.m.mPreferences add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-label {Reset to defaults} \
	-state disabled \
	-command "shexport:reset_prefs_cmd $rwin" \
	-underline {-1}

    lappend menu_names $menu_name.mHelp
    # build widget $menu_name.mHelp
    menubutton $menu_name.mHelp \
	-background {LightGray} \
	-foreground {black} \
	-menu "$menu_name.mHelp.m" \
	-text {Help} \
	-underline {0}

    # build widget $menu_name.mHelp.m
    menu $menu_name.mHelp.m \
	-background {LightGrey} \
	-foreground {black} \
	-tearoff 0

    $menu_name.mHelp.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command "shexport:about_cmd $rwin" \
	-label {About} \
	-state disabled \
	-underline {0}

    $menu_name.mHelp.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command "shexport:about_author_cmd $rwin" \
	-label {Author} \
	-state disabled \
	-underline {0}

    $menu_name.mHelp.m add command \
	-activebackground {SteelBlue2} \
	-background {LightGray} \
	-command "shexport:help_on_topmenu_cmd $rwin" \
	-label {On pulldown menu} \
	-state disabled \
	-underline {3}

    # pack widget $menu_name
    pack append $menu_name \
	$menu_name.mFile {left frame center} \
	$menu_name.mOptions {left frame center} \
	$menu_name.mHelp {right frame center}

    focus $menu_name

    #
    # install the global bindings (accelerators).
    #
    bind all	<Control-x>	"shexport:exit_cmd $rwin"
    bind Entry	<Control-x>	"shexport:exit_cmd $rwin"
    bind Text	<Control-x>	"shexport:exit_cmd $rwin"

    return $menu_name
}

# 
# Make the DATA INFORMATION window.
#
proc shexport:GUI:info {rwin w} {

    set title [label $w.title -text "Data Information" \
	-font -*-helvetica-bold-r-normal-*-18-*-*-*-*-*-*-* \
	-bd 1]
    set info [frame $w.info -bd 20]

    set headers {
	"Parameter"
	"NpCol"
	"Minimum"
	"Maximum"
	"Center"
	"Std. Dev."
    }

    # Add table header first.
    set i 0
    foreach h $headers {
	set hlabel [label $info.h$i -text $h]
	grid $hlabel -column $i -row 0 -ipadx 0.25c
	incr i
    }

    set colnames {
	"X" "Y" "Z" "X'" "Y'" "Z'" "Photon Energy (eV)" "Numerical Aperture"
    }
    set nnames [llength $colnames]

    set nheaders [llength $headers]
    for {set i 1} {$i <= $nnames} {incr i} {
	set colname [label $info.v${i}0 -text [lindex $colnames [expr $i - 1]]]
	grid $colname -column 0 -row $i -ipadx 0.25c

	set npcol [label $info.v${i}1 -text "0"]
	grid $npcol -column 1 -row $i -ipadx 0.25c

	#
	# the value grid points are named $w.info.v$i$c, where i is the
	# row number and c is the column number.
	#
	set val [format "%12.5E" 0.0]
	for {set c 2} {$c < $nheaders} {incr c} {
	    set vlabel [label $info.v$i$c -text $val]
	    grid $vlabel -column $c -row $i -ipadx 0.25c
	}
    }

    grid $title -column 0 -row 0 -sticky ns
    grid $info -column 0 -row 1 -sticky news
    grid columnconfigure $w 0 -weight 1 
    grid rowconfigure $w 0 -weight 0 
    grid rowconfigure $w 1 -weight 1
}

proc shexport:GUI:choices {rwin w} {

    tixLabelFrame $w.rays   -label "Rays"
    tixLabelFrame $w.format -label "# of Columns"
    tixLabelFrame $w.format -label "Output Format"

    shexport:GUI:ray_choices    $rwin [$w.rays   subwidget frame]
    shexport:GUI:ncol_choices   $rwin [$w.ncols  subwidget frame]
    shexport:GUI:format_choices $rwin [$w.format subwidget frame]

    grid columnconfig $w 0 -weight 1 -minsize 0
    grid rowconfig $w 0 -weight 1 -minsize 0

    grid $w.rays -row 0 -column 0
    grid $w.format -row 0 -column 1
}

proc shexport:GUI:format_choices {rwin w} {
    global shexport
    set name [tixOptionName $w]
    option add *$name*TixOptionMenu*label.anchor e

    tixOptionMenu $w.format -label "Format: "  \
	-disablecallback true -command [list shexport:choose_format $rwin SPACE]

    foreach fmt "SPACE COMMA TAB" {
	$w.format add command $fmt -label "$fmt"
    }

    pack $w.format -padx 5 -pady 3 -fill x -side top
}

proc shexport:GUI:axis_choices {rwin w} {
    global shexport
    set name [tixOptionName $w]
    option add *$name*TixOptionMenu*label.anchor e

    tixOptionMenu $w.xmenu -label "X axis: "  \
	-disablecallback true -command [list shexport:choose_axis $rwin col_x]

    tixOptionMenu $w.ymenu -label "Y axis: " \
	-disablecallback true -command [list shexport:choose_axis $rwin col_y]

    foreach menu "xmenu ymenu" {
	set cnt 1
        foreach axis "X Y Z X' Y' Z'" {
	    $w.${menu} add command $cnt -label "$axis  "
	    incr cnt
	}
    }

    #$w.xmenu config -value $shexport($rwin,col_x)
    #$w.ymenu config -value $shexport($rwin,col_y)
    #$w.xmenu config -disablecallback false
    #$w.ymenu config -disablecallback false

    pack $w.xmenu -padx 5 -pady 3 -fill y -side top
    pack $w.ymenu -padx 5 -pady 3 -fill y -side top

    button $w.apply -text "Apply" -command "shexport:graph:plot $rwin"
    pack $w.apply -padx 5 -pady 3 -fill x -side bottom
}

proc shexport:GUI:ray_choices {rwin w} {
    global shexport
    set name [tixOptionName $w]
    option add *$name*TixSelect*label.anchor c
    option add *$name*TixSelect*orientation vertical
    option add *$name*TixSelect*labelSide top

    tixSelect $w.rays -label "" -radio true \
	-disablecallback true -command [list shexport:choose_rays $rwin]

    $w.rays add good -text Good
    $w.rays add lost -text Lost
    $w.rays add all  -text All

    $w.rays config -disablecallback false
    set shexport($rwin,ray_choices_cmd) "$w.rays config -value"
    pack $w.rays -side left -padx 5 -pady 3 -fill x
}

proc shexport:GUI:plot_choices {rwin w} {
    global shexport
    set name [tixOptionName $w]
    option add *$name*TixSelect*label.anchor c
    option add *$name*TixSelect*orientation vertical
    option add *$name*TixSelect*labelSide top

    tixSelect $w.plots -label "" -allowzero true -radio false \
	-disablecallback true -command [list shexport:choose_plots $rwin]

    $w.plots add scatter -text "Scatter Plot"
    $w.plots add histo_t -text "Top Histogram"
    $w.plots add histo_s -text "Side Histogram"
    $w.plots add info    -text "Image Info"

    set plots ""
    foreach plot "scatter histo_t histo_s info" {
	if $shexport($rwin,plot_$plot) {lappend plots $plot}
    }
    #$w.plots config -value $plots
    #$w.plots config -disablecallback false

    set shexport($rwin,plot_choices_cmd) "$w.plots config -value"

    pack $w.plots -side left -padx 5 -pady 3 -fill x
}

proc shexport:GUI:scale_choices {rwin w} {
    global shexport
    set name [tixOptionName $w]
    option add *$name*TixSelect*label.anchor c
    option add *$name*TixSelect*orientation vertical
    option add *$name*TixSelect*labelSide top

    tixSelect $w.scale -label "" -radio true \
	-disablecallback true -command [list shexport:choose_scale $rwin]

    $w.scale add auto  -text "Automatic"
    $w.scale add cart  -text "Cartesian"
    $w.scale add fill  -text "Fill"
    $w.scale add user  -text "User-defined" -state disabled

    #$w.scale config -value $shexport($rwin,scale)
    #$w.scale config -disablecallback false

    set shexport($rwin,scale_choices_cmd) "$w.scale config -value"

    pack $w.scale -side left -padx 5 -pady 3 -fill x
}

proc shexport:GUI:plotopts {rwin w} {

    tixLabelFrame $w.axis  -label "Columns"
    tixLabelFrame $w.rays  -label "Rays"
    tixLabelFrame $w.plots -label "Plots"
    tixLabelFrame $w.scale -label "Scaling"

    shexport:GUI:axis_choices  $rwin [$w.axis subwidget frame]
    shexport:GUI:ray_choices   $rwin [$w.rays subwidget frame]
    shexport:GUI:plot_choices  $rwin [$w.plots subwidget frame]
    shexport:GUI:scale_choices $rwin [$w.scale subwidget frame]

    #pack $w.axis  -side top -padx 5 -pady 3 -fill x
    #pack $w.rays  -side top -padx 5 -pady 3 -fill x
    #pack $w.plots -side top -padx 5 -pady 3 -fill x
    #pack $w.scale -side top -padx 5 -pady 3 -fill x

    #grid columnconfig $w 0 -weight 1 -minsize 0
    #grid rowconfig $w 0 -weight 1 -minsize 0

    grid $w.axis -row 0 -column 0 -columnspan 1 -rowspan 1 -sticky news
    grid $w.rays -row 1 -column 0 -columnspan 1 -rowspan 1 -sticky news
    grid $w.plots -row 2 -column 0 -columnspan 1 -rowspan 1 -sticky news
    grid $w.scale -row 3 -column 0 -columnspan 1 -rowspan 1 -sticky news
}

proc shexport:GUI:plotbody {rwin w} {
    global shexport shexport_prefs

    scrollbar ${w}.hscroll -orient horizontal \
        -command "${w}.c xview"

    scrollbar ${w}.vscroll -orient vertical \
        -command "${w}.c yview"

    canvas ${w}.c \
        -xscrollcommand "${w}.hscroll set" \
        -yscrollcommand "${w}.vscroll set" \
        -width $shexport(plot_body_width) \
	-height $shexport(plot_body_height) \
        -scrollregion [list \
	    0 0 $shexport(plot_body_width) $shexport(plot_body_height)]
    
    set shexport($rwin,plot_master_w) ${w}.c
    
    grid columnconfig $w 0 -weight 1 -minsize 0
    grid rowconfig $w 0 -weight 1 -minsize 0
    grid ${w}.c -column 0 -row 0 \
        -rowspan 1 -columnspan 1 -sticky news
    grid ${w}.vscroll -column 1 -row 0 \
        -rowspan 1 -columnspan 1 -sticky news
    grid ${w}.hscroll -column 0 -row 1 \
        -rowspan 1 -columnspan 1 -sticky news
}

proc shexport:GUI:make {rwin} {
    global shexport

    wm title $rwin $shexport($rwin,win_title)
    wm protocol $rwin WM_SAVE_YOURSELF "shexport:exit_cmd $rwin"
    #
    # BUG WORKAROUND (FIXME/CHECK):
    # Don't install a wm protocol handler for WM_DELETE_WINDOW if the
    # toplevel is not "." (ie., if it's running under SHADOW GUI) to
    # avoid a bug Tix that causes core-dumps. 
    if {$rwin == "."} {
	wm protocol $rwin WM_DELETE_WINDOW "shexport:exit_cmd $rwin"
    }

    set topmenu    [frame ${rwin}.topmenu -relief raised -bd 1]
    set body       [frame ${rwin}.body]
    set info       [frame ${body}.info -relief sunken -bd 1]
    set choices    [frame ${body}.choices -relief sunken -bd 1]

    shexport:GUI:topmenu $rwin $topmenu
    shexport:GUI:info $rwin $info
    shexport:GUI:choices $rwin $choices

    grid $info -row 0 -column 0 -sticky news
    grid $choices -row 1 -column 0 -sticky news
    grid columnconfigure $body 0 -weight 1

    grid $topmenu -row 0 -column 0 -sticky ew
    grid $body -row 1 -column 0 -sticky news
    grid columnconfigure ${rwin} 0 -weight 1

    update
}

################################ Misc #####################################

proc shexport:misc:add_file_to_title {rwin file} {
    global shexport
    if {$file == ""} {
	set filepart "\[No dataset loaded\]"
    } else {
	set filepart "\[$file\]"
    }
    wm title $rwin "$shexport($rwin,win_title)   $filepart"
}

################################ Main #####################################

#
# parse the command line and initialize the interface etc
#
proc shexport:main {{rwin .}} {
    global shexport
    if [string compare $rwin "."] {		;# need toplevel
	toplevel $rwin -class ShadowExporter
    }

    shexport:init:env $rwin
    shexport:init:globals $rwin
    shexport:parse_args $rwin
    shexport:init:preferences $rwin
    shexport:init:fonts $rwin
    shexport:GUI:make $rwin
    shexport:init:startup $rwin
    set shexport(initialized) 1
}

######################## Start program ####################################
#

#
# DO NOT call shexport:main if running under SHADOW GUI!
#
if ![info exists gvars] {
    shexport:main
}
