#
# toolbox.tcl: routines for SHADOW Toolbox.
#
# ------------------------------------------------
# Mumit Khan <khan@xraylith.wisc.edu>
# Center for X-ray Lithography
# University of Wisconsin-Madison
# 3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Copyright (c) 1994-1996 Mumit Khan
#
#

set toolbox_priv(enabled_style)		""
set toolbox_priv(disabled_style)	""

#
# create the toolbox for the GUI.
#
# w: Enclosing frame. Already created.
# toolbox: The initial toolbox elements. Each list item describes a
#     Toolbox item, and is of the following format:
#	    {Path Pixmap Header/Entry enabled/disabled {callback_data}}
#     eg.,
#	    {Beamline folder header enabled {beamline}}
#	    {Beamline.Source file entry enabled {beamline source}}
#
# callback: The callback routine.
#
# The callback data is passed back to the callback routine and used to
# identify the item selected on the client side.
#

proc toolbox:make {w {toolbox {}} {callback ""}} {
    global toolbox_priv
    # We create the frame and the ScrolledHList widget
    # at the top of the dialog box
    #
    frame $w.toolbox

    set tree $w.toolbox.a
    tixTree $tree -options {
	hlist.separator "."
	hlist.selectMode single
	hlist.header true
	relief flat
    }
    pack $tree -expand yes -fill both -side left 

    set hlist [$tree subwidget hlist]
    $tree config  \
        -browsecmd "toolbox:_select $hlist [list $callback]"  \
	-command "toolbox:_select $hlist [list $callback]"
    
    #
    # Add the tools to the hlist.
    #

    # First some styles for the headers
    set style(header) [tixDisplayStyle text  -refwindow $hlist \
        -fg black -anchor c \
        -padx 8 -pady 5 \
        -font [tix option get bold_font ]
    ]

    set toolbox_priv(disabled_style) [tixDisplayStyle imagetext \
	-refwindow $hlist -fg grey]

    set toolbox_priv(enabled_style) [tixDisplayStyle imagetext \
	-refwindow $hlist -fg black]

    $hlist header create 0 -itemtype text \
	-text "Toolbox" -style $style(header)

    foreach t $toolbox {
        set tool [lindex $t 0]
	set image [tix getimage [lindex $t 1]]
	set type [lindex $t 2]
	set state [lindex $t 3]
	set data [lindex $t 4]
	set text [lindex [split $tool .] end]
	$hlist add $tool -itemtype imagetext \
	    -text "$text" -image $image \
	    -data $data \
	    -state $state
	if {$state == "disabled" && $type != "header"} {
	    $hlist entryconfigure $tool -style $toolbox_priv(disabled_style)
	}
    }

    #
    # DO NOT SET "autosetmode"! Otherwise the idea of opening/closing
    # can confuse issues. Better left completely opened.
    #
    # $tree autosetmode

    return $w.toolbox
}

#
# toolbox:_get_tool_name: Gets the Tool name from the supplied path.
# Simply get the last token of a path separated by "." (or whatever
# the separator character may be).
#
proc toolbox:_get_tool_name {path {sep .}} {
    return [lindex [split $path $sep] end]
}

#
# toolbox:_select: Tix browse/select callback. This is what calls the
# client callback routine with the item data. 
#
# since the toolbox is in single-click selection mode, we clear both
# the anchor and selection right after the selection is made.
#
proc toolbox:_select {hlist callback path} {
    set data [$hlist info data $path]
    set toolname [toolbox:_get_tool_name $path]
    if {$callback != ""} {
        eval $callback $data
    }
    after idle [list $hlist selection clear]
    after idle [list $hlist anchor clear]
}

#
# state better be "enabled" or "disabled"
#
proc toolbox:set_state {w path state} {
    global toolbox_priv

    set hlist [$w subwidget hlist]
    set style [set toolbox_priv(${state}_style)]
    $hlist entryconfigure $path -state $state \
        -style $style
}
