#
# settings.tcl: routines for SHADOW GUI settings
#
# ------------------------------------------------
# Mumit Khan <khan@xraylith.wisc.edu>
# Center for X-ray Lithography
# University of Wisconsin-Madison
# 3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Copyright (c) 1994-1997 Mumit Khan
#
#

#
# public routines:
#
# settings:load		    -  load settings from file/registry
# settings:save		    -  load settings to file/registry
#
# The settings are the following:
#
#  last_directory1 last working directory
#  last_directory2   -2
#  last_directory3   -3
#  last_directory4   -4
#  last_directory5   -5
#  last_workspace1 last workspace file loaded
#  last_workspace2   -2
#  last_workspace3   -3
#  last_workspace4   -4
#  last_workspace5   -5
#  last_geometry   last geometry
# 

proc settings:_init {a} {
    global gvars 

    upvar 1 $a settings
    for {set i 1} {$i < 6} {incr i} {
	set settings(last_directory$i) ""
	set settings(last_workspace$i) ""
    }
    set settings(last_directory1) [pwd]
    set settings(last_geometry) ""
}

proc settings:_load_last_settings {a} {
    global gvars 

    upvar 1 $a settings

    for {set i 1} {$i < 6} {incr i} {
        set dir [string trim $settings(last_directory$i)]
	if [string length $dir] {
	    lappend gvars(last_directories) \
	        [cvt_filename_to_tcl $dir]
	} else {
	    break
	}
    }
    for {set i 1} {$i < 6} {incr i} {
        set wkspc [string trim $settings(last_workspace$i)]
	if [string length $wkspc] {
	    lappend gvars(last_workspaces) \
	        [cvt_filename_to_tcl $wkspc]
	} else {
	    break
	}
    }
    set gvars(last_geometry) $settings(last_geometry)
}

#
# LOAD_SETTINGS: load user-specific settings
#
proc settings:load {} {
    global tcl_platform
    settings:_init settings
    if {[string compare $tcl_platform(platform) "unix"] == 0} {
        settings:load_unix settings
    } elseif {[string compare $tcl_platform(platform) "windows"] == 0} {
        settings:load_win settings
    }
    settings:_load_last_settings settings
}

proc settings:_load_cur_settings {a} {
    global gvars

    upvar 1 $a settings

    vputs "Cur settings: $gvars(last_directories), $gvars(last_workspaces)"

    set settings(last_directory1) [pwd]
    set dirs [lrange $gvars(last_directories) 0 3]
    set ndirs [llength $dirs]
    for {set i 1} {$i < $ndirs} {incr i} {
	set idx [expr $i + 1]
        set dir [lindex $dirs $i]
	set settings(last_directory$idx) $dir
    }
        
    set wksps [lrange $gvars(last_workspaces) 0 4]
    set nwksps [llength $wksps]
    for {set i 0} {$i < $nwksps} {incr i} {
	set idx [expr $i + 1]
        set wksp [lindex $wksps $i]
	set settings(last_workspace$idx) $wksp
    }
    set settings(last_geometry) [wm geometry .]
}

#
# SAVE_SETTINGS: save current settings
#
proc settings:save {} {

    settings:_init settings
    settings:_load_cur_settings settings

    global tcl_platform
    if {[string compare $tcl_platform(platform) "unix"] == 0} {
        settings:save_unix settings
    } elseif {[string compare $tcl_platform(platform) "windows"] == 0} {
        settings:save_win settings
    }
}
