#
# settings-unix.tcl: routines for SHADOW GUI personal settings for unix
#
# ------------------------------------------------
# Mumit Khan <khan@xraylith.wisc.edu>
# Center for X-ray Lithography
# University of Wisconsin-Madison
# 3731 Schneider Dr., Stoughton, WI, 53589
# ------------------------------------------------
#
# Copyright (c) 1994-1997 Mumit Khan
#
#

#
# These are private routines called by settings: module (settings.tcl)
#
# settings:load_unix
# settings:save_unix
#

#
# settings:_load_unix_settings:
#
# Loads the settings file, which under Unix is a simple text file with
# the following format:
#
#    <Variable>	<Separator> <Value>
#
# Where <Separator> = {"="":"}
#
# Split each line, and then assign a($variable) = $value, where "a" is
# passed from the caller, usually "settings"
#
proc settings:_load_unix_settings {file a} {
    upvar 2 $a array
    if {[catch {open $file r} fin] != 0} {
	return
    }
    while {[gets $fin line] >= 0} {
	# is it a comment or empty line?
	if {[regexp {^[ \t]*#} $line] == 0 &&
	    [regexp {^[ \t]*$} $line] == 0} {
	    set line [split $line "=:"]
	    if {[llength $line] != 2} {
		close $fin
		error \
		    "Illegal syntax in settings \"$file\". wrong format file?"
	    }
	    # use case-indepent variable names.
	    set lname [string tolower [string trim [lindex $line 0]]]
	    set lvalue [string trim [lindex $line 1]]
	    #
	    # now simply add a variable in gvars array with the same
	    # name as the one in the gfile and the corresponding value.
	    vputs "set array($lname) $lvalue"
	    set array($lname) $lvalue
	}
    }
    return
}

#
# LOAD_SETTINGS: load user-specific settings
#
proc settings:load_unix {a} {
    global gvars 

    set settings_file $gvars(settings_file)
    vputs "Loading settings from $settings_file..."
    if ![file readable $settings_file] {
	return
    }
    upvar 1 $a settings
    if {[catch {settings:_load_unix_settings [list $settings_file] settings} msg]} {
	dialog .settings {Load settings} \
	    [build_msg \
	    "Error loading settings from \"$settings_file\".($msg)\n" \
	    "This may be caused by files created by an older version of \n"\
	    "SHADOW. It will be ok next time you start SHADOW"] \
	    {} 0 Dismiss
	return 1
    }
    vputs "Loaded settings from $settings_file."
    return 0
}

#
# SAVE_SETTINGS: save current settings for unix
#
proc settings:save_unix {a} {
    global gvars 
    upvar 1 $a settings

    set settings_file $gvars(settings_file)
    vputs "Saving settings to $settings_file..."
    if {[catch {open $settings_file w} fid]} {
	dialog .settings {Load settings} \
	    "Error opening settings file \"$settings_file\"." \
	    {} 0 Dismiss
	return 1
    }
    parray_names settings $fid
    close $fid
    vputs "Saved settings to $settings_file."
    return 0
}

