/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *			http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2001-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: R. Wilcke (wilcke@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */

/*
Update 08/10/2017 P. Boesecke
                  strlib_numberstring included in keyvalue function
                  for compatibility, ignoring first number in key 
                  value strings.
Update 07/10/2017 P. Boesecke
                  string copy replaced with strncpy
Update 07/06/2017 P. Boesecke
                  free replaced with strlib_free
Update 07/02/2017 P. Boesecke
                  correction in outname: inparams => *inparams
Update 07/10/2016 P. Boesecke
                  keyvalue function added
Update 02/12/2015 P. Boesecke
                  outname: outfile always initialized with an empty string
Update 30/11/2015 P. Boesecke
                  outname changed: splitting outext in path and extension 
                  outext_path is inserted between outdir and base name,
                  outext_name is appended to base name.
                  Outname creates the directory outext_path, if it does 
                  not exist.
Update 26/11/2015 P. Boesecke
                  Use of temporary data buffers.
                  scan_argument: do not apply filename_name to
                  string history values.
Update 19/11/2015 P. Boesecke (boesecke@esrf.fr)
                  unused function isbigendian removed
Update 09/03/2015 P. Boesecke (boesecke@esrf.fr)
                  outname(): separate infile into pattern and params,
                  returns outfile with params
Update 20/07/2014 P. Boesecke (boesecke@esrf.fr)
                  fnampat: default image number in filename patterns 
                  corrected from 0 to 1. 
Update 01/08/2013 P. Boesecke (boesecke@esrf.fr)
                  bench: gettimeofday does not exist for WIN32
                  function get_time( void ) added,
                  print_memlist(): return value added
                  basename(x) -> filename_name(buf,len,x)
                  strncat -> strlib_nconcat
                  strdup -> strlib_newstr
                  strspn -> strlib_spn
                  gettimeofday: 2nd parameter &tzp -> NULL
Update 12/10/2011 P. Boesecke (boesecke@esrf.fr)
                  print_memlist() added
Update 30/09/2009 R. Wilcke (wilcke@esrf.fr)
                  scan_argument(): for "%s" format, return an empty string
                  buffer if the input argument is empty (left the "result"
                  argument unchanged before).
Update 24/04/2009 R. Wilcke (wilcke@esrf.fr)
                  outname(): increase size of output name buffer from 512 to
                  1024 characters;
                  outname(): duplicate result buffer before passing it to the
                  "basename()" routine (the routine can change its argument);
                  outname(): add new input argument "outdir" to the argument
                  list and code to add it in front of the output file name.
Update 21/01/2008 R. Wilcke (wilcke@esrf.fr)
                  _pmalloc(), _prealloc() and _pfree(): slight modifications of
                  error messages.
Update 11/12/2007 R. Wilcke (wilcke@esrf.fr)
                  fnampat(): initialize filename buffer only after test for NULL
                  pointer.
Update 06/12/2007 R. Wilcke (wilcke@esrf.fr)
                  add function fnampat().
Update 24/08/2004 R. Wilcke (wilcke@esrf.fr)
                  _pfree(), print_memsize(), byte_swap2N():
                  add "return(0)" at the end of the routine;
                  bench(): add "return(iret)" at the end of the routine;
                  isbigendian(): return -1 if type cannot be determined;
                  scan_argument(): add "u" format conversion and size modifiers
                  "h" and "l".
Update 23/08/2004 R. Wilcke (wilcke@esrf.fr)
                  scan_argument(): add "g" format conversion, write all input
                  arguments into a special history buffer.
Update 17/09/2002 R. Wilcke (wilcke@esrf.fr)
                  bench(): return error code of gettimeofday() or 0 if no error.
Update 16/09/2002 R. Wilcke (wilcke@esrf.fr)
                  reallocate the message levels for the prmsg() calls.
Update 20/11/2001 R. Wilcke (wilcke@esrf.fr)
                  outname(): handle the case when a name is an empty string.
Update 11/12/2001 R. Wilcke (wilcke@esrf.fr)
                  protect against taking strlen() from NULL pointer.
Update 20/11/2001 R. Wilcke (wilcke@esrf.fr)
                  outname(): protect against infile = NULL pointer.
                  outname(): handle the cases when "inext" or "outext" are NULL;
Update 08/11/2001 R. Wilcke (wilcke@esrf.fr)
                  split the code of "util.c" in two files:
                  - "inout.c" contains all input/output related routines,
                  - "util.c" contains the other routines of the old "util.c".
Update 09/08/2001 R. Wilcke (wilcke@esrf.fr)
                  declare struct *pmem_head as "static".
*/

#include "spd.h"
#include <string.h>

#if MY_MALLOC
/*==============================================================================
 * If MY_MALLOC is defined, an alternative set of routines for memory
 * management is used. The following routines are defined:
 *
 * - pmalloc(), prealloc(), pfree() replace the standard routines malloc(),
 *   realloc() and free();
 * - print_memsize() prints the total allocated memory size.
 *
 * The differences to the standard routines are:
 * - the alternative set provides error printout indicating the current file
 *   and line number where the error occurred;
 * - the alternative set operates on a linked list of data structures of type
 *   "pmem". Each structure contains the size of its data segment, a pointer to
 *   the next element in the list, and a pointer to its own data segment. This
 *   allows to keep track of the overall memory usage by the program with the
 *   routine print_memsize().
 *
 * Otherwise, the input parameters and return values of the routines in the
 * alternative set are the same as the ones in the standard set.
 *
 * If MY_MALLOC is not defined, pmalloc(), prealloc() and pfree() are defined
 * to be the standard set of memory management routines, and print_memsize()
 * is a dummy.
 */

static struct pmem *pmem_head = NULL;

void *_pmalloc(int size,char *file,int line)
{
  struct pmem *ptr;
  /*
   * Allocate the new memory segment, put the address of the last memory
   * segment in the link pointer, and save the address of the new memory
   * segment for use in the link pointer at the next call.
   */
  if((ptr = (struct pmem *)malloc(size + sizeof(struct pmem))) == NULL)
    __prmsg(FATAL,file,line,("cannot allocate %d bytes\n",size));
  ptr->next = pmem_head;
  pmem_head = ptr;
  ptr->data = (void *)(((char *)ptr) + sizeof(struct pmem));
  ptr->size = size;

  return ptr->data;
}

void *_prealloc(void *ptr,int size,char *file,int line)
{
  struct pmem *loop_ptr,*new_ptr,**ref_ptr;

  /*
   * Loop throuth the linked list until the requested data segment is found or
   * the end of the list is reached (then loop_ptr == NULL).
   */
  for(loop_ptr = pmem_head, ref_ptr = &pmem_head; loop_ptr;
    loop_ptr = *(ref_ptr = &loop_ptr->next))
    if(ptr == loop_ptr->data)
      break;
  if(loop_ptr == NULL)
    __prmsg(FATAL,file,line,("pointer was never allocated: 0x%x \n",ptr));

  /*
   * Reallocate the requested data segment.
   */
  if((new_ptr = (struct pmem *)
    realloc(loop_ptr,size + sizeof(struct pmem))) == NULL)
    __prmsg(FATAL,file,line,("cannot reallocate %d bytes \n",
      size + sizeof(struct pmem)));

  *ref_ptr = new_ptr;
  new_ptr->data  = (void*)(((char*)new_ptr) + sizeof(struct pmem));
  new_ptr->size = size;

  return(new_ptr->data);
}

int _pfree(void *ptr,char *file,int line)
{
  struct pmem *loop_ptr, **ref_ptr;
  /*
   * Loop through the linked list until the requested data segment is found or
   * the end of the list is reached (then loop_ptr == NULL).
   */

  for(loop_ptr = pmem_head, ref_ptr = &pmem_head; loop_ptr;
    loop_ptr = *(ref_ptr = &loop_ptr->next)) {
    if(ptr == loop_ptr->data)
      break;
  }
  if(loop_ptr == NULL)
    __prmsg(FATAL,file,line,("pointer was never allocated: 0x%x\n",ptr));
  /*
   * Free the requested data segment, and re-adjust the link pointer.
   */
  *ref_ptr = loop_ptr->next;
  free(loop_ptr);
  return(0);
}

/*==============================================================================
 * Print the total memory size of all allocated data segments.
 */
int print_memsize()
{
  struct pmem *loop_ptr;
  float sum = 0;

  for(loop_ptr = pmem_head; loop_ptr; loop_ptr = loop_ptr->next)
    sum += loop_ptr->size;
  prmsg(DMSG,("Memory usage is now: %7.2f MiB\n",sum / 1024 / 1024));
  return(0);
}

/*==============================================================================
 * Print the list of all allocated data segments.
 */
int print_memlist()
{
  struct pmem *loop_ptr;
  int loop_cnt;

  printf("================================================================================\n");

  for(loop_ptr=pmem_head,loop_cnt=1; loop_ptr; loop_ptr=loop_ptr->next,loop_cnt++)
  {
    printf(" memory segment %3d: %p\n",loop_cnt,loop_ptr);
    printf("           data %3s: %p\n"," ",loop_ptr->data);
    printf("           size %3s: %ld\n"," ",loop_ptr->size);
    printf("           next %3s: %p\n"," ",loop_ptr->next);
    printf("--------------------------------------------------------------------------------\n");
  }
  return(0);
}
#else
int print_memsize()
{
  return(0);
}

int print_memlist()
{
  return(0);
}
#endif /* MY_MALLOC */

/*==============================================================================
 * Switch the byte order in the input array "s" containing "n" elements of type
 * short.
 *
 * Input : s: input data array, type short
 *         n: number of data items in the input data array
 * Output: s: input data array with bytes swapped
 * Return: 0
 */

union swap2_data {
  unsigned char  c[2];
  unsigned short x;
};

int byte_swap2N(register union swap2_data *s,int n)
{
  register int i;
  register unsigned char t;

  for(i = 0; i < n; i++, s++) {
    t = s->c[0];
    s->c[0] = s->c[1];
    s->c[1] = t;
  }
  return(0);
}

/*==============================================================================
 * Determines the minimum and the maximum of the four input values.
 *
 * Input : f1, f2, f3, f4: the four values to be searched for miminum and
 *                         maximum
 * Output: p_minf: the minimum of the four input values
 *         p_maxf: the maximum of the four input values
 * Return: 0
 */

int minmax4(float f1,float f2,float f3,float f4,float *p_minf,float *p_maxf)
{
  int i;
  float ft[3];
  float minf = f1;
  float maxf = f1;

  ft[0] = f2; ft[1] = f3; ft[2] = f4;
  for(i=0; i<3; i++) {
    if(minf > ft[i])
      minf = ft[i];
    if(maxf < ft[i])
      maxf = ft[i];
  }
  *p_minf = minf;
  *p_maxf = maxf;
  return(0);
}

/*==============================================================================
 * Obtains a parameter value from an input argument string.
 *
 * It searches the input argument "arg" for a string of the form
 *
 *   parameter_name=parameter_value
 *
 * where parameter_name is the string contained in the input argument "str".
 *
 * If found, it prints the parameter value according to the format given in the
 * input argument "format" and returns the value in "result".
 *
 * Special case for "%s" format: if "parameter_value" is an empty string, the
 * routine retuns an empty string in "result".
 *
 * Input : arg:    string to be searched for the parameter name and value
 *         name:   string with the parameter name
 *         format: format for the parameter's value, composed of
 *                 - a '%' in the first character
 *                 - optionally followed by the size modifier 'h', or 'l'
 *                 - followed by the conversion letter 's', 'd', 'u', 'f' or 'g'
 * Output: result: the value of the requested parameter
 * Return: 1  if the requested parameter was found
 *         0  otherwise
 */

int scan_argument(char *arg,char *name,char *format,void *result)
{
  char *pfmt,*rescop;
  char name_eq[256];
  static char history_arg[EdfMaxValLen + 1];
  int retval;

  /*
   * Test for valid format specification.
   */
  pfmt = format;
  if(*pfmt++ != '%' || strlib_spn(pfmt + strlib_spn(pfmt,"hl"),"sdufg") != 1) {
    prmsg(ERROR,("unknown parameter format %s wrong\n",format));
    return(0);
  }

  /*
   * Add a "=" (equal sign) to "name" and then search for the string in "arg".
   */
  strncpy(name_eq,name,256);
  strcat(name_eq,"=");

  if(strncmp(arg,name_eq,strlen(name_eq)) == 0) {
    strcat(name_eq,format);

    /*
     * Read the parameter value from the string "arg" according to the format
     * given.
     *
     * Return 0 if parameter name cannot be found.
     */
    if((retval = sscanf(arg,name_eq,result)) == 0) {
      prmsg(ERROR,("unknown parameter %s=xxx\n",name));
      return(0);

    /*
     * If found, print parameter value.
     */
    } else {
      strcat(name_eq,"\n");
      if(*pfmt == 's') {
        /*
         * The sscanf() function skips whitespace for "%s" format, and returns
         * an EOF error if an argument contains only whitespace. This would
         * therefore mean that the input value of "result" remains unchanged.
         *
         * To get an empty string argument from the input, check therefore if
         * the last character in the input is an equal sign "=".
         */
	if(retval == EOF && *(arg + strlen(arg) - 1) == '=') {
	  *(char *)result = '\0';
          sprintf(history_arg,"%s=",name);
	} else {
	  rescop = strlib_newstr(result);
          sprintf(history_arg,"%s=%.*s",name,EdfMaxValLen - 11,rescop);

	  strlib_free(rescop);
	}
	prmsg(DMSG,(name_eq,result));
      } else {
        sprintf(history_arg,"%s%s",name,strchr(arg,'='));
        if(*pfmt == 'f' || *pfmt == 'g')
 	  prmsg(DMSG,(name_eq,*(float *)result));
        else if(*pfmt == 'd')
	  prmsg(DMSG,(name_eq,*(int *)result));
        else if(*pfmt == 'u')
	  prmsg(DMSG,(name_eq,*(unsigned int *)result));
        if(strcmp(pfmt,"lf") == 0 || strcmp(pfmt,"lg") == 0)
 	  prmsg(DMSG,(name_eq,*(double *)result));
        if(strcmp(pfmt,"hd") == 0)
 	  prmsg(DMSG,(name_eq,*(short *)result));
        if(strcmp(pfmt,"hu") == 0)
 	  prmsg(DMSG,(name_eq,*(short unsigned *)result));
        if(strcmp(pfmt,"ld") == 0)
 	  prmsg(DMSG,(name_eq,*(long *)result));
        if(strcmp(pfmt,"lu") == 0)
 	  prmsg(DMSG,(name_eq,*(long unsigned *)result));
      }
    }
    edf_history_argv("InputArg",history_arg);
    return(1);
  } else
    return(0);
} // scan_argument

/*==============================================================================
 * Analyzes a list of filename patterns and extracts filenames from them.
 *
 * At the first call, or the next call after a reset, the routine starts at the
 * beginning of the list of filename patterns, analyzes the first pattern and
 * if successful returns with the first filename.
 *
 * For the following calls, the routine keeps internally trace of how far the
 * pattern list has been analyzed, and returns subsequent filenames until an
 * error occurs or the pattern list is exhausted.
 *
 * When the pattern list has been exhausted, the internal state of the routine
 * is reset. This can also be done manually by calling the routine with a non-
 * positive number of file name patterns.
 *
 * There can be an arbitrary number of filename patterns as input to the
 * routine.
 *
 * The routine creates a sequence of filenames from a filename template with
 * some wildcard characters. These wildcard characters are replaced by numbers
 * according to the numerical information contained in the filename pattern.
 *
 * The filename pattern consists of up to four elements, separated by commas:
 * 1) the filename template: essentially a filename (possibly with path), but
 *    the name itself can contain one or several wildcard characters "%"
 *    (percent sign). These need not be contiguous in the template;
 * 2) the start value of the numerical sequence;
 * 3) the end value of the numerical sequence;
 * 4) the increment between two sequence numbers.
 *
 * In a filename pattern with wildcards, the first filename returned will be
 * the template string with the wildcard characters replaced by the start value
 * of the numerical sequence. The second filename will then contain the start
 * value plus the increment, and so on, until the generated number passes the
 * end value of the sequence.
 *
 * Negative numbers are allowed, even for the increment. The sequence then
 * counts downwards.
 *
 * Under the following circumstances the routine will return exactly one
 * filename (with the start value of the sequence):
 * - if the increment is 0;
 * - if (start value) > (end value) and the increment is positive;
 * - if (start value) < (end value) and the increment is negative;
 *
 * If the wildcard pattern is longer than the significant digits in the number,
 * it will be left padded with zeros. If the number is negative, the first
 * wildcard is replaced by a "-" (minus). If the wildcard pattern is shorter
 * than the significant digits in the number, the number will be truncated.
 *
 * Not all elements of the sequence need to be specified. Missing elements are
 * replaced by default values:
 * - start value = 1;
 * - end value = start value;
 * - increment = 1.
 *
 * If the filename template does not contain any wildcards, the template string
 * is returned as filename.
 *
 * Examples:   pattern                filenames created
 *             abc%%d,1,3,2           abc01d    abc03d
 *             abc%%de%%,497,495,-1   ab04de97  ab04de96  ab04de95
 *             abcdef                 abcdef
 *             abc%%%                 abc001
 *             abc%%%,8               abc008
 *             abc%%%,4,7             abc004    abc005    abc006 abc007
 *             abc%%%,-7,-6           abc-07    abc-06
 *
 * Input : filnam: buffer to receive character string. Must be allocated by
 *                 calling program
 *         fillen: length of buffer filnam
 *         patc:   number of file name patterns to analyze. If <= 0, the
 *                 internal state of the routine is reset
 *         patv:   array of strings with the file name patterns
 * Output: filnam: most recent filename obtained from the pattern analysis
 * Return: -1  (= error) if input arguments are incorrect or invalid
 *         -2  (= end) if pattern list has been exhausted (end of present
 *             analysis) or if manual reset has been done
 *          0  else (= OK, new filename obtained)
 */

int fnampat(char *filnam,size_t fillen,int patc,char *patv[])
{
  static char filbuf[FILENAME_MAX];
  char number[1024];
  char *ptrpatv;
  static int patn = 0;
  int err,numlen = sizeof(number);
  static long num,fst,lst,inc,maxloop,loop = 0;

  /*
   * Check for incorrect input buffer arguments. If so, return error.
   */
  if(fillen <= 0 || filnam == NULL)
    return(-1);
  *filnam = '\0';

  /*
   * Check if the input pattern list is exhausted, or a manual reset has been
   * requested. Return with end code.
   */
  if(patn >= patc || patc <= 0) {
    patn = 0;
    loop = 0;
    return(-2);
  }

  /*
   * Check if the filename pattern has wildcards. If not, return just this
   * pattern string and advance the pattern list pointer to the next pattern.
   */
  ptrpatv = *(patv + patn);
  if(filename_has_pattern(ptrpatv) == 0) {
    strlib_nconcat(filnam,fillen,filnam,ptrpatv,fillen - 1);
    patn++;
    return(0);
  }

  /*
   * Check if at the beginning of a pattern analysis (then the pattern loop
   * counter is 0). If so, extract the filename template and possibly the
   * numerical arguments. Set default values for the numerical arguments if they
   * are not provided in the pattern.
   */
  if(loop == 0) {
    if(filename_parameter(filbuf,sizeof(filbuf),ptrpatv,0) == NULL)
      return(-1);
    fst = num_str2long(filename_parameter(number,numlen,ptrpatv,1),NULL,&err);
    if(err)
      fst = 1;
    lst = num_str2long(filename_parameter(number,numlen,ptrpatv,2),NULL,&err);
    if(err)
      lst = fst;
    inc = num_str2long(filename_parameter(number,numlen,ptrpatv,3),NULL,&err);
    if(err)
      inc = 1;

    /*
     * Determine number of iterations for the pattern, and set file index number
     * to the first value.
     */
    maxloop = inc == 0 ? 1 : (lst - fst) / inc + 1;
    if(maxloop < 1)
      maxloop = 1;
    num = fst;
  }

  /*
   * Construct the actual filename from the pattern with the present file index
   * number.
   */
  filename_pattern(filnam,fillen,filbuf,num);

  /*
   * Check if at the end of a pattern analysis. If so, advance the pattern list
   * pointer to the next pattern and reset loop counter. If not, set the file
   * index number to its next value for the next call of the routine.
   */
  if(++loop >= maxloop) {
    loop = 0;
    patn++;
  } else
    num+=inc;
  return(0);
}

/*==============================================================================
 * Creates a name for an output file by taking the string "infile" with the
 * name of the input file and replacing the extension contained in "inext" by
 * the string contained in "outext". The resulting file name can optionally be
 * preceded by the directory path contained in "outdir".
 *
 * More precisely, the "infile" is searched for the occurrence of the substring
 * given by "inext". If found, "inext" in the string "infile" is replaced by
 * "outext". It does not need to be separated by e.g. a dot, and the substring
 * does not even have to be at the end of "infile". As an example, with "infile"
 * containing "datafile3.dat", "inext" containing "e3" and "outext" containing
 * "_cor", the output file name is "datafil_cor.dat".
 *
 * If "infile" is a NULL pointer or empty, the output file name is "outext".
 * If "outext" is also a NULL pointer, the routine returns a NULL pointer.
 *
 * If "inext" is a NULL pointer or is not found as a substring of "infile", then
 * the output file name is created by concatenating "infile" with "outext". If
 * "outext" is empty or NULL, this means that the output file name is the same
 * as the input file name.
 *
 * If "outext" is a NULL pointer or empty, the output file name is "infile",
 * with the substring "inext" removed if found.
 *
 * The resulting file name will be preceded by the directory path contained in
 * "outdir", if
 * - "outdir" does contain a real string (not NULL pointer and not empty);
 * - "infile" does not contain a directory part (i.e. the "infile" string does
 *   not contain the directory separator "/" anywhere).
 *
 * No tests are made that the string in "outdir" is of the type "directory path"
 * (e.g. with at least one "/" separator at the end of the string), thus the
 * routine could also be used to add a prefix to the output file name.
 *
 * Note that the routine returns a pointer to a static character array. It will
 * therefore be overwritten by each subsequent call, but it needs not (cannot)
 * be freed.
 *
 * If the infile name is followed by a parameter list which is separated with 
 * commas the full parameter list is first removed from infile and finally
 * appended to the generated output file name.
 *
 * NOTE: Each parameter specifies a single file number or image number. It does 
 *       not specify fst, lst, incr!
 *
 * Input : infile: string with the name for the input file[,params]
 *         outdir: string with the (optional) directory path for the output file
 *         inext:  string with the extension for the input file name
 *         outext: string with the extension for the output file name[,params]
 *                 If outext contains a directory, e.g. <dir>/<ext>,
 *                 <dir> is inserted after outdir and <ext> is used
 *                 as filename extension: 
 *                 <outfile>=<outdir>/<dir>/<infile><ext>.
 *                 
 * Output: none
 * Return: string with the name for the output file
 *         NULL   if no output file name could be created
 */

char *outname(const char *infile,const char *outdir,const char *inext,const char *outext)
{
  char *pos;
  size_t len;
  static char outfile[1024];
  char inpattern[1024]="", inparams[1024]="";
  char out[1024]="", ext[1024]="";
  char filename[1024];

  /*
   * Reset outfile to empty string
   */
  *outfile='\0';

  /*
   * Separate infile in pattern and params
   */
  if ( !(filename_parameter ( inpattern, 1024, infile, 0 ))) {
    *inpattern='\0'; // empty string
  }
  if ( !(filename_parameter ( inparams, 1024, infile, -1 ))) {
    *inparams='\0'; // empty string
  }

  /*
   * Separate outext in directory out and extension ext
   */
  if ( !(filename_path (out,1024,outext)) ) {
    *out='\0'; // empty string
  }
  if (!strcmp(out,"./"))
    *out='\0'; // empty string

  if ( !(filename_name (ext,1024,outext)) ) {
    *ext='\0'; // empty string
  }

  /*
   * replace %-pattern in out with 0
   */
  filename_pattern ( out, 1024, out, 0 ); 

  /*
   * Put output directory path at the beginning of the filename if 
   * - it exists (not NULL pointer and not empty);
   * - the file name itself does not contain a directory path.
   */
  if (filename_has_path( inpattern )) {
    // use path of infile
    filename_path(outfile,1024,inpattern);
    filename_name(inpattern,1024,inpattern);
  } else {
    // use outdir
    filename_path(outfile,1024,outdir);
  }

  /*
   * Append out
   */
  strlib_nconcat(outfile,1024,outfile,out,strlen(out));

  /*
   * Create directory outfile
   * (see man 2 stat)
   */
  if ( !strlib_is_empty(out) ) {
    // mkdir outfile
    struct stat st = {0};

    if (stat(outfile, &st) == -1) {
        mkdir(outfile, 0755); // drwxr-xr-x
    }
  }

  /*
   * Remove inext from inpattern (infile) and append the rest
   */
   len = strlen(inpattern);
   if ( !strlib_is_empty(inext) ) {
     if ( (pos = (char *)strstr(inpattern,inext)) != NULL) {
       len = pos - inpattern;
     }
   }
   strlib_nconcat(outfile,1024,outfile,inpattern,len);

  /*
   * Append ext to outfile
   */
  strlib_nconcat(outfile,1024,outfile,ext,strlen(ext));

  /*
   * Check created filename
   */
  if (strlen(filename_name(filename,1024,outfile))<1)
    return((char *)NULL); // filename is empty

  /*
   * Concatenate outfile with inparams
   */
  if (*inparams!='\0') {
    strlib_nconcat(outfile,1024,outfile,",",1);
    strlib_nconcat(outfile,1024,outfile,inparams,strlen(inparams));
  }
  
  return(outfile);
} // outname

/*==============================================================================
 * Measures and prints the time difference between two consecutive calls.
 *
 * At the first call to this routine, the input argument "str" must be a NULL
 * pointer. This sets the start time.
 *
 * All subsequent calls with "str" not equal to a NULL pointer measure the time
 * between the start time and the time of the present call, print it together
 * with "str" and store the time of the present call into the start time as
 * preparation for the next call. It is thus possible to time a whole sequence
 * of events.
 *
 * To end such a sequence and start a new one, the routine can at any time be
 * called again with a NULL pointer, thus resetting the start time.
 *
 * Input : str: if NULL: resets the start time to the time of the present call
 *              else:    string to be printed with the time difference
 * Output: none
 * Return: if no error: 0
 *         else       : error code of gettimeofday() function
 */

int bench(char *str)
{
  int iret=-1;

#ifndef WIN32
  static struct timeval  start, stop;

  if(str == (char *)NULL) {
    iret = gettimeofday(&start,NULL);
  }
  else {
    if((iret = gettimeofday(&stop,NULL)) == 0) {
      prmsg(DMSG,("Time in %s : %10.3f\n",str,(double)(stop.tv_sec-start.tv_sec)
        + (double)(stop.tv_usec-start.tv_usec) * (double)0.000001));
      start.tv_sec = stop.tv_sec;
      start.tv_usec = stop.tv_usec;
    }
  }
#endif

  return(iret);
}

/*==============================================================================
 * get the time in seconds with us resolution to do some simple benchmarking
 * =============================================================================
 */

double get_time( void ) 
{ 
  double value=-1.0;

#ifndef WIN32
    struct timeval now;
    gettimeofday(&now,NULL);
    value =  (double) now.tv_sec+now.tv_usec*1e-6;
#endif

  return( value );
}

/*==============================================================================
 * Temporary data buffers
 * =============================================================================
 */

struct raw_buffer_descr {
  int valid;       // >0 indicates that the descriptor is valid
  void *buffer;    // pointer to data
  size_t bufsize;  // size of data buffer in bytes
  int mtype;       // machine type of allocated data buffer elements
  int rows;        // number of rows (dim_2)
  int cols;        // number of columns (dim_1)
  int imgnum;      // image number
  int freedata;    // >0 indicates that the buffer can be deallocated, e.g.
                     // deallocation of shared memory buffers is not allowed. 
};

// Define a raw data buffer descriptor for each type.
// If RawBufferDescrInitialized is 0 the raw data descriptors are internally 
// initialized with _init_raw_buffer_descrs().
static struct raw_buffer_descr RawBufferDescr[MAXTYP];
static int RawBufferDescrInitialized=0;

/* Defined in spd.h
  int raw_buffer_is_available(int type);
  int release_raw_buffer(int type);
  int set_raw_buffer(int type, void *data, int imgnum, int rows, int cols, int mtype);
  int get_raw_buffer_params(int type, void **buffer, int *imgnum, int *rows, int *cols, int *mtype);
*/

/*
 * Sets *pdescr to default values
 */
int _default_raw_buffer_descr(struct raw_buffer_descr *pdescr) {

  if (pdescr) {
    pdescr->valid=0;
    pdescr->buffer=NULL;
    pdescr->bufsize=0;
    pdescr->mtype=InValidMType;
    pdescr->rows=0;
    pdescr->cols=0;
    pdescr->imgnum=1;
    pdescr->freedata=0;
  }

  return( 0 );

} // _default_raw_buffer_descr

/*
 * According to C default this routine only needs to
 * set the RawBufferDescrInitialized flag.
 */
int _init_raw_buffer_descrs() {
  int type;
  struct raw_buffer_descr *pdescr;

  for (type=0;type<MAXTYP;type++) {
    pdescr=&(RawBufferDescr[type]);
    _default_raw_buffer_descr(pdescr);
  }
  RawBufferDescrInitialized=1;
  return(0);
} // _init_raw_buffer_descrs

/*
 * Returns the pointer to the raw buffer descriptor of type
 * or NULL if it is not available.
 */         
struct raw_buffer_descr *_praw_buffer_descr( int type ) {
  struct raw_buffer_descr *pdescr=NULL;
  if (!RawBufferDescrInitialized) _init_raw_buffer_descrs();
  if ((0<=type)&&(type<MAXTYP)) {
    pdescr=&(RawBufferDescr[type]);
  }
  return( pdescr );
} // _praw_buffer_descr

/*
 * Return:  1  data buffer is available
 *          0  no data buffer available
 */
int raw_buffer_is_available(int type) {
  struct raw_buffer_descr *pdescr;
  if (!RawBufferDescrInitialized) _init_raw_buffer_descrs();
  pdescr =  _praw_buffer_descr( type );
  if (!pdescr) return(0); // no data buffer available
  return( pdescr->valid );
} // raw_buffer_is_available

/*
 * Release the raw data buffer
 * Return: -1  if error
 *          0  else
 */
int release_raw_buffer(int type) {
  struct raw_buffer_descr *pdescr;
  if (!RawBufferDescrInitialized) _init_raw_buffer_descrs();
  pdescr =  _praw_buffer_descr( type );
  if (!pdescr) return(-1);

  if (pdescr->valid > 0) {
    if (pdescr->freedata > 0 ) {
      if (pdescr->buffer) pfree(pdescr->buffer);
    }
  }
  _default_raw_buffer_descr(pdescr);

  return(0);

} // release_raw_buffer

/*
 * Creates a temporary data buffer for type and saves data in the allocated 
 * buffer.
 * Return: -1  if error
 *          0  else
 */
int set_raw_buffer(int type, void *data, int imgnum, int rows, int cols, int mtype) {
  struct raw_buffer_descr *pdescr;
  if ( release_raw_buffer( type )) return(-1);
  pdescr =  _praw_buffer_descr( type );
  if (!pdescr) return(-1);

  if (data) {
    size_t item_size;
    size_t data_size=0;
    void *buffer_ptr=NULL;
    item_size = edf_machine_sizeof(mtype);
    if (!item_size) return(-1);
    data_size = item_size * rows * cols;
    buffer_ptr = (void *)pmalloc(data_size);
    if (!buffer_ptr) return(-1);

    if (!memcpy(buffer_ptr,data,data_size)) return(-1);

    pdescr->buffer=buffer_ptr;
    pdescr->bufsize=data_size;
    pdescr->mtype=mtype;
    pdescr->rows=rows;
    pdescr->cols=cols;
    pdescr->imgnum=imgnum;
    pdescr->freedata=1;

    pdescr->valid=1;
  }

  return(0);

} // set_raw_buffer

/*
 * For type get the buffer pointer, image number, rows, columns and machine data type.
 * Return: -1  error
 *          1  descr is not valid
 *          0  OK
 */
int get_raw_buffer_params(int type, void **buffer, int *imgnum, int *rows, int *cols, int *mtype) {
  struct raw_buffer_descr *pdescr;
  if (!RawBufferDescrInitialized) _init_raw_buffer_descrs();
  pdescr =  _praw_buffer_descr( type );
  if (!pdescr) return(-1);

  if (buffer) *buffer = pdescr->buffer;
  if (imgnum) *imgnum = pdescr->imgnum;
  if (rows) *rows     = pdescr->rows;
  if (cols) *cols     = pdescr->cols;
  if (mtype) *mtype   = pdescr->mtype;

  return((pdescr->valid>0)?0:1);
} // get_raw_buffer_params

/*==============================================================================
 * Replacement of header keys in the input string with their values
 *
 * If the input string contains one or more substrings between square brackets
 * these substrings are interpreted as header keys and it is tried replacing
 * each substring (including the square brackets) with the corresponding header 
 * key value. If the key cannot be found in the header nothing is replaced
 * and the number of unchanged keys is returned in *pstatus. If the key in the
 * input string is followed by two comma separated numbers, the value is split 
 * at white spaces in parameters and only the parameter corresponding to 
 * the second number is returned. The first number is ignored to be compatible
 * to the saxs programs, where the first number denotes the number of the input
 * image, which has no meaning here.
 * The input string is copied to a temporary buffer, then the output buffer
 * is filled.
 * The number 0 returns the full value string.
 *
 * Examples:
 *
 *  npar = 0
 *  header:             {  ..., "Title" : "parn1 parn2 parn3", ... }
 *  string:             "This is [Title]"
 *  *pstatus            0
 *  return value:       "This is parn1 parn2 parn3"
 *
 *  npar = 1
 *  header :            { ..., "Dim_1" : "512 pixels", ... }
 *  string :            "3*[Dim_1]"
 *  *pstatus            0
 *  return value :      "3*512"
 *
 *  npar = <any> (using parameter after Title)
 *  header:             {  ..., "Title" : "parn1 parn2 parn3", ... }
 *  string:             "This is [Title,,2]"
 *  *pstatus            0
 *  return value:       "This is parn2"
 *
 *  npar = <any>
 *  header:             {  ... } (without Info)
 *  string:             "This is [Info,,2]"
 *  *pstatus            1
 *  return value:       "This is [Info,,2]"
 *
 * Input :   buffer : output buffer
 *           buflen : output buffer size in bytes
 *           string : input string containing None or several header keys
 *             npar : default number of parameter (0 takes the full string)
 * Return: *pstatus : output status, 
 *                    0 if all header keys could be replaced
 *                    >0 number of header keys that could not be replaced
 *                    <0 error
 *
 * Return value:   success: pointer to output buffer
 *                 error:   NULL
 */
# define SPDRPK_START                   '['
# define SPDRPK_END                     ']'
# define SPDRPK_SEPARATOR               ','
void _spnum( long *value, long defval,
            char ** pb, const char **ps, long npar,
            const char *header, int level );
char * _spkey( char ** pb, const char **ps, long npar,
              const char *header, int level, int *pstatus);
char * _spstring( char **pb, const char **ps, long npar,
                 const char *header, int level, int *pstatus);
char *keyvalue ( char *buffer, size_t buflen, const char *header,
                 const char *string, long npar, int *pstatus )
{ char tmpbuf[1024];
  char *pb, *value=NULL;
  const char *ps;
  size_t j;

  int status = -1;

  prmsg(DMSG,(" -- keyvalue\n"));

  if ((!buffer)||(!buflen)) goto keyvalue_error;

  // prepare output string
  //pb = buffer;
  pb = tmpbuf;
  for (j=1;j<1024;j++)
    *pb++ = ' '; // fill with spaces
  *pb = '\0'; // terminate with NUL

  // copy
  pb = tmpbuf;
  ps = string;
  value = _spstring( &pb, &ps, npar, header, 0, &status);
  if (status<0) goto keyvalue_error;

  // calculate substrings of type num((<expression>))
  value = strlib_numberstring( buffer, buflen, value, NULL );

  prmsg(DMSG,(" -- keyvalue END (status = %d, value=\"%s\")\n",status,value));
  if (pstatus) *pstatus=status;

  return( value );

keyvalue_error:

  prmsg(DMSG,(" -- keyvalue END (status = %d)\n",status));
  if (pstatus) *pstatus=status;
  return( NULL );

} // keyvalue

/*+++------------------------------------------------------------------------
NAME

   _spstring --- copy input string to output string and insert header values

SYNOPSIS
   *ps points to the current position in the input string (updated)
   *pb points to the current position in the output string (updated)
       The output string must be initialized over its useful length with
       non-zero characters and must be terminated by character NULL.
   if npar is 0, the full value string is copied.
   If npar is >=1, only the substring between the npar-th-1 and the
   npar-th delimeter is copied.
   Delimeters are white-spaces.
   The pointer to the output string is returned. The pointers *pb and *ps are
   incremented. *pb is only incremented until **pb is character NULL.
   The parameters itself can be header values.

----------------------------------------------------------------------------*/
char * _spstring( char **pb, const char **ps, long npar,
                 const char *header, int level, int *pstatus)
{  char *value;
   int status=0;

   prmsg(DMSG,(" -- _spstring (level = %d) in = \"%s\"\n",level,*ps));

   value = *pb;
   while ((**pb)&&(**ps))
     switch (**ps) {
       case SPDRPK_START:
         (*ps)++; _spkey(pb,ps,npar,header,level+1,&status); break;
       default :
         **pb = **ps; (*pb)++; (*ps)++; break;
     } /* switch */

   if (status<0) goto _spstring_error;

   **pb = '\0'; // terminate with NUL

   prmsg(DMSG,(" -- _spstring END (level = %d, status = %d, value=\"%s\") next = \"%s\"\n",level,status,value,*ps));
   if (pstatus) *pstatus=status;
   return( value );

_spstring_error:

   prmsg(DMSG,(" -- _spstring END (level = %d, status = %d, value=\"%s\")\n",level,status,value));
   if (pstatus) *pstatus=status;
   return( value );

} /* _spstring */

/*+++------------------------------------------------------------------------
NAME

   _spkey --- extract key from input string and insert value in output string

SYNOPSIS
   *ps points to the current position in the input string (updated)
   *pb points to the current position in the output string (updated)
       The output string must be initialized over its useful length with
       non-zero characters and must be terminated by character NULL.
   if npar is 0, the full value string is copied.
   If npar is >=1, only the substring between the npar-th-1 and the
   npar-th delimeter is copied.
   Delimeters are white-spaces.
   The pointer to the output string is returned. The pointers *pb and *ps are
   incremented. *pb is only incremented until **pb is character NULL.

USAGE
   input string  "[<key>,<npar>]"
          <key>: header key (string), no default
         <npar>: parameter number (long integer), default npar

----------------------------------------------------------------------------*/
char * _spkey( char ** pb, const char **ps, long npar, 
              const char *header, int level, int *pstatus)
{  const char *psstart, *psstop;
   const char *keyval, *pL;
   char *value;
   char key[EdfMaxKeyLen+1], *pk; //, *pkend; unused-but-set-variable

   long i, ipar;
   int j;

   int status = 0;
   int edferror, edfstatus;

   prmsg(DMSG,(" -- _spkey (level = %d) in = \"%s\"\n",level,*ps));

   // remember start position
   psstart = *ps;
 
   // prepare key string 
   pk = key;
   for (j=0;j<EdfMaxKeyLen;j++)
     *pk++ = ' '; // fill with spaces
   *pk = '\0'; // terminate with NUL

   ipar      = npar;
 
   value = *pb;
   pk = key;
   // pkend = key+EdfMaxKeyLen; unused-but-set-variable
   while ((*pk)&&(**ps))
     switch (**ps) {
       case SPDRPK_START:
         // in levels > 1 copy only the first parameter
         (*ps)++; _spkey(&pk,ps,1,header,level+1,&status); break;
       case SPDRPK_SEPARATOR: 

         // skip first number
         (*ps)++;
         _spnum(NULL,0,&pk,ps,1,header,level+1);

         // read second number 
         if (**ps == SPDRPK_SEPARATOR ) {
           (*ps)++; 
           _spnum(&ipar,ipar,&pk,ps,1,header,level+1);
         }         

         // ignore further numbers and search SPDRPK_END:
         while (**ps == SPDRPK_SEPARATOR ) {
           (*ps)++;
           _spnum(NULL,0,&pk,ps,1,header,level+1);
         }
         break;
       case SPDRPK_END:
         *pk = '\0'; (*ps)++; break;
       default :
         *(pk++) = **ps; (*ps)++; break;
     } /* switch */

   // get end position
   psstop = *ps;

   if (status<0) goto _spkey_error;

   prmsg(DMSG,("    _spkey: key = %s, ipar = %ld, next = \"%s\"\n",key,ipar,*ps));

   // copy keys to *pb
   if ( edf_search_header_element(header,key,&keyval,&edferror,&edfstatus)&&(!edfstatus) ) {
     // copy keyval to output buffer
     pL = keyval;
     if ( ipar == 0 ) { // copy full string
        while ( (**pb) && (*pL) ) *(*pb)++ = *pL++;
     } else { // copy ipar-th substring
       // search start position
       i=1;
       while ( (i<ipar)&&(*pL) ) 
         if (strlib_is_white(*pL++)) i++;
       // copy substring 
       while ( (**pb) && (*pL) && !(strlib_is_white(*pL)) )
         *(*pb)++ = *pL++;
     }
   } else {
     // cannot replace key, copy input string to output buffer
     *ps = psstart;
     if (**pb) *(*pb)++ = SPDRPK_START;
     while ((**pb)&&(*ps<psstop))
       *(*pb)++ = *(*ps)++;
     if (status>=0) status++; // count not replaced keys
   }

   prmsg(DMSG,(" -- _spkey END (level = %d, status = %d, value=\"%s\") next = \"%s\"\n",level,status,value,*ps));
 
   if (pstatus) *pstatus=status;
   return( value );

_spkey_error:

   prmsg(DMSG,(" -- _spkey END (level = %d, status = %d, value=\"%s\")\n",level,status,value));

   if (pstatus) *pstatus=status;
   return( value );
 
} /* _spkey */

/*+++------------------------------------------------------------------------
NAME

   _spnum --- scans input string for a long integer expression

SYNOPSIS
   *ps points to the current position in the input string (updated). It is
   always incremented either until a string terminator '\0' is found
   or until the characters SPDRPK_SEPARATOR or SPDRPK_END are found. In
   both cases *ps points to the termination character.
   In case of success value is updated with the read integer number,
   otherwise it contains the default.
   If value is NULL, only ps is updated.
----------------------------------------------------------------------------*/
void _spnum( long *value, long defval,
            char ** pb, const char **ps, long npar,
            const char *header, int level )
{  char num[EdfMaxValLen+1], *pn, *pnend;
   int j;

   int status = 0, err = 0;

   prmsg(DMSG,(" -- _spnum     (level = %d) in = \"%s\"\n",level,*ps));

   // prepare num string
   pn = num;
   for (j=0;j<EdfMaxValLen;j++)
     *pn++ = ' '; // fill with spaces
   *pn = '\0'; // terminate with NUL

   // copy number string
   pn = num;
   pnend = num+EdfMaxValLen;
   while ((*pn)&&(**ps))
     switch (**ps) {
       case SPDRPK_START:
         // in levels > 1 copy only the first parameter (default)
         (*ps)++; _spkey(&pn,ps,1,header,level+1,&status); break;
       case SPDRPK_SEPARATOR:
       case SPDRPK_END:
         *pn = '\0'; break;
       default :
         if (pn<pnend) *(pn++) = **ps; (*ps)++; break;
     } /* switch */

   if ( value ) {
     *value = num_str2long ( num, NULL, &err);
     if (err) *value = defval;
   }

   prmsg(DMSG,(" -- _spnum END (level = %d) next = \"%s\"\n",level,*ps));

   return;

} /* _spnum */
