/*
 *   Project: The SPD Image correction and azimuthal regrouping
 *                      http://forge.epn-campus.eu/projects/show/azimuthal
 *
 *   Copyright (C) 2001-2010 European Synchrotron Radiation Facility
 *                           Grenoble, France
 *
 *   Principal authors: P. Boesecke (boesecke@esrf.fr)
 *                      R. Wilcke (wilcke@esrf.fr)
 *                      J. Kieffer (kieffer@esrf.fr)
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published
 *   by the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   and the GNU Lesser General Public License  along with this program.
 *   If not, see <http://www.gnu.org/licenses/>.
 */

/*
Update   07/10/2017 P. Boesecke (boesecke@esrf.fr)
                    read_esrf_file: argument list updated
Update   16/06/2017 P. Boesecke (boesecke@esrf.fr)
                    average_init: the averaging buffers are allocated 
                    using new_avg_data.
Update   07/06/2017 P. Boesecke (boesecke@esrf.fr)
                    read_esrf_file: argument list updated
Update   18/06/2016 P. Boesecke (boesecke@esrf.fr)
                    edf_read_data: datasize is the size of the data buffer
Creation 16/06/2016 P. Boesecke (boesecke@esrf.fr)
                    The function average_esrf_file averages all (default) or 
                    selected images of an input data file, return a single 
                    header and a single data buffer.
                    All images must have identical size.
*/

# include "spd.h"
# define SIZE_F (unsigned long)


const char percentchar='%';

/*
 * Keeping averaging parameters in a static structure.
 * All elements of average_params are initialized
 * to NULL (C language default).
 */
static SPDAvgParams average_params;

/*==============================================================================
 * This routine returns a pointer to the static structure average_params.
 * The init flag is set to 1.
 */
SPDAvgParams *spd_avg_params()
{ return(&average_params);
} // spd_avg_params

/*==============================================================================
 * This routine sets the defaults of all averaging parameters in ap.
 * The init flag is set to 1.
 */
void _spd_avg_params_clear( SPDAvgParams *ap )
{
  if (ap) {
    // clear ap
    memset(ap,0,sizeof(SPDAvgParams));
    // set ap default values
    sprintf(ap->Framename.V,"%s","framebuffer.edf");
    sprintf(ap->AMode.V,"%s","mean");
    sprintf(ap->FMode.V,"%s","None");
    ap->NM1.V=1;
    ap->SIGM.V=5.0;
    ap->SIGP.V=5.0;
    ap->MINS.V=0.5;
    ap->LOQ.V=0.1;
    ap->UPQ.V=0.9;
    ap->PQ.V=0.5;
    ap->FILL.V=0;
    ap->FIL1.V=3.0;
    ap->FIL2.V=3.0;
    ap->init=1;
  }
} // spd_avg_params_clear

void spd_avg_params_clear()
{ _spd_avg_params_clear(&average_params);
} // spd_avg_params_clear

/*==============================================================================
 * This routine initializes ap only if the flag ap->init is not set.
 */
void _spd_avg_params_init( SPDAvgParams *ap )
{
  if (ap) {
    if (!(ap->init)) spd_avg_params_clear( ap );
  }
} // spd_avg_params_init

void spd_avg_params_init()
{ _spd_avg_params_init(&average_params);
} // spd_avg_params_init

/*==============================================================================
 * This routine prints ap. 
 */
void _spd_avg_params_print( SPDAvgParams *ap )
{
  if (ap) {
    printf("average mode            : %s\n", avgmode2string(string2avgmode(ap->AMode.V)));
    printf("filter mode             : %s\n", avgmode2string(string2avgmode(ap->FMode.V)));
//    printf("create variances        : %s\n", pcb->CreateVariance.V?"yes":"no");
    printf("variance calculation    : %s\n", ap->NM1.V?"n-1":"n");
    printf("sigma-                  : %f\n", ap->SIGM.V);
    printf("sigma+                  : %f\n", ap->SIGP.V);
    printf("minimum sigma           : %f\n", ap->MINS.V);
    printf("lower quantil loq       : %f\n", ap->LOQ.V);
    printf("upper quantil upq       : %f\n", ap->UPQ.V);
    printf("quantil (amode)         : %f\n", ap->PQ.V);
    printf("fill empty points       : %s\n", ap->FILL.V?"yes":"no");
    if (ap->FILL.V) {
      printf("fill averaging length 1 : %f\n", ap->FIL1.V);
      printf("fill averaging length 2 : %f\n", ap->FIL2.V);
    }
  }
} // spd_avg_params_print

void spd_avg_params_print()
{ _spd_avg_params_print(&average_params);
} // spd_avg_params_print

/*---------------------------------------------------------------------------
NAME
  average --- Creates average and variance from a series of images.

PURPOSE
  Creates average and variance from a series of images.

average_init     : initialization of ihb variables in first image
average_esrf_file: loop
average_term     : finalization (final calculation, releasing variables ...)
---------------------------------------------------------------------------*/
AvgData *average_init ( long dim_1, long dim_2, long dim_3, int verbose, int *pstatus )
{
  AvgData *frames=NULL;

  int status=-1;

  prmsg(DMSG,("average_init BEGIN\n"));

  if (!(frames = new_avg_data(NULL, dim_1, dim_2, dim_3)))
    goto average_init_error;

  prmsg(DMSG,("average_init END\n"));

  status=0;

  if (pstatus) *pstatus = status;

  return(frames);

average_init_error:

  free_avg_data(frames);

  if (pstatus) *pstatus = status;

  prmsg(DMSG,("average_init END (%d)\n",status));

  return(NULL);

} // average_init

/*---------------------------------------------------------------------------
NAME
  average_term --- Averages all frames and write the result to data and vardat

 Output:         *data: output data buffer, destination of averaging
                        (dimension dim_1*dim_2)
 Output:       *vardat: output variance buffer, destination of averaging 
                        or NULL variance calculation is omitted.
                        (dimension dim_1*dim_2)
 Input:  dummy, ddummy: dummy and ddummy values of the output data buffer
                        (constant VarDummy is used for vardat)
 Input:         frames: frame buffer with averaging data 
                        (dimension dim_1*dim_2*dim_3)
 Input:             ap: averaging parameters
 Input:       verbose : verbose level as used in inout.c
 Output:      *pstatus: return status, 0 in case of Success
---------------------------------------------------------------------------*/
void average_term (float *data, float *vardat, float dummy, float ddummy, 
                   AvgData *frames, long dim_1, long dim_2, long dim_3, 
                   SPDAvgParams *ap, int verbose, int *pstatus)
{
  float *I0Data, *E0Data;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;

  int  errval;
  int  stream; // optional output stream for writing framebuffer

  AvgListOfPoints *emptypoints=NULL;
  int  amode;
  int  fmode;

  long used=0, ignored=0, filled=0;

  int i_1, i_2;
  int status=-1;

  prmsg(DMSG,("average_term BEGIN\n"));

  amode = string2avgmode( ap->AMode.V );
  fmode = string2avgmode( ap->FMode.V );

  if (verbose>=2)
    spd_avg_params_print( ap );

  prmsg(DMSG,("\n Averaging %d images (%s)\n",frames->counter,avgmode2string(amode)));

  I0Data  = data;
  E0Data = vardat;
  I0Dummy = dummy;
  I0DDummy = ddummy;

    /* Average frames */

    long dim[4];
    dim[0]=3;dim[1]=dim_1;dim[2]=dim_2;dim[3]=dim_3;
    if (ap->Framename.I) {
      /* 
       * write 3d frame buffer
       */

      prmsg(DMSG,("    Writing 3D frame buffer file \'%s\'\n", ap->Framename.V));

      // open edf file
      stream=edf_open_data_file  ( ap->Framename.V, "new", &errval, &status );
      if (status) {
        prmsg(ERROR,("Error opening \'%s\'\n",ap->Framename.V));
        goto average_term_error;
      }

      // write edf file
      edf_write_data ( stream, 1, 1, dim, frames->data, MFloat,
                       &errval, &status );
      if (status) {
        prmsg(ERROR,("Error writing data to \'%s\'\n",ap->Framename.V));
        goto average_term_error;
      }

      // close edf file
      edf_close_data_file ( stream, &errval, &status );
      if (status) {
        prmsg(ERROR,("Error closing \'%s\'\n",ap->Framename.V));
        goto average_term_error;
      }
    }

    /* 
     * Calculate averages, variances and 
     * copy to I0Data and E0Data
     */

    emptypoints = new_avglistofpoints();

    pI0Data = I0Data;
    pE0Data = E0Data;
    for (i_2=0;i_2<dim_2;i_2++) {
      AvgValue ave;
      for (i_1=0;i_1<dim_1;i_1++) {

        if ( E0Data ) {
          // value and variance (ap->NM1.V==0: n values, ap->NM1.V!=0: n-1 values)
          ave = average( dim, frames->data, frames->dummy, frames->ddummy,
                         i_1, i_2, amode, fmode, ap->NM1.V, 1,
                         ap->SIGM.V, ap->SIGP.V, ap->MINS.V, ap->LOQ.V, 
                         ap->UPQ.V, ap->PQ.V, &status );
          if (status) {
            prmsg(ERROR,("Error averaging frames at pixel %dx%d\n",i_1,i_2));
            goto average_term_error;
          }
          if (ave.n>0) {
            *pI0Data = ave.val;
            *pE0Data = ave.var;
          } else {
            if (!(append_avgpoint( emptypoints, i_1, i_2 ))) {
              prmsg(ERROR,("Error appending pixel %dx%d to list of empty points (%p)\n",
                      i_1,i_2,emptypoints));
              goto average_term_error;
            }
          }
          used+=ave.n;
          ignored+=ave.i;
        } else {
          // value
          ave = average( dim, frames->data, frames->dummy, frames->ddummy,
                         i_1, i_2, amode, fmode, ap->NM1.V, 0,
                         ap->SIGM.V, ap->SIGP.V, ap->MINS.V, ap->LOQ.V, 
                         ap->UPQ.V, ap->PQ.V, &status );
          if (status) {
            prmsg(ERROR,("Error averaging frames at pixel %dx%d\n",i_1,i_2));
            if (status==-2) prmsg(ERROR,("NotEnoughMemoryAvailable\n"));
            goto average_term_error;
          }
          if (ave.n>0) {
            *pI0Data = ave.val;
          } else {
            if (!(append_avgpoint( emptypoints, i_1, i_2 ))) {
              prmsg(ERROR,("Error appending pixel %dx%d to list of empty points (%p)\n",
                      i_1,i_2,emptypoints));
              goto average_term_error;
            }
          }
          used+=ave.n;
          ignored+=ave.i;
        }
  
        pI0Data++;
        pE0Data++;
  
      } /* for i_1 ... */

    } /* for i_2 ... */
 
    if (ap->FILL.V) {
      if (verbose>2)  print_avglistofpoints( stdout, emptypoints );

      filled =  fill_empty_avgpoints( I0Data, dim_1, dim_2, emptypoints,
                                   I0Dummy, I0DDummy, ap->FIL1.V, ap->FIL2.V, 
                                   0, &status );
      if (status) {
        if (status==-2) prmsg(ERROR,("NotEnoughMemoryAvailable\n"));
        goto average_term_error;
      }
    }

  if (fmode) {
    long total = used+ignored;
    double percent=0.0;
    if (total>0)
      percent = 100.0 * (((double) ignored)/((double) total));
    else percent = 100.0;
    prmsg(DMSG,("   %ld of %ld input pixels ignored (%3.1lf%c)\n",
      ignored,total,percent,percentchar));
    if (ap->FILL.V) {
      long nempty;
      nempty = length_avglistofpoints( emptypoints );
      prmsg(DMSG,("   %ld of %ld empty points filled (%gx%g) ",
        filled,nempty,ap->FIL1.V,ap->FIL2.V));
      if (nempty!=filled) {
        prmsg(DMSG,("(%ld empty points could not be filled) ",
          nempty-filled));
      }
      prmsg(DMSG,("\n"));
    }
  }
  prmsg(DMSG,("\n"));

  free_avglistofpoints( emptypoints ); 

  if (pstatus) *pstatus = 0;

  prmsg(DMSG,("average_term END\n"));

  return;

average_term_error:

  free_avglistofpoints( emptypoints ); 

  if (pstatus) *pstatus = status;

  prmsg(DMSG,("average_term END (%d)\n",status));

  return;

} // average_term

/*==============================================================================
 * Average all (default) or selected images of an input data file, return a 
 * single header and a single data buffer.
 *
 * The data file header is read to obtain the information on the number of
 * rows and columns of the input image, the total size in bytes and the byte
 * ordering scheme used when storing the data. All images must have identical
 * dimensions (Dim_1, Dim_2), offsets (Offset_1, Offset_2) and binning sizes
 * (BSize_1, BSize_2), otherwise the routines fails. A set of images with
 * specific image numbers can be selected by appending a comma separated list
 * of numbers to the input filename (fname).
 *
 * The dimensions of the buffer ("rows" and "columns") are set from the
 * dimensions of the input image.
 *
 * Then the image data are read from the input file, averaged and written
 * to a data buffer that is returned to the calling program. If a buffer
 * of suitable size is provided in the input arguments, it is used, else
 * a buffer will be allocated.
 *
 * If any of these operations fails, the routine will return an error
 * indication and the "err" variable will be set to reflect the error that
 * occurred.
 *
 * Input : fname: name of the input data file, if followed by a comma separated
 *                list of numbers they are interpreted as image numbers to average.
 *                It is an error if the list contains image numbers that cannot
 *                be found in the file.
 *         type : type of file to be read (source, flood field, ...)
 *      verbose : verbose level as used in inout.c
 *         mtype: requested type of data to be read. Only MFloat is accepted.
 *                
 * Output: imgnum: data block number of read image (only updated on success)
 *         data: new buffer of type MFloat containing the data read from the 
 *               input data file
 *         rows: number of elements in the second dimension of the image (this
 *               is the "slow-moving" index, i.e. the first index in a two-
 *               dimensional C data buffer)
 *         cols: number of elements in the first dimension of the image (this is
 *               the "fast-moving" index, i.e. the second index in a two-
 *               dimensional C data buffer)
 *         err:  error indicator. Only meaningful if there is an error. Values:
 *               *  1 if the data header could not be obtained correctly
 *               *  3 if the data record could not be obtained correctly
 *               *  5 if the data could not be averaged 
 * Return: -1  if error
 *          0  else
 */
int average_esrf_file(const char *fname, long *imgnum, void **data,
  int *rows, int *cols,int type, int mtype, SPDAvgParams *ap, 
  int verbose, int *err)
{
  int stream=-1;
  int status=0, errval=0;

  size_t datasize=0;
  struct data_head user_head, cmd_head;

  long ChnNum = 1;
  long ImNum = 1;
  const long *ImNums=NULL;
  long frame_no=1, next_frame_no=0;
  long *Params=NULL;
  long nparams=0;

  char param[EdfMaxLinLen+1];
  char filnam[EdfMaxLinLen+1];

  AvgData *frames=NULL;

  long dim_1=0l, dim_2=0l, dim_3=1l;

  float *dest;
  long  nelements=0;
  float *pdummy;
  float *pddummy;

  float dummy=0.0, ddummy=DDSET(0.0);

  prmsg(DMSG,("average_esrf_file BEGIN\n"));

  /*
   * Check parameter mtype
   */

  if (mtype != MFloat) {
    prmsg(ERROR,("ERROR: only mtype MFloat can be averaged.\n"));
    goto average_esrf_file_error;
  }

  /*
   * Read first frame using read_esrf_file
   * It will also allocate an output buffer (data) 
   * for the corresponding machine type
   */
  frame_no=1;
  if (read_esrf_file(fname, NULL, NULL, imgnum, data, rows, cols,
                      type, mtype, frame_no, &next_frame_no, &errval)) {
    goto average_esrf_file_error;
  }

  /*
   * Average all input frames if there are more than one
   */
  if (next_frame_no > frame_no) {

    // get filename (parameter 0) from fname
    filename_parameter ( filnam, EdfMaxLinLen+1, fname , 0 );

    /*
     * Open data file. Return with error if it fails.
     */
    if((stream = edf_open_data_file(filnam,"read",&errval,&status)) == -1) {
      prmsg(ERROR,("Error opening file \"%s\".\n",filnam));
      goto average_esrf_file_error;
    }

    /*
     * Get image_numbers, either from the list of fname parameters or
     * using edf_block_numbers.
     */

    // determine the number of parameters
    nparams=0;
    while( (filename_parameter ( param, EdfMaxLinLen+1, fname , nparams+1 )) ) {
      nparams++;
    }
    if (nparams>0) {
      long parnum;

      // get ImNums from list of parameters
      Params = (long*) pmalloc((nparams+1)*sizeof(long));
      if (!Params) {
        prmsg(ERROR,("Error allocating parameter table of file \"%s\".\n",filnam));
        goto average_esrf_file_error;
      }

      Params[0] = nparams;
      for (parnum=1;parnum<=nparams;parnum++) {
        if ( filename_parameter ( param, EdfMaxLinLen+1, fname , frame_no )) {
          Params[parnum] = num_str2long(param,NULL,&errval);
        } else {
          Params[parnum] = 0; // this should never happen!
        }
      }
      ImNums = Params;
    } else {
      // get ImNums using edf_block_numbers
      if ( (ImNums = edf_block_numbers(stream,ChnNum,&errval,&status)) == NULL ) {
        prmsg(ERROR,("Error determining image numbers of file \"%s\".\n",filnam));
        goto average_esrf_file_error;
      } else if ( ImNums[0] == 0 ) {
        prmsg(ERROR,("No images in file \"%s\".\n",filnam));
        goto average_esrf_file_error;
      }
    }

    // Get dimensions of first frame
    dim_1 = (long) *cols;
    dim_2 = (long) *rows;
    dim_3 = ImNums[0];

    ImNum = ImNums[frame_no];

    /*
     * Get the header of the already read image "type".
     * (updated by calling read_esrf_file)
     */
    get_headval(&user_head,type);

    /*
     * Allocating float buffer for averaging dim_3*dim_2*dim_1 * machinesizeof(mtype)
     */
    frames = average_init ( dim_1, dim_2, dim_3, verbose, &status );
    if ( !frames ) {
        prmsg(ERROR,("ERROR allocating float buffer %ld*%ld*%ld \n",dim_1, dim_2, dim_3));
        goto average_esrf_file_error;
    }

    /*
     * Keep input images in 3d frame buffer until end of loop
     */
 
    // Increment frame counter

    if (user_head.init & FL_DUMMY) {
      dummy  = user_head.Dummy; // Dummy is not updated from command line!
    } else dummy  = 0.0;
    if (user_head.init & FL_DDUMM) {
      ddummy  = user_head.DDummy; // DDummy is not updated from command line!
    } else ddummy  = DDSET(dummy);

    frames->counter+=1l;

    nelements = dim_1*dim_2;
    dest = frames->data + (frame_no-1)*nelements;
    pdummy = frames->dummy + (frame_no-1);
    pddummy = frames->ddummy + (frame_no-1);

    // copy data of first input frame_no to frame buffer
    memmove(dest,*data,sizeof(float)*nelements);
    *pdummy = dummy;
    *pddummy = ddummy;

    /*
     * Get the command line header
     */
    get_headval(&cmd_head,CMDTYP);

    // fill float buffer with images
    for (frame_no=next_frame_no;frame_no<=dim_3;frame_no++) {
      float dummyin=0.0, ddummyin=DDSET(0.0);
      long *dim = NULL;
      float off1=0.0, off2=0.0;
      float bis1=1.0, bis2=1.0;

      ImNum = ImNums[frame_no];

      /*
       *  Check region of current image (dim,off,bis)
       */
      edf_read_data_dimension(stream,ImNum,ChnNum,&dim,NULL,&errval,&status);
      if(status != status_success) {
        prmsg(ERROR,("Error reading dimensions of image %d of file \"%s\".\n",ImNum,filnam));
        goto average_esrf_file_error;
      }

      // Preferably, use command head values for offset and bsize, otherwise read them from the file 
      if ((cmd_head.init != 0)&&(cmd_head.init & FL_OFFS1)) {
        off1 = cmd_head.Offset_1;
      } else {
        if (!edf_read_header_float(stream,ImNum,ChnNum,"Offset_1",&off1,&errval,&status)) {
          off1=0.0;
        } else if(status != status_success) {
          prmsg(ERROR,("Error reading Offset_1 of image %d of file \"%s\".\n",ImNum,filnam));
          goto average_esrf_file_error;
        }
      }

      if ((cmd_head.init != 0)&&(cmd_head.init & FL_OFFS2)) {
        off2 = cmd_head.Offset_2;
      } else {
        if (!edf_read_header_float(stream,ImNum,ChnNum,"Offset_2",&off2,&errval,&status)) {
          off2=0.0;
        } else if(status != status_success) {
          prmsg(ERROR,("Error reading Offset_2 of image %d of file \"%s\".\n",ImNum,filnam));
          goto average_esrf_file_error;
        }
      }

      if ((cmd_head.init != 0)&&(cmd_head.init & FL_BSIZ1)) {
        bis1 = cmd_head.BSize_1;
      } else {
        if (!edf_read_header_float(stream,ImNum,ChnNum,"BSize_1",&bis1,&errval,&status)) {
          bis1=1.0;
        } else if(status != status_success) {
          prmsg(ERROR,("Error reading BSize_1 of image %d of file \"%s\".\n",ImNum,filnam));
          goto average_esrf_file_error;
        }
      }

      if ((cmd_head.init != 0)&&(cmd_head.init & FL_BSIZ2)) {
        bis2 = cmd_head.BSize_2;
      } else {
        if (!edf_read_header_float(stream,ImNum,ChnNum,"BSize_2",&bis2,&errval,&status)) {
          bis2=1.0;
        } else if(status != status_success) {
          prmsg(ERROR,("Error reading BSize_2 of image %d of file \"%s\".\n",ImNum,filnam));
          goto average_esrf_file_error;
        }
      }

      if (!edf_read_header_float(stream,ImNum,ChnNum,"Dummy",&dummyin,&errval,&status)) {
        dummyin = 0.0;
      } else if(status != status_success) {
        prmsg(ERROR,("Error reading Dummy of image %d of file \"%s\".\n",ImNum,filnam));
        goto average_esrf_file_error;
      }
      if (!edf_read_header_float(stream,ImNum,ChnNum,"DDummy",&ddummyin,&errval,&status)) {
        ddummyin = DDSET(dummyin);
      } else if(status != status_success) {
        prmsg(ERROR,("Error reading DDummy of image %d of file \"%s\".\n",ImNum,filnam));
        goto average_esrf_file_error;
      }

      int badcnt=0;
      if ( user_head.BSize_1 != bis1 ) {
        badcnt++;
        prmsg(ERROR,("BSize 1 mismatch of image %d of file \"%s\": %g != %g.\n",
          ImNum,filnam,bis1,user_head.BSize_1));
      }
      if ( user_head.BSize_2 != bis2 ) {
        badcnt++;
        prmsg(ERROR,("BSize_2 mismatch of image %d of file \"%s\": %g != %g.\n",
          ImNum,filnam,bis2,user_head.BSize_2));
      }
      if (user_head.Offset_1 != off1) {
        badcnt++;
        prmsg(ERROR,("Offset_1 mismatch of image %d of file \"%s\": %g != %g.\n",
          ImNum,filnam,off1,user_head.Offset_1));
      }
      if (user_head.Offset_2 != off2) {
        badcnt++;
        prmsg(ERROR,("Offset_2 mismatch of image %d of file \"%s\": %g != %g.\n",
          ImNum,filnam,off2,user_head.Offset_2));
      }
      if (dim_1 != dim[1]) {
        badcnt++;
        prmsg(ERROR,("Dimension 1 mismatch of image %d of file \"%s\": %ld != %ld.\n",
          ImNum,filnam,dim[1],dim_1));
      }
      if (dim_2 != dim[2]) {
        badcnt++;
        prmsg(ERROR,("Dimension 2 mismatch of image %d of file \"%s\": %ld != %ld.\n",
          ImNum,filnam,dim[2],dim_2));
      }

      if (badcnt>0) goto average_esrf_file_error;

      /*
       *  Read image data
       *  use 'data' as output buffer (allocated as float buffer by read_esrf_file)
       *  datasize is the size of the 'data' buffer in bytes
       */
      datasize = dim_1*dim_2*sizeof(float);
      
      edf_read_data(stream,ImNum,ChnNum,&dim,&datasize,data,mtype,&errval,&status);
      if(status) {
        errval = 3;
        prmsg(ERROR,("Error reading image %d data from file \"%s\".\n",ImNum,filnam));
        goto average_esrf_file_error;
      }

      /*
       * Keep input images in 3d frame buffer until end of loop
       */

      // Increment frame counter
      frames->counter+=1l;

      nelements = dim_1*dim_2;
      dest = frames->data + (frame_no-1)*nelements;
      pdummy = frames->dummy + (frame_no-1);
      pddummy = frames->ddummy + (frame_no-1);

      // move data of frame frame_no to frame buffer
      memmove(dest,*data,nelements*sizeof(float));
      *pdummy  = dummyin;
      *pddummy  = ddummyin;

    } // for frame_no

    edf_close_data_file(stream,&errval,&status);
    if(status) {
      errval = 3;
      prmsg(ERROR,("Error closing data file \"%s\".\n",filnam));
      goto average_esrf_file_error;
    }

    // average_term average frames and write to data and vardat
    average_term ((float*) *data, NULL, dummy, ddummy, frames, dim_1, dim_2, dim_3, ap, verbose, &status);
    if(status) {
      errval = 5;
      prmsg(ERROR,("The data of \"%s\" could not be averaged.\n",filnam));
      goto average_esrf_file_error;
    }

    if (frames) free_avg_data(frames);

    if (Params) pfree(Params);

  }

  if (err) *err=errval;

  prmsg(DMSG,("average_esrf_file END\n"));

  return(0);

average_esrf_file_error:

  if (stream!=-1) {
    edf_close_data_file(stream,NULL,NULL);
  }
  if (frames) free_avg_data(frames);

  if (Params) pfree(Params);

  if (err) *err=errval;
  prmsg(DMSG,("average_esrf_file END (err=%d)\n",errval));

  return(-1);

} // average_esrf_file
