/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/* NAME
   specscan

   DESCRIOPTION
   Extraction of scans from spec files and conversion to PC readable text 
   files 

   HISTORY
   1999-04-23 V1.0 Peter Boesecke 
   1999-04-27 V1.1 PB argument counting corrected
   1999-04-27 V1.2 PB space added after number ("#S 4" != "#S 49")
   2004-10-30 V1.3 PB MAKE_FUNCTION
   2008-06-19 V1.4 PB output file is only open if input scan was found
   2008-06-25 V1.5 PB extracted file starts with an empty line, otherwise
                      it cannot be open with newplot,
                      option ext=<extension> added, option filter->grep.
   2008-07-02 V1.6 PB if range is omitted, scan numbers are read. This
                      allows to start with an unknown scan number.
   2008-12-05 V1.7 PB open output file as binary file to avoid
                      translation of \n to \r\n when compiled on windows.
   2009-03-20 V1.8 PB char *ps -> const char *ps

*/
/*=========================================================================*/
/* Public                                                                  */
/*=========================================================================*/

# include <stdlib.h>
# include <errno.h>
# include <stdio.h>
# include <unistd.h>

# include <ctype.h>
# include <string.h>
# include <time.h>

# include "filename.h"
# include "numio.h"

# define BUFLEN 1024

# define STATUS_ERROR -1
# define STATUS_SUCCESS 0
# define STATUS_OPENERROR 1
# define STATUS_READERROR 2
# define STATUS_STOPPEDAFTERHELP 39

# define SPECSCAN_VERSION "1.8 (PB 2009-03-20)"

char end_of_line[3] = { '\r', '\n', '\0' };

/* removes end_of_line from line */
char * rmeoln ( char * line )
{ char * pc = line;

  
  if (pc)
    while (*pc) {
      if (strchr( end_of_line, (int) *pc )) *pc='\0';
      pc++;
    }
  return( line );

} /* rmeoln */

/* markncmp
   markncmp compares marklen character of line with the string marker. 
   If they match the scan number is read from the next parameter
   and returned. 
   If line does not start with marker or if the scan number is missing
   the returned value is -1, otherwise the scan number is returned
   as a positive integer including 0.
*/
long markncmp( const char *line, const char *marker, size_t marklen )
{ long number=-1;
  int errval=0;
  if (strlen(line)>=marklen) {
    if (!strncmp(line,marker,marklen)) {
      number = num_str2long ( line+marklen, NULL, &errval);
    }
    if (errval) number=-1;
  }
  return ( number );
} // markncmp

/* spec2kal

   Description
   Extraction of a scan from a spec file and adding cr-lf at line end 

   Arguments
   const char * odir : output directory 
   const char * fnam : spec file name
   const char * marker : marker string, where extraction 
                will start (inclusive) and end (exclusive)
   const char * extension : file extension
   long skip : number of start marker that should be skipped before the first
               scan starts. It is used if the spec file contains several scan 
               files with the same start marker
   long first, last, inc : loop parameters
   int debug         : flag

   Return Value
   Number of written files.
*/

long spec2kal ( const char * odir, const char * fnam, const char * extension,
                const char * marker, 
                long skip, long first, long last, long inc, const char *grep,
                int debug, int *pstatus )
{ 
  FILE *input, *output;
  char buffer[BUFLEN];
  char fnam_body[BUFLEN];
  char fnam_name[BUFLEN];
  char onam[BUFLEN];
  char line[BUFLEN];
  int  found = 0;


  long loop, loopmax, l0, l1, linc;
  long num, marknum;
  size_t marklen;

  long file_cnt=0l;

  if (*pstatus) return( file_cnt );

  sprintf(buffer,"function spec2kal, input file %s", fnam);
  input = fopen( fnam, "r");
  if (!input) { perror(buffer); *pstatus = STATUS_OPENERROR; return(file_cnt); }

  /* read first line */
  if (!(feof(input))) {
    fgets (line, BUFLEN, input);
    if (ferror(input)) {
      perror(buffer); *pstatus=STATUS_READERROR; return(file_cnt);
    }
    if (debug>3) printf("line=%s\r\n",rmeoln(line)); 
  }

  if (first<=last) {
    l0=first;l1=last;linc=inc;} else {l0=last;l1=first;linc=-inc;
  }
  if (linc>0) { 
    loopmax=(l1-l0)/linc+1l; 
  } else { 
    loopmax = 1; 
  }

  if (debug>2) {
    printf("l0=%ld, l1=%ld, linc=%ld, loopmax=%ld, last=%ld\n",
      l0,l1,linc,loopmax,last);
  }

  loop=1l;
  num=first;
  while ( (loop<=loopmax)||(last<0) ) {

    if (debug>1) 
      printf("loop = %ld, loopmax=%ld, last=%ld\n",loop,loopmax,last);

    /* skip first occurences of marker==num, skip is decremented to -1 */
    for (;skip>=0;skip--) {

      /* search scan */
      marklen = strlen(marker);
      marknum = markncmp( line, marker, marklen);
      if ( (marknum>=0)&&(last<0) ) num=marknum;

      while (!((feof(input))||(marknum==num))) {
        if (debug>4) printf("skipping %s\r\n",rmeoln(line)); 

        fgets (line, BUFLEN, input);
        if (ferror(input)) {
          perror(buffer); *pstatus = STATUS_READERROR; return(file_cnt); 
        }
        if (debug>3) printf("line=%s\r\n",rmeoln(line)); 
        if (!(feof(input))) {
          marknum = markncmp( line, marker, marklen);
          if ( (marknum>=0)&&(last<0) ) num=marknum;
        }
      }
      if (skip>0) { /* skip this scan */
        if (!(feof(input))) {
          fgets (line, BUFLEN, input);
          if (ferror(input)) {
            perror(buffer);*pstatus=STATUS_READERROR;return(file_cnt);
          }
          if (debug>3) printf("line=%s\r\n",rmeoln(line)); 
        }
      }
    }
    skip=0;

    /* search the substring "grep" in the first line */

    if (strstr(line,grep)) { // substring found

      /* convert lines until next line starting with marker */
  
      filename_name ( fnam_name, BUFLEN, fnam ); 
      filename_body ( fnam_body, BUFLEN, fnam_name );
      sprintf(onam,"%s/%s_%04ld%s",odir,fnam_body,num,extension);

      sprintf(buffer,"spec2kal, input=>>%s<<, output=>>%s<<",fnam,onam);
      if (!(feof(input))) {
        if (debug) {
          printf(" onam   = %s\n",onam);
        }
        output = fopen ( onam, "wb" );
        if (!output) { 
          perror(buffer); *pstatus = STATUS_OPENERROR; return(file_cnt); 
        }

        fprintf(output,"\r\n"); // start with an empty line
        fprintf(output,"%s\r\n",rmeoln(line)); // scan number and title

        fgets (line, BUFLEN, input);
        if (ferror(input)) {
          perror(buffer); *pstatus=STATUS_READERROR; return(file_cnt);
        } 
        if (debug>3) printf("line=%s\r\n",rmeoln(line)); 
        while (!((marknum=markncmp( line, marker, marklen)>=0)||(feof(input)))) {
          fprintf(output,"%s\r\n",rmeoln(line));

          fgets (line, BUFLEN, input);
          if (ferror(input)) {
            perror(buffer); *pstatus=STATUS_READERROR; return(file_cnt);
          }
          if (debug>3) printf("line=%s\r\n",rmeoln(line)); 

        }
        fclose( output );
        file_cnt++;
      } else {
        if (debug>0)
          if (last>=0)
            printf("Start marker >>%s %d<< not found\n",marker,num);
        last=1; loopmax=-1; // stop here
      }
    } else { // skip this scan
      if (debug>2) {
        printf("grep=>>%s<< does not match line=>>%s<<\n",
          grep, line);
      }

      if (!(feof(input))) {
        fgets (line, BUFLEN, input);
        if (ferror(input)) {
          perror(buffer);*pstatus=STATUS_READERROR;return(file_cnt);
        }
        if (debug>3) printf("line=%s\r\n",rmeoln(line)); 
      } else {
        if (debug>0)
          if (last>=0)
            printf("Start marker >>%s %d<< not found\n",marker,num);
        last=1; loopmax=-1; // stop here
      }

    }
    loop++;
    num+=inc;
  } // for loop 

  fclose( input );
 
  return( file_cnt );
} /* spec2kal */

int specscan_help ( FILE * out , const char * name ) {
      fprintf(out," %s [--help]\\\n",name);
      fprintf(out,"    [debug=<0|1>]\\\n");
      fprintf(out,"    [skip=<skip>]\\\n");
      fprintf(out,"    [ext=<extension>]\\\n");
      fprintf(out,"    [marker=<marker>]\\\n");
      fprintf(out,"    [grep=<grep string>\\\n");
      fprintf(out,"    [range=<fst>[,<lst>[,<inc>]]]\\\n");
      fprintf(out,"    [odir=<output directory>] {<fnam>}\n");
      fprintf(out,"%s version: %s\n", name, SPECSCAN_VERSION);
      return ( STATUS_STOPPEDAFTERHELP );
} // specscan_help

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_specscan
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  int errval=0;
  int status=STATUS_SUCCESS;
  int argcnt=1;
  int debug = 0;

  char * marker="#S";
  char * extension=".txt";
  char * grep="";
  char * fnam="datafile.spec";
  char * odir=".";
  char * range="1";
  const char * ps;

  long first=1, last=-1, inc=1;

  long skip = 0;
  long file_cnt=0;

  #define BUFLEN 1024
  char buffer[BUFLEN];

  // skip all options
  while (argc > argcnt) {
    if ( strncmp( argv[argcnt],"debug=", 6 ) == 0 ) {
      sscanf(&argv[argcnt][6],"%d",&debug);
      numio_debug( debug );
      argcnt++;
    } else if ( strncmp( argv[argcnt],"--help", 6 ) == 0 ) {
      status=specscan_help( stdout, argv[0] );
      argcnt++;
      break; 
    } else if ( strncmp( argv[argcnt],"skip=", 5 ) == 0 ) {
      sscanf(&argv[argcnt][5],"%d",&skip);
      argcnt++;
    } else if ( strncmp( argv[argcnt],"grep=", 5 ) == 0 ) {
      grep=&argv[argcnt][5];
      argcnt++;
    } else if ( strncmp( argv[argcnt],"marker=", 7 ) == 0 ) {
      marker=&argv[argcnt][7];
      argcnt++;
    } else if ( strncmp( argv[argcnt],"ext=", 4 ) == 0 ) {
      extension=&argv[argcnt][4];
      argcnt++;
    } else if ( strncmp( argv[argcnt],"range=", 6 ) == 0 ) {
      range=&argv[argcnt][6];
      ps=range;
      first = num_str2long ( ps, &ps, &errval);
      if (errval) {
        status=STATUS_ERROR;
        fprintf(stderr,"ERROR: >>%s<< status %d, exiting %s\n",
          range, status, argv[0]);
        return(status);
      }
      
      last = first;
      if (*ps==',') {
        last = num_str2long ( ++ps, &ps, &errval);
        if (errval) {
          status=STATUS_ERROR;
          fprintf(stderr,"ERROR: >>%s<< status %d, exiting %s\n",
            range, status, argv[0]);
          return(status);
        }
      }

      if (*ps==',') {
        inc = num_str2long ( ++ps, &ps, &errval);
        if (errval) {
          status=STATUS_ERROR;
          fprintf(stderr,"ERROR: >>%s<< status %d, exiting %s\n",
            range, status, argv[0]);
          return(status);
        }
      }

      argcnt++;
    } else if ( strncmp( argv[argcnt],"odir=", 5 ) == 0 ) {
      odir=&argv[argcnt][5];
      argcnt++;
    } else break;
  }

  if (debug) {
    printf(" debug  = %d\n",debug);
    printf(" skip   = %ld\n",skip);
    printf(" marker = %s\n",marker);
    printf(" grep   = %s\n",grep);
    printf(" range  = %s\n",range);
    printf(" first  = %ld\n",first);
    printf(" last   = %ld\n",last);
    printf(" inc    = %ld\n",inc);
    printf(" odir   = %s\n",odir);
  }

  if (status == STATUS_SUCCESS) {
    if (argc>argcnt) {
      while (argc>argcnt) {
        fnam = argv[argcnt];

        if (debug) {
          printf(" fnam   = %s\n",fnam);
        }
        file_cnt+=
          spec2kal (odir, fnam, extension, marker, skip, first, last, inc, grep,
                    debug, &status);
        if (status) {
          fprintf(stderr,"ERROR: status %d, exiting %s\n",status, argv[0]);
          return(-1);
        }

        argcnt++;
      }
      printf("%ld files written.\n", file_cnt);
    } else status=specscan_help( stdout, argv[0] );
  }
  return ( status );
} /* MAIN */
