/***************************************************************************/
/* Written 1994++ by Peter Boesecke                                        */
/* Copyright (C) 2011 European Synchrotron Radiation Facility              */
/*                       Grenoble, France                                  */
/*                                                                         */
/*    Principal authors: Peter Boesecke  (boesecke@esrf.eu)                */
/*                                                                         */
/*    This program is free software: you can redistribute it and/or modify */
/*    it under the terms of the GNU General Public License as published by */
/*    the Free Software Foundation, either version 3 of the License, or    */
/*    (at your option) any later version.                                  */
/*                                                                         */
/*    This program is distributed in the hope that it will be useful,      */
/*    but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/*    GNU General Public License for more details.                         */
/*                                                                         */
/*    You should have received a copy of the GNU General Public License    */
/*    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/
/***************************************************************************/
/*+++
1 saxs_waxs
  Projection of the Ewald sphere that is visible by an inclined detector 
  to the s_1 | s_2 plane in reciprocal space. The length of the radial 
  coordinate corresponds to the length of the scattering vector s. The 
  projection is either done assuming isotropic scattering symmetry (default)
  or rotational symmetry around a symmetry axis.

  The scattering vectors are calculated with saxs-coordinates (-rsys saxs).

2 PURPOSE

  Arguments:
  saxs_waxs [options] <i1nam> <onam> <i1fst> <i1lst> <i1inc> 
             <odum> <odim1> <odim2> <i1rot1> <i1rot2> <i1rot3>

  Defaults:
  <input file name>   : input.edf 
  <output file name>  : output.edf
  <first image>       : <first image number in input file>
  <last image>                              : <last image number in input file>
     if argument list ends with first image : <first image>
  <increment>         : 1
  <dummy>             : <dummy value in first image of input file>
  <dimension 1>       : <horizontal dimension of first image in input file>
  <dimension 2>       : <vertical dimension of first image in input file>

  All rotations and directions are defined for orientation 1

  <i1rot1>            : <ccw rotation around direction 1> (default header value)
  <i1rot2>            : <ccw rotation around direction 2> (default header value)
  <i1rot3>            : <ccw rotation around direction 3> (default header value)

  Axis of rotational symmetry
  It is possible to project the saxs pattern to the s_1 | s_2 plane by 
  rotating the intensities around an axis of rotational symmetry in reciprocal
  space. In analogy to the detector rotation this axis can be defined by three
  rotations around s_1, s_2 and s_3:

  <sro1>            : <ccw rotation around direction 1> (default 0_deg)
  <sro2>            : <ccw rotation around direction 2> (default 0_deg)
  <sro3>            : <ccw rotation around direction 3> (default 0_deg)

  The symmetry axis is either axis 1 or axis 2, depending on the parameter 
  <styp>            : 0 == isotropic scattering (default),
                      1 == rotational symmetry around axis 1,
                      2 == rotational symmetry around axis 2

2 HISTORY
  2001-04-16  Peter Boesecke from saxs_refract V3.1 
  2001-04-19  PB V3.0
  2001-04-19  PB V3.1 K taken from input image ihb[1]
  2001-07-09  PB V4.00 new loop, no repetition
  2002-05-09  PB V4.01 inverse transformation
  2002-09-06  PB V4.02 xyfile, the xyfile is always written to image number 1, 
                       memory 1
  2003-02-09  PB V4.03 line args for Dummy, DDummy, Dim
  2003-03-30  PB V4.04 ImageLoop: Function call changed
  2003-08-14  PB V4.05 option_define, all option numbers removed
  2003-11-30  PB V4.06 BinSiz
  2004-04-15  PB V4.07 xyfiles: - The displacement is now correctly calculated 
                                for the full region of the binned source image
                                and can be used as input of the spatial 
                                distortion correction program, known as spd.
                                - BSize_1 and BSize_2 are also calculated. 
                                The region parameters of the binned source
                                image (dimension, offset, binning size) are
                                used for the xy-files.
                                - The xyfile is written with the image number
                                of the output image to memory 1
                       new option +/-xypi: if xypi is set (default) the
                       pixel size and the center of the unbinned output image 
                       are written to the xy-files and can be used by spd as 
                       the pixel size of the corrected image.
  2004-04-16  PB V4.08 write center to xyfiles when +xypi is set
  2004-05-08  PB V4.09 indexing error of xyfiles corrected (i_1 starts with 0) 
                       write center and binsize also to yfile
  2004-07-27  PB V4.10 write sample distance to xyfiles when +xypi is set
  2004-09-26  PB V4.11 Write geometric parameters of transformed image always
                       to special output keywords of xy-files,
                       Write RasterProjection key, WaveLength
                       Displacements are now calculated from array coordinates,
                       no longer from image coordinates.
                       Option +-rev (Inv) removed and replaced by combinations
                       of options -i1pro <projection> and -opro <projection>,
                       default: opro==waxs
                       a) i1pro==saxs, opro==waxs: Inv=FALSE
                       b) i1pro==saxs, opro==saxs: no action 
                       c) i1pro==waxs, opro==saxs: Inv=TRUE 
                       d) i1pro==waxs, opro==waxs: no action 
  2004-10-14  PB V4.12 Using DetectorRotation header parameters as defaults
                       for each image.
                       The options rot1, rot2, rot3 are temporarily kept as 
                       float options for backward compatibility. The rotations 
                       must be given in degrees. The use of units with these
                       options is not well defined and should be avoided.
                       The option iNrot <rad> should be used instead.
                       The option xypi is temporily kept with default TRUE.
                       Description of detector rotations improved.
  2004-10-31  PB V4.13 MAKE_FUNCTION
  2004-10-31  PB V4.14 option +-rot
  2004-12-06  PB V4.15 waxs_SymInit 
  2005-04-10  PB V4.16 waxs_range, set_default_pixsiz, set_default_center
                       +options pdef and cdef to set the defaults or not.
                       default of pdef is FALSE, default of pdef is TRUE
  2005-07-15  PB V4.17 ReportSaxsStatus parameter list updated,
                       SetSaxsErrorExternal added
  2005-09-17  PB V4.18 VarDat
  2005-10-14  PB V4.19 FREE( a) sets a to NULL
  2005-10-14  PB V4.20 option pp added
  2007-04-23  PB V4.21 function concat corrected (to avoid buffer ovfl)
  2007-06-11  PB V4.22 ImageLoop parameters updated
  2010-03-18  PB V4.23 waxs_Range, waxs_Transform -> waxs.c, 
                       rot1, rot2, rot3 are now obsolete,
                       unused variables removed,
                       saxs_waxs: || -> && (because there is only 
                       1 alternative there was no logical 
                       differenence), waxs_Range, waxs_Transform
                       arguments updated
  2010-05-31  PB V4.24 update for waxs.c V1.10
  2011-06-07  PB V4.25 strncat->strncpy
  2011-06-30  PB V4.26 waxs_Transform updated,
                       concat replaced with corrected version strlib_concat
  2012-01-31  PB V4.27 ImageLoop parameters updated
  2014-02-07  PB V4.28 saxs_waxs: waxs_Init and waxs_SymInit updated
  2015-10-03  PB V4.29 FileAlreadyOpen
  2017-05-21  PB V4.30 ioalloc

---*/
# define Version  "saxs_waxs V4.30 2017-05-21, Peter Boesecke"

# include <errno.h>
# include <stdio.h>
# include <unistd.h>
# include <fcntl.h>
# include <string.h>

# include "SaxsPrograms.h"
# include "waxs.h"
# include "strlib.h"
# include "ioalloc.h"

# define Usage "[options] \n\
                <i1nam> <onam> <i1fst> <i1lst> <i1inc> \n\
                <odum> <odim1> <odim2> <i1rot1> <i1rot2> <i1rot3> \n\n\
                Projection of a SAXS pattern to the Ewald-sphere\n"

# define BlockNum 2       /* 1 input sequence + 1 output sequence */

/* Special options */
# define SXyname "xyna"  /* line */  // xy-file name
# define SXy   "xy"      /* flag */  // xy-file flag
# define SXypix "xypi"   /* flag */
# define SRot1 "rot1"    /* float */ // in degree for backwards compatibility
# define SRot2 "rot2"    /* float */ // in degree for backwards compatibility
# define SRot3 "rot3"    /* float */ // in degree for backwards compatibility
# define SRot  "rot"     /* flag */
# define SSymType "styp" /* line */
# define SSymRot1 "sro1" /* float */
# define SSymRot2 "sro2" /* float */
# define SSymRot3 "sro3" /* float */
# define SPdef "pdef"    /* flag */  // set output pixel size to default  
# define SCdef "cdef"    /* flag */  // set output center to default  
# define SPp "pp"        /* flag */  // prompt prang: fail, if opro==i1pro 

/*---------------------------------------------------------------------------
 NAME

   set_default_pixsiz --- sets default pixel size in image header block 
  
 SYNOPSIS

   void set_default_pixsiz(int i, ImgHeadBlk ihb[],
                           WaxsCoord Wmin, WaxsCoord Wmax, int * pstatus);

 DESCRIPTION
   Calculates a default pixel size calculatd from the saxs-coordinate range
   given by Wmin and Wmax and the parameters Dim, SampleDistance and 
   WaveLength of the header ihb[i]. An appropriate default center needs 
   to be set in a subsequent step with set_default_center.

 RETURN VALUE
   void
   status returned in *pstatus

---------------------------------------------------------------------------*/
void set_default_pixsiz(int i, ImgHeadBlk ihb[], 
                        WaxsCoord Wmin, WaxsCoord Wmax, int * pstatus)
{ const float eps=1e-32;
  long  dim_1, dim_2;
  float dis, wvl;
  float saxs_pix_1, saxs_pix_2, def_pix_1, def_pix_2, pix_1, pix_2;
  float smin_1, smin_2, smax_1, smax_2;

  *pstatus = Failed;

  dim_1 = ihb[i].Dim[1];
  dim_2 = ihb[i].Dim[2];
  pix_1 = ihb[i].PixSiz[1].V;
  if (fabs(pix_1)<=eps) return;
  pix_2 = ihb[i].PixSiz[2].V;
  if (fabs(pix_2)<=eps) return;
  wvl = ihb[i].WaveLength.V;
  if (fabs(wvl)<=eps) return;
  dis = ihb[i].SampleDistance.V;
  if (fabs(dis)<=eps) return;

  /* smin and smax are the minimum and maximum moduli of the scattering
     vectors in the image */

  smin_1 = Wmin.s_1; smin_2 = Wmin.s_2;
  smax_1 = Wmax.s_1; smax_2 = Wmax.s_2;

  if (dim_1>0) {
    saxs_pix_1 = (smax_1-smin_1)/dim_1;
    def_pix_1 = S2PSIZE(saxs_pix_1,dis,wvl);
    /* Set PixSiz if it was not set by image block (.I>=0) */
    if (ihb[i].PixSiz[1].I>=0) ihb[i].PixSiz[1].V = def_pix_1;
    /* Update BinSiz if it was not set by image block (.I>=0) */
    if (ihb[i].BinSiz[1].I>=0) ihb[i].BinSiz[1].V *= def_pix_1/pix_1;
  }

  if (dim_2>0) {
    saxs_pix_2 = (smax_2-smin_2)/dim_2;
    def_pix_2 = S2PSIZE(saxs_pix_2,dis,wvl);
    /* Set PixSiz if it was not set by image block (.I>=0) */
    if (ihb[i].PixSiz[2].I>=0) ihb[i].PixSiz[2].V = def_pix_2;
    /* Update BinSiz if it was not set by image block (.I>=0) */
    if (ihb[i].BinSiz[2].I>=0) ihb[i].BinSiz[2].V *= def_pix_2/pix_2;
  }

  *pstatus = Success;

} /* set_default_pixsiz */

/*---------------------------------------------------------------------------
 NAME

   set_default_center --- sets default center in image header block

 SYNOPSIS

   void set_default_center(int i, ImgHeadBlk ihb[],
                           WaxsCoord Wmin, WaxsCoord Wmax, int * pstatus);

 DESCRIPTION
   Calculates a default center calculatd from the saxs-coordinate range
   given by Wmin and Wmax and the parameters Dim, SampleDistance and
   WaveLength of the header ihb[i]. An appropriate pixel size can be 
   set in a previous step, e.g. with set_default_pixsiz.

   lower edge
   C = (ARRAYSTART+LOWERBORDER+O)-((smin/P)*(D)/WAVENUMBER(W))
   upper edge
   C = (ARRAYSTART+LOWERBORDER+O+dim)-((smax/P)*(D)/WAVENUMBER(W))

 RETURN VALUE
   void
   status returned in *pstatus

---------------------------------------------------------------------------*/
void set_default_center(int i, ImgHeadBlk ihb[], 
                        WaxsCoord Wmin, WaxsCoord Wmax, int * pstatus)
{ const float eps=1e-32;
  float def_cen_1, def_cen_2, cen_11, cen_12, cen_21, cen_22;

  long  dim_1, dim_2;
  float off_1, off_2;
  float dis, wav;
  float pix_1, pix_2;
  float smin_1, smin_2, smax_1, smax_2;

  *pstatus = Failed;

  dim_1 = ihb[i].Dim[1];
  dim_2 = ihb[i].Dim[2];
  off_1 = ihb[i].Offset[1].V;
  off_2 = ihb[i].Offset[2].V;
  pix_1 = ihb[i].PixSiz[1].V;
  if (fabs(pix_1)<=eps) return;
  pix_2 = ihb[i].PixSiz[2].V;
  if (fabs(pix_2)<=eps) return;
  if (fabs(ihb[i].WaveLength.V)<=eps) return;
  wav = WAVENUMBER(ihb[i].WaveLength.V);
  dis = ihb[i].SampleDistance.V;
  if (fabs(dis)<=eps) return;

  /* smin and smax are the minimum and maximum moduli of the scattering
     vectors in the image */

  smin_1 = Wmin.s_1; smin_2 = Wmin.s_2;
  smax_1 = Wmax.s_1; smax_2 = Wmax.s_2;

  cen_11 = ARRAYSTART+LOWERBORDER+off_1-(smin_1/pix_1)*(dis/wav);
  cen_12 = ARRAYSTART+LOWERBORDER+dim_1+off_1-(smax_1/pix_1)*(dis/wav);
  def_cen_1 = (cen_11+cen_12)*0.5;
  /* Set Center if it was not set by image block (.I>=0) */
  if (ihb[i].Center[1].I>=0) ihb[i].Center[1].V = def_cen_1;

  cen_21 = ARRAYSTART+LOWERBORDER+off_2-(smin_2/pix_2)*(dis/wav);
  cen_22 = ARRAYSTART+LOWERBORDER+dim_2+off_2-(smax_2/pix_2)*(dis/wav);
  def_cen_2 = (cen_21+cen_22)*0.5;
  /* Set Center if it was not set by image block (.I>=0) */
  if (ihb[i].Center[2].I>=0) ihb[i].Center[2].V = def_cen_2;

  *pstatus = Success;

} /* set_default_center */

/*---------------------------------------------------------------------------
1 saxs_waxs

2 PURPOSE
  Projection of an inclined detector surface
---------------------------------------------------------------------------*/
void saxs_waxs (CmdBlk * pcb, long num[], ImgHeadBlk ihb[], int * pstatus)
{ const char * RoutineName = "saxs_waxs";
  int   i, imax; 
  float *I0Data, *E0Data;
  int   I0Dim_1, I0Dim_2;
  float *pI0Data, *pE0Data;
  float I0Dummy, I0DDummy;
  float Sum, Weight, Value;
  float VarSum, VarWeight, E0Value;

  float *I1Data, *E1Data;
  int   I1Dim_1,I1Dim_2;
  float *pI1Data;
  float I1Dummy, I1DDummy;

  double rot1, rot2, rot3;
  long  I0Projection, I1Projection;

  int i_1, i_2;

  float f0_1, f0_2;
  float f1_1, f1_2;

  float Off_1[BlockNum], Off_2[BlockNum];
  float Ps_1[BlockNum], Ps_2[BlockNum];

  WaxsCoord W0, W1;
  WaxsCoord Wmin, Wmax;
  double K;                                                       

  WParams I1wparams, I0wparams;

  int transform;

 /* xyfile parameters */
  char xypath[IO_size], xyname[IO_size];
  char xy_pattern[IO_size], xyfull[IO_size];
  char xfile[IO_size], yfile[IO_size];
  int xstream, ystream;
  float *xdata, *ydata;
  float *pxdata, *pydata;
  int idata;
  long xynum;
  size_t xysize;
  int ErrorValue;

  IO_long Xy, Xypix, Pdef, Cdef, Pp;
  IO_line Xyname;

  IO_long  SymType;
  IO_float SymRot1, SymRot2, SymRot3;

  char key[IO_size];

  float I0Off_1, I0Ps_1;
  float I0Off_2, I0Ps_2;
  float I1Off_1, I1Ps_1;
  float I1Off_2, I1Ps_2;
  float xypix_1, xypix_2;
  float xycen_1, xycen_2;
  float xydis, xywvl;
  float xyrot_1, xyrot_2, xyrot_3;

  SetSaxsError( Version, RoutineName, NULL );

  *pstatus = Success;

  imax = pcb->ImgBlkLen;
  printf("\n Calculating ihb[0,%ld] = Function(",ihb[0].ImNum);
  for(i=1;i<imax;i++) {
    printf("ihb[%d,%ld] ", i, ihb[i].ImNum); }

  printf(")\n\n");

 /* Check the number of images */
  if (pcb->ImgBlkLen!=2) {
     printf("%d images found, 1 input and 1 output image required\n",
             pcb->ImgBlkLen); *pstatus=Failed; return; }

 /* --- Get option values */ 
  Xy = option_flag( SXy, pcb, pstatus);
  if (*pstatus!=Success) return;
  Xypix = option_flag( SXypix, pcb, pstatus);
  if (*pstatus!=Success) return;
  Xyname = option_line ( SXyname, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  Pdef = option_flag( SPdef, pcb, pstatus);
  if (*pstatus!=Success) return;
  Cdef = option_flag( SCdef, pcb, pstatus);
  if (*pstatus!=Success) return;
  Pp = option_flag( SPp, pcb, pstatus);
  if (*pstatus!=Success) return;

 /* 1 input, 1 output */
  I0Data  = ihb[0].Data;
  E0Data  = ihb[0].VarDat;
  I0Dummy = ihb[0].Dummy.V;
  I0DDummy = ihb[0].DDummy.V;
  I0Dim_1  = (int) ihb[0].Dim[1];
  I0Dim_2  = (int) ihb[0].Dim[2];
  I1Data  = ihb[1].Data;
  E1Data  = ihb[1].VarDat;
  I1Dummy = ihb[1].Dummy.V;
  I1DDummy = ihb[1].DDummy.V;
  I1Dim_1  = (int) ihb[1].Dim[1];
  I1Dim_2  = (int) ihb[1].Dim[2];

  /* check the reference system: Saxs required */
  if (pcb->RSys.V != IO_Saxs) {
    printf("ERROR: Reference system %s required\n",reftostr(IO_Saxs));
    printf("       Start program with option -rsys %s\n",reftostr(IO_Saxs)); 
    }

  /* open xy-files */
  xyname[0] = xypath[0] = '\0';
  xfile[0]  = yfile[0]  = '\0';
  xstream   = ystream   = -1;
  xdata     = ydata     = (float *) NULL;
  if (Xy.V) {
    filename_parameter (xy_pattern,IO_size,Xyname.V,0);
    filename_pattern (xy_pattern,IO_size,xy_pattern,pcb->ib[0].FileNumber);
    filename_full ( xyfull, IO_size, xy_pattern );
    filename_path ( xypath, IO_size, xyfull );
    filename_name ( xyname, IO_size, xyfull );
    xysize = edf_dim_product ( ihb[1].Dim ); // identical to input array
    /* open x-file */
    sprintf(xfile,"%s%s%s",xypath,"x",xyname);
    xstream = edf_search_stream( xfile,  "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return; 
    if (xstream>=0) { *pstatus = FileAlreadyOpen; return; }
    xstream = edf_open_data_file ( xfile, "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) return;
    if (xysize>0) {
      if (!(xdata = (float *) MALLOC(sizeof(float)*xysize))) {
        *pstatus = NotEnoughMemoryAvailable; return; }
      pxdata = xdata;
      for (idata=0;idata<xysize;idata++) *(pxdata++)=0.0;
    }
    /* open y-file */
    sprintf(yfile,"%s%s%s",xypath,"y",xyname);
    ystream = edf_search_stream( yfile,  "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) {  FREE(xdata); return; }
    if (ystream>=0) { FREE(xdata); *pstatus = FileAlreadyOpen; return; }
    ystream = edf_open_data_file ( yfile, "new", &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) {  FREE(xdata); return; }
    if (xysize>0) {
      if (!(ydata = (float *) MALLOC(sizeof(float)*xysize))) {
        FREE(xdata); *pstatus = NotEnoughMemoryAvailable; return; }
      pydata = ydata;
      for (idata=0;idata<xysize;idata++) *(pydata++)=0.0;
    }
  }

  K    = (double) WAVENUMBER(ihb[1].WaveLength.V);
  rot1 = (double) ihb[1].DetRot[1].V;
  rot2 = (double) ihb[1].DetRot[2].V;
  rot3 = (double) ihb[1].DetRot[3].V;
 
  /* set inverse and transform */
  if (ihb[0].Projection.I>=0) {
    ihb[0].Projection.V = IO_ProWaxs;
    ihb[0].Projection.I = TRUE;
  }
  I1Projection = ihb[1].Projection.V;
  I0Projection = ihb[0].Projection.V;

  if (Pp.V) {
    if ( (I0Projection==IO_ProWaxs)&&(I1Projection==IO_ProWaxs) ) {
      fprintf(ERROUT,
"ERROR: The image will not be transformed (%s projection -> %s projection)\n",
        protostr(I1Projection),protostr(I0Projection));
      fprintf(ERROUT,
"       If this is not an error, restart the programs with the option -%s.\n",SPp);
      fprintf(ERROUT,
"       or redefine the projection type of the input or output image\n");
      *pstatus=Failed; return;
    }
  }
  
  waxs_Init ( &I1wparams, ihb[1].Orientation.V, K, rot1, rot2, rot3 );

  waxs_Init ( &I0wparams, ihb[0].Orientation.V,
              (double) WAVENUMBER(ihb[0].WaveLength.V), 
              (double) ihb[0].DetRot[1].V,
              (double) ihb[0].DetRot[2].V,
              (double) ihb[0].DetRot[3].V );

  SymType  = option_long  (  SSymType, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  SymRot1 = option_float ( SSymRot1, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  SymRot2 = option_float ( SSymRot2, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;
  SymRot3 = option_float ( SSymRot3, pcb, ihb, num, 1, pstatus );
  if (*pstatus!=Success) return;

  /* symmetry operation are only used for waxs projection coordinates and
     ignored for saxs projections. Because input and output projections are 
     not fixed, both parameter sets must be initialized */
  waxs_SymInit ( &I1wparams, (int) SymType.V, ihb[1].Orientation.V,
                 (double) SymRot1.V, (double) SymRot2.V, (double) SymRot3.V );
  waxs_SymInit ( &I0wparams, (int) SymType.V, ihb[0].Orientation.V,
                 (double) SymRot1.V, (double) SymRot2.V, (double) SymRot3.V );

  /* Get output range */
  if (pcb->TestBit)
    printf("I1Projection=%s, I0Projection=%s\n",protostr(I1Projection),protostr(I0Projection));

  transform=waxs_Range( &I1wparams, &I0wparams,
                        I1Projection, I0Projection, ihb[1].Dim[1], ihb[1].Dim[2],
                        ihb[1].Offset[1].V, ihb[1].PixSiz[1].V, ihb[1].Center[1].V,
                        ihb[1].Offset[2].V, ihb[1].PixSiz[2].V, ihb[1].Center[2].V,
                        ihb[1].SampleDistance.V, ihb[1].WaveLength.V,
                        &Wmin, &Wmax, pstatus);
  if (*pstatus!=Success) return;

  if (pcb->TestBit) {
    printf("waxs_Range:Wmin.s_1=%g, Wmax.s_1=%g, Wmin.s_2=%g, Wmax.s_2=%g, transform=%d\n",
      Wmin.s_1,Wmax.s_1,Wmin.s_2,Wmax.s_2,transform);
  }

  /* Set default values of output image */
  if (Pdef.V) {
    set_default_pixsiz(0, ihb, Wmin, Wmax, pstatus);
    if (*pstatus!=Success) return;

    if (pcb->TestBit) {
      printf("set_default_pixsiz:pix_1=%g, pix_2=%g\n",
        ihb[0].PixSiz[1].V,ihb[0].PixSiz[2].V);
    }
  }

  if (Cdef.V) {
    set_default_center(0, ihb, Wmin, Wmax, pstatus);
    if (*pstatus!=Success) return;

    if (pcb->TestBit) {
      printf("set_default_center:cen_1=%g, cen_2=%g\n",
        ihb[0].Center[1].V,ihb[0].Center[2].V);
    }
  }

  GetReferenceParameters( pcb, ihb, 0, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  UpdateImageParameters ( pcb, ihb, 1, imax-1,
                          Off_1, Off_2, Ps_1,  Ps_2, pstatus );
  if (*pstatus!=Success) return;

  SetSaxsError( Version, RoutineName, NULL );

  if (pcb->TestBit) {
    printf("I0wparams\n");
    waxs_PrintParams ( stdout, I0wparams );
    printf("I1wparams\n");
    waxs_PrintParams ( stdout, I1wparams );
    printf("\n"); }

  /* loop over the output array  */
  for (i_2=0;i_2<I0Dim_2;i_2++) {
    W0.s_2 =  (double) WORLD( i_2, Off_2[0], Ps_2[0] );
    pI0Data = ABSPTR(I0Data,I0Dim_1,I0Dim_2,0,i_2);
    pE0Data = E0Data-I0Data+pI0Data;
    for (i_1=0;i_1<I0Dim_1;i_1++) {
      W0.s_1 =  (double) WORLD( i_1, Off_1[0], Ps_1[0] ); 

      W1=waxs_Transform(&I0wparams, &I1wparams, transform, W0);

      if (!W1.status) {

        f1_1 = INDEX(W1.s_1, Off_1[1], Ps_1[1]);
        f1_2 = INDEX(W1.s_2, Off_2[1], Ps_2[1]);

        if ( E0Data ) {
          if ( Isum2ldwE (I1Data,E1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                 f1_1-0.5, f1_2-0.5, f1_1+0.5, f1_2+0.5, 
                 &Sum, &Weight, &VarSum, &VarWeight) ) {
            /* then do something with the data */
            Value = Sum/Weight;
            UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
            if (VarSum>=0.0) {
              E0Value = VarSum/(VarWeight*VarWeight);
              UPDATE( *pE0Data, E0Value, I0Dummy, I0DDummy );
            }
          } /* if Isum ... */
        } else {
          if ( Ipol2ld (I1Data,I1Dim_1,I1Dim_2,I1Dummy,I1DDummy,
                 f1_1, f1_2, &Value) ) {
            /* then do something with the data */
            UPDATE( *pI0Data, Value, I0Dummy, I0DDummy );
          }
        } /* if E0Data */

      } /* if (!W1.status) */

      pI0Data++;
      pE0Data++;
    } /* for i_1 ... */
  } /* for i_2 ... */

  /* loop over the input array  */
  if (Xy.V) {

    /* The displacements are expressed in array coordinates */ 
    /* calculate I0 and I1 array coordinates */
    ARRAYREF(I0Off_1,I0Ps_1);
    ARRAYREF(I0Off_2,I0Ps_2);
    ARRAYREF(I1Off_1,I1Ps_1);
    ARRAYREF(I1Off_2,I1Ps_2);

    for (i_2=0;i_2<I1Dim_2;i_2++) {
      W1.s_2 =  (double) WORLD( i_2, Off_2[1], Ps_2[1] );
      pI1Data = ABSPTR(I1Data,I1Dim_1,I1Dim_2,0,i_2);
      for (i_1=0;i_1<I1Dim_1;i_1++) {
        W1.s_1 =  (double) WORLD( i_1, Off_1[1], Ps_1[1] );
  
        W0=waxs_Transform(&I1wparams, &I0wparams, -transform, W1);
  
        if (!W0.status) {
  
          /* calculate I0 indices */
          f0_1 = INDEX(W0.s_1, Off_1[0], Ps_1[0]);
          f0_2 = INDEX(W0.s_2, Off_2[0], Ps_2[0]);

          pxdata = xdata + (pI1Data - I1Data);
          *pxdata = WORLD(f0_1, I0Off_1, I0Ps_1) - WORLD(i_1, I1Off_1, I1Ps_1);
          pydata = ydata + (pI1Data - I1Data);
          *pydata = WORLD(f0_2, I0Off_2, I0Ps_2) - WORLD(i_2, I1Off_2, I1Ps_2);
  
        } /* if (!W0.status) */

        pI1Data++;
      } /* for i_1 ... */
    } /* for i_2 ... */
  } /* if (Xy.V) */

  /* write and close xy-files */
  if (Xy.V) {
    /* xfile */
    xynum = ihb[0].ImNum;
    printf("Writing : %s%s, Image :  %ld\n","x",xyname,xynum);
    edf_write_header_history ( xstream, xynum, 1l,
                               SaxsHistoryKey, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    /* The xy-displacement files cover the same region as the input image, 
       i.e. dimensions, offsets and binning sizes are the same. */
    edf_write_header_float(xstream,xynum,1l,KOffset_1,ihb[1].Offset[1].V,
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KOffset_2,ihb[1].Offset[2].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    if ( (ihb[1].BinSiz[1].I)||(ihb[1].BinSiz[2].I)||\
         (ihb[1].BinSiz[1].V!=1.0)||(ihb[1].BinSiz[2].V!=1.0) ) {
      edf_write_header_float(xstream,xynum,1l,KBinSiz_1,ihb[1].BinSiz[1].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
      edf_write_header_float(xstream,xynum,1l,KBinSiz_2,ihb[1].BinSiz[2].V,
                             &ErrorValue, pstatus );
      SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    }
    /* Normally, the geometrical parameters of the input image and of the
       xy-displacement output images should be identical. If Xypix is set
       PixSiz, Center and SampleDistance of the output image are written 
       to the xy-files. The output binning size and the geometrical
       parameters of the output image are always written to keys starting
       with the prefix KDisplaced followed by the corresponding keyword, e.g.  
       Psize_1 -> OutputPsize_1. */

    if (Xypix.V) {
      xypix_1 = ihb[0].PixSiz[1].V; xypix_2 = ihb[0].PixSiz[2].V;
      xycen_1 = ihb[0].Center[1].V; xycen_2 = ihb[0].Center[2].V;
      xydis   = ihb[0].SampleDistance.V;
      xywvl   = ihb[0].WaveLength.V;
      xyrot_1 = ihb[0].DetRot[1].V; xyrot_2 = ihb[0].DetRot[2].V; 
      xyrot_3 = ihb[0].DetRot[3].V;
    } else {
      xypix_1 = ihb[1].PixSiz[1].V; xypix_2 = ihb[1].PixSiz[2].V;
      xycen_1 = ihb[1].Center[1].V; xycen_2 = ihb[1].Center[2].V;
      xydis   = ihb[1].SampleDistance.V;
      xywvl   = ihb[1].WaveLength.V;
      xyrot_1 = ihb[1].DetRot[1].V; xyrot_2 = ihb[1].DetRot[2].V; 
      xyrot_3 = ihb[1].DetRot[3].V;
    }
    edf_write_header_float(xstream,xynum,1l,KPixSiz_1,xypix_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KPixSiz_2,xypix_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KCenter_1,xycen_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KCenter_2,xycen_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KSampleDistance,xydis,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,KWaveLength,xywvl,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_line(xstream,xynum,1l,
                          KProjection,protostr(I1Projection),
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_1,xyrot_1,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_2,xyrot_2,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_unit(xstream,xynum,1l,KDetRot_3,xyrot_3,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

    /* Write parameters of transformed image to x-displacement file */
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_size,KDisplaced,KOffset_1),ihb[0].Offset[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                strlib_concat(key,IO_size,KDisplaced,KOffset_2),ihb[0].Offset[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KBinSiz_1),ihb[0].BinSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KBinSiz_2),ihb[0].BinSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KPixSiz_1),ihb[0].PixSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KPixSiz_2),ihb[0].PixSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KCenter_1),ihb[0].Center[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KCenter_2),ihb[0].Center[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
      strlib_concat(key,IO_size,KDisplaced,KSampleDistance),ihb[0].SampleDistance.V,
      &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_float(xstream,xynum,1l,
      strlib_concat(key,IO_size,KDisplaced,KWaveLength),ihb[0].WaveLength.V,
      &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    edf_write_header_line(xstream,xynum,1l,
           strlib_concat(key,IO_size,KDisplaced,KProjection),protostr(I0Projection),
           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }

    edf_write_data( xstream,xynum,1l,ihb[1].Dim,xdata,MFloat, 
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(xdata); FREE(ydata); return; }
    FREE(xdata);
    edf_close_data_file ( xstream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    /* yfile */
    printf("Writing : %s%s, Image :  %ld\n","y",xyname,xynum);
    edf_write_header_history ( ystream, xynum, 1l,
                               SaxsHistoryKey, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KOffset_1,ihb[1].Offset[1].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KOffset_2,ihb[1].Offset[2].V,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    if ( (ihb[1].BinSiz[1].I)||(ihb[1].BinSiz[2].I)||\
         (ihb[1].BinSiz[1].V!=1.0)||(ihb[1].BinSiz[2].V!=1.0) ) {
      edf_write_header_float(ystream,xynum,1l,KBinSiz_1,ihb[1].BinSiz[1].V,
                             &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
      edf_write_header_float(ystream,xynum,1l,KBinSiz_2,ihb[1].BinSiz[2].V,
                             &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
      if (*pstatus!=Success) { FREE(ydata); return; }
    }
    edf_write_header_float(ystream,xynum,1l,KPixSiz_1,xypix_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KPixSiz_2,xypix_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KCenter_1,xycen_1,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KCenter_2,xycen_2,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KSampleDistance,xydis,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,KWaveLength,xywvl,
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_line(ystream,xynum,1l,
                           KProjection,protostr(I1Projection),
                           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_1,xyrot_1,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_2,xyrot_2,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_unit(ystream,xynum,1l,KDetRot_3,xyrot_3,"deg",
                          &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

    /* Write parameters of transformed image to y-displacement file */
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_size,KDisplaced,KOffset_1),ihb[0].Offset[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_size,KDisplaced,KOffset_2),ihb[0].Offset[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_size,KDisplaced,KBinSiz_1),ihb[0].BinSiz[1].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                strlib_concat(key,IO_size,KDisplaced,KBinSiz_2),ihb[0].BinSiz[2].V,
                &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KPixSiz_1),ihb[0].PixSiz[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KPixSiz_2),ihb[0].PixSiz[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KCenter_1),ihb[0].Center[1].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
                 strlib_concat(key,IO_size,KDisplaced,KCenter_2),ihb[0].Center[2].V,
                 &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
      strlib_concat(key,IO_size,KDisplaced,KSampleDistance),ihb[0].SampleDistance.V,
      &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_float(ystream,xynum,1l,
      strlib_concat(key,IO_size,KDisplaced,KWaveLength),ihb[0].WaveLength.V,
      &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_write_header_line(ystream,xynum,1l,
           strlib_concat(key,IO_size,KDisplaced,KProjection),protostr(I0Projection),
           &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }

    edf_write_data( ystream, xynum, 1l, ihb[1].Dim, ydata, MFloat, 
                    &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    edf_close_data_file ( ystream, &ErrorValue, pstatus );
    SetSaxsErrorExternal( ErrorValue, ReportSaxsImageError );
    if (*pstatus!=Success) { FREE(ydata); return; }
    FREE(ydata);
  }

} /* saxs_waxs*/

/*---------------------------------------------------------------------------
user_io
Do all the keyboard io and return cb, and ib
---------------------------------------------------------------------------*/

void user_io(CmdBlk * pcb, ImgBlk * ib, int * pstatus)
{
  char  progfn[IO_size];
  ImgHeadBlk ihb[BlockNum];

  float ROff_1, RPs_1, UOff_1, UPs_1;
  float ROff_2, RPs_2, UOff_2, UPs_2;

  int errval;
  IO_float *pRot1, *pRot2, *pRot3; // in degree for backwards compatibility
  IO_long Rot;
  static const double deg2rad   = SAXS_PI/180.0; 

  /*--- Determine program name without directory ---*/
  filename_name ( progfn, IO_size,  pcb->argv[0] );

  /*--- Show usage if no arguments are given ---*/
  if (pcb->argc<=1) printf("Usage: %s %s\n",progfn,Usage);

  /*--- Write name of program ---*/
  printf("\n");
  printf("   %s %s\n",progfn,Version);
  printf("\n");

  ArgvFilenames ( pcb, ib, ihb, 0, BlockNum-1, pstatus);
  if (*pstatus!=Success) return;
  GetReference(pcb->RSys.V,1,ihb,&ROff_1,&ROff_2,&RPs_1,&RPs_2,pstatus );
  if (*pstatus!=Success) return;
  GetReference(pcb->USys.V,1,ihb,&UOff_1,&UOff_2,&UPs_1,&UPs_2,pstatus );
  if (*pstatus!=Success) return;

  /*---       input rotation angles of detector */
  if (pcb->Prompt.V) {
    printf("Detector Rotations\n");
    printf("The angles %s, %s and %s\n", 
      KDetRot_1, KDetRot_2, KDetRot_3);
    printf("describe the rotations of the detector for a fixed %s around\n", KSampleDistance);
    printf("the sample. The rotations are defined in the lab coordinate system for\n");
    printf("%s 1. Images with %s different from 1 are \n",
      KOrientation, KOrientation);
    printf("automatically transformed. The rotations are applied sequentially. The \n");
    printf("first rotation is around axis 1 of the lab system, the second rotation \n");
    printf("is around axis 2 of the lab system and the third rotation is around \n");
    printf("axis 3 of the lab system.\n");
    printf("The coordinate system with orientation 1 is defined in the following way:\n");
    printf("  origin (0,0) at lower left corner of the image\n");
    printf("  axis 1: horizontally to the right\n");
    printf("  axis 2: vertically up\n");
    printf("  axis 3: against beam direction\n");
    printf("\n");
    printf("             ^ 2 (vertical)\n");
    printf("             |\n");
    printf("             |\n");
    printf("             |\n");
    printf("             |\n");
    printf("             +------> 1 (horizontal)\n");
    printf("             \\\n"); 
    printf("              \\\n");
    printf("               \\\n");
    printf("                _| 3 (against primary beam)\n");
    printf("\n");
    printf("ID01:  -x <=> axis 1,  y <=> axis 2, -z <=> axis 3\n");            
    printf("ID02:   x <=> axis 1, -y <=> axis 3,  z <=> axis 2\n");
    printf("\n");                                                       
  }

  /*--- Argument  : rotations */
  Rot = option_flag( SRot, pcb, pstatus);
  if (*pstatus!=Success) return;
  pRot1 = (IO_float*) option_parameter_search( SRot1, pcb, pstatus );
  if (*pstatus!=Success) return;
  pRot2 = (IO_float*) option_parameter_search( SRot2, pcb, pstatus );
  if (*pstatus!=Success) return;
  pRot3 = (IO_float*) option_parameter_search( SRot3, pcb, pstatus );
  if (*pstatus!=Success) return;

  if ((pRot1->I)||(pRot2->I)||(pRot3->I)) {
    printf("\
ATTENTION: You are using the option -rot1 <rot1[deg]>, -rot2 <rot2[deg]> or\n\
           -rot3 <rot3[deg]>. They are replaced by the option \n\
           -i1rot <rot1[rad]> <rot2[rad]> <rot2[rad]>. rot1, rot2 and rot3 \n\
           are for a short time available for backward compatibility. Please \n\
           use the option i1rot which offers the possibility to use units. \n\
           The values <rot1>, <rot2> and <rot3> must be given in radian or \n\
           with a unit, e.g. -i1rot 0_deg 32.5_deg 0_deg.\n");
    if (!Rot.V) {
      printf("\
           The options -rot1, -rot2 and -rot3 are obsolete.\n\
           Use the option -i1rot <rot1> <rot2> <rot3> with angles in radian\n");
      *pstatus=Failed; return;
    }
    printf("\n");
  }

  /*--- Argument  : rotation around axis 1 */
  if (pRot1->I) { 
    ib[1].DetRot[1].I = pRot1->I;
    num_double2str( ib[1].DetRot[1].V, IO_size, pRot1->V*deg2rad, 
      "deg", EdfNDigitsFloat, &errval);
    if (errval) { *pstatus=Failed; return; }
  }
  argv_flexp(pcb," first rotation: ccw around axis 1 (rad or \'_\' and unit)",
             &ib[1].DetRot[1],(ib[1].DetRot[1].I)?ib[1].DetRot[1].V:"",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : rotation around axis 2 */
  if (pRot2->I) {
    ib[1].DetRot[2].I = pRot2->I;
    num_double2str( ib[1].DetRot[2].V, IO_size, pRot2->V*deg2rad, 
      "deg", EdfNDigitsFloat, &errval);
    if (errval) { *pstatus=Failed; return; }
  }
  argv_flexp(pcb,"second rotation: ccw around axis 2 (rad or \'_\' and unit)",
             &ib[1].DetRot[2],(ib[1].DetRot[2].I)?ib[1].DetRot[2].V:"",pstatus);
  if (*pstatus!=Success) return;

  /*--- Argument  : rotation around axis 3 */
  if (pRot3->I) {
    ib[1].DetRot[3].I = pRot3->I;
    num_double2str( ib[1].DetRot[3].V, IO_size, pRot3->V*deg2rad, 
      "deg", EdfNDigitsFloat, &errval);
    if (errval) { *pstatus=Failed; return; }
  }
  argv_flexp(pcb," third rotation: ccw around axis 3 (rad or \'_\' and unit)",
             &ib[1].DetRot[3],(ib[1].DetRot[3].I)?ib[1].DetRot[3].V:"",pstatus);
  if (*pstatus!=Success) return;

  printf("\n");
  if (ib[1].Name.I)       printf("i/p file           : %s\n",ib[1].Name.V);
  if (ib[0].Name.I)       printf("o/p file           : %s\n",ib[0].Name.V);
  if (ib[1].First.I)      printf("first image        : %ld\n",ib[1].First.V);
  if (ib[1].Last.I)       printf("last image         : %ld\n",ib[1].Last.V);
  if (ib[1].Inc.I)        printf("increment          : %ld\n",ib[1].Inc.V);
  if (ib[0].Dummy.I)      printf("output dummy       : %s\n",ib[0].Dummy.V);
  if (ib[0].Dim[1].I)     printf("output dimension 1 : %s\n",ib[0].Dim[1].V);
  if (ib[0].Dim[2].I)     printf("output dimension 2 : %s\n",ib[0].Dim[2].V);
  if (ib[0].DetRot[1].I)  printf("detctor rotation 1 : %s\n",ib[0].DetRot[1].V);
  if (ib[0].DetRot[2].I)  printf("detctor rotation 2 : %s\n",ib[0].DetRot[2].V);
  if (ib[0].DetRot[3].I)  printf("detctor rotation 3 : %s\n",ib[0].DetRot[3].V);

  printf("\n");

  if (pcb->TestBit) {
    PrintBlocks ( pcb, ib );
    printf("\n"); }

  return;
}

/*---------------------------------------------------------------------------
main
---------------------------------------------------------------------------*/

#if MAKE_FUNCTION
# define MAIN main_saxs_waxs
#else
# define MAIN main
#endif

int MAIN (int argc, char *argv[])
{
  CmdBlk cb;                /* command block  */
  ImgBlk ib[BlockNum];      /* image blocks */

  int status;
  int arg_no = 0;

  IO_long Xy;
  IO_line Xyname;

 /* Init options, control block and image blocks */
  InitOptions( Usage, Version, NULL, TRUE, &cb, ib, BlockNum );

 /* Default reference system is SAXS */
  cb.RSys.V = IO_Saxs; /* default */

 /* Define special options */
  option_define ( IO_tpflag,  SRot,    "FALSE",     &cb ); // activate rot1, rot2, rot3
  option_define ( IO_tpfloat, SRot1,   "0.0",      &cb ); // obsolete option
  option_define ( IO_tpfloat, SRot2,   "0.0",      &cb ); // obsolete option
  option_define ( IO_tpfloat, SRot3,   "0.0",      &cb ); // obsolete option

  option_define ( IO_tplvexp, SSymType,     "0",      &cb );
  option_define ( IO_tpflexp, SSymRot1,   "0.0",      &cb );
  option_define ( IO_tpflexp, SSymRot2,   "0.0",      &cb );
  option_define ( IO_tpflexp, SSymRot3,   "0.0",      &cb );

  option_define ( IO_tpline,  SXyname, "file.edf", &cb );
  option_define ( IO_tpflag,  SXy,     "FALSE",    &cb );
  /* Write pixel size of output image to xy-files (pixel size is used by spd) */
  option_define ( IO_tpflag,  SXypix,  "TRUE",     &cb );
  /* Set output pixel size to default */ 
  option_define ( IO_tpflag,  SPdef,   "FALSE",    &cb );
  /* Set output center to default */ 
  option_define ( IO_tpflag,  SCdef,   "TRUE",    &cb );
  /* Set prompt prang to default */ 
  option_define ( IO_tpflag,  SPp,     "TRUE",    &cb );

 /* Read options from argument list */
  ReadOptions( argv, &arg_no, &cb, ib, &status);

 /* Set xy-flag if xy-file name is given */
  if (status==Success)
    Xy = option_flag(SXy,&cb,&status );
  if (status==Success)
    if (!Xy.I) Xyname = option_line(SXyname,&cb,NULL,NULL,0,&status);
  if (status==Success)
    if (!Xy.I) option_flag_update(Xyname.I?TRUE:FALSE,Xy.I,SXy,&cb,&status);

  /* USER KEYBOARD I/O */
  if (status==Success) {
    argv_start ( &cb, 1 );
    user_io( &cb, ib, &status);
    argv_end( &cb ); /* must be called after user_io */
  }

  /* SEQUENCE CALCULATION */
  if (status==Success) 
    IMAGELOOP( &cb, ib, saxs_waxs, NULL, NULL, TRUE, &status );

  return( ReportSaxsStatus( status, 0 ) );

} /* MAIN */

